/*
  oql.c - Object Query Language.

  Copyright (C) 2001 Free Software Foundation

  This file is part of the GNU Enterprise Application Server (GEAS)

  GEAS is free software; you can redistribute it and/or modify it under
  the terms of the GNU General Public License as published by the Free
  Software Foundation; either version 2, or (at your option) any later
  version.

  GEAS is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
  for more details.

  You should have received a copy of the GNU General Public License
  along with GEAS; if not, write to the Free Software Foundation, Inc.,
  59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

  $Id: oql.c,v 1.65 2001/07/02 08:20:14 treshna Exp $
*/

#include "config.h"
#include "oql.h"
#include <string.h>
#include <ctype.h>
#include <glib.h>
#include "classdata.h"
#include "geas-server.h"

static _QueryData *create_query_data (const char *orderby);
static _QueryCondition *create_query_condition (const char *targetclass,
                                                const char *field,
                                                const char *relation,
                                                const char *value);
static void free_query_condition (_QueryCondition * condition);

int oql_sort_string_name (gpointer a, gpointer b);

static void oql_add_conditions_to_sql (_QueryData * q, DBType db,
                                       GString * buf);
static void oql_add_complex_conditions_to_sql (_QueryData * q, DBType db,
                                               GString * buf);
static void oql_add_complex_constraint_to_sql (_QueryData * q, odl_class * cl,
                                               DBType db, oql_constraint * c,
                                               GString * buf);
static GString *oql_make_select (_QueryData * q, DBType db);
static GString *oql_make_delete (_QueryData * q, DBType db);
static GString *oql_make_write (_QueryData * q, DBType db);

static char *oql_quote_value (const char *value, DBType db);

static guint g_strcase_hash (gconstpointer key);
static gboolean g_strcase_equal (gconstpointer v1, gconstpointer v2);

/* ------------------------------------------------------------------------- *\
 * Generate SQL query form OQL query.
\* ------------------------------------------------------------------------- */
const char *
oql_query_as_sql (_QueryData * q, DBType database)
{
  char *retval = NULL;
  char *type = NULL, *dbtype = NULL;
  GString *buf = NULL;

  switch (q->type)
    {
    case OQL_SELECT:
      type = "SELECT";
      break;
    case OQL_WRITE:
      type = "WRITE";
      break;
    case OQL_DELETE:
      type = "DELETE";
      break;
    default:
      type = "<unknown>";
      break;
    }
  switch (database)
    {
    case OQL_DBTYPE_MYSQL:
      dbtype = "MySQL";
      break;
    case OQL_DBTYPE_POSTGRESQL:
      dbtype = "PostgreSQL";
      break;
    default:
      dbtype = "<unknown>";
    }

  if (q->sql)
    {
      return (q->sql);
    }

  switch (q->type)
    {
    case OQL_SELECT:
      buf = oql_make_select (q, database);
      break;
    case OQL_DELETE:
      buf = oql_make_delete (q, database);
      break;
    case OQL_WRITE:
      buf = oql_make_write (q, database);
      break;
    }
  if (buf)
    {
      if (database == OQL_DBTYPE_CACHEONLY)
        {
          retval = buf->str;
          buf->str = NULL;
          g_string_free (buf, TRUE);
        }
      else
        {
          q->sql = buf->str;
          retval = q->sql;
          buf->str = NULL;
          g_string_free (buf, TRUE);
        }
    }

  if (!retval)
    {
      fatal_error ("error : no SQL was generated\n"
                   "(serious error somewhere in the OQL module - hi, jamest, is this more helpful :)\n");
    }
  return (retval);
}

/* ------------------------------------------------------------------------- *\
 * Add conditions to the SQL query.
\* ------------------------------------------------------------------------- */
static void
oql_add_conditions_to_sql (_QueryData * q, DBType db, GString * buf)
{
  GList *l = NULL;
  gboolean need_and = FALSE;
  char *quoted = NULL;
  char *temp;

  g_string_append (buf, " WHERE ");

  /* link classes together by objectid field */
  if (g_list_length (q->classes) > 1)
    {
      l = q->classes->next;
      g_string_append (buf, "(");
      while (l)
        {
          quoted = oql_quote_column (q->classes->data, "objectid", db);
          g_string_append (buf, quoted);
          g_free (quoted);

          g_string_append (buf, "=");

          quoted = oql_quote_column (l->data, "objectid", db);
          g_string_append (buf, quoted);
          g_free (quoted);

          if (l->next)
            g_string_append (buf, " AND ");

          l = g_list_next (l);
        }
      g_string_append (buf, ")");
      need_and = TRUE;
    }

  if (q->conditions)
    {
      if (need_and)
        g_string_append (buf, " AND ");
      need_and = TRUE;

      l = q->conditions;
      while (l)
        {
          _QueryCondition *c = l->data;

          g_string_append (buf, "(");

          switch (db)
            {
            case OQL_DBTYPE_MYSQL:
              if (!c->casesensitive)
                g_string_append (buf, "upper(");
              break;
            case OQL_DBTYPE_POSTGRESQL:
              if (!c->casesensitive)
                g_string_append (buf, "upper(rtrim(");
              break;
            default:
            }

          quoted = oql_quote_column (c->targetclass, c->field, db);
          g_string_append (buf, quoted);
          g_free (quoted);
          switch (db)
            {
            case OQL_DBTYPE_MYSQL:
              if (!c->casesensitive)
                g_string_append (buf, ")");
              break;
            case OQL_DBTYPE_POSTGRESQL:
              if (!c->casesensitive)
                g_string_append (buf, ",' '))");
              break;
            default:
            }

          switch (c->test)
            {
            case q_hackish:
              g_string_append (buf, c->relation);
              quoted = oql_quote_value (c->value, db);
              break;
            case q_equals:
              g_string_append (buf, "=");
              quoted = oql_quote_value (c->value, db);
              break;
            case q_lessthan:
              g_string_append (buf, "<");
              quoted = oql_quote_value (c->value, db);
              break;
            case q_greaterthan:
              g_string_append (buf, ">");
              quoted = oql_quote_value (c->value, db);
              break;
            case q_contains:
              g_string_append (buf, " LIKE ");
              temp = g_strdup_printf ("%%%s%%", c->value);
              quoted = oql_quote_value (temp, db);
              g_free (temp);
              break;
            case q_startswith:
              g_string_append (buf, " LIKE ");
              temp = g_strdup_printf ("%s%%", c->value);
              quoted = oql_quote_value (temp, db);
              g_free (temp);
              break;
            case q_endswith:
              g_string_append (buf, " LIKE ");
              temp = g_strdup_printf ("%s%%", c->value);
              quoted = oql_quote_value (temp, db);
              g_free (temp);
              break;
            case q_notequals:
              /* TODO */
              break;
            case q_notlessthan:
              /* TODO */
              break;
            case q_notgreaterthan:
              /* TODO */
              break;
            case q_notcontains:
              /* TODO */
              break;
            case q_notstartswith:
              /* TODO */
              break;
            case q_notendswith:
              /* TODO */
              break;
            }

          switch (db)
            {
            case OQL_DBTYPE_MYSQL:
              if (!c->casesensitive)
                g_string_append (buf, "upper(");
              break;
            case OQL_DBTYPE_POSTGRESQL:
              if (!c->casesensitive)
                g_string_append (buf, "upper(");
              break;
            default:
            }
          g_string_append (buf, quoted);
          g_free (quoted);

          switch (db)
            {
            case OQL_DBTYPE_MYSQL:
              if (!c->casesensitive)
                g_string_append (buf, ")");
              break;
            case OQL_DBTYPE_POSTGRESQL:
              if (!c->casesensitive)
                g_string_append (buf, ")");
              break;
            default:
            }
          g_string_append (buf, ")");

          if (l->next)
            g_string_append (buf, " AND ");
          l = g_list_next (l);
        }
    }
  if (q->complex)
    {
      if (need_and)
        g_string_append (buf, " AND ");
      oql_add_complex_conditions_to_sql (q, db, buf);
    }
}

/* ------------------------------------------------------------------------- *\
 * Transform any illegal column names ("table.column") to a legal form.
\* ------------------------------------------------------------------------- */
char *
oql_quote_column (const char *table, const char *column, DBType db)
{
  char *mangled = NULL;
  char *retval = NULL;
  char *quotedttable = NULL;
  char *quotedcolumn = NULL;
  char *p;
  int dotcount = 0;

  p = (char *) column;
  while (p && *p != '\0')
    {
      if (*p == '.')
        dotcount++;
      p++;
    }

  if (table)
    {
      mangled = odl_mangle_qualified_name (table);

      switch (db)
        {
        case OQL_DBTYPE_CACHEONLY:
        case OQL_DBTYPE_MYSQL:
          if (is_word_reserved (mangled, db))
            /* quotedttable = g_strdup_printf ("__%s", mangled, db); */
            quotedttable = g_strdup_printf ("__%s", mangled);
          else
            quotedttable = g_strdup (mangled);
          break;

        case OQL_DBTYPE_POSTGRESQL:
          quotedttable = g_strdup (mangled);
          if (is_word_reserved (mangled, db))
            /* quotedttable = g_strdup_printf ("\"%s\"", mangled, db); */
	    quotedttable = g_strdup_printf ("\"%s\"", mangled);
          else
            quotedttable = g_strdup (mangled);
          break;
        }
    }

  if (column)
    {
      switch (db)
        {
        case OQL_DBTYPE_CACHEONLY:
        case OQL_DBTYPE_MYSQL:
          /* TODO: if column == illegal column name, change it */
          if (is_word_reserved (column, db))
            /* quotedcolumn = g_strdup_printf ("__%s", column, db); */
            quotedcolumn = g_strdup_printf ("__%s", column);
          else
            quotedcolumn = g_strdup (column);
          break;
        case OQL_DBTYPE_POSTGRESQL:
          /* TODO: if column == illegal column name, change it */
          if (is_word_reserved (column, db))
            /* quotedcolumn = g_strdup_printf ("\"%s\"", column, db); */
	    quotedcolumn = g_strdup_printf ("\"%s\"", column);
          else
            quotedcolumn = g_strdup (column);
          break;
        }
    }

  /* replace a dot in a column name with an underscore */
  /* cost.amount -> cost_amount */
  if (dotcount > 0)
    {
      char *newf = g_malloc (strlen (quotedcolumn) + 1 + dotcount);

      memcpy (newf, quotedcolumn, strlen (quotedcolumn) + 1);
      g_free (quotedcolumn);
      quotedcolumn = newf;

      for (p = strchr (quotedcolumn, '.'); p != NULL;
           p = strchr (quotedcolumn, '.'))
        {
          /* make space for __ */
          memmove ((void *) (p + 2), (const void *) (p + 1), strlen (p));

          /* replace . with __ */
          *p = '_';
          p++;
          *p = '_';
        }
    }

  if (table && column)
    retval = g_strdup_printf ("%s.%s", quotedttable, quotedcolumn);
  else if (table)
    retval = g_strdup (quotedttable);
  else
    retval = g_strdup (quotedcolumn);

  /* convert entire string to lower case */
  p = retval;
  while (*p != '\0')
    {
      *p = tolower (*p);
      p++;
    }

  if (mangled)
    g_free (mangled);
  if (quotedttable)
    g_free (quotedttable);
  if (quotedcolumn)
    g_free (quotedcolumn);

  return (retval);
}

/* ------------------------------------------------------------------------- *\
 * Quote a value in an SQL statement.
\* ------------------------------------------------------------------------- */
static char *
oql_quote_value (const char *value, DBType db)
{
  int count, i;
  int len;
  char *str, *ptr, *out;

  len = strlen (value);
  switch (db)
    {
    case OQL_DBTYPE_CACHEONLY:
    case OQL_DBTYPE_MYSQL:
      /* count characters to quote */
      count = 0;
      for (i = 0; i < len; i++)
        if (value[i] == '\'' || value[i] == '\\')
          count++;
      if (count == 0)
        return (g_strdup_printf ("'%s'", value));

      /* count: extra characters len: original chars 1: space for \0 2: space for quotes (') */
      out = (char *) g_malloc (sizeof (char) * (count + len + 1 + 2));
      ptr = out;
      str = (char *) value;
      *ptr = '\'';
      ptr++;
      while (*str != '\0')
        {
          if (*str == '\'')
            {
              *ptr = '\\';
              ptr++;
            }
          if (*str == '\\')
            {
              *ptr = '\\';
              ptr++;
            }
          *ptr = *str;
          ptr++;
          str++;
        }
      *ptr++ = '\'';
      *ptr = '\0';
      return (out);

      break;
    case OQL_DBTYPE_POSTGRESQL:
      /* really a TODO */
      count = 0;
      for (i = 0; i < len; i++)
        if (value[i] == '\'' || value[i] == '\\')
          count++;
      if (count == 0)
        return (g_strdup_printf ("'%s'", value));

      /* count: extra characters len: original chars 1: space for \0 2: space for quotes (') */
      out = (char *) g_malloc (sizeof (char) * (count + len + 1 + 2));
      ptr = out;
      str = (char *) value;
      *ptr = '\'';
      ptr++;
      while (*str != '\0')
        {
          if (*str == '\'')
            {
              *ptr = '\\';
              ptr++;
            }
          if (*str == '\\')
            {
              *ptr = '\\';
              ptr++;
            }
          *ptr = *str;
          ptr++;
          str++;
        }
      *ptr++ = '\'';
      *ptr = '\0';

      return (out);
      break;
    default:
      fatal_error ("Unsupported databsae type : %d", db);
    }
  return (NULL);
}

/* ------------------------------------------------------------------------- *\
 * Build a "SELECT" SQL statement.
\* ------------------------------------------------------------------------- */
static GString *
oql_make_select (_QueryData * q, DBType db)
{
  GString *buf;
  GList *tmp;
  char *loadclass = NULL;
  char *quoted;
  odl_class *cl;

  if (q->classes)
    loadclass = q->classes->data;
  else
    return (NULL);

  buf = g_string_new ("SELECT ");
  if (!buf)
    return (NULL);

  /* add fields to read */
  tmp = q->fields;
  while (tmp)
    {
      quoted = oql_quote_column (loadclass, tmp->data, db);
      g_string_append (buf, quoted);
      g_free (quoted);

      if (tmp->next)
        g_string_append (buf, ",");

      tmp = g_list_next (tmp);
    }

  g_string_append (buf, " FROM ");

  /* add table names */
  tmp = q->classes;
  while (tmp)
    {
      quoted = oql_quote_column (tmp->data, NULL, db);
      g_string_append (buf, quoted);
      g_free (quoted);

      if (tmp->next)
        g_string_append (buf, ",");

      tmp = g_list_next (tmp);
    }

  /* add restrictions to the query */
  if (g_list_length (q->classes) > 1 || q->conditions || q->complex)
    oql_add_conditions_to_sql (q, db, buf);

  /* add order by clause */
  if (q->orderby)
    {
      char *quoted;

      g_string_append (buf, " ORDER BY ");
      quoted = oql_quote_column (loadclass, q->orderby, db);
      g_string_append (buf, quoted);
      g_free (quoted);
    
      /* if reverse : if descending, then sort in ascending order
       * else       : if descending, then sort in descending order
       */
      cl = odl_find_class( all_classes , loadclass , NULL );
      if ( (q->reversesearch) || (!q->reversesearch) ) /*  commented out code that didn't work, which was cl->desc */
        g_string_append (buf, " DESC");
    }

  if (q->uselimit)
    {
      quoted = g_strdup_printf (" LIMIT %lu", (long)q->maxresults);
      g_string_append (buf, quoted);
      g_free (quoted);
    }

  /* done */
  return (buf);
}

/* ------------------------------------------------------------------------- *\
 * Build a "DELETE" SQL statement.
\* ------------------------------------------------------------------------- */
static GString *
oql_make_delete (_QueryData * q, DBType db)
{
  GString *buf;
  /* GList *tmp; */
  char *loadclass = NULL;
  char *quoted, *classname, *key;

  if (q->classes)
    loadclass = q->classes->data;
  else
    return (NULL);

  buf = g_string_new ("DELETE FROM ");
  if (!buf)
    return (NULL);

  classname = (char *) g_list_nth_data (q->classes, 0);
  key = (char *) g_list_nth_data (q->values, 0);

  /* class to delete */
  quoted = oql_quote_column (classname, NULL, db);
  g_string_append (buf, quoted);
  g_free (quoted);

  /* add restrictions to the query */
  if (g_list_length (q->classes) > 1 || q->conditions)
    oql_add_conditions_to_sql (q, db, buf);

  /* done */
  return (buf);
}

/* ------------------------------------------------------------------------- *\
 * Build a "WRITE" SQL statement.
\* ------------------------------------------------------------------------- */
static GString *
oql_make_write (_QueryData * q, DBType db)
{
  char *writeclass = NULL, *quoted = NULL;
  GList *tmp = NULL, *tmp2 = NULL;
  GString *buf = NULL;
  odl_class *c;
  char *translated = NULL;

  if (q->classes)
    writeclass = q->classes->data;
  else
    return (NULL);

  /* get class definition */
  c = odl_find_class (all_classes, writeclass, NULL);

  if (db == OQL_DBTYPE_MYSQL)
    {
      /* make REPLACE query for MySQL - inserts new data or changes old data */
      buf = g_string_new ("REPLACE INTO ");
      if (!buf)
        return (NULL);

      /* class to write to */
      quoted = oql_quote_column (writeclass, NULL, db);
      g_string_append (buf, quoted);
      g_free (quoted);

      /* fields */
      g_string_append (buf, "(");
      tmp = q->fields;
      while (tmp)
        {
          quoted = oql_quote_column (NULL, tmp->data, db);
          g_string_append (buf, quoted);
          g_free (quoted);

          if (tmp->next)
            g_string_append (buf, ",");

          tmp = g_list_next (tmp);
        }
      g_string_append (buf, ")");

      /* values */
      g_string_append (buf, "VALUES (");
      tmp = q->values;
      tmp2 = q->fields;
      while (tmp)
        {
          translated = oql_translate_for_write (tmp->data, c, tmp2->data, db);
          if (translated)
            {
              quoted = oql_quote_value (translated, db);
              g_free (translated);
            }
          else
            quoted = g_strdup ("");
          g_string_append (buf, quoted);
          g_free (quoted);

          if (tmp->next)
            g_string_append (buf, ",");

          tmp = g_list_next (tmp);
          tmp2 = g_list_next (tmp2);
        }
      g_string_append (buf, ")");
    }
  else if (db == OQL_DBTYPE_POSTGRESQL && !q->indatabase)
    {
      /* make an INSERT query for PostgreSQL - inserts new data */
      buf = g_string_new ("INSERT INTO ");
      if (!buf)
        return (NULL);

      /* class to write to */
      quoted = oql_quote_column (writeclass, NULL, db);
      g_string_append (buf, quoted);
      g_free (quoted);

      /* fields */
      g_string_append (buf, "(");
      tmp = q->fields;
      while (tmp)
        {
          quoted = oql_quote_column (NULL, tmp->data, db);
          g_string_append (buf, quoted);
          g_free (quoted);

          if (tmp->next)
            g_string_append (buf, ",");

          tmp = g_list_next (tmp);
        }
      g_string_append (buf, ")");

      /* values */
      g_string_append (buf, "VALUES (");
      tmp = q->values;
      tmp2 = q->fields;
      while (tmp)
        {
          translated = oql_translate_for_write (tmp->data, c, tmp2->data, db);
          if (translated)
            {
              quoted = oql_quote_value (translated, db);
              g_free (translated);
            }
          else
            quoted = g_strdup ("");
          g_string_append (buf, quoted);
          g_free (quoted);

          if (tmp->next)
            g_string_append (buf, ",");

          tmp = g_list_next (tmp);
          tmp2 = g_list_next (tmp2);
        }
      g_string_append (buf, ")");
    }
  else if (db == OQL_DBTYPE_POSTGRESQL && q->indatabase)
    {
      GList *f, *v;

      /* make an UPDATE query for PostgreSQL - changes old data */
      buf = g_string_new ("UPDATE ");
      if (!buf)
        return (NULL);

      quoted = oql_quote_column (writeclass, NULL, db);
      g_string_append (buf, quoted);
      g_free (quoted);

      f = q->fields;
      v = q->values;

      g_string_append (buf, " SET ");
      while (f && v)
        {
          quoted = oql_quote_column (NULL, f->data, db);
          g_string_append (buf, quoted);
          g_free (quoted);

          g_string_append (buf, "=");

          /* translate from GEAS format to database format, as appropriate */
          translated = oql_translate_for_write (v->data, c, f->data, db);
          if (translated)
            {
              quoted = oql_quote_value (translated, db);
              g_free (translated);
            }
          else
            quoted = g_strdup ("");

          g_string_append (buf, quoted);
          g_free (quoted);

          if (f->next)
            g_string_append (buf, ",");

          f = g_list_next (f);
          v = g_list_next (v);
        }

      /* add restrictions to the query */
      if (g_list_length (q->classes) > 1 || q->conditions)
        oql_add_conditions_to_sql (q, db, buf);
    }

  /* done */
  return (buf);
}

/* ------------------------------------------------------------------------- *\
 * Get the classname in the OQL query.
\* ------------------------------------------------------------------------- */
const char *
oql_query_get_classname (_QueryData * query)
{
  return (char *) g_list_nth_data (((_QueryData *) query)->classes, 0);
}

/* ------------------------------------------------------------------------- *\
 * Free the OQL query.
\* ------------------------------------------------------------------------- */
void
oql_free_query (_QueryData * q)
{
  if (q)
    {
      if (q->sql)
        {
          g_free (q->sql);
        }
      if (q->fields)
        {
          GList *f = q->fields;

          while (f)
            {
              if (f->data)
                {
                  g_free (f->data);
                }
              f = g_list_next (f);
            }
          g_list_free (q->fields);
        }
      if (q->values)
        {
          GList *f = q->values;

          while (f)
            {
              if (f->data)
                {
                  g_free (f->data);
                }
              f = g_list_next (f);
            }
          g_list_free (q->values);
        }
      if (q->conditions)
        {
          GList *f = q->conditions;

          while (f)
            {
              if (f->data)
                {
                  free_query_condition (f->data);
                }
              f = g_list_next (f);
            }
          g_list_free (q->conditions);
        }
      if (q->classes)
        {
          GList *f = q->classes;

          while (f)
            {
              if (f->data)
                {
                  g_free (f->data);
                }
              f = g_list_next (f);
            }
          g_list_free (q->classes);
        }

      if (q->orderby)
        {
          g_free (q->orderby);
        }

      g_free (q);
    }
}

/* ------------------------------------------------------------------------- *\
 * Get the "field" from the OQL query.
\* ------------------------------------------------------------------------- */
const char *
oql_query_get_field_name (_QueryData * query, int field)
{
  return (g_list_nth_data (query->fields, field));
}

/* ------------------------------------------------------------------------- *\
 * Get the index for a given field in an OQL query.
\* ------------------------------------------------------------------------- */
int
oql_query_get_field_position (_QueryData * query, const char *fieldname)
{
  int n = 0;
  GList *l = query->fields;

  while (l)
    {
    /* printf( "'%s' = '%s' ?\n" , fieldname , (char *)l->data  ); */
      if (g_strcasecmp (fieldname, (char *) l->data) == 0)
        {
          return (n);
        }
      n++;
      l = g_list_next (l);
    }
  return (-1);
}

/* ========================================================================= *\
 * Create OQL queries.
\* ========================================================================= */

/* ------------------------------------------------------------------------- *\
 * Create OQL query for writing an object.
\* ------------------------------------------------------------------------- */
_QueryData *
oql_write_object (const gchar * classname, const gchar * key,
                  GHashTable * values, gboolean update)
{
  _QueryData *query;
  odl_class *c;
  const gchar *fieldname;
  gchar *value;
  GList *all_fields;
  GList *field;
  _QueryCondition *condition;

  g_return_val_if_fail (classname, NULL);
  g_return_val_if_fail (key, NULL);
  g_return_val_if_fail (values, NULL);

  query = create_query_data (NULL);

  query->type = OQL_WRITE;
  query->indatabase = update;
  query->classes = g_list_append (query->classes, g_strdup (classname));

  query->fields = g_list_append (query->fields, g_strdup ("objectid"));
  query->values = g_list_append (query->values, g_strdup (key));

  c = odl_find_class (all_classes, classname, NULL);
  all_fields = odl_class_get_fields (c, FT_basic);

  field = g_list_first (all_fields);
  while (field)
    {
      fieldname = odl_field_get_name ((odl_field *) field->data);
      value = g_hash_table_lookup (values, fieldname);
      if (value)
        {
          query->fields = g_list_append (query->fields, g_strdup (fieldname));
          query->values = g_list_append (query->values, g_strdup (value));
        }
      field = g_list_next (field);
    }
  odl_fieldlist_free (all_fields);

  condition = create_query_condition (g_strdup (classname),
                                      g_strdup ("objectid"),
                                      g_strdup ("="), g_strdup (key));
  query->conditions = g_list_append (query->conditions, condition);

  return (query);
}

_QueryData *
oql_make_sql_query (const char *query)
{
  _QueryData *q = create_query_data (NULL);
  if (q)
    {
      q->sql = g_strdup (query);
    }
  return (q);
}

/* ------------------------------------------------------------------------- *\
 * Create OQL query for deleting an object.
\* ------------------------------------------------------------------------- */
_QueryData *
oql_delete_object (const char *classname, const char *key)
{
  _QueryData *q = NULL;
  _QueryCondition *condition;

  q = create_query_data (NULL);
  if (!q)
    {
      errormsg ("Out of memory");
      return (NULL);
    }

  q->type = OQL_DELETE;

  q->classes = g_list_append (q->classes, g_strdup (classname));
  condition =
    create_query_condition (g_strdup (classname), g_strdup ("objectid"),
                            g_strdup ("="), g_strdup (key));
  q->conditions = g_list_append (q->conditions, condition);
  return (q);
}

/* ------------------------------------------------------------------------- *\
 * Create OQL query for deleting all objects with matching classname,
 * where the fieldname has the value of 'key'.
\* ------------------------------------------------------------------------- */
_QueryData *
oql_delete_all_objects (const char *classname, const char *fieldname,
                        const char *key)
{
  _QueryData *q = NULL;
  _QueryCondition *condition;

  q = create_query_data (NULL);
  if (!q)
    {
      errormsg ("Out of memory");
      return (NULL);
    }

  q->type = OQL_DELETE;

  q->classes = g_list_append (q->classes, g_strdup (classname));
  condition =
    create_query_condition (g_strdup (classname), g_strdup (fieldname),
                            g_strdup ("="), g_strdup (key));
  q->conditions = g_list_append (q->conditions, condition);
  return (q);
}

/* ------------------------------------------------------------------------- *\
 * Create OQL query for loading a single object with objectid equal to 'key'.
\* ------------------------------------------------------------------------- */
_QueryData *
oql_load_object_by_key (const char *classname, const char *key)
{
  GList *l;
  GList *parents;
  _QueryData *q = NULL;
  _QueryCondition *condition;
  odl_class *c;
  GList *fields;

  q = create_query_data (NULL);
  if (!q)
    {
      errormsg ("Out of memory");
    }
  q->type = OQL_SELECT;

  c = odl_find_class (all_classes, classname, NULL);
  if (!c)
    {
      errormsg ("Class %s not found", classname);
      oql_free_query (q);
      abort ();
      return (NULL);
    }

  add_order_by (q, c->orderby, FALSE);

  fields = odl_class_get_fields (c, FT_basic);
  l = fields;
  while (l)
    {
      char *fieldname;

      fieldname = g_strdup (odl_field_get_name ((odl_field *) l->data));
      q->fields = g_list_append (q->fields, fieldname);
      l = g_list_next (l);
    }
  odl_fieldlist_free (fields);
  q->fields = g_list_sort (q->fields, (GCompareFunc) oql_sort_string_name);

  q->classes = g_list_append (q->classes, g_strdup (classname));
  parents = odl_class_get_parentnames (c);
  if (parents)
    {
      l = parents;
      while (l)
        {
          q->classes = g_list_append (q->classes, g_strdup (l->data));
          l = g_list_next (l);
        }
      odl_namelist_free (parents);
    }
  condition =
    create_query_condition (g_strdup (classname), g_strdup ("objectid"),
                            g_strdup ("="), g_strdup (key));
  q->conditions = g_list_append (q->conditions, condition);
  return (q);
}

/* ------------------------------------------------------------------------- *\
 * Create OQL query for loading all instances of a given class.
\* ------------------------------------------------------------------------- */
_QueryData *
oql_find_all_objects (const char *classname)
{
  _QueryData *q = NULL;
  odl_class *c;
  GList *parents, *l;
  GList *fields;

  q = create_query_data (NULL);
  if (!q)
    {
      errormsg ("Out of memory");
    }
  q->type = OQL_SELECT;

  c = odl_find_class (all_classes, classname, NULL);
  if (!c)
    {
      errormsg ("Class %s not found", classname);
      oql_free_query (q);
      return (NULL);
    }

  add_order_by (q, c->orderby, FALSE);

  fields = odl_class_get_fields (c, FT_basic);
  l = fields;
  while (l)
    {
      char *fieldname;

      fieldname = g_strdup (odl_field_get_name ((odl_field *) l->data));
      q->fields = g_list_append (q->fields, fieldname);
      l = g_list_next (l);
    }
  odl_fieldlist_free (fields);
  q->fields = g_list_sort (q->fields, (GCompareFunc) oql_sort_string_name);

  q->classes = g_list_append (q->classes, g_strdup (classname));
  parents = odl_class_get_parentnames (c);
  if (parents)
    {
      l = parents;
      while (l)
        {
          q->classes = g_list_append (q->classes, g_strdup (l->data));
          l = g_list_next (l);
        }
      odl_namelist_free (parents);
    }
  return (q);
}

/* ------------------------------------------------------------------------- *\
 * Create OQL query to get a single field from an object.
\* ------------------------------------------------------------------------- */
_QueryData *
oql_load_object_field_by_key (const char *classname, const char *field,
                              const char *key)
{
  GList *l;
  GList *parents;
  _QueryData *q = NULL;
  _QueryCondition *condition;
  odl_class *c;

  q = create_query_data (NULL);
  if (!q)
    {
      errormsg ("Out of memory");
    }
  q->type = OQL_SELECT;

  c = odl_find_class (all_classes, classname, NULL);
  if (!c)
    {
      errormsg ("Class %s not found", classname);
      oql_free_query (q);
      return (NULL);
    }

  add_order_by (q, c->orderby, FALSE);
  q->fields = g_list_append (q->fields, g_strdup ("objectid"));
  q->fields = g_list_append (q->fields, g_strdup (field));
  q->fields = g_list_sort (q->fields, (GCompareFunc) oql_sort_string_name);
  q->classes = g_list_append (q->classes, g_strdup (classname));
  parents = odl_class_get_parentnames (c);
  if (parents)
    {
      l = parents;
      while (l)
        {
          q->classes = g_list_append (q->classes, g_strdup (l->data));
          l = g_list_next (l);
        }
      odl_namelist_free (parents);
    }

  condition =
    create_query_condition (g_strdup (classname), g_strdup ("objectid"),
                            g_strdup ("="), g_strdup (key));
  q->conditions = g_list_append (q->conditions, condition);
  return (q);
}

/* ------------------------------------------------------------------------- *\
 * Create OQL query that loads all instances of a class.
\* ------------------------------------------------------------------------- */
_QueryData *
oql_load_object (const char *classname)
{
  GList *l, *tmp, *parents;
  _QueryData *q = NULL;
  odl_class *c;
  /* int nnn = 0; */

  q = create_query_data (NULL);
  if (!q)
    {
      errormsg ("Out of memory");
      return (NULL);
    }
  q->type = OQL_SELECT;

  c = odl_find_class (all_classes, classname, NULL);
  if (!c)
    {
      errormsg ("Class %s not found", classname);
      oql_free_query (q);
      return (NULL);
    }

  add_order_by (q, c->orderby, FALSE);
  l = odl_class_get_fields (c, FT_basic);
  tmp = l;
  while (tmp)
    {
      char *fieldname;

      fieldname = g_strdup (odl_field_get_name ((odl_field *) tmp->data));
      q->fields = g_list_append (q->fields, fieldname);
      tmp = g_list_next (tmp);
    }
  q->fields = g_list_sort (q->fields, (GCompareFunc) oql_sort_string_name);

  odl_fieldlist_free (l);
  q->classes = g_list_append (q->classes, g_strdup (classname));

  parents = odl_class_get_parentnames (c);
  if (parents)
    {
      l = parents;
      while (l)
        {
          q->classes = g_list_append (q->classes, g_strdup (l->data));
          l = g_list_next (l);
        }
      odl_namelist_free (parents);
    }

  return (q);
}

/* ------------------------------------------------------------------------- *\
 * Add query constraint to OQL query.
\* ------------------------------------------------------------------------- */
gboolean
do_oql_add_query_constraint (_QueryData * q, const char *loadclass,
                             const char *fieldname, const char *relation,
                             const char *value, enum querytest test,
                             gboolean casesensitive)
{
  _QueryCondition *condition;

  if (!q)
    return (FALSE);

  condition =
    create_query_condition (g_strdup (loadclass), g_strdup (fieldname),
                            g_strdup (relation), g_strdup (value));
  condition->test = test;
  condition->casesensitive = casesensitive;
  q->conditions = g_list_append (q->conditions, condition);
  return (TRUE);
}


/* ------------------------------------------------------------------------- *\
 * Parse OQL query and create.
\* ------------------------------------------------------------------------- */
_QueryData *
oql_load (const char *classname, const char *OQL_query)
{
  fatal_error ("OQL text queries not yet supported.");
  return NULL;
}


/* ------------------------------------------------------------------------- *\
 * Create OQL query that searches for objects that have a particular value 
 * in a particular field.
\* ------------------------------------------------------------------------- */
_QueryData *
oql_objects_field_search (const char *classname, const char *field,
                          const char *value)
{
  _QueryData *q = oql_load_object (classname);

  if (q)
    {
      _QueryCondition *condition;

      condition =
        create_query_condition (g_strdup (classname), g_strdup (field),
                                g_strdup ("="), g_strdup (value));
      if (condition)
        {
          q->conditions = g_list_append (q->conditions, condition);
        }
      else
        {
          oql_free_query (q);
          return (NULL);
        }
      return (q);
    }

  return (NULL);
}

/* ------------------------------------------------------------------------- *\
 * Add ordeby clause to OQL query.
\* ------------------------------------------------------------------------- */
void
add_order_by (_QueryData * query, const char *orderby, gboolean reverse)
{
  /* static int count = 0; */

  if (query->orderby)
    g_free (query->orderby);

  query->reversesearch = reverse;
  if (orderby)
    {
      query->orderby = g_strdup (orderby);
    }
  else
    {
      query->orderby = NULL;
    }
}

/* ------------------------------------------------------------------------- *\
 * Add limit clause to OQL query.
\* ------------------------------------------------------------------------- */
void
oql_limit_results (_QueryData * q, unsigned long int limit)
{
  q->uselimit = TRUE;
  q->maxresults = limit;
}

/* ------------------------------------------------------------------------- *\
 * Create the OQL query structure.
\* ------------------------------------------------------------------------- */
static _QueryData *
create_query_data (const char *orderby)
{
  _QueryData *q = (_QueryData *) g_new0 (_QueryData, 1);

  if (q)
    {
      q->complex = NULL;
      q->sql = NULL;
      q->fields = NULL;
      q->conditions = NULL;
      q->classes = NULL;
      if (orderby)
        q->orderby = g_strdup (orderby);
      else
        q->orderby = NULL;
      q->values = NULL;

      q->uselimit = FALSE;
      q->maxresults = 0;
    }
  return (q);
}

/* ------------------------------------------------------------------------- *\
 * Create a _QueryCondition structure.
\* ------------------------------------------------------------------------- */
static _QueryCondition *
create_query_condition (const char *targetclass, const char *field,
                        const char *relation, const char *value)
{
  _QueryCondition *c = (_QueryCondition *) g_new0 (_QueryCondition, 1);

  if (c)
    {
      c->targetclass = (char *) targetclass;
      c->field = (char *) field;
      c->relation = (char *) relation;
      c->value = (char *) value;
      c->casesensitive = TRUE;
      c->test = q_hackish;
    }
  return (c);
}

/* ------------------------------------------------------------------------- *\
 * Free the _QueryCondition structure.
\* ------------------------------------------------------------------------- */
static void
free_query_condition (_QueryCondition * c)
{
  if (c)
    {
      if (c->targetclass)
        g_free (c->targetclass);
      if (c->field)
        g_free (c->field);
      if (c->relation)
        g_free (c->relation);
      if (c->value)
        g_free (c->value);
      g_free (c);
    }
}

/* ------------------------------------------------------------------------- *\
 * Sorts strings, used in sorting fields in an OQL query.
\* ------------------------------------------------------------------------- */
int
oql_sort_string_name (gpointer a, gpointer b)
{
  return (g_strcasecmp (a, b));
}

GHashTable *oql_reservedwords = NULL;

/* ------------------------------------------------------------------------- *\
 * 31 bit hash function.
\* ------------------------------------------------------------------------- */
guint
g_strcase_hash (gconstpointer key)
{
  const char *p = key;
  guint h = tolower (*p);

  if (h)
    for (p += 1; *p != '\0'; p++)
      h = (h << 5) - h + tolower (*p);

  return h;
}

/* ------------------------------------------------------------------------- *\
 * Are the strings equal.
\* ------------------------------------------------------------------------- */
gboolean
g_strcase_equal (gconstpointer v1, gconstpointer v2)
{
  const gchar *string1 = v1;
  const gchar *string2 = v2;

  return g_strcasecmp (string1, string2) == 0;
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
void
quick_reserved_word (const char *word, gboolean mysql, gboolean postgresql)
{
  _sql_reserved_word *w = NULL;

  if (oql_reservedwords == NULL)
    {
      oql_reservedwords = g_hash_table_new (g_strcase_hash, g_strcase_equal);
    }

  w = g_hash_table_lookup (oql_reservedwords, word);
  if (!w)
    {
      w = g_new0 (_sql_reserved_word, 1);
      if (w)
        {
          w->word = g_strdup (word);
          w->mysql = FALSE;
          w->postgresql = FALSE;
          if (!w->word)
            {
              g_free (w);
              return;
            }
          g_hash_table_insert (oql_reservedwords, w->word, w);
        }
      else
        return;
    }

  w->mysql = mysql;
  w->postgresql = postgresql;
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
void
add_reserved_word (const char *word, int dbtype, gboolean reserved)
{
  _sql_reserved_word *w = NULL;

  if (oql_reservedwords == NULL)
    oql_reservedwords = g_hash_table_new (g_strcase_hash, g_strcase_equal);

  w = g_hash_table_lookup (oql_reservedwords, word);
  if (!w)
    {
      w = g_new0 (_sql_reserved_word, 1);
      if (w)
        {
          w->word = g_strdup (word);
          w->mysql = FALSE;
          w->postgresql = FALSE;
          if (!w->word)
            {
              g_free (w);
              return;
            }
          g_hash_table_insert (oql_reservedwords, w->word, w);
        }
      else
        return;
    }
  switch (dbtype)
    {
    case OQL_DBTYPE_MYSQL:
      w->mysql = reserved;
      break;
    case OQL_DBTYPE_POSTGRESQL:
      w->postgresql = reserved;
      break;
    }
}

/* ------------------------------------------------------------------------- *\
 * Is the word a reserved word.
\* ------------------------------------------------------------------------- */
gboolean
is_word_reserved (const char *word, int dbtype)
{
  _sql_reserved_word *w = NULL;

  w = g_hash_table_lookup (oql_reservedwords, word);
  if (!w)
    return (FALSE);

  switch (dbtype)
    {
    case OQL_DBTYPE_MYSQL:
      return (w->mysql);
    case OQL_DBTYPE_POSTGRESQL:
      return (w->postgresql);
    }

  return (FALSE);
}

/* ------------------------------------------------------------------------- *\
 * Create an OQL field.
\* ------------------------------------------------------------------------- */
oql_field *
oql_make_field (char *field, enum querytest test, char *value,
                gboolean casesensitive, gboolean invert)
{
  oql_field *f = g_new0 (oql_field, 1);

  if (f)
    {
      f->field = g_strdup (field);
      f->test = test;
      f->value = g_strdup (value);
      f->casesensitive = casesensitive;
      f->invert = invert;
    }
  return (f);
}

/* ------------------------------------------------------------------------- *\
 * Create an OQL constraint.
\* ------------------------------------------------------------------------- */
oql_constraint *
oql_make_constraint (enum oql_logic logic)
{
  oql_constraint *o = g_new0 (oql_constraint, 1);

  if (o)
    {
      o->logic = logic;
      o->fields = NULL;
      o->constraints = NULL;
    }
  return (o);
}

/* ------------------------------------------------------------------------- *\
 * Add the fields from a constraint.
\* ------------------------------------------------------------------------- */
void
oql_add_field (oql_constraint * c, oql_field * f)
{
  c->fields = g_list_append (c->fields, f);
}

/* ------------------------------------------------------------------------- *\
 * Add a constraint.
\* ------------------------------------------------------------------------- */
void
oql_add_constraint (oql_constraint * c, oql_constraint * addme)
{
  c->constraints = g_list_append (c->constraints, addme);
}

/* ------------------------------------------------------------------------- *\
 * Add complex constraint.
\* ------------------------------------------------------------------------- */
gboolean
oql_add_complex_constraint (_QueryData * q, oql_constraint * c)
{
  q->complex = g_list_append (q->complex, c);
  return FALSE;
}

/* ------------------------------------------------------------------------- *\
 * Add complex conditions to SQL query.
\* ------------------------------------------------------------------------- */
static void
oql_add_complex_conditions_to_sql (_QueryData * q, DBType db, GString * buf)
{
  GList *l;
  odl_class *cl;
  char *classname;

  if (!q)
    return;
  if (!q->classes)
    return;
  if (!q->classes->data)
    return;
  classname = q->classes->data;

  l = q->complex;
  while (l)
    {
      g_string_append (buf, "(");

      cl = odl_find_class (all_classes, classname, NULL);
      oql_add_complex_constraint_to_sql (q, cl, db, l->data, buf);

      g_string_append (buf, ")");
      if (l->next)
        g_string_append (buf, " AND ");
      l = l->next;
    }
}

/* ------------------------------------------------------------------------- *\
 * Add complex constraint to SQL query.
\* ------------------------------------------------------------------------- */
static void
oql_add_complex_constraint_to_sql (_QueryData * q, odl_class * cl, DBType db,
                                   oql_constraint * c, GString * buf)
{
  GList *l;
  /* gboolean need_and = FALSE; */
  char *logic;
  char *quoted;
  const char *classname = odl_class_get_full_name (cl);
  char *temp;

  if (c->logic == oql_and)
    logic = " AND ";
  else
    logic = " OR ";

  /* add fields */
  l = c->fields;

  while (l)
    {
      /* add this field */
      oql_field *f = l->data;

      /* field */
      if (!f->casesensitive)
        {
          if (db == OQL_DBTYPE_MYSQL)
            g_string_append (buf, "upper(");
          else if (db == OQL_DBTYPE_POSTGRESQL)
            g_string_append (buf, "upper(rtrim(");
        }

      quoted = oql_quote_column (classname, f->field, db);
      g_string_append (buf, quoted);
      g_free (quoted);

      if (!f->casesensitive)
        {
          if (db == OQL_DBTYPE_MYSQL)
            g_string_append (buf, ")");
          else if (db == OQL_DBTYPE_POSTGRESQL)
            g_string_append (buf, ",' '))");
        }

      /* test, and prepare 'quoted' with value, and extra characters needed by the test */
      switch (f->test)
        {
        case q_equals:
          if (f->invert)
            g_string_append (buf, "<>");
          else
            g_string_append (buf, "=");
          quoted = oql_quote_value (f->value, db);
          break;
        case q_lessthan:
          if (f->invert)
            g_string_append (buf, ">=");
          else
            g_string_append (buf, "<");
          quoted = oql_quote_value (f->value, db);
          break;
        case q_greaterthan:
          if (f->invert)
            g_string_append (buf, "<=");
          else
            g_string_append (buf, ">");
          quoted = oql_quote_value (f->value, db);
          break;
        case q_contains:
          if (f->invert)
            g_string_append (buf, " not like ");
          else
            g_string_append (buf, " like ");
          temp = g_strdup_printf ("%%%s%%", f->value);
          quoted = oql_quote_value (temp, db);
          g_free (temp);
          break;
        case q_startswith:
          if (f->invert)
            g_string_append (buf, " not like ");
          else
            g_string_append (buf, " like ");
          temp = g_strdup_printf ("%%%s", f->value);
          quoted = oql_quote_value (temp, db);
          g_free (temp);
          break;
        case q_endswith:
          if (f->invert)
            g_string_append (buf, " not like ");
          else
            g_string_append (buf, " like ");
          temp = g_strdup_printf ("%s%%", f->value);
          quoted = oql_quote_value (temp, db);
          g_free (temp);
          break;
        case q_notequals:
          /* TODO */
          break;
        case q_notlessthan:
          /* TODO */
          break;
        case q_notgreaterthan:
          /* TODO */
          break;
        case q_notcontains:
          /* TODO */
          break;
        case q_notstartswith:
          /* TODO */
          break;
        case q_notendswith:
          /* TODO */
          break;
        case q_hackish:
          /* TODO */
          break;
        }

      /* add the value to test */
      if (!f->casesensitive)
        {
          if (db == OQL_DBTYPE_MYSQL)
            g_string_append (buf, "upper(");
          else if (db == OQL_DBTYPE_POSTGRESQL)
            g_string_append (buf, "upper(rtrim(");
        }
      g_string_append (buf, quoted);
      if (!f->casesensitive)
        {
          if (db == OQL_DBTYPE_MYSQL)
            g_string_append (buf, ")");
          else if (db == OQL_DBTYPE_POSTGRESQL)
            g_string_append (buf, ",' '))");
        }

      /* next */
      if (l->next)
        g_string_append (buf, logic);
      l = g_list_next (l);
    }

  /* add sub constraints */
  if (c->fields && c->constraints)
    g_string_append (buf, logic);

  l = c->constraints;
  while (l)
    {
      /* add this constraint */
      g_string_append (buf, "(");
      oql_add_complex_constraint_to_sql (q, cl, db, l->data, buf);
      g_string_append (buf, ")");

      /* next */
      if (l->next)
        g_string_append (buf, logic);
      l = g_list_next (l);
    }
}

/* ------------------------------------------------------------------------- *\
 * Convert a value from the GEAS internal format to a format used by the
 * database.
\* ------------------------------------------------------------------------- */
char *
oql_translate_for_write (char *value, odl_class * c, const char *fieldname,
                         int database)
{
  odl_field *f;

  f = odl_class_get_field (c, fieldname);
  if (!f)
    {
      abort ();
      return (g_strdup (""));   /* don't like this, much */
    }

  switch (odl_field_get_datatype (f))
    {
    case DT_date:
      /* if( database == OQL_DBTYPE_MYSQL ) { } else if( database == OQL_DBTYPE_POSTGRESQL ) { } */
      if (strlen (value) == 0)
        {
          debug_output (DEBUGLEVEL_8, "convert '%s' to '%s'", value,
                        "0001-01-01");
          return (g_strdup ("0001-01-01"));
        }
      break;
    case DT_time:
      if (strlen (value) == 0)
        {
          debug_output (DEBUGLEVEL_8, "convert '%s' to '%s'", value,
                        "0001-01-01");
          return (g_strdup ("00:00:00"));
        }
      break;
    case DT_datetime:
      if (strlen (value) == 0)
        {
          debug_output (DEBUGLEVEL_8, "convert '%s' to '%s'", value,
                        "0001-01-01");
          return (g_strdup ("0001-01-01 00:00:00"));
        }
      break;
    default:
      break;
    }

  /* just passes data through */
  return g_strdup (value);
}

/* ------------------------------------------------------------------------- *\
 * Convert a value from a format used by a database to a format used by GEAS.
\* ------------------------------------------------------------------------- */
char *
oql_translate_from_read (char *value, odl_class * c, const char *fieldname,
                         int database)
{
  odl_field *f;

  if (c == NULL)
    return g_strdup (value);

  f = odl_class_get_field (c, fieldname);
  if (!f)
    {
      abort ();
      return (g_strdup (""));   /* don't like this, much */
    }
  switch (odl_field_get_datatype (f))
    {
    case DT_date:
      /* if( database == OQL_DBTYPE_MYSQL ) { } else if( database == OQL_DBTYPE_POSTGRESQL ) { } */
      if (strcmp (value, "0001-01-01") == 0)
        {
          debug_output (DEBUGLEVEL_8, "convert '%s' to '%s'", value, "");
          return (g_strdup (""));
        }
      break;
    case DT_time:
      if (strcmp (value, "00:00:00") == 0)
        {
          debug_output (DEBUGLEVEL_8, "convert '%s' to '%s'", value, "");
          return (g_strdup (""));
        }
      break;
    case DT_datetime:
      if (strcmp (value, "0001-01-01 00:00:00") == 0)
        {
          debug_output (DEBUGLEVEL_8, "convert '%s' to '%s'", value, "");
          return (g_strdup (""));
        }
      break;
    default:
      break;
    }

  /* just passes data through */
  return g_strdup (value);
}
