/*
   objectstore.c - Interface to the database backend(s)

   Copyright (C) 2001 Free Software Foundation

   Part of the GNU Enterprise Application Server (GEAS)

   GEAS is free software; you can redistribute it and/or modify it under
   the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2, or (at your option) any later
   version.

   GEAS is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.

   You should have received a copy of the GNU General Public License
   along with GEAS; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   $Id: objectstore.c,v 1.68 2001/06/20 22:26:23 reinhard Exp $
   
*/

#include "config.h"
#include "objectstore.h"
#include <stdio.h>
#include <limits.h>
#include <string.h>
#include "geas-server.h"
#include "classdata.h"
#include "datamonitor/datamonitor.h"
#include "objectstore_private.h"

/* compile time hard limit */
#ifdef MAX_DATABASE_CONNECTIONS
static unsigned int max_database_connections = MAX_DATABASE_CONNECTIONS;
#else
static unsigned int max_database_connections = 1;        /* default of 1 connection per
                                                   database */
#endif


#define add_field_to_result_row(row,field) \
row = g_list_append(row,field)



/* maintain a list of databases in use */
static GList *database_list = NULL;

/* private function prototypes */
static struct database_handle *find_database_handle (GList * list,
                                                     const char *database,
                                                     int *err, char **errmsg);


/* ========================================================================= *\
 * public functions
\* ========================================================================= */

/* ------------------------------------------------------------------------- *\
 *
\* ------------------------------------------------------------------------- */
void
update_objectstore_databases (gboolean remove_items)
{
  FILE *changefile;
  struct database_handle *h = NULL;
  const char *database = get_first_active_database (configdata);     /* TODO HACK
                                                                        HACK
                                                                        HACK */

  /* write header to change file */
  debug_output (DEBUGLEVEL_HIGH, "writing header to change file");
  changefile =
    fopen (get_global_option_str
           (configdata, "databasechangefile", "database.changes.txt"), "w");

  if (changefile)
    {
      fprintf (changefile,
               "===========================================================================\n");
      fprintf (changefile, "GEAS Database change requirements information\n");
      fprintf (changefile,
               "===========================================================================\n\n");
      fclose (changefile);
    }

  /* automatic class storage installer */

  /* find the database */
  debug_output (DEBUGLEVEL_HIGH, "getting db handle");

  h = find_database_handle (database_list, database, NULL, NULL);
  if (!h)
    {
      errormsg ("Could not find database '%s'", database);
      return;
    }

  /* make it update (TODO: error msg handling) */
  /* message( "updating..." ); */
  debug_output (DEBUGLEVEL_HIGH, "updating the tables");

  h->update_tables (h, remove_items, NULL, NULL);
}

/* ------------------------------------------------------------------------- *\
 *
\* ------------------------------------------------------------------------- */
struct query_result *
query_objectstore (QueryData * query, int *err, char **errmsg)
{
  struct database_handle *h = NULL;
  struct query_result *retval = NULL;
  struct query_result *storeresult = NULL;
  const char *database = get_first_active_database (configdata);        /* HACK */

  /* TODO: database dependant quote mode selection */
  trace_functioncall ();

  /* clear error indicators */
  if (err)
    (*err) = 0;
  if (errmsg)
    (*errmsg) = NULL;

  /* find database handle named 'database' */
  /* self_test_message("Looking for database '%s'", database); */
  h = find_database_handle (database_list, database, err, errmsg);
  if (!h)
    return (NULL);

  /* finish off the query */
  /* add_order_by(query, NULL); */

  /* TODO: */
  /* cacheresult = search_cache( query ); */

  /*message( "-----------------------------------------" ); */
  /* execute query (errors passed back automatically) */
  /* TODO: if error == unexpected disconnection then try to reconnect */
  /* self_test_message("Executing"); */
  /*   printf( "SQL: %s\n" , oql_query_as_sql(query,OQL_DBTYPE_CACHEONLY) ); */
  storeresult = h->execute (h, query, err, errmsg);
  /*   printf( "query done\n" ); */

#ifdef DEBUG
  if (errmsg && *errmsg && debuglevel > 1)
    errormsg ("Query error: %s", *errmsg);
#endif

  /* TODO: */
  /* retval = merge_results(cacheresult,storeresult); */
  retval = storeresult;                /* temp: return just the store result */

  if (retval)
    retval->query = query;

  /* done */
  return (retval);
}

/* ------------------------------------------------------------------------- *\
 *
\* ------------------------------------------------------------------------- */
struct query_result *
delete_from_objectstore (const char *classname, const char *key, int *err,
                         char **errmsg)
{
  struct database_handle *h = NULL;
  struct query_result *retval = NULL;
  const char *database = get_first_active_database (configdata);        /* HACK */

  trace_functioncall ();
  if (!classname || !key)
    return (NULL);

  /*   message("Deleting %s/%s from database",classname,key); */

  /* clear error indicators */
  if (err)
    (*err) = 0;
  if (errmsg)
    (*errmsg) = NULL;

  /* find database handle named 'database' */
  /* self_test_message("Looking for database '%s'", database); */
  h = find_database_handle (database_list, database, err, errmsg);
  if (!h)
    return (NULL);

  /* execute query (errors passed back automatically) */
  /* self_test_message("Executing"); */
  retval = h->delete_object (h, classname, key, err, errmsg);

  /* done */
  return (retval);
}

/* ------------------------------------------------------------------------- *\
 *
\* ------------------------------------------------------------------------- */
struct query_result *
delete_all_from_objectstore (const char *classname,
                             const char *fieldname,
                             const char *key, int *err, char **errmsg)
{
  struct database_handle *h = NULL;
  struct query_result *retval = NULL;
  const char *database = get_first_active_database (configdata);        /* TODO HACK */

  trace_functioncall ();
  if (!classname || !key)
    return (NULL);

  /* message("Deleting %s from database where %s.%s = %s",classname,classname,fieldname,key); */

  /* clear error indicators */
  if (err)
    (*err) = 0;
  if (errmsg)
    (*errmsg) = NULL;

  /* find database handle named 'database' */
  /* self_test_message("Looking for database '%s'", database); */
  h = find_database_handle (database_list, database, err, errmsg);
  if (!h)
    return (NULL);

  /* execute query (errors passed back automatically) */
  /* self_test_message("Executing"); */
  retval = h->delete_all_objects (h, classname, fieldname, key, err, errmsg);

  /* done */
  return (retval);
}

/* ------------------------------------------------------------------------- *\
 * Write data to the backend
\* ------------------------------------------------------------------------- */
struct query_result *
write_to_objectstore (const char *classname, const char *key,
                      GHashTable *values, gboolean update,
                      int *err, char **errmsg)
{
  struct database_handle *h = NULL;
  struct query_result *retval = NULL;
  const char *database = get_first_active_database (configdata);     /* HACK */

  trace_functioncall ();

  g_return_val_if_fail (classname, NULL);
  g_return_val_if_fail (key, NULL);
  g_return_val_if_fail (values, NULL);

  /* clear error indicators */
  if (err)
    (*err) = 0;
  if (errmsg)
    (*errmsg) = NULL;

  /* find database handle named 'database' */
  /* self_test_message("Looking for database '%s'", database); */
  h = find_database_handle (database_list, database, err, errmsg);
  if (!h)
    {
      errormsg ("Could not find database '%s'", database);
      return (NULL);
    }

  /* execute query (errors passed back automatically) */
  /* self_test_message("Executing"); */
  retval = h->write_object (h, classname, key, values, update, err, errmsg);

  /* done */
  return (retval);
}

/* ------------------------------------------------------------------------- *\
 *
\* ------------------------------------------------------------------------- */
void
free_query_result (struct query_result *result)
{
  GList *ll, *fl;

  if (result)
    {
      if (result->data)
        {
          ll = result->data;
          while (ll)
            {
              fl = (GList *) ll->data;
              while (fl)
                {
                  if (fl->data)
                    g_free (fl->data);
                  fl = g_list_next (fl);
                }
              if (ll->data)
                g_list_free (ll->data);
              ll = g_list_next (ll);
            }
          g_list_free (result->data);
        }
      g_free (result);
    }
}

/* ------------------------------------------------------------------------- *\
 *
\* ------------------------------------------------------------------------- */
gboolean
initialise_objectstore (configuration config)
{
  int i;
  struct database_handle *h = NULL;
  gboolean succeeded = FALSE;

  /* for each database, create a new instance of the appropriate type */
  /* self_test_message("%d databases", count_configured_databases(config)); */

  for (i = 0; i < count_configured_databases (config); i++)
    {
      const char *dbname = get_database_name (config, i);
      const char *type;

      /* self_test_message("init: database %d name: '%s'",i, dbname); */

      /* if no name, ignore - should never happen, anyway */
      if (!dbname)
        {
          continue;                /* skip */
        }

      /* if not active, skip this entire entry */
      if (get_database_option_bool (config, dbname, "active", FALSE) == FALSE)
        {
          continue;                /* skip */
        }

      /* find which database code to use */
      type = get_database_option (config, dbname, "type");
      if (!type)
        {
          continue;                /* skip */
        }

      /* self_test_message("Database %s: type = %s", dbname, type); */

      /* create an appropriate handle for the database access data */
      /* somewhat messy nested #ifdefs, to control selection of databases
         to use */

      if (g_strcasecmp (type, "MySQL") == 0)
        {
          h =
            (struct database_handle *) MySQL_create_database_handle (config,
                                                                     dbname);
          succeeded = TRUE;
        }

      if (!succeeded && g_strcasecmp (type, "PostgreSQL") == 0)
        {
          h = postgresql_create_database_handle (config, dbname);
          succeeded = TRUE;
        }


      /* ******************** */

      if (!succeeded)
        {
          errormsg
            ("Database '%s' is of type %s, but this is not supported or could\nnot be initialised.\n",
             dbname, type);
          h = NULL;
          continue;                /* go to next item in list */
        }

      /* add h to list of known databases */
      if (h)
        {
          h->type = g_strdup (type);
          database_list = g_list_append (database_list, h);
        }
    }
  /* self_test_message( "Initialised %d databases" ,
     g_list_length(database_list) ); */
  return (succeeded);
}

/* ------------------------------------------------------------------------- *\
 *
\* ------------------------------------------------------------------------- */
gboolean
close_object_store (void)
{
  GList *l;
  unsigned int i;

  trace_functioncall ();
  l = database_list;
  while (l)
    {
      struct database_handle *h = (struct database_handle *) l->data;

      if (h)
        {
          for (i = 0; i < h->num_connections; i++)
            {
              h->disconnect (h, i);
            }
          h->delete_database (h);
          l->data = NULL;
        }

      l = g_list_next (l);
    }
  if (database_list)
    g_list_free (database_list);
  database_list = NULL;
  return (TRUE);
}

int using_postgresql( void )
{
  struct database_handle *h = NULL;

  const char *database = get_first_active_database(configdata); /* HACK */
  h = find_database_handle (database_list, database, NULL,NULL);

  if( g_strcasecmp(h->type,"postgresql") == 0 ) return 1;
  return 0;
}


/* hack!   another hack!  */
void start_objectstore_transaction( void )
{
  struct database_handle *h = NULL;
  const char *database = get_first_active_database(configdata); /* HACK */

  h = find_database_handle (database_list, database, NULL,NULL);
  if (!h)
    return;

  h->begintransaction (h);
}

/* and some more of the hack! */
void commit_objectstore_transaction( void )
{
  struct database_handle *h = NULL;
  const char *database = get_first_active_database(configdata); /* HACK */

  h = find_database_handle (database_list, database, NULL,NULL);
  if (!h)
    return;

  h->committransaction (h);
}

/* ========================================================================= *\
 * private functions
\* ========================================================================= */

/* ------------------------------------------------------------------------- *\
 * find a database configuration by name
\* ------------------------------------------------------------------------- */
static struct database_handle *
find_database_handle (GList * list, const char *database, int *err,
                      char **errmsg)
{
  struct database_handle *h = NULL;

  while (list)
    {
      h = (struct database_handle *) list->data;
      /* self_test_message( "%s = %s ?" , h->name, database ); */
      if (h && h->name && (g_strcasecmp (h->name, database) == 0))
        {
          /* self_test_message( "yep" ); */
          return (h);
        }
      else
        {
          /* self_test_message( "nope" ); */
          h = NULL;
        }
      list = g_list_next (list);
    }
  if (!h)
    {
      if (err)
        (*err) = -1;                /* TODO: meaningful error codes */
      if (errmsg)
        (*errmsg) =
          g_strdup_printf ("Database '%s' was not configured.", database);
      return (NULL);
    }
  return (h);
}

/* ------------------------------------------------------------------------- *\
 *
\* ------------------------------------------------------------------------- */
gboolean
generic_database_config (configuration config, const char *database,
                         struct database_handle *h)
{
  h->name = h->username = h->hostname = NULL;
  h->unixsocket = h->dbname = h->password = NULL;

  h->name = g_strdup (database);
  h->dbname =
    (char *) get_database_option_str (config, database, "dbname", NULL);
  h->username =
    (char *) get_database_option_str (config, database, "username", NULL);
  h->password =
    (char *) get_database_option_str (config, database, "password", NULL);
  h->hostname =
    (char *) get_database_option_str (config, database, "hostname", NULL);
  h->unixsocket =
    (char *) get_database_option_str (config, database, "unixsocket", NULL);
  h->port = get_database_option_int (config, database, "port", -1);
  h->retries = get_database_option_int (config, database, "retries", 0);
  h->num_connections =
    get_database_option_int (config, database, "connections", 5);

  /* copy data */
  if (h->dbname)
    h->dbname = g_strdup (h->dbname);
  else
    errormsg ("Property 'database name' was not defined for database %s",
              database);

  if (h->username)
    h->username = g_strdup (h->username);
  /* not required */

  if (h->password)
    h->password = g_strdup (h->password);
  /* not required */

  if (h->hostname)
    h->hostname = g_strdup (h->hostname);
  /* not required */

  if (h->unixsocket)
    h->unixsocket = g_strdup (h->unixsocket);
  /* not required */

  /* force this to a valid range */
  if (h->num_connections > max_database_connections)
    h->num_connections = max_database_connections;

  if (h->num_connections < 1)
    h->num_connections = 1;

  /* clear this, for now */
  h->connections = NULL;

  /* check for valid data in required fields */
  if (!h->name)
    return (FALSE);
  else
    return (TRUE);
}

/* ------------------------------------------------------------------------- *\
 *
\* ------------------------------------------------------------------------- */
void
free_generic_database (struct database_handle *h)
{
  if (h)
    {
      if (h->name)
        g_free (h->name);

      if (h->username)
        g_free (h->username);
      if (h->password)
        g_free (h->password);
      if (h->hostname)
        g_free (h->hostname);
      if (h->dbname)
        g_free (h->dbname);
      if (h->unixsocket)
        g_free (h->unixsocket);

      if (h->type)
        g_free (h->type);

      /* do not free h->connections or h here */
    }
}

/* ------------------------------------------------------------------------- *\
 *
\* ------------------------------------------------------------------------- */
struct query_result *
new_query_result (void)
{
  struct query_result *n =
    (struct query_result *) g_malloc (sizeof (struct query_result));
  if (n)
    {
      n->success = FALSE;
      n->rows_affected = 0;
      n->data = NULL;
    }
  return (n);
}

/* ------------------------------------------------------------------------- *\
 *
\* ------------------------------------------------------------------------- */
void
remove_message (FILE * fp, DBchange * c)
{
  GList *l;

  if (c->type == DBCH_UNKNOWN)
    {
      fprintf (fp, "Unknown remove change command.\n");
      return;
    }

  fprintf (fp, "Remove: ");
  if (c->type == DBCH_REMOVE_TABLE)
    {
      fprintf (fp, "Table %s", c->name);
      if (c->columns)
        {
          fprintf (fp, "\nColumns:\n");
          l = c->columns;
          while (l)
            {
              fprintf (fp, "  %s", ((DBchange *) l->data)->name);
              l = g_list_next (l);
            }
        }
      else
        {
          fprintf (fp, " (entire table)");
        }
    }
  fprintf (fp, "\n");
}

/* ------------------------------------------------------------------------- *\
 *
\* ------------------------------------------------------------------------- */
void
add_message (FILE * fp, DBchange * c)
{
  GList *l;
  GString *buf;

  if (c->type == DBCH_UNKNOWN)
    {
      fprintf (fp, "Unknown remove change command.\n");
      return;
    }

  buf = g_string_new ("");

  if (c->type == DBCH_ADD_TABLE || c->type == DBCH_MODIFY_TABLE)
    {
      if (c->type == DBCH_MODIFY_TABLE)
        {
          g_string_append (buf, "Modify table ");
          g_string_append (buf, c->name);
          g_string_append (buf, ": Add columns ");
          fprintf (fp, "Modify: ");
        }
      else
        {
          g_string_append (buf, "Create table ");
          g_string_append (buf, c->name);
          g_string_append (buf, ": Columns ");
          fprintf (fp, "Add: ");
        }

      fprintf (fp, "Table %s", c->name);
      if (c->columns)
        {
          fprintf (fp, "\n  Columns:\n");
          l = c->columns;
          while (l)
            {
              g_string_append (buf, ((DBchange *) l->data)->name);
              g_string_append (buf, " ");
              g_string_append (buf,
                               odl_datatype_name (((DBchange *) l->data)->
                                                  datatype));

              if (((DBchange *) l->data)->format)
                {
                  g_string_append (buf, "[");
                  g_string_append (buf, ((DBchange *) l->data)->format);
                  g_string_append (buf, "]");
                }
              fprintf (fp, "    %s (%s%s%s)\n",
                       ((DBchange *) l->data)->name,
                       odl_datatype_name (((DBchange *) l->data)->datatype),
                       ((DBchange *) l->data)->format !=
                       NULL ? " : " : "",
                       ((DBchange *) l->data)->format !=
                       NULL ? ((DBchange *) l->data)->format : "");

              l = g_list_next (l);
              if (l)
                g_string_append (buf, ",");
            }
        }
      else
        {
          fprintf (fp, " (entire table)\n");
        }
    }

  dm_logentry (DM_EVENT_DATABASE, NULL, buf->str);
  g_string_free (buf, TRUE);
}

/* end of private functions */

#ifdef SELF_TEST

/* ------------------------------------------------------------------------- *\
 *
\* ------------------------------------------------------------------------- */
static void
display_objectstore_status (void)
{
  GList *l;
  /* int i; */

  l = database_list;
  while (l)
    {
#ifdef USE_MYSQL
      struct database_handle *h = (struct database_handle *) l->data;
      struct MySQL_handle *mysql;
      struct MySQL_connection *c;

      /*
         if (h)
         message("database: %s", h->name);
         else
         message("unknown database");
       */
      if (g_strcasecmp (h->type, "MySQL") == 0)
        {
          mysql = (struct MySQL_handle *) h;
          for (i = 0; i < h->num_connections; i++)
            {
              struct MySQL_connection *c =
                (struct MySQL_connection *) &((struct MySQL_connection *)
                                              h->connections)[i];

              /* message("MySQL database %d: %s is %sconnected and is
               *  %savailable", i, h->name, c->base.connected ? "" :
               *  "not " , c->base.available ? "" : "not "); */
            }
        }
#endif
#ifdef USE_POSTGRESQL
#warning "TODO: Probably something should be here. - neilt"
#endif
      l = g_list_next (l);
    }
}

/* ------------------------------------------------------------------------- *\
 *
\* ------------------------------------------------------------------------- */
void
self_test_objectstore (void)
{
  /* char buf[1024]; */
  int err = 0;
  char *errmsg = NULL;
  struct QueryData *query;
  struct query_result *result = NULL;
  int passed = 0, failed = 0;
  int count = 0;

  while (1)
    {
      message ("Testing object store: %d", ++count);

      display_objectstore_status ();

      query = oql_find_all_objects ("test");
      if (!query)
        {
          /* TODO */
          continue;
        }
      result = query_objectstore (query, &err, &errmsg);
      if (err != 0)
        {
          failed++;
        }
      else
        {
          int i, j;

          for (i = 0; i < result->rows_affected; i++)
            {
              printf ("row %2d : ", i);
              for (j = 0; j < result->field_count; j++)
                {
                  printf ("%s.%s = %s\t", get_result_class_name (result),
                          get_result_field_name (result, j),
                          get_result_field (result, i, j));
                }
              printf ("\n");
            }
        }
      if (errmsg)
        g_free (errmsg);
      if (result)
        passed = result->success;

      close_object_store ();
      free_query_result (result);

      initialise_objectstore (configdata);
      break;
    }

  message ("Object store tests completed. %d passed. %d failed.", passed,
           failed);
}

#endif
/* end of self test */
