/*
  methods_unstable.h - Unstable code for python method support.

  Copyright (C) 2001 Free Software Foundation

  This file is part of the GNU Enterprise Application Server (GEAS)

  GEAS is free software; you can redistribute it and/or modify it under
  the terms of the GNU General Public License as published by the Free
  Software Foundation; either version 2, or (at your option) any later
  version.

  GEAS is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
  for more details.

  You should have received a copy of the GNU General Public License
  along with GEAS; if not, write to the Free Software Foundation, Inc.,
  59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

  $Id: methods_unstable.h,v 1.9 2001/08/21 21:20:24 reinhard Exp $
*/

/*
  this code works, but does not guarantee server stablity, and can do minimal error
  checking. it should not be used in a production system, but instead serves as a
  minimal system to start writing useable code
*/

/*
#ifndef USE_UNSTABLE_METHODS
#error "USE_UNSTABLE_METHODS was not defined."
#endif
*/

#define DEFAULT_IDL_FILE   "geas.idl"
#define DEFAULT_METHOD_DIR "../examples/businessobjects/methods"
#define ORBIT_PYTHON_SO    "/usr/lib/python1.5/site-packages/CORBAmodule.so"

#define ERR_OK              (0)
#define ERR_NO_GMODULE      (-1)
#define ERR_NO_ORBIT_PYTHON (-2)
#define ERR_NO_FUNCTIONS    (-3)
#define ERR_MEMORY          (-4)

#include <glib.h>
#include <gmodule.h>
#include <Python.h>
#include <dirent.h>

static GList *provider_list = NULL;
static PyThreadState *mainThreadState = NULL;
static GModule *orbit_python_library = NULL;
static PyObject *(*CORBA_to_Python_func) (CORBA_Object obj);
static GList *list;
static provider_t *current_provider = NULL;
static PyObject *function;
static PyThreadState *py_threadstate = NULL;
static PyInterpreterState *mainInterpreterState;

static int load_python_binding (configuration config);


/* START PYTHON MODULE */
static PyObject *
methods_register_python (PyObject * self, PyObject * args)
{
  PyObject *function;
  char *methodname;

/*  g_assert (self); */
  g_assert (args);

  if (!PyArg_ParseTuple
      (args, "sO:methods_register_python", &methodname, &function))
    {
      return (NULL);
    }
  Py_XINCREF (function);
  /* PyObject_Print(function, stdout, 0); */
  if (!PyCallable_Check (function))
    {
      Py_XDECREF (function);
      return Py_BuildValue ("i", 0);
    }

  if (current_provider)
    current_provider->extra = (void *) function;

  return Py_BuildValue ("i", 1);        /* succeed */
}

PyMethodDef method_system_methods[] = {
  {"register_method", methods_register_python, METH_VARARGS}
  ,
  {NULL, NULL}
};

/* END PYTHON MODULE */


gboolean
initialise_method_handling (configuration config)
{
  char *buf = NULL;
  PyObject *m, *py_dict;

  g_return_val_if_fail (config, FALSE);

  /* start up embedded python interpreter */
  /* this is why this code is considered unstable: it can not enforce */
  /* more than minimal security restrictions, and can not prevent */
  /* faulty business methods from crashing the server */

  /* initialise python */
  Py_Initialize ();
  PyEval_InitThreads ();
  mainThreadState = PyThreadState_Get ();

  /* load ORBit-python binding */
  switch (load_python_binding (config))
    {
    case ERR_OK:
      /* success */
      break;
    case ERR_NO_GMODULE:
    case ERR_NO_ORBIT_PYTHON:
    case ERR_NO_FUNCTIONS:
    case ERR_MEMORY:
      /* error message already disoplayed */
      return (-1);
    }

  m = Py_InitModule ("method_system", method_system_methods);
  if (PyErr_Occurred () || !m)
    {
      fprintf (stdout, "Python error while initialising sub interpreter: ");
      PyErr_Print ();
      exit (0);
    }
  py_dict = PyModule_GetDict (m);
  if (PyErr_Occurred ())
    {
      fprintf (stdout, "Python error while initialising sub interpreter: ");
      PyErr_Print ();
    }

  /* python header/initialisation code */
  buf =
    g_strdup_printf ("CORBA._load_idl(\"%s\")",
                     get_global_option_str (config, "rootIDLfile",
                                            DEFAULT_IDL_FILE));
  if (!buf)
    {
      errormsg ("Out of memory");
      return (ERR_MEMORY);
    }
  PyRun_SimpleString ("import sys");
  PyRun_SimpleString ("import method_system");
  PyRun_SimpleString ("import CORBA");
  message (buf);
  PyRun_SimpleString (buf);
  PyRun_SimpleString ("import GEAS");
  g_free (buf);
  if (PyErr_Occurred ())
    {
      printf ("Python error occured:\n");
      fflush (NULL);
      PyErr_Print ();
      abort ();
    }
  mainInterpreterState = mainThreadState->interp;
  py_threadstate = PyThreadState_New (mainInterpreterState);
  PyEval_ReleaseLock ();
  
  return (TRUE);
}

gboolean
load_method_handlers (configuration config)
{
  GList *lst;
  char *dirname;
  char *name;
  odl_field *f;
  struct dirent *entry;
  DIR *dir;
  FILE *fp;
  GList *l, *l2;
  odl_class *c;

  g_return_val_if_fail (config, FALSE);

  PyEval_AcquireLock ();
  PyThreadState_Swap (py_threadstate);

  /* get directory where code files are stored */
  dirname =
    (char *) get_global_option_str (config, "methoddir", DEFAULT_METHOD_DIR);
  if (dirname == NULL)
    {
      return (FALSE);
    }
  dirname = g_strdup (dirname);
  while (dirname[strlen (dirname) - 1] == '\\' ||
         dirname[strlen (dirname) - 1] == '/')
    dirname[strlen (dirname) - 1] = '\0';
  /* message( "method dir name: '%s'" , dirname ); */

  /* execute each .py file */
  dir = opendir (dirname);
  if (!dir)
    {
      errormsg ("Could not open directory '%s'", dirname);
      g_free (dirname);
      return (FALSE);
    }
  while ((entry = readdir (dir)) != NULL)
    {
      if (!strcmp (&entry->d_name[strlen (entry->d_name) - 3], ".py"))
        {
          name = g_strdup_printf ("%s/%s", dirname, entry->d_name);
          if (!name)
            {
              errormsg ("Out of memory");
              closedir (dir);
              g_free (dirname);
              return (FALSE);
            }

          /* run file 'name' */
          fp = fopen (name, "r");
          if (!name)
            {
              errormsg ("Could not open file '%s'", name);
              closedir (dir);
              g_free (dirname);
              g_free (name);
              return (FALSE);
            }
          message ("Loading file %s", name);
          PyRun_SimpleFile (fp, name);

          /* done this file */
          fclose (fp);
          g_free (name);
        }
    }
  closedir (dir);
  g_free (dirname);

  /* done loading, now register */

  lst = l = odl_tree_list_classes (all_classes);
  while (l)
    {
      c = odl_find_class (all_classes, l->data, NULL);
      debug_output (DEBUGLEVEL_2, "    class %s", odl_class_get_name (c));

      l2 = list = odl_class_get_fields (c, FT_method);
      while (l2)
        {
          char *buf;
          f = l2->data;
          debug_output (DEBUGLEVEL_2, "        method: %s",
                        odl_field_get_name (f));

          current_provider =
            alloc_provider_data (odl_class_get_name (c), NULL,
                                 odl_field_get_name (f), NULL);
          provider_list = g_list_append (provider_list, current_provider);

          buf =
            g_strdup_printf
            ("method_system.register_method( \"%s_%s\" , %s_%s )",
             odl_class_get_name (c), odl_field_get_name (f),
             odl_class_get_name (c), odl_field_get_name (f));
          g_strdown (buf);
          PyRun_SimpleString (buf);
          g_free (buf);
          current_provider = NULL;

          l2 = l2->next;
        }
      odl_fieldlist_free (list);

      l = g_list_next (l);
    }
  odl_namelist_free (l);

  PyThreadState_Swap (NULL);
  PyEval_ReleaseLock ();

  return (TRUE);
}

gboolean
shutdown_method_handling (void)
{
  return (TRUE);
}

CORBA_char *
execute_method (GEAS_object_reference * obj,
                const char *methodname, GEAS_Arguments * args,
                CORBA_Environment * ev)
{
  char *retval = NULL;
  int i;
  PyObject *result;
  PyObject *l, *tuple;
  PyObject *pyobj, *server;
  char *resultstr = NULL;
  GEAS_DataObject obj2;
  GEAS_Connection serverobj;

  g_return_val_if_fail (obj, NULL);
  g_return_val_if_fail (methodname, NULL);
  g_return_val_if_fail (args, NULL);
  g_return_val_if_fail (ev, NULL);

  obj2 =
    make_dataobject_reference (obj->classname, obj->objectid, obj->username,
                               obj->sessionid, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    /* didn't work */
    return (NULL);

  serverobj =
    make_connection_reference (obj->username, "server_allocated", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      /* didn't work */
      CORBA_Object_release (obj2, ev);
      return (NULL);
    }

  /* find 'call' object */
  list = provider_list;
  while (list)
    {
      g_message ("%s = %s", ((provider_t *) list->data)->classname, obj->classname);
      g_message ("%s = %s", ((provider_t *) list->data)->methodname, methodname);
      if (!g_strcasecmp
          (((provider_t *) list->data)->classname, obj->classname)
          && !g_strcasecmp (((provider_t *) list->data)->methodname,
                            methodname))
        {
          function = ((provider_t *) list->data)->extra;
          list = NULL;
        }
      if (list)
        list = list->next;
    }
  if (function != 0)
    {
      int i;

      /* lock python interpreter in case it's needed */
      PyEval_AcquireLock ();
      PyThreadState_Swap (py_threadstate);

      pyobj = CORBA_to_Python_func (obj2);
      server = CORBA_to_Python_func (serverobj);

      /* PyObject_Print(pyobj, stdout, 0); */
      l = PyList_New (0);
      PyList_Append (l, pyobj);

      for (i = 0; i < args->_length; i++)
        {
          PyList_Append (l, PyString_FromString (args->_buffer[i]));
        }
      PyList_Append (l, server);
      tuple = PyList_AsTuple (l);

      result = PyObject_CallObject (function, tuple);
      Py_DECREF (pyobj);
      Py_DECREF (server);
      Py_DECREF (l);
      Py_DECREF (tuple);
      if (result != 0)
        {
	  /*	     printf( "result: " );	     PyObject_Print(result, stdout, 0);
		     printf( "\n" ); */

          if (PyString_AsString (result))
            {
              resultstr = g_strdup (PyString_AsString (result));
              Py_DECREF (result);
            }
          else
            {
              fprintf (stderr,
                       "Python methods _must_ return a string value\n");
              abort ();
            }
        }
      else
        {
          PyErr_Print ();
          make_ServerError_exception (ev,
                                      "Failed to call method %s.%s (unknown reason)",
                                      obj->classname, methodname);
          PyThreadState_Swap (NULL);
          PyEval_ReleaseLock ();
          return (NULL);
        }
    }
  else
    make_ServerError_exception (ev, "Couldn't find the function to call");

  /* make method call */
  PyThreadState_Swap (NULL);
  PyEval_ReleaseLock ();

  retval = CORBA_string_dup (resultstr);
  free (resultstr);
  return (retval);
}

static int
load_python_binding (configuration config)
{
  gpointer gsym;
  CORBA_to_Python_func = NULL;

  g_assert (config);

  if (!g_module_supported ())
    {
      errormsg ("glib dynamic module loading is not supported.");
      return (ERR_NO_GMODULE);
    }

  orbit_python_library =
    g_module_open (get_global_option_str
                   (config, "ORBit-python", ORBIT_PYTHON_SO),
                   G_MODULE_BIND_LAZY);

  if (!orbit_python_library)
    {
      errormsg ("Could not load ORBit-Python module.");
      return (ERR_NO_ORBIT_PYTHON);
    }

  g_module_symbol (orbit_python_library, "CORBA_Object_to_PyObject", &gsym);
  CORBA_to_Python_func = gsym;

  if (!CORBA_to_Python_func)
    {
      errormsg ("Could not locate required functions");
      return (ERR_NO_FUNCTIONS);
    }

  return (ERR_OK);
}
