/*
  methods.c - Dummy file if method support is not compiled in.

  Copyright (C) 2001 Free Software Foundation

  This file is part of the GNU Enterprise Application Server (GEAS)

  GEAS is free software; you can redistribute it and/or modify it under
  the terms of the GNU General Public License as published by the Free
  Software Foundation; either version 2, or (at your option) any later
  version.

  GEAS is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
  for more details.

  You should have received a copy of the GNU General Public License
  along with GEAS; if not, write to the Free Software Foundation, Inc.,
  59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

  $Id: methods.c,v 1.11 2001/08/21 21:20:24 reinhard Exp $
*/

#include "config.h"

#include <glib.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "geas-server.h"
#include "classdata.h"
#include "config/configuration.h"
#include "methods.h"
#include "exceptions.h"

/* base data structures and related private utility methods */
typedef struct
{
  char *classname;              /* full class name      */
  char *methodname;             /* name of method field */
  char *cid;
  CORBA_Object object;
  void *extra;                  /* misc */
  void *function;               /* glib */
}
provider_t;

static provider_t *alloc_provider_data (const char *classname,
                                        const char *cid,
                                        const char *methodname, void *extra);

static void free_provider_data (provider_t * p);

#if USE_OAF
/* use the OAF technique for locating objects */
#include "methods_oaf.h"
#elif USE_PYTHON_METHODS
/* use a somewhat unreliable fallback technique */
#include "methods_unstable.h"
#elif USE_GLIBMODULE_METHODS
/* loads glib modules as method sources */
#include "methods_glibmodule.h"
#else

/* default functions, if no method code system is selected */

gboolean
initialise_method_handling (configuration config)
{
  /* this doesn't really do much exciting stuff */
  criticalerror ("No method code handling has been compiled.");
  message ("No business class methods will work.");
  return (FALSE);
}

gboolean
load_method_handlers (configuration config)
{
  /* can't load if we're not ready, and we can't be ready (see above :) */
  return (FALSE);
}

gboolean
shutdown_method_handling (void)
{
  /* no code here */
  return (FALSE);
}

/* assumes everything has been validated already */
CORBA_char *
execute_method (GEAS_object_reference * obj,
                const char *methodname, GEAS_Arguments * args,
                CORBA_Environment * ev)
{
  /* hi, mom! */
  fatal_error ("No method code handling has been compiled.");
  return (NULL);
}

#endif


static provider_t *
alloc_provider_data (const char *classname, const char *cid,
                     const char *methodname, void *extra)
{
  provider_t *p = g_new0 (provider_t, 1);
  if (p)
    {
      p->classname = NULL;
      p->cid = NULL;
      p->methodname = NULL;
      p->object = CORBA_OBJECT_NIL;
      p->extra = NULL;
      p->function = NULL;

      if (classname)
        {
          p->classname = g_strdup (classname);
          if (!p->classname)
            {
              free_provider_data (p);
              return (NULL);
            }
        }
      if (cid)
        {
          p->cid = g_strdup (cid);
          if (!p->cid)
            {
              free_provider_data (p);
              return (NULL);
            }
        }
      if (methodname)
        {
          p->methodname = g_strdup (methodname);
          if (!p->methodname)
            {
              free_provider_data (p);
              return (NULL);
            }
        }
      return (p);
    }
  return (NULL);
}

static void
free_provider_data (provider_t * p)
{
  CORBA_Environment ev;         /* really just ignoring exceptions */

  CORBA_exception_init (&ev);
  if (p)
    {
      if (p->classname)
        g_free (p->classname);
      if (p->cid)
        g_free (p->cid);
      if (p->methodname)
        g_free (p->methodname);
      if (!CORBA_Object_is_nil (p->object, &ev))
        {
          CORBA_Object_release (p->object, &ev);
        }
      g_free (p);
    }
  CORBA_exception_free (&ev);
}
