/*
   geas-skeleton.c -
   
   Part of GNU Enterprise Application Server (GEAS)

   Copyright (C) 2000-2001 Free Software Foundation
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
   
   $Id: geas-skeleton.c,v 1.68 2001/07/22 21:16:45 reinhard Exp $
 
*/

/** \file geas-skeleton.c
 * \brief CORBA interface code and GEAS object reference code
 * Most of the functions in this file are called by the ORBit generated
 * skeleton. They all perform a similar task - extract the object ID,
 * and call an appropriate implementation function., then free the
 * identifier once finished. additionally, the function that performs this
 * operation also identifies any active transaction for the user, and 
 * makes a note of the transaction's id.
 */

#include <ctype.h>

#include "config.h"

#include <glib.h>
#include <stdio.h>
#include <time.h>
#include <sys/times.h>
#include <uuid/uuid.h>

#include "geas-server.h"
#include "objectcache/objectcache.h"
#include "geas.h"
#include "geas-skeleton.h"
#include "geas-skelimpl.h"
#include "exceptions.h"

#include "factory.h"
#include "objectlist.h"
#include "dataobject.h"
#include "admin.h"
#include "transaction.h"
#include "connection.h"
#include "query.h"

#ifdef USE_SERVANT_MANAGER
#include "servantlocator.h"
#endif

/** \brief Data used by ORBit to manage object references */
struct poa_data_s
{
  PortableServer_POA transaction_poa;
  PortableServer_POA connection_poa;
  PortableServer_POA dataobject_poa;
  PortableServer_POA objectlist_poa;
  PortableServer_POA admin_poa;
  PortableServer_POA query_poa;
  PortableServer_POA constraint_poa;
};

struct poa_data_s poa_data;

static GEAS_object_reference *alloc_geas_object_reference (void);
static GEAS_object_reference *parse_object_id (const char *id);
ObjectKey old_create_new_object_key (void);

gboolean allow_oid_refill = TRUE;

/* private function: convert current object ID to a GEAS_object_reference */

/** \brief Update GEAS object reference to reflect current transaction
 * After identifying the user associated with an operation, this function
 * identifies the status of the current transaction, if any.
 */

static void
get_users_current_transaction_status (GEAS_object_reference * id)
{
  /* QueryData *query = NULL; */
  /* ObjectData *user=NULL; */
  /* ObjectData *transaction=NULL; */
  /* char *transactionid = NULL; */
  /* char *open; */

  id->transactionopen = FALSE;
  id->currenttransaction = NULL;
  return;
}

/** \brief get all identifiers for the current object
 * When an operation is performed on an object, this function is called to
 * extract the ID from the object reference.
 * 
 * Additionally, it looks up the user record for the username associated with the
 * object reference, and can look up additional data - currently, this is the
 * identifier of the currently active transaction, if any.
 * 
 * TODO: use the CORBA::Current interface, once implemented in ORBit
 * to extract the ID directly. In combination with the security system, this
 * will mean usernames do not have to be included in object references.
 */
static GEAS_object_reference *
get_current_object_reference (PortableServer_Servant servant,
                              PortableServer_POA poa, CORBA_Environment * ev)
{
  PortableServer_ObjectId *id;
  GEAS_object_reference *retval = NULL;

  /* convert it to an ID */
  timer_start_operation(TIMER_SKELMISC,"get_current_object_reference");
  id = PortableServer_POA_servant_to_id (poa, servant, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      timer_fail_operation( TIMER_SKELETON );
      return (NULL);
    }

  retval = corba_id_to_geas_reference (id, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      if (id)
        CORBA_free (id);
      timer_fail_operation( TIMER_SKELMISC );
      return (NULL);
    }
  if (!retval)
    {
      timer_fail_operation( TIMER_SKELMISC );
      return (NULL);
    }

  if (retval->username != NULL)
    {
      /* make a server reference, so we can talk to the server */
      retval->server =
        make_connection_reference (retval->username,
                                   "7b33d03c-547b-46b4-bd7a-01f1e4ffc73f",
                                   ev);
      retval->serverid = alloc_geas_object_reference ();
      if (retval->serverid)
        {
          retval->serverid->username = g_strdup (retval->username);
        }

      /* if a real user, look for transaction ID */
#if 0
      if (strcmp (retval->username, SERVER_PRIVATE_USERNAME) != 0)
        {
          get_users_current_transaction_status (retval);
        }
#endif
    }

  /* Free extra memory */
  if (id)
    CORBA_free (id);

  /* return new ID data */
  timer_done_operation( TIMER_SKELMISC );
  return (retval);
}

/* ZYX ------------------------------- */

static void
impl_GEAS_Query_Constraint__destroy (impl_POA_GEAS_Query_Constraint *
                                     servant, CORBA_Environment * ev)
{
  /* NOT USED */
}

static GEAS_ObjectReference
impl_GEAS_Query_Query__get_id (impl_POA_GEAS_Query_Query * servant,
                               CORBA_Environment * ev)
{
  GEAS_ObjectReference retval = NULL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();
  timer_start_operation(TIMER_SKELETON,"Query::Query::_get_id");
  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }
  retval = Query__get_id (id, ev);
  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static GEAS_Query_logictype
impl_GEAS_Query_Constraint__get_logic (impl_POA_GEAS_Query_Constraint *
                                       servant, CORBA_Environment * ev)
{
  GEAS_Query_logictype retval = 0;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = Constraint__get_logic (id, ev);

  free_geas_object_reference (id);

  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static void
impl_GEAS_Query_Constraint__set_logic (impl_POA_GEAS_Query_Constraint *
                                       servant, GEAS_Query_logictype value,
                                       CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  Constraint__set_logic (id, value, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

static void
impl_GEAS_Query_Constraint_addField (impl_POA_GEAS_Query_Constraint * servant,
                                     GEAS_Query_Field * field,
                                     CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  Constraint_addField (id, field, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

static void
impl_GEAS_Query_Query__destroy (impl_POA_GEAS_Query_Query *
                                servant, CORBA_Environment * ev)
{
  /* NOT USED */
}

static void
impl_GEAS_Query_Query_release (impl_POA_GEAS_Query_Query * servant,
                               CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  Query_release (id, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

static void
impl_GEAS_Query_Query_clear (impl_POA_GEAS_Query_Query * servant,
                             CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  Query_clear (id, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

static GEAS_Query_Constraint
impl_GEAS_Query_Query_newConstraint (impl_POA_GEAS_Query_Query * servant,
                                     CORBA_Environment * ev)
{
  GEAS_Query_Constraint retval;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (CORBA_OBJECT_NIL);
  }

  retval = Constraint_newConstraint (id, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return (retval);
}

static GEAS_Query_Constraint
impl_GEAS_Query_Constraint_newConstraint (impl_POA_GEAS_Query_Constraint *
                                          servant, CORBA_Environment * ev)
{
  GEAS_Query_Constraint retval;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (CORBA_OBJECT_NIL);
  }

  retval = Constraint_newConstraint (id, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return (retval);
}

static CORBA_char *
impl_GEAS_Query_Query__get_orderby (impl_POA_GEAS_Query_Query * servant,
                                    CORBA_Environment * ev)
{
  CORBA_char *retval = NULL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = Query__get_orderby (id, ev);

  free_geas_object_reference (id);

  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static void
impl_GEAS_Query_Query__set_orderby (impl_POA_GEAS_Query_Query *
                                    servant, CORBA_char * value,
                                    CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  Query__set_orderby (id, value, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

static CORBA_boolean
impl_GEAS_Query_Query__get_reverse (impl_POA_GEAS_Query_Query * servant,
                                    CORBA_Environment * ev)
{
  CORBA_boolean retval = CORBA_FALSE;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = Query__get_reverse (id, ev);

  free_geas_object_reference (id);

  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static void
impl_GEAS_Query_Query__set_reverse (impl_POA_GEAS_Query_Query * servant,
                                    CORBA_boolean value,
                                    CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  Query__set_reverse (id, value, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

static GEAS_Query_logictype
impl_GEAS_Query_Query__get_logic (impl_POA_GEAS_Query_Query * servant,
                                  CORBA_Environment * ev)
{
  GEAS_Query_logictype retval = GEAS_Query_UNKNOWN;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = Constraint__get_logic (id, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return (retval);
}

static void
impl_GEAS_Query_Query__set_logic (impl_POA_GEAS_Query_Query * servant,
                                  GEAS_Query_logictype value,
                                  CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  Constraint__set_logic (id, value, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

static CORBA_char *
impl_GEAS_Query_Query__get_classname (impl_POA_GEAS_Query_Query * servant,
                                      CORBA_Environment * ev)
{
  CORBA_char *retval = NULL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = Query__get_classname (id, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return (retval);
}

static void
impl_GEAS_Query_Query__set_classname (impl_POA_GEAS_Query_Query * servant,
                                      CORBA_char * value,
                                      CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  Query__set_classname (id, value, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

static void
impl_GEAS_Query_Query_addField (impl_POA_GEAS_Query_Query * servant,
                                GEAS_Query_Field * field,
                                CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  Constraint_addField (id, field, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

/* XYZ ------------------------------- */

static CORBA_long
impl_GEAS_Admin__get_debuglevel (impl_POA_GEAS_Admin * servant,
                                 CORBA_Environment * ev)
{
  CORBA_long retval = 0;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = Admin__get_debuglevel (id, ev);

  free_geas_object_reference (id);

  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static void
impl_GEAS_Admin__set_debuglevel (impl_POA_GEAS_Admin * servant,
                                 CORBA_long value, CORBA_Environment * ev)
{
  GEAS_object_reference *id = NULL;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  Admin__set_debuglevel (id, value, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

static CORBA_boolean
impl_GEAS_Admin__get_verbose (impl_POA_GEAS_Admin * servant,
                              CORBA_Environment * ev)
{
  CORBA_boolean retval = FALSE;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = Admin__get_verbose (id, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return (retval);
}

static void
impl_GEAS_Admin__set_verbose (impl_POA_GEAS_Admin * servant,
                              CORBA_boolean value, CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

static void
impl_GEAS_Admin_restart (impl_POA_GEAS_Admin * servant,
                         CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  Admin_restart (id, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

static void
impl_GEAS_Admin_shutdown (impl_POA_GEAS_Admin * servant,
                          CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  Admin_shutdown (id, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

static CORBA_unsigned_long
impl_GEAS_Connection_countAllInstances (impl_POA_GEAS_Connection * servant,
                                        CORBA_char * classname,
                                        CORBA_Environment * ev)
{
  CORBA_unsigned_long retval = 0;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = Connection_countAllInstances (id, classname, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return (retval);
}

static CORBA_char *
impl_GEAS_Connection__get_currentUserID (impl_POA_GEAS_Connection * servant,
                                         CORBA_Environment * ev)
{
  CORBA_char *retval = NULL;    /* default value */
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = Connection__get_currentUserID (id, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return (retval);
}

static GEAS_classnames *
impl_GEAS_Connection__get_classes (impl_POA_GEAS_Connection * servant,
                                   CORBA_Environment * ev)
{
  GEAS_classnames *retval = NULL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = Connection__get_classes (id, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static GEAS_classlist *
impl_GEAS_Connection__get_schema (impl_POA_GEAS_Connection * servant,
                                  CORBA_Environment * ev)
{
  GEAS_classlist *retval = NULL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = Connection__get_schema (id, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static GEAS_Query_Query
impl_GEAS_Connection_newQuery (impl_POA_GEAS_Connection * servant,
                               CORBA_Environment * ev)
{
  GEAS_Query_Query retval = NULL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = Connection_newQuery (id, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static GEAS_ObjectList
impl_GEAS_Connection_executeQuery (impl_POA_GEAS_Connection * servant,
                                   GEAS_Query_Query querytoperform,
                                   CORBA_Environment * ev)
{
  GEAS_ObjectList retval = NULL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  allow_oid_refill = FALSE;
  retval = Connection_executeQuery (id, querytoperform, ev);
  allow_oid_refill = TRUE;

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static GEAS_ClassDefinition *
impl_GEAS_Connection_getClassDefinition (impl_POA_GEAS_Connection * servant,
                                         CORBA_char * name,
                                         CORBA_Environment * ev)
{
  GEAS_ClassDefinition *retval = NULL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = Connection_getClassDefinition (id, name, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static GEAS_ClassDefinition *
impl_GEAS_Connection_getFullClassDefinition (impl_POA_GEAS_Connection *
                                             servant, CORBA_char * name,
                                             CORBA_Environment * ev)
{
  GEAS_ClassDefinition *retval = NULL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = Connection_getFullClassDefinition (id, name, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static GEAS_LookupOptions *
impl_GEAS_Connection_getLookupFieldOptions (impl_POA_GEAS_Connection *
                                            servant, CORBA_char * classname,
                                            CORBA_char * fieldname,
                                            CORBA_Environment * ev)
{
  GEAS_LookupOptions *retval = NULL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = Connection_getLookupFieldOptions (id, classname, fieldname, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static CORBA_boolean
impl_GEAS_Connection_supportsFeature (impl_POA_GEAS_Connection * servant,
                                      GEAS_Feature featureID,
                                      CORBA_Environment * ev)
{
  CORBA_boolean retval = CORBA_FALSE;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = Connection_supportsFeature (id, featureID, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

/*
 * this is just another nasty hack. it's not my fault, and I did not want
 * this.  go bother someone else if it doesn't make sense
 */
static CORBA_boolean
impl_GEAS_DataObject__get_inDatabase (impl_POA_GEAS_DataObject * servant,
                                      CORBA_Environment * ev)
{
  CORBA_boolean retval = CORBA_FALSE;
  GEAS_object_reference *id = NULL;
  ObjectData *ob;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();
  if (allow_oid_refill)
    refill_oid_buffer (5);
  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  ob = oc_find_object_by_key (id->classname, id->objectid);
  if (ob)
    {
    
      if (oc_get_object_flag (ob, of_indatabase) == FLAG_ON)
        retval = CORBA_TRUE;
    }

  timer_done_operation( TIMER_SKELETON );
#ifdef STUPID_CRAP
printf( "in database :- %s/%s : %d   (T:%d  F:%d)\n" , id->classname, id->objectid , retval , CORBA_TRUE,CORBA_FALSE );
   {
   char xyz[64];
   printf( "press enter\n" );
   gets(xyz);
   }
#endif
  free_geas_object_reference (id);
  return retval;
}

static GEAS_DataObject
impl_GEAS_Connection_newObject (impl_POA_GEAS_Connection * servant,
                                CORBA_char * classname,
                                CORBA_Environment * ev)
{
  GEAS_DataObject retval = CORBA_OBJECT_NIL;
  GEAS_object_reference *id = NULL;

  timer_start_operation(TIMER_SKELETON,"newObject(%s)",classname);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = Connection_newObject (id, classname, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static GEAS_ObjectList
impl_GEAS_Connection_newEmptyList (impl_POA_GEAS_Connection * servant,
                                   CORBA_char * classname,
                                   CORBA_Environment * ev)
{
  GEAS_DataObject retval = CORBA_OBJECT_NIL;
  GEAS_object_reference *id = NULL;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = Connection_newEmptyList (id, classname, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}


static GEAS_ObjectList
impl_GEAS_Connection_loadObjects (impl_POA_GEAS_Connection * servant,
                                  CORBA_char * classname,
                                  CORBA_char * fieldname,
                                  CORBA_char * searchvalue,
                                  CORBA_Environment * ev)
{
  GEAS_ObjectList retval = CORBA_OBJECT_NIL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  allow_oid_refill = FALSE;
  retval = Connection_loadObjects (id, classname, fieldname, searchvalue, ev);
  allow_oid_refill = TRUE;

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

int sillytest = 0;

static GEAS_ObjectList
impl_GEAS_Connection_loadAll (impl_POA_GEAS_Connection * servant,
                              CORBA_char * classname, CORBA_Environment * ev)
{
  GEAS_ObjectList retval = CORBA_OBJECT_NIL;
  GEAS_object_reference *id;

#ifdef TIMER_TESTING
  /* yet another quick hack - isn't this fun? */
  if( strcmp(classname,"averages") == 0 )
   {
     timer_display_profiles();
     make_ServerError_exception(ev,"not really an error");
     return retval;
   }
  if( strcmp(classname,"start") == 0 )
   {
     sillytest = 1;
     timer_start_profile(TIMER_FUNC_OPERATION);
     make_ServerError_exception(ev,"not really an error");
     return retval;
   }
  if( strcmp(classname,"done") == 0 )
   {
     timer_update_profile(TIMER_FUNC_OPERATION);
     make_ServerError_exception(ev,"not really an error");
     return retval;
   }
#endif

  timer_start_operation(TIMER_SKELETON,"loadAll(%s)",classname);
  timer_start_profile( TIMER_FUNC_LOADALL );
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);

  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  allow_oid_refill = FALSE;
  retval = Connection_loadAll (id, classname, ev);
  allow_oid_refill = TRUE;

  free_geas_object_reference (id);
  timer_update_profile( TIMER_FUNC_LOADALL );
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static GEAS_DataObject
impl_GEAS_Connection_loadSingleObject (impl_POA_GEAS_Connection * servant,
                                       CORBA_char * classname,
                                       CORBA_char * fieldname,
                                       CORBA_char * searchvalue,
                                       CORBA_Environment * ev)
{
  GEAS_DataObject retval = CORBA_OBJECT_NIL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval =
    Connection_loadSingleObject (id, classname, fieldname, searchvalue, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

/* ------------------------ */

/* raises: */
static GEAS_Transaction
impl_GEAS_Connection_new (impl_POA_GEAS_Connection * servant,
                          CORBA_Environment * ev)
{
  GEAS_Transaction retval = CORBA_OBJECT_NIL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = TransactionFactory_new (id, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

/* raises: */
static GEAS_Transaction
impl_GEAS_Connection_current (impl_POA_GEAS_Connection * servant,
                              CORBA_Environment * ev)
{
  GEAS_Transaction retval = CORBA_OBJECT_NIL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = TransactionFactory_current (id, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

/* ------------------------ */

/* raises: */
static void
impl_GEAS_TransactionFactory__destroy (impl_POA_GEAS_TransactionFactory *
                                       servant, CORBA_Environment * ev)
{
  /* empty function */
}

/* raises: */
static GEAS_Transaction
impl_GEAS_TransactionFactory_new (impl_POA_GEAS_TransactionFactory * servant,
                                  CORBA_Environment * ev)
{
  GEAS_Transaction retval = CORBA_OBJECT_NIL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = TransactionFactory_new (id, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

/* raises: */
static GEAS_Transaction
impl_GEAS_TransactionFactory_current (impl_POA_GEAS_TransactionFactory *
                                      servant, CORBA_Environment * ev)
{
  GEAS_Transaction retval = CORBA_OBJECT_NIL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = TransactionFactory_current (id, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

/* ------------------------ */

/* raises: */
static void
impl_GEAS_Transaction__destroy (impl_POA_GEAS_Transaction *
                                servant, CORBA_Environment * ev)
{
  /* empty function */
}

/* raises: TransactionNotInProgress */
static void
impl_GEAS_Transaction_begin (impl_POA_GEAS_Transaction * servant,
                             CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  Transaction_begin (id, ev);
  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

/* raises: TransactionNotInProgress */
static void
impl_GEAS_Transaction_commit (impl_POA_GEAS_Transaction * servant,
                              CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  Transaction_commit (id, ev);
  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

/* raises: TransactionNotInProgress */
static void
impl_GEAS_Transaction_abort (impl_POA_GEAS_Transaction * servant,
                             CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  Transaction_abort (id, ev);
  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

/* raises: TransactionNotInProgress */
static void
impl_GEAS_Transaction_checkpoint (impl_POA_GEAS_Transaction * servant,
                                  CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  Transaction_checkpoint (id, ev);
  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

/* raises: TransactionNotInProgress */
static void
impl_GEAS_Transaction_join (impl_POA_GEAS_Transaction * servant,
                            CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  Transaction_join (id, ev);
  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

/* raises: TransactionNotInProgress */
static void
impl_GEAS_Transaction_leave (impl_POA_GEAS_Transaction * servant,
                             CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  Transaction_leave (id, ev);
  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

/* raises: */
static CORBA_boolean
impl_GEAS_Transaction_isOpen (impl_POA_GEAS_Transaction * servant,
                              CORBA_Environment * ev)
{
  CORBA_boolean retval = CORBA_FALSE;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = Transaction_isOpen (id, ev);
  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

/* ------------------------ */

static void
impl_GEAS_Connection_flush (impl_POA_GEAS_Connection * servant,
                            CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  if (allow_oid_refill)
    refill_oid_buffer (5);

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }
  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

static void
impl_GEAS_Connection_logout (impl_POA_GEAS_Connection * servant,
                             CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  Connection_logout (id, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

static GEAS_LoadDefinition *
impl_GEAS_Connection_getReferenceDefinition (impl_POA_GEAS_Connection *
                                             servant, CORBA_char * classname,
                                             CORBA_char * fieldname,
                                             CORBA_Environment * ev)
{
  GEAS_object_reference *id;
  GEAS_LoadDefinition *retval = NULL;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = Connection_getReferenceDefinition (id, classname, fieldname, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return (retval);
}

static void
impl_GEAS_Connection_clearCache (impl_POA_GEAS_Connection * servant,
                                 CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();
  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  Connection_clearCache (id, ev);
  free_geas_object_reference (id);
  if (allow_oid_refill)
    refill_oid_buffer (1024 * 1024);
  timer_done_operation( TIMER_SKELETON );
}

static void
impl_GEAS_Connection_activateEventLog (impl_POA_GEAS_Connection * servant,
                                       CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();
  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  Connection_activateEventLog (id, ev);
  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

static GEAS_Changes *
impl_GEAS_Connection__get_latestChanges (impl_POA_GEAS_Connection *
                                         servant, CORBA_Environment * ev)
{
  GEAS_object_reference *id = NULL;
  GEAS_Changes *retval = NULL;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();
  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = Connection__get_latestChanges (id, ev);
  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return (retval);
}

static CORBA_boolean
impl_GEAS_DataObject_identity (impl_POA_GEAS_DataObject * servant,
                               CORBA_char * classname, CORBA_char * oid,
                               CORBA_Environment * ev)
{
  CORBA_boolean retval = CORBA_FALSE;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  if (g_strcasecmp (classname, id->classname) == 0 &&
      strcmp (id->objectid, oid) == 0)
    retval = CORBA_TRUE;
  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}


static GEAS_ObjectReference
impl_GEAS_DataObject__get_objectID (impl_POA_GEAS_DataObject * servant,
                                    CORBA_Environment * ev)
{
  GEAS_ObjectReference retval = NULL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = DataObject__get_objectID (id, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static CORBA_boolean
impl_GEAS_DataObject_ofclass (impl_POA_GEAS_DataObject * servant,
                              CORBA_char * classname, CORBA_Environment * ev)
{
  CORBA_boolean retval = FALSE;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = DataObject_ofclass (id, classname, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static CORBA_char *
impl_GEAS_DataObject__get_classname (impl_POA_GEAS_DataObject * servant,
                                     CORBA_Environment * ev)
{
  CORBA_char *retval = NULL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = DataObject__get_classname (id, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static GEAS_classnames *
impl_GEAS_DataObject__get_classes (impl_POA_GEAS_DataObject * servant,
                                   CORBA_Environment * ev)
{
  GEAS_classnames *retval = NULL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = DataObject__get_classes (id, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static GEAS_fieldlist *
impl_GEAS_DataObject__get_allfields (impl_POA_GEAS_DataObject * servant,
                                     CORBA_Environment * ev)
{
  GEAS_fieldlist *retval = NULL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = DataObject__get_allfields (id, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static GEAS_LookupOptions *
impl_GEAS_DataObject_getLookupFieldOptions (impl_POA_GEAS_DataObject *
                                            servant, CORBA_char * fieldname,
                                            CORBA_Environment * ev)
{
  GEAS_LookupOptions *retval = NULL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = DataObject_getLookupFieldOptions (id, fieldname, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static void
impl_GEAS_DataObject_setLookupField (impl_POA_GEAS_DataObject * servant,
                                     GEAS_LookupOption * setting,
                                     CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  DataObject_setLookupField (id, setting, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

static CORBA_char *
impl_GEAS_DataObject_getField (impl_POA_GEAS_DataObject * servant,
                               CORBA_char * fieldname, CORBA_Environment * ev)
{
  CORBA_char *retval = NULL;
  GEAS_object_reference *id;

  trace_functioncall ();

//  if (allow_oid_refill)
//   refill_oid_buffer (5);

//if( sillytest == 0 )
  id = get_current_object_reference (servant, servant->poa, ev);

  timer_start_operation(TIMER_SKELETON,"%s.getField(%s)",id->classname,fieldname);
  timer_start_profile( TIMER_FUNC_GETFIELD );

  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

//if( sillytest == 0 )
    retval = DataObject_getField (id, fieldname, ev);
//else
//    retval = CORBA_string_dup("hello there");

//if( sillytest == 0 )
  free_geas_object_reference (id);
  timer_update_profile( TIMER_FUNC_GETFIELD );
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static void
impl_GEAS_DataObject_setField (impl_POA_GEAS_DataObject * servant,
                               CORBA_char * fieldname,
                               CORBA_char * value, CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,"setField(%s,%s)",fieldname,value);
  trace_functioncall ();
  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  if (allow_oid_refill)
    refill_oid_buffer (5);

  DataObject_setField (id, fieldname, value, ev);

  free_geas_object_reference (id);

  trace_functiondone ();
  timer_done_operation( TIMER_SKELETON );
}

static GEAS_DataObject
impl_GEAS_DataObject_getReference (impl_POA_GEAS_DataObject * servant,
                                   CORBA_char * fieldname,
                                   CORBA_Environment * ev)
{
  GEAS_DataObject retval = NULL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = DataObject_getReference (id, fieldname, ev);
  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static GEAS_ObjectList
impl_GEAS_DataObject_getList (impl_POA_GEAS_DataObject * servant,
                              CORBA_char * fieldname, CORBA_Environment * ev)
{
  GEAS_ObjectList retval = NULL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  allow_oid_refill = FALSE;
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = DataObject_getList (id, fieldname, ev);

  free_geas_object_reference (id);
  allow_oid_refill = TRUE;
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static void
impl_GEAS_DataObject_insert (impl_POA_GEAS_DataObject * servant,
                             CORBA_char * fieldname,
                             GEAS_DataObject obj, CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  DataObject_insert (id, fieldname, obj, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

static void
impl_GEAS_DataObject_setReference (impl_POA_GEAS_DataObject * servant,
                                   CORBA_char * fieldname,
                                   GEAS_DataObject obj,
                                   CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  DataObject_setReference (id, fieldname, obj, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

static void
impl_GEAS_DataObject_flush (impl_POA_GEAS_DataObject * servant,
                            CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  if (allow_oid_refill)
    refill_oid_buffer (5);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

static void
impl_GEAS_DataObject_delete (impl_POA_GEAS_DataObject * servant,
                             CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  DataObject_delete (id, ev);

  if (allow_oid_refill)
    refill_oid_buffer (5);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

static CORBA_char *
impl_GEAS_DataObject_callMethod (impl_POA_GEAS_DataObject * servant,
                                 CORBA_char * methodname,
                                 GEAS_Arguments * args,
                                 CORBA_Environment * ev)
{
  CORBA_char *retval = NULL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  if (allow_oid_refill)
    refill_oid_buffer (5);

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = DataObject_callMethod (id, methodname, args, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static CORBA_unsigned_long
impl_GEAS_ObjectList__get_length (impl_POA_GEAS_ObjectList * servant,
                                  CORBA_Environment * ev)
{
  CORBA_unsigned_long retval = 0;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  if (allow_oid_refill)
    refill_oid_buffer (1);

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = ObjectList__get_length (id, ev);
  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static GEAS_objectslist *
impl_GEAS_ObjectList__get_objects (impl_POA_GEAS_ObjectList * servant,
                                   CORBA_Environment * ev)
{
  GEAS_objectslist *retval = NULL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  timer_start_profile( TIMER_FUNC_GET_OBJECTS );
  trace_functioncall ();

  if (allow_oid_refill)
    refill_oid_buffer (5);

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = ObjectList__get_objects (id, ev);

  free_geas_object_reference (id);
  timer_update_profile( TIMER_FUNC_GET_OBJECTS );
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static CORBA_char *
impl_GEAS_ObjectList__get_classname (impl_POA_GEAS_ObjectList * servant,
                                     CORBA_Environment * ev)
{
  CORBA_char *retval = NULL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = ObjectList__get_classname (id, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static GEAS_DataObject
impl_GEAS_ObjectList_getEntry (impl_POA_GEAS_ObjectList * servant,
                               CORBA_unsigned_long index,
                               CORBA_Environment * ev)
{
  GEAS_DataObject retval = NULL;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  timer_start_profile(TIMER_FUNC_GETENTRY);
  trace_functioncall ();

  if (allow_oid_refill)
    refill_oid_buffer (2);

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    timer_update_profile(TIMER_FUNC_GETENTRY);
    return (retval);
  }

  retval = ObjectList_getEntry (id, index, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  timer_update_profile(TIMER_FUNC_GETENTRY);
  return retval;
}

static void
impl_GEAS_ObjectList_appendObject (impl_POA_GEAS_ObjectList * servant,
                                   GEAS_DataObject obj,
                                   CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  ObjectList_appendObject (id, obj, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

static void
impl_GEAS_ObjectList_removeEntry (impl_POA_GEAS_ObjectList * servant,
                                  CORBA_unsigned_long index,
                                  CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  ObjectList_removeEntry (id, index, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

static void
impl_GEAS_ObjectList_resort (impl_POA_GEAS_ObjectList * servant,
                             CORBA_char * fieldname,
                             CORBA_boolean ascending, CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  ObjectList_resort (id, fieldname, ascending, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

static CORBA_unsigned_long
impl_GEAS_ObjectList_hideObjects (impl_POA_GEAS_ObjectList * servant,
                                  CORBA_char * fieldname,
                                  CORBA_char * compare,
                                  CORBA_char * value, CORBA_Environment * ev)
{
  CORBA_unsigned_long retval = 0;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = ObjectList_hideObjects (id, fieldname, compare, value, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static CORBA_unsigned_long
impl_GEAS_ObjectList_showObjects (impl_POA_GEAS_ObjectList * servant,
                                  CORBA_char * fieldname,
                                  CORBA_char * compare,
                                  CORBA_char * value, CORBA_Environment * ev)
{
  CORBA_unsigned_long retval = 0;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = ObjectList_showObjects (id, fieldname, compare, value, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static void
impl_GEAS_ObjectList_showAll (impl_POA_GEAS_ObjectList * servant,
                              CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

static CORBA_unsigned_long
impl_GEAS_ObjectList_applyFilter (impl_POA_GEAS_ObjectList * servant,
                                  CORBA_Environment * ev)
{
  CORBA_unsigned_long retval = 0;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = ObjectList_applyFilter (id, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static CORBA_boolean
impl_GEAS_ObjectList__get_delayFilter (impl_POA_GEAS_ObjectList * servant,
                                       CORBA_Environment * ev)
{
  CORBA_boolean retval = FALSE;
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return (retval);
  }

  retval = ObjectList__get_delayFilter (id, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static void
impl_GEAS_ObjectList__set_delayFilter (impl_POA_GEAS_ObjectList * servant,
                                       CORBA_boolean value,
                                       CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

static void
impl_GEAS_ObjectList_release (impl_POA_GEAS_ObjectList * servant,
                              CORBA_Environment * ev)
{
  GEAS_object_reference *id;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  id = get_current_object_reference (servant, servant->poa, ev);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_fail_operation( TIMER_SKELETON );
    return;
  }

  ObjectList_release (id, ev);

  free_geas_object_reference (id);
  timer_done_operation( TIMER_SKELETON );
}

GEAS_ConnectionFactory
impl_GEAS_ConnectionFactory__create (PortableServer_POA poa,
                                     CORBA_Environment * ev)
{
  GEAS_ConnectionFactory retval;
  impl_POA_GEAS_ConnectionFactory *newservant;
  PortableServer_ObjectId *objid;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  newservant = g_new0 (impl_POA_GEAS_ConnectionFactory, 1);
  newservant->servant.vepv = &impl_GEAS_ConnectionFactory_vepv;
  newservant->poa = poa;
  POA_GEAS_ConnectionFactory__init ((PortableServer_Servant) newservant, ev);
  objid = PortableServer_POA_activate_object (poa, newservant, ev);
  CORBA_free (objid);
  retval = PortableServer_POA_servant_to_reference (poa, newservant, ev);

  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static void
impl_GEAS_ConnectionFactory__destroy (impl_POA_GEAS_ConnectionFactory *
                                      servant, CORBA_Environment * ev)
{
  PortableServer_ObjectId *objid;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  objid = PortableServer_POA_servant_to_id (servant->poa, servant, ev);
  PortableServer_POA_deactivate_object (servant->poa, objid, ev);
  CORBA_free (objid);

  POA_GEAS_ConnectionFactory__fini ((PortableServer_Servant) servant, ev);
  g_free (servant);
}

static GEAS_Connection
impl_GEAS_ConnectionFactory_guestLogin (impl_POA_GEAS_ConnectionFactory *
                                        servant, CORBA_Environment * ev)
{
  GEAS_Connection retval;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  retval = ConnectionFactory_guestLogin (ev);

  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static GEAS_Connection
impl_GEAS_ConnectionFactory_unsecureLogin (impl_POA_GEAS_ConnectionFactory *
                                           servant,
                                           CORBA_char *
                                           username,
                                           CORBA_char *
                                           password, CORBA_Environment * ev)
{
  GEAS_Connection retval;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  retval = ConnectionFactory_unsecureLogin (username, password, ev);
  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static CORBA_char *impl_GEAS_ConnectionFactory_getAuthenticationKey
  (impl_POA_GEAS_ConnectionFactory * servant, CORBA_char * username,
   CORBA_Environment * ev)
{
  CORBA_char *retval;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  retval = ConnectionFactory_getAuthenticationKey (username, ev);

  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static GEAS_Connection
impl_GEAS_ConnectionFactory_getConnection (impl_POA_GEAS_ConnectionFactory *
                                           servant, CORBA_char * username,
                                           CORBA_char * passkey,
                                           CORBA_Environment * ev)
{
  GEAS_Connection retval;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  retval = ConnectionFactory_getConnection (username, passkey, ev);

  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static GEAS_Admin
impl_GEAS_ConnectionFactory_adminLogin (impl_POA_GEAS_ConnectionFactory *
                                        servant, CORBA_char * username,
                                        CORBA_char * passkey,
                                        CORBA_Environment * ev)
{
  GEAS_Admin retval;

  timer_start_operation(TIMER_SKELETON,__PRETTY_FUNCTION__);
  trace_functioncall ();

  retval = ConnectionFactory_adminLogin (username, passkey, ev);

  timer_done_operation( TIMER_SKELETON );
  return retval;
}

static CORBA_boolean
impl_GEAS_MethodProvider_verify (impl_POA_GEAS_MethodProvider * servant,
                                 CORBA_char * name, CORBA_long argcount,
                                 CORBA_Environment * ev)
{
  /* TODO not used (yet?) */
  return FALSE;
}

static CORBA_char *
impl_GEAS_MethodProvider_execute (impl_POA_GEAS_MethodProvider * servant,
                                  CORBA_char * name, GEAS_Arguments * args,
                                  CORBA_Environment * ev)
{
  /* TODO another one that's not used (yet?) */
  return NULL;
}

/******************************************************************/

POA_GEAS_Transaction poa_geas_transaction_servant =
  { NULL, &impl_GEAS_Transaction_vepv };
POA_GEAS_Connection poa_geas_connection_servant =
  { NULL, &impl_GEAS_Connection_vepv };
POA_GEAS_DataObject poa_geas_dataobject_servant =
  { NULL, &impl_GEAS_DataObject_vepv };
POA_GEAS_ObjectList poa_geas_objectlist_servant =
  { NULL, &impl_GEAS_ObjectList_vepv };
POA_GEAS_Admin poa_geas_admin_servant = { NULL, &impl_GEAS_Admin_vepv };
POA_GEAS_Query_Query poa_geas_query_query_servant =
  { NULL, &impl_GEAS_Query_Query_vepv };
POA_GEAS_Query_Constraint poa_geas_query_constraint_servant =
  { NULL, &impl_GEAS_Query_Constraint_vepv };

#ifdef USE_SERVANT_MANAGER

/* Workaround for lack of Current interface support in ORBit */

int
initialise_object_servants (PortableServer_POA root_poa)
{
  static int count = 0;
  CORBA_PolicyList policies;
  PortableServer_POAManager mgr = NULL;
  CORBA_Environment ev;

  /* prevent duplicates */
  if (count != 0)
    {
      return 0;
    }
  count = 1;

  CORBA_exception_init (&ev);

  mgr = PortableServer_POA__get_the_POAManager (root_poa, &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error creating poa: %s", CORBA_exception_id (&ev));
    }

  /* prepare the POA for using a servant locator */
  policies._maximum = 5;
  policies._length = 5;
  policies._buffer = g_new0 (CORBA_Policy, 5);

  policies._buffer[0] =
    (CORBA_Policy) PortableServer_POA_create_lifespan_policy (root_poa,
                                                              PortableServer_PERSISTENT,
                                                              &ev);
  policies._buffer[1] =
    (CORBA_Policy) PortableServer_POA_create_id_assignment_policy (root_poa,
                                                                   PortableServer_USER_ID,
                                                                   &ev);
  policies._buffer[2] =
    (CORBA_Policy) PortableServer_POA_create_id_uniqueness_policy (root_poa,
                                                                   PortableServer_MULTIPLE_ID,
                                                                   &ev);
  policies._buffer[3] =
    (CORBA_Policy)
    PortableServer_POA_create_request_processing_policy (root_poa,
                                                         PortableServer_USE_SERVANT_MANAGER,
                                                         &ev);
  policies._buffer[4] =
    (CORBA_Policy)
    PortableServer_POA_create_servant_retention_policy (root_poa,
                                                        PortableServer_NON_RETAIN,
                                                        &ev);

  /* check policies were created ok */
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error creating policies: %s", CORBA_exception_id (&ev));
    }

  /* create POAs */
  poa_data.constraint_poa =
    PortableServer_POA_create_POA (root_poa, "ConstraintPOA", mgr, &policies,
                                   &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error creating constraint poa: %s",
                   CORBA_exception_id (&ev));
    }
  poa_data.query_poa =
    PortableServer_POA_create_POA (root_poa, "QueryPOA", mgr, &policies, &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error creating query poa: %s", CORBA_exception_id (&ev));
    }

  poa_data.transaction_poa =
    PortableServer_POA_create_POA (root_poa, "TransactionPOA", mgr, &policies,
                                   &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error creating transaction poa: %s",
                   CORBA_exception_id (&ev));
    }
  poa_data.connection_poa =
    PortableServer_POA_create_POA (root_poa, "ConnectionPOA", mgr, &policies,
                                   &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error creating connection poa: %s",
                   CORBA_exception_id (&ev));
    }
  poa_data.dataobject_poa =
    PortableServer_POA_create_POA (root_poa, "DataObjectPOA", mgr, &policies,
                                   &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error creating DataObject poa: %s",
                   CORBA_exception_id (&ev));
    }
  poa_data.objectlist_poa =
    PortableServer_POA_create_POA (root_poa, "ObjectListPOA", mgr, &policies,
                                   &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error creating ObjectList poa: %s",
                   CORBA_exception_id (&ev));
    }
  poa_data.admin_poa =
    PortableServer_POA_create_POA (root_poa, "AdminPOA", mgr, &policies, &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error creating admin poa: %s", CORBA_exception_id (&ev));
    }

  /* initialise servants */
  POA_GEAS_Query_Constraint__init (&poa_geas_query_constraint_servant, &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error initialising constraint servant: %s",
                   CORBA_exception_id (&ev));
    }

  POA_GEAS_Query_Query__init (&poa_geas_query_query_servant, &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error initialising query servant: %s",
                   CORBA_exception_id (&ev));
    }

  POA_GEAS_Transaction__init (&poa_geas_transaction_servant, &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error initialising transaction servant: %s",
                   CORBA_exception_id (&ev));
    }

  POA_GEAS_Connection__init (&poa_geas_connection_servant, &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error initialising connection servant: %s",
                   CORBA_exception_id (&ev));
    }

  POA_GEAS_DataObject__init (&poa_geas_dataobject_servant, &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error initialising data object servant: %s",
                   CORBA_exception_id (&ev));
    }

  POA_GEAS_ObjectList__init (&poa_geas_objectlist_servant, &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error initialising object list servant: %s",
                   CORBA_exception_id (&ev));
    }

  POA_GEAS_Admin__init (&poa_geas_admin_servant, &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error initialising admin servant: %s",
                   CORBA_exception_id (&ev));
    }

  /* create servant managers */
  create_fake_servants (root_poa, &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error creating servant locators: %s",
                   CORBA_exception_id (&ev));
    }

  /* set servant managers */
  /* PortableServer_POA_set_servant_manager( ); */

  PortableServer_POA_set_servant_manager (poa_data.constraint_poa,
                                          poa_servantmanager_constraint, &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error setting constraint servant manager: %s",
                   CORBA_exception_id (&ev));
    }

  PortableServer_POA_set_servant_manager (poa_data.query_poa,
                                          poa_servantmanager_query, &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error setting constraint servant manager: %s",
                   CORBA_exception_id (&ev));
    }

  PortableServer_POA_set_servant_manager (poa_data.transaction_poa,
                                          poa_servantmanager_transaction,
                                          &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error setting transaction servant manager: %s",
                   CORBA_exception_id (&ev));
    }

  PortableServer_POA_set_servant_manager (poa_data.connection_poa,
                                          poa_servantmanager_connection, &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error setting connection servant manager: %s",
                   CORBA_exception_id (&ev));
    }

  PortableServer_POA_set_servant_manager (poa_data.dataobject_poa,
                                          poa_servantmanager_dataobject, &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error setting data object servant manager: %s",
                   CORBA_exception_id (&ev));
    }

  PortableServer_POA_set_servant_manager (poa_data.objectlist_poa,
                                          poa_servantmanager_objectlist, &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error setting object list servant manager: %s",
                   CORBA_exception_id (&ev));
    }

  PortableServer_POA_set_servant_manager (poa_data.admin_poa,
                                          poa_servantmanager_admin, &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error setting admin servant manager: %s",
                   CORBA_exception_id (&ev));
    }

  /*
     PortableServer_POA_set_servant(poa_data.dataobject_poa,
     &poa_geas_dataobject_servant, &ev); if( ev._major != CORBA_NO_EXCEPTION ) 
     { fatal_error( "Error setting DataObject servant manager: %s" ,
     CORBA_exception_id(&ev) ); }
     PortableServer_POA_set_servant(poa_data.objectlist_poa,
     &poa_geas_objectlist_servant, &ev); if( ev._major != CORBA_NO_EXCEPTION ) 
     { fatal_error( "Error setting ObjectList servant manager: %s" ,
     CORBA_exception_id(&ev) ); }
     PortableServer_POA_set_servant(poa_data.admin_poa,
     &poa_geas_admin_servant, &ev); if( ev._major != CORBA_NO_EXCEPTION ) {
     fatal_error( "Error setting Admin servant manager: %s" ,
     CORBA_exception_id(&ev) ); } */
  CORBA_exception_free (&ev);

  return 1;
}

#else

/* don't compile default servant initialisation */

int
initialise_object_servants (PortableServer_POA root_poa)
{
  static int count = 0;
  CORBA_PolicyList policies;
  PortableServer_POAManager mgr = NULL;
  CORBA_Environment ev;

  /* prevent duplicates */
  if (count != 0)
    {
      return 0;
    }
  count = 1;

  CORBA_exception_init (&ev);

  mgr = PortableServer_POA__get_the_POAManager (root_poa, &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error creating poa: %s", CORBA_exception_id (&ev));
    }

  /* prepare the POA for creating default objects */
  policies._maximum = 5;
  policies._length = 5;
  policies._buffer = g_new0 (CORBA_Policy, 5);

  policies._buffer[0] =
    (CORBA_Policy) PortableServer_POA_create_lifespan_policy (root_poa,
                                                              PortableServer_PERSISTENT,
                                                              &ev);
  policies._buffer[1] =
    (CORBA_Policy) PortableServer_POA_create_id_assignment_policy (root_poa,
                                                                   PortableServer_USER_ID,
                                                                   &ev);
  policies._buffer[2] =
    (CORBA_Policy) PortableServer_POA_create_id_uniqueness_policy (root_poa,
                                                                   PortableServer_MULTIPLE_ID,
                                                                   &ev);
  policies._buffer[3] =
    (CORBA_Policy)
    PortableServer_POA_create_request_processing_policy (root_poa,
                                                         PortableServer_USE_DEFAULT_SERVANT,
                                                         &ev);
  policies._buffer[4] =
    (CORBA_Policy)
    PortableServer_POA_create_servant_retention_policy (root_poa,
                                                        PortableServer_NON_RETAIN,
                                                        &ev);

  /* policies._buffer[5] = (CORBA_Policy)
     ORBit_PortableServer_POA_create_okeyrand_policy(root_poa,0,0,&ev); */

  /* check policies were created ok */
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error creating policies: %s", CORBA_exception_id (&ev));
    }

  /* create POAs */
  poa_data.connection_poa =
    PortableServer_POA_create_POA (root_poa, "ConnectionPOA", mgr, &policies,
                                   &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error creating connection poa: %s",
                   CORBA_exception_id (&ev));
    }
  poa_data.dataobject_poa =
    PortableServer_POA_create_POA (root_poa, "DataObjectPOA", mgr, &policies,
                                   &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error creating DataObject poa: %s",
                   CORBA_exception_id (&ev));
    }
  poa_data.objectlist_poa =
    PortableServer_POA_create_POA (root_poa, "ObjectListPOA", mgr, &policies,
                                   &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error creating ObjectList poa: %s",
                   CORBA_exception_id (&ev));
    }
  poa_data.admin_poa =
    PortableServer_POA_create_POA (root_poa, "AdminPOA", mgr, &policies, &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error creating admin poa: %s", CORBA_exception_id (&ev));
    }

  /* initialise servants */
  POA_GEAS_Connection__init (&poa_geas_connection_servant, &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error initialising connection servant: %s",
                   CORBA_exception_id (&ev));
    }

  /* set servants */
  PortableServer_POA_set_servant (poa_data.connection_poa,
                                  &poa_geas_connection_servant, &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error setting connection servant: %s",
                   CORBA_exception_id (&ev));
    }

  PortableServer_POA_set_servant (poa_data.dataobject_poa,
                                  &poa_geas_dataobject_servant, &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error setting DataObject servant: %s",
                   CORBA_exception_id (&ev));
    }
  PortableServer_POA_set_servant (poa_data.objectlist_poa,
                                  &poa_geas_objectlist_servant, &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error setting ObjectList servant: %s",
                   CORBA_exception_id (&ev));
    }
  PortableServer_POA_set_servant (poa_data.admin_poa, &poa_geas_admin_servant,
                                  &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("Error setting Admin servant: %s",
                   CORBA_exception_id (&ev));
    }

  /*
     mgr = PortableServer_POA__get_the_POAManager(poa_data.connection_poa,
     &ev); if( ev._major != CORBA_NO_EXCEPTION ) { fatal_error( "Error getting
     manager: %s" , CORBA_exception_id(&ev) ); } */

  CORBA_exception_free (&ev);
  return 0;
}
#endif

#if 0
/* Convert a CORBA object ID to an ASCII string (as passed to make_*reference() below) */
static char *
objectid_to_string (PortableServer_ObjectId * oid)
{
  CORBA_Environment ev;
  char *retval = NULL;

  /* on any error, return a NULL, otherwise return the string */
  CORBA_exception_init (&ev);
  retval = PortableServer_ObjectId_to_string (oid, &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    retval = NULL;
  CORBA_exception_free (&ev);

  return (retval);
}
#endif

/* convert GEAS's internal ID to a CORBA object reference */
static CORBA_Object
GEASid_to_reference (const char *string, const char *corba_classname,
                     PortableServer_POA poa, CORBA_Environment * ev)
{
  GEAS_Connection retval = CORBA_OBJECT_NIL;
  PortableServer_ObjectId *object_id;

timer_start_profile( TIMER_FUNC_MAKEOID );
  /* create ID */
  object_id = PortableServer_string_to_ObjectId ((CORBA_char *) string, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
timer_update_profile( TIMER_FUNC_MAKEOID );
      return (CORBA_OBJECT_NIL);
    }

  /* create object reference */
  retval =
    PortableServer_POA_create_reference_with_id (poa,
                                                 object_id,
                                                 (CORBA_char *)
                                                 corba_classname, ev);
  CORBA_free (object_id);
  if (ev->_major != CORBA_NO_EXCEPTION) {
    timer_update_profile( TIMER_FUNC_MAKEOID );
    return (CORBA_OBJECT_NIL);
  }

  /* done */
  timer_update_profile( TIMER_FUNC_MAKEOID );
  return (retval);
}

static GEAS_object_reference *
alloc_geas_object_reference (void)
{
  GEAS_object_reference *oi;

  oi = (GEAS_object_reference *) g_new0 (GEAS_object_reference, 1);
  if (!oi)
    return (NULL);

  /* clear data */
  oi->parentid = NULL;
  oi->listid = NULL;
  oi->username = NULL;
  oi->classname = NULL;
  oi->objectid = NULL;
  oi->uniqueid = NULL;
  oi->sessionid = NULL;
  oi->currenttransaction = NULL;
  oi->transactionopen = FALSE;
  oi->server = CORBA_OBJECT_NIL;
  oi->serverid = NULL;

  return (oi);
}

GEAS_object_reference *
corba_id_to_geas_reference (PortableServer_ObjectId * oid,
                            CORBA_Environment * ev)
{
  char *str;
  GEAS_object_reference *retval;

  /* id to string */
  str = PortableServer_ObjectId_to_string (oid, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      errormsg ("%s", CORBA_exception_id (ev));
      return (NULL);
    }
  /* message( "id = '%s'" , str ); */

  if (!str)
    {
      return (NULL);
    }

  /* string to GEAS id */
  retval = parse_object_id (str);
  CORBA_free (str);
  return (retval);
}

/* TODO: fix this */
GEAS_object_reference *
corba_object_to_geas_reference (PortableServer_POA poa, CORBA_Object obj,
                                CORBA_Environment * ev)
{
  PortableServer_ObjectId *oid = NULL;
  GEAS_object_reference *retval;

  /* object to id */
  oid = PortableServer_POA_reference_to_id (poa, obj, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      return (NULL);
    }
  /* id to GEAS id */
  retval = corba_id_to_geas_reference (oid, ev);

  CORBA_free (oid);
  return (retval);
}

GEAS_object_reference *
current_object_to_geas_reference (CORBA_Environment * ev)
{
  PortableServer_ObjectId *oid;
  GEAS_object_reference *retval;

  fprintf (stderr,
           "\n********************\n\nerror: current_object_to_geas_reference is not yet implemented\n\n");
  exit (0);

  /* get Current ID */
  oid = NULL;

  /* TODO: requires PortableServer::Current::get_object_id() in ORBit */

  /* id to GEAS id */
  retval = corba_id_to_geas_reference (oid, ev);

  CORBA_free (oid);
  return (retval);
}

void
free_geas_object_reference (GEAS_object_reference * ref)
{
  if (ref)
    {
      if (ref->username)
        g_free (ref->username);

      if (ref->sessionid)
        g_free (ref->sessionid);

      if (ref->classname)
        g_free (ref->classname);

      if (ref->objectid)
        g_free (ref->objectid);

      if (ref->listid)
        g_free (ref->listid);

      if (ref->parentid)
        g_free (ref->parentid);

      if (ref->uniqueid)
        g_free (ref->uniqueid);

      if (ref->currenttransaction)
        g_free (ref->currenttransaction);

      if (ref->serverid)
        free_geas_object_reference (ref->serverid);

      if (ref->server != CORBA_OBJECT_NIL)
        {
          CORBA_Environment ev;

          CORBA_exception_init (&ev);
          CORBA_Object_release (ref->server, &ev);
          CORBA_exception_free (&ev);
        }

      g_free (ref);
    }
}

/* Business Object Identifier */
ObjectIdentifier *
create_object_identifier (const char *classname, const char *key)
{
  ObjectIdentifier *id;

  if (!classname || !key)
    return (NULL);

  id = g_new0 (ObjectIdentifier, 1);
  if (id)
    {
      id->classname = g_strdup (classname);
      id->key = g_strdup (key);
      if (!id->classname || !id->key)
        {
          free_object_identifier (id);
          return (NULL);
        }
    }
  return (id);
}

void
free_object_identifier (ObjectIdentifier * id)
{
  if (id)
    {
      if (id->classname)
        g_free (id->classname);
      if (id->key)
        g_free (id->key);
      g_free (id);
    }
}

/* Create/manage object keys using libuuid */
#define OID_BUF_LEN 4096
static unsigned char oid_buffer[OID_BUF_LEN + 2][64];
static int oid_count = (-1);

void
refill_oid_buffer (int max)
{
  int count = 0;
  /* create at most max oids */

  while (count < max && oid_count < (OID_BUF_LEN - 1))
    {
//      timer_start_profile( TIMER_FUNC_MAKEID );
      oid_count++;
      count++;
      uuid_generate (&oid_buffer[oid_count][0]);
//      timer_update_profile( TIMER_FUNC_MAKEID );
    }
  debug_output (DEBUGLEVEL_2, "Created %d oids", count);
}

ObjectKey
create_new_object_key ()
{
  ObjectKey key = g_new0 (unsigned char, 16);
  if (oid_count >= 1)
    {
      memcpy (key, &oid_buffer[oid_count][0], 16);
      oid_count--;
    }
  else
    {
      refill_oid_buffer (2);
      key = create_new_object_key ();
    }
  return (key);
}

ObjectKey
old_create_new_object_key (void)
{
  static unsigned char buf[64];
  static gboolean first = TRUE;
  ObjectKey key = g_new0 (unsigned char, 16);
  int i;

  if (first == TRUE)
    {
      printf ("using uuid_generate_time()\n");
      uuid_generate_time (buf);
      /* first = FALSE; */
    }
  else
    {
      printf ("incrementing buffer: ");
      i = 15;
      while (i >= 0)
        {
          printf ("%d ", i);
          if (buf[i] == 255)
            {
              buf[i] = 0;
              i--;
            }
          else
            {
              buf[i]++;
              i = (-1);
            }
        }
      printf ("\n");
    }

  memcpy (key, buf, 16);
  return (key);

  /*  real code 

     ObjectKey key = g_new0(unsigned char, 16);

     if (key)
     uuid_generate(key);
     return (key);
   */
}

void
cleanup_oid (char *oid)
{
  char *p;

  /* convert to lower case */
  for (p = oid; p && *p != '\0'; p++)
    *p = tolower (*p);

  /* remove '-' characters */
  p = oid;
  while (p && *p != '\0')
    {
      while (*p == '-')
        memmove ((char *) p, (const char *) (p + 1), strlen (p));
      p++;
    }
}

const char *
object_key_as_string (ObjectKey key)
{
  static char buf[37];
  buf[36] = '\0';

  uuid_unparse (key, buf);
  cleanup_oid (buf);
  return (buf);
}

void
free_object_key (ObjectKey key)
{
  if (key)
    g_free (key);
}

unsigned long int
object_key_size (ObjectKey key)
{
  /* currently a fixed size */
  return (16);
}

void
print_object_key (FILE * fp, ObjectKey key)
{
  const char *k = object_key_as_string (key);

  if (k)
    fprintf (fp, k);
}

ObjectKey
object_key_from_string (const char *str)
{
  ObjectKey key = g_malloc (16);

  if (key)
    uuid_parse ((char *) str, key);
  return (key);
}

#ifdef DEBUG
void
display_geas_object_reference (GEAS_object_reference * gref)
{
  if (gref)
    {
      debug_output (DEBUGLEVEL_4, "  GEAS object reference:");
      if (gref->listid)
        debug_output (DEBUGLEVEL_4, "    listid    : %s", gref->listid);
      if (gref->username)
        debug_output (DEBUGLEVEL_4, "    username  : %s", gref->username);
      if (gref->classname)
        debug_output (DEBUGLEVEL_4, "    classname : %s", gref->classname);
      if (gref->objectid)
        debug_output (DEBUGLEVEL_4, "    objectid  : %s", gref->objectid);
      if (gref->uniqueid)
        debug_output (DEBUGLEVEL_4, "    uniqueid  : %s", gref->uniqueid);
      if (gref->parentid)
        debug_output (DEBUGLEVEL_4, "    parentid  : %s", gref->parentid);
      if (gref->sessionid)
        debug_output (DEBUGLEVEL_4, "    sessionid : %s", gref->sessionid);

      if (gref->currenttransaction)
        debug_output (DEBUGLEVEL_4, "transactionid : %s",
                      gref->currenttransaction);
    }
  else
    {
      debug_output (DEBUGLEVEL_4, "Unknown GEAS object reference");
    }

}
#endif


static CORBA_Object
make_1_element_reference (const char *type, const char *idlclass,
                          const char *e1,
                          PortableServer_POA poa, CORBA_Environment * ev)
{
  GEAS_Connection retval = CORBA_OBJECT_NIL;
  char *buf;

  buf = g_strdup_printf ("%s*%s", type, e1);
  if (buf)
    {
      retval = GEASid_to_reference (buf, idlclass, poa, ev);
      if (ev->_major != CORBA_NO_EXCEPTION)
        retval = CORBA_OBJECT_NIL;
      g_free (buf);
    }

  return (retval);
}


static CORBA_Object
make_2_element_reference (const char *type, const char *idlclass,
                          const char *e1,
                          const char *e2,
                          PortableServer_POA poa, CORBA_Environment * ev)
{
  GEAS_Connection retval = CORBA_OBJECT_NIL;
  char *buf;

  buf = g_strdup_printf ("%s*%s*%s", type, e1, e2);
  if (buf)
    {
      retval = GEASid_to_reference (buf, idlclass, poa, ev);
      if (ev->_major != CORBA_NO_EXCEPTION)
        retval = CORBA_OBJECT_NIL;
      g_free (buf);
    }

  return (retval);
}


static CORBA_Object
make_3_element_reference (const char *type, const char *idlclass,
                          const char *e1,
                          const char *e2,
                          const char *e3,
                          PortableServer_POA poa, CORBA_Environment * ev)
{
  GEAS_Connection retval = CORBA_OBJECT_NIL;
  char *buf;

  buf = g_strdup_printf ("%s*%s*%s*%s", type, e1, e2, e3);
  if (buf)
    {
      retval = GEASid_to_reference (buf, idlclass, poa, ev);
      if (ev->_major != CORBA_NO_EXCEPTION)
        retval = CORBA_OBJECT_NIL;
      g_free (buf);
    }

  return (retval);
}

#if 0
static CORBA_Object
make_4_element_reference (const char *type, const char *idlclass,
                          const char *e1,
                          const char *e2,
                          const char *e3,
                          const char *e4,
                          PortableServer_POA poa, CORBA_Environment * ev)
{
  GEAS_Connection retval = CORBA_OBJECT_NIL;
  char *buf;

  buf = g_strdup_printf ("%s*%s*%s*%s*%s", type, e1, e2, e3, e4);
  if (buf)
    {
      retval = GEASid_to_reference (buf, idlclass, poa, ev);
      if (ev->_major != CORBA_NO_EXCEPTION)
        retval = CORBA_OBJECT_NIL;
      g_free (buf);
    }

  return (retval);
}
#endif

static char *
get_last_element (const char *buf)
{
  char *p;

  p = strrchr (buf, '*');
  if (!p)
    return (g_strdup (buf));

  return (g_strdup (p + 1));
}

static void
unparse_1_element_reference (const char *buf, char **e1)
{
  *e1 = get_last_element (buf);
}

static void
unparse_2_element_reference (const char *buf, char **e1, char **e2)
{
  char *tmp = g_strdup (buf);
  char *p;

  *e2 = get_last_element (tmp);

  p = strrchr (tmp, '*');
  if (p)
    *p = '\0';

  *e1 = get_last_element (tmp);

  g_free (tmp);
}

static void
unparse_3_element_reference (const char *buf, char **e1, char **e2, char **e3)
{
  char *tmp = g_strdup (buf);
  char *p;

  *e3 = get_last_element (tmp);

  p = strrchr (tmp, '*');
  if (p)
    *p = '\0';
  *e2 = get_last_element (tmp);

  p = strrchr (tmp, '*');
  if (p)
    *p = '\0';
  *e1 = get_last_element (tmp);

  g_free (tmp);
}

#if 0
static void
unparse_4_element_reference (const char *buf,
                             char **e1, char **e2, char **e3, char **e4)
{
  char *tmp = g_strdup (buf);
  char *p;

  *e4 = get_last_element (tmp);

  p = strrchr (tmp, '*');
  if (p)
    *p = '\0';
  *e3 = get_last_element (tmp);

  p = strrchr (tmp, '*');
  if (p)
    *p = '\0';
  *e2 = get_last_element (tmp);

  p = strrchr (tmp, '*');
  if (p)
    *p = '\0';
  *e1 = get_last_element (tmp);

  g_free (tmp);
}
#endif

CORBA_Object
make_admin_reference (const char *username, const char *sessionid,
                      CORBA_Environment * ev)
{
  char *uid = NULL;
  CORBA_Object retval;

  uid = g_strdup_printf ("%s-%s", sessionid, username);
  if (!uid)
    return (CORBA_OBJECT_NIL);
  retval = make_1_element_reference (OBJECTTYPE_ADMIN, "IDL:GEAS/Admin:1.0",
                                     uid, poa_data.admin_poa, ev);
  if (uid)
    g_free (uid);
  return (retval);
}

CORBA_Object
make_transaction_reference (const char *username, const char *objectid,
                            const char *sessionid, CORBA_Environment * ev)
{
  char *uid = NULL;
  CORBA_Object retval;

  uid = g_strdup_printf ("%s-%s", sessionid, username);
  if (!uid)
    return (CORBA_OBJECT_NIL);
  retval =
    make_2_element_reference (OBJECTTYPE_TRANSACTION,
                              "IDL:GEAS/Transaction:1.0", uid, objectid,
                              poa_data.transaction_poa, ev);
  if (uid)
    g_free (uid);
  return (retval);
}

CORBA_Object
make_connection_reference (const char *username, const char *sessionid,
                           CORBA_Environment * ev)
{
  char *uid = NULL;
  CORBA_Object retval;

  uid = g_strdup_printf ("%s-%s", sessionid, username);
  if (!uid)
    return (CORBA_OBJECT_NIL);
  retval =
    make_2_element_reference (OBJECTTYPE_CONNECTION,
                              "IDL:GEAS/Connection:1.0", uid, sessionid,
                              poa_data.connection_poa, ev);
  if (uid)
    g_free (uid);
  return (retval);
}

CORBA_Object
make_list_reference (const char *listid, const char *username,
                     const char *sessionid, CORBA_Environment * ev)
{
  char *uid = NULL;
  CORBA_Object retval;

  uid = g_strdup_printf ("%s-%s", sessionid, username);
  if (!uid)
    return (CORBA_OBJECT_NIL);
  retval =
    make_2_element_reference (OBJECTTYPE_LIST, "IDL:GEAS/ObjectList:1.0",
                              listid, uid, poa_data.objectlist_poa, ev);
  if (uid)
    g_free (uid);
  return (retval);
}

CORBA_Object
make_dataobject_reference (const char *objectclass, const char *objectid,
                           const char *username, const char *sessionid,
                           CORBA_Environment * ev)
{
  char *uid = NULL;
  CORBA_Object retval;

  uid = g_strdup_printf ("%s-%s", sessionid, username);
  if (!uid)
    return (CORBA_OBJECT_NIL);
  retval =
    make_3_element_reference (OBJECTTYPE_DATA, "IDL:GEAS/DataObject:1.0",
                              objectclass, objectid, uid,
                              poa_data.dataobject_poa, ev);
  if (uid)
    g_free (uid);
  return (retval);
}

CORBA_Object
make_query_reference (const char *uniqueid, const char *username,
                      const char *sessionid, CORBA_Environment * ev)
{
  char *uid = NULL;
  CORBA_Object retval;

  uid = g_strdup_printf ("%s-%s", sessionid, username);
  if (!uid)
    return (CORBA_OBJECT_NIL);
  retval =
    make_2_element_reference (OBJECTTYPE_QUERY, "IDL:GEAS/Query/Query:1.0",
                              uniqueid, uid, poa_data.query_poa, ev);
  if (uid)
    g_free (uid);
  return (retval);
}

CORBA_Object
make_constraint_reference (const char *uniqueid, const char *parentid,
                           const char *username, const char *sessionid,
                           CORBA_Environment * ev)
{
  char *uid = NULL;
  CORBA_Object retval;

  uid = g_strdup_printf ("%s-%s", sessionid, username);
  if (!uid)
    return (CORBA_OBJECT_NIL);
  retval =
    make_3_element_reference (OBJECTTYPE_CONSTRAINT,
                              "IDL:GEAS/Query/Constraint:1.0", uniqueid,
                              parentid, uid, poa_data.constraint_poa, ev);
  if (uid)
    g_free (uid);
  return (retval);
}

static void
reprocess_username (char **username, char **sessionid)
{
  char *p, *tmp;

  tmp = *username;

  if (*sessionid)
    g_free (*sessionid);
  *sessionid = NULL;

  p = strchr (*username, '-');
  if (!p)
    return;

  *p = '\0';
  p++;
  *sessionid = g_strdup (*username);
  *username = g_strdup (p);

  g_free (tmp);
}

/* convert string to GEAS id */
static GEAS_object_reference *
parse_object_id (const char *id)
{
  GEAS_object_reference *oi;
  /* char *p; */
  /* char *tmp; */

  oi = alloc_geas_object_reference ();
  if (!oi)
    return (NULL);

  if (strncmp (id, OBJECTTYPE_DATA, strlen (OBJECTTYPE_DATA)) == 0
      && id[strlen (OBJECTTYPE_DATA)] == '*')
    {
      /* DataObject : 02*classname*objectid*username   */
      unparse_3_element_reference (id, &oi->classname, &oi->objectid,
                                   &oi->username);
      reprocess_username (&oi->username, &oi->sessionid);
    }
  else if (strncmp (id, OBJECTTYPE_LIST, strlen (OBJECTTYPE_LIST)) == 0
           && id[strlen (OBJECTTYPE_LIST)] == '*')
    {
      /* ObjectList : 03*listid*username               */
      unparse_2_element_reference (id, &oi->listid, &oi->username);
      reprocess_username (&oi->username, &oi->sessionid);
    }
  else if (strncmp (id, OBJECTTYPE_CONSTRAINT, strlen (OBJECTTYPE_CONSTRAINT))
           == 0 && id[strlen (OBJECTTYPE_CONSTRAINT)] == '*')
    {
      /* Constraint : 07*uniqueid*parentid             */
      unparse_3_element_reference (id, &oi->uniqueid, &oi->parentid,
                                   &oi->username);
      reprocess_username (&oi->username, &oi->sessionid);
    }
  else if (strncmp (id, OBJECTTYPE_QUERY, strlen (OBJECTTYPE_QUERY)) == 0
           && id[strlen (OBJECTTYPE_QUERY)] == '*')
    {
      /* Query      : 06*uniqueid*username             */
      unparse_2_element_reference (id, &oi->uniqueid, &oi->username);
      reprocess_username (&oi->username, &oi->sessionid);
    }
  else
    if (strncmp (id, OBJECTTYPE_TRANSACTION, strlen (OBJECTTYPE_TRANSACTION))
        == 0 && id[strlen (OBJECTTYPE_TRANSACTION)] == '*')
    {
      /* Transaction: 05*username*objectid             */
      unparse_2_element_reference (id, &oi->username, &oi->objectid);
      reprocess_username (&oi->username, &oi->sessionid);
    }
  else if (strncmp (id, OBJECTTYPE_CONNECTION, strlen (OBJECTTYPE_CONNECTION))
           == 0 && id[strlen (OBJECTTYPE_CONNECTION)] == '*')
    {
      /* Connection : 01*username*sessionid            */
      unparse_2_element_reference (id, &oi->username, &oi->sessionid);
      reprocess_username (&oi->username, &oi->sessionid);
    }
  else if (strncmp (id, OBJECTTYPE_ADMIN, strlen (OBJECTTYPE_ADMIN)) == 0
           && id[strlen (OBJECTTYPE_ADMIN)] == '*')
    {
      /* Admin      : 04*username                      */
      unparse_1_element_reference (id, &oi->username);
      reprocess_username (&oi->username, &oi->sessionid);
    }

#ifdef DEBUG
  display_geas_object_reference (oi);
#endif
  return (oi);
}
