
/*
   geas-server.h
   
   Part of GNU Enterprise Application Server (GEAS)
 
   Copyright (C) 2000 Free Software Foundation
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
 
   $Id: geas-server.h,v 1.52 2001/07/06 04:48:43 dneighbors Exp $
*/

/** \file geas-server.h
 *  \brief Main header file
 * This file contains defines, extern's for global variables,
 * and prototypes for global utility functions.
 */

#ifndef GEAS_SERVER_H
#define GEAS_SERVER_H

#include "classdata.h"

#ifdef TIMER_TESTING
  #define TIMER_MISC       0 /* geas-server.c     */
  #define TIMER_SKELETON   1 /* geas-skeleton.c   */
  #define TIMER_SKELMISC   2 /* geas-skeleton.c   */
  #define TIMER_CACHE      3 /* objectcache/      */
  #define TIMER_LISTS      4 /* collectiondata.c  */
  #define TIMER_SECTIONS  20 /* count of sections */

  #define TIMER_FUNC_GETFIELD      0
  #define TIMER_FUNC_SETFIELD      1
  #define TIMER_FUNC_NEWOBJECT     2
  #define TIMER_FUNC_LOADALL       3
  #define TIMER_FUNC_GET_OBJECTS   4

  #define TIMER_FUNC_FIND_BY_KEY   6
  #define TIMER_FUNC_FIND_CACHED_BY_KEY  7
  #define TIMER_FUNC_GET_FIELD     8
  #define TIMER_FUNC_SET_FIELD     9
  #define TIMER_FUNC_FLUSH        10
  #define TIMER_FUNC_MAKELIST     11
  #define TIMER_FUNC_RELEASELIST  12
  #define TIMER_FUNC_MAKEID       13
  #define TIMER_FUNC_EXECQUERY    14
  #define TIMER_FUNC_HANDLERESULTS     15
  #define TIMER_FUNC_LOOKUP       16
  #define TIMER_FUNC_TEMP1        17
  #define TIMER_FUNC_GETENTRY     18

  #define TIMER_FUNC_A            42
  #define TIMER_FUNC_B            43
  #define TIMER_FUNC_C            44
  #define TIMER_FUNC_D            45
  #define TIMER_FUNC_E            46
  #define TIMER_FUNC_F            47
  #define TIMER_FUNC_G            48
  #define TIMER_FUNC_H            49
  #define TIMER_FUNC_I            50
  #define TIMER_FUNC_J            51
  #define TIMER_FUNC_K            52
  #define TIMER_FUNC_L            53

  #define TIMER_FUNC_MAKEOID      97
  #define TIMER_FUNC_MAKEREF      98
  #define TIMER_FUNC_OPERATION    99
  #define TIMER_FUNC_COUNT       100


  void timer_start_operation ( int section , char *name , ... );
  void timer_done_operation  ( int section );
  void timer_fail_operation  ( int section );
  void timer_start_profile   ( int function );
  void timer_update_profile  ( int function );
  void timer_display_profiles( void );

#else
  #define timer_start_operation(x, dummy...) /* empty */
  #define timer_done_operation(dummy)        /* empty */
  #define timer_fail_operation(dummy)        /* empty */

  #define timer_start_profile(dummy)         /* empty */
  #define timer_update_profile(dummy)        /* empty */
  #define timer_display_profiles()           /* empty */
#endif

/** \brief username reserved for the server */
#define SERVER_PRIVATE_USERNAME "server_private_user"

/* error messages */
void do_fatal_error (char *file, int line, char *func, char *fmt, ...);

/* displays the message if settings.verbose is not TRUE */

/* adds a newline to the end of the message                  */
void message (char *fmt, ...);  /* to stdout                  */

/* always printed */
void criticalerror (char *fmt, ...);    /* to stderr   */

#ifdef SELF_TEST
void do_self_test_message (char *fmt, ...);     /* to stdout */
#define self_test_message(fmt,args...)  do_self_test_message(fmt,##args)
#else
#define self_test_message(fmt,args...)  /* empty */
#endif

/** \brief Display an error message
 * In DEBUG mode, the file and line number of the call to errormsg() is
 * printed. The function acts like printf() but will only display
 * to the screen if the 'quiet' mode was not selected by command line.
 * options. It will also write the message to any log file configured to
 * display error messages.
 */
#define errormsg(fmt,args...) \
error_message_out("error",__FILE__,__PRETTY_FUNCTION__,__LINE__ , fmt , ##args )

/** \brief Display warning messages
 * This acts like errormsg() above, but labels messages as 'warning'
 * rather than 'error'
 */
#define warnmsg(fmt,args...) \
error_message_out("warning",__FILE__,__PRETTY_FUNCTION__,__LINE__ , fmt , ##args )

/** \brief display message then exit immediately. */
#define fatal_error(fmt,args...)  \
do_fatal_error(__FILE__,__LINE__,__PRETTY_FUNCTION__,fmt,##args)

/** \brief Holds current configuration information
 */
struct ServerSettings
{
  gboolean verbose, quiet, showconfig, loggeasclasses, showclasses;
  gchar *configfile;
  guint64 cache_maximum_size;
};

/** \brief global variables */
extern struct ServerSettings settings;

extern int debuglevel;          /** \brief configurable debugging output detail :
                                   1-10, 1 = low detail, 10 = high detail */

extern odl_tree *all_classes;

/* tracing functions */

/* output function - not to be used directly, in most cases */

/* see macros below */
void debug_output_f (char *type, int minlevel, char *file, char *func,
                     unsigned int line, const char *msg, ...);
void error_message_out (char *type, char *file, char *func,
                        unsigned int line, char *msg, ...);

/* the higher the level (first num) the more debugging output is produced
 * 1  = most general
 * 10 = absolutely everything
 */

/* set debuglevel in geas.conf to the numbers to get that level of detail */

/* intermediate levels are, of course, allowed */
#define DEBUGLEVEL_ALWAYS (-2)  /* always display this message */
#define DEBUGLEVEL_OFF    (-1)  /* no debug info */
#define DEBUGLEVEL_LOW       1  /* lowest debugging detail  */
#define DEBUGLEVEL_MEDIUM    5  /* medium detail message */
#define DEBUGLEVEL_HIGH     10  /* high detail message   */

#define DEBUGLEVEL_1   1        /* low */
#define DEBUGLEVEL_2   2
#define DEBUGLEVEL_3   3
#define DEBUGLEVEL_4   4
#define DEBUGLEVEL_5   5        /* medium */
#define DEBUGLEVEL_6   6
#define DEBUGLEVEL_7   7
#define DEBUGLEVEL_8   8
#define DEBUGLEVEL_9   9
#define DEBUGLEVEL_10 10        /* high */

/* set debug level for features */
#define TRACE_FUNCTIONCALL 3    /* MEDIUM   */
#define TRACE_NOTDONE      0    /* ALWAYS   */

#ifdef DEBUG
# define trace_functioncall() \
debug_output_f("function call", TRACE_FUNCTIONCALL, __FILE__, __PRETTY_FUNCTION__, __LINE__ , "" )
# define trace_functiondone() \
debug_output_f("function done", TRACE_FUNCTIONCALL, __FILE__, __PRETTY_FUNCTION__, __LINE__ , "" )
#else
#define trace_functioncall()    /* empty */
#define trace_functiondone()    /* empty */
#endif

/* note: trace_debug priorities should be between 1 and 10, although this
 * is not enforced
 */
#ifdef DEBUG
#define debug_output(level,str,args...) \
debug_output_f(NULL,level,NULL,NULL,0,str,##args)
#else
#define debug_output(level,str,args...) /* empty */
#endif

gboolean allow_log_class_event (const char *classname);

#endif
