/*
   event_log_code.h
   
   Part of GNU Enterprise Application Server (GEAS)
 
   Copyright (C) 2000-2001 Free Software Foundation
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
 
   $Id: event_log_code.h,v 1.7 2001/06/13 04:28:21 treshna Exp $
*/

#include <time.h>

static GList *event_logs = NULL;

static GHashTable *edits = NULL;
static GHashTable *deletes = NULL;

typedef struct EventLog
{
  char *sessionid;
  GList *edits, *deletes;
}
EventLog;

typedef struct LoggableEvent
{
  signed int usagecount;
  char *classname;
  char *oid;
  char *fieldname;
  char *when;
}
LoggableEvent;

static void
free_event_entry (LoggableEvent * e)
{
  if (e)
    {
      if (e->classname)
        g_free (e->classname);
      if (e->oid)
        g_free (e->oid);
      if (e->fieldname)
        g_free (e->fieldname);
      if (e->when)
        g_free (e->when);
      g_free (e);
    }
}

static const char *
evt_get_now (void)
{
  static char buf[1024];
  char x[32];
  static int y = 1;
  struct tm *tm;
  static time_t last = 0;
  time_t t = 0;

  t = time (NULL);
  tm = localtime (&t);
  strftime (buf, 1020, "%Y-%m-%d %H:%M:%S", tm);
  sprintf (x, ".%d", y);
  strcat (buf, x);
  if (last < t)
    {
      y = 1;
    }
  else
    {
      y++;
    }
  last = t;
  return (buf);
}

static LoggableEvent *
log_edit_event (const char *classname, const char *oid, const char *fieldname)
{
  LoggableEvent *e = g_new0 (LoggableEvent, 1);
  if (e)
    {
      e->classname = g_strdup (classname);
      e->oid = g_strdup (oid);
      e->fieldname = g_strdup (fieldname);
      e->when = g_strdup (evt_get_now ());
      e->usagecount = 0;
    }
g_assert( e != NULL );
  return (e);
}

static guint
logevt_hash_func (gconstpointer ptr)
{
  return 0;
}

static gint
logevt_cmp_func (gconstpointer ptr1, gconstpointer ptr2)
{
  return 0;
}


static LoggableEvent *
log_delete_event (const char *classname, const char *oid)
{
  LoggableEvent *e = g_new0 (LoggableEvent, 1);
  if (e)
    {
      e->classname = g_strdup (classname);
      e->oid = g_strdup (oid);
      e->fieldname = NULL;
      e->when = g_strdup (evt_get_now ());
      e->usagecount = 0;
    }
  return (e);
}

void
Connection_activateEventLog (GEAS_object_reference * id,
                             CORBA_Environment * ev)
{
  EventLog *elog = g_new0 (EventLog, 1);
  if (elog)
    {
      elog->sessionid = g_strdup (id->sessionid);
      elog->edits = NULL;
      elog->deletes = NULL;
      event_logs = g_list_append (event_logs, elog);
      printf ("elog: Added log '%s'\n", elog->sessionid);
    }
}


GEAS_Changes *
Connection__get_latestChanges (GEAS_object_reference * id,
                               CORBA_Environment * ev)
{
  LoggableEvent *evt;
  GEAS_Changes *c = GEAS_Changes__alloc ();
  GList *l = event_logs;
  EventLog *elog = NULL;
  int n1, n2, count;

  while (l)
    {
      elog = l->data;
      if (strcmp (elog->sessionid, id->sessionid) == 0)
        {
          break;
        }
      else
        {
          elog = NULL;
        }
      l = g_list_next (l);
    }
  if (!elog)
    {
      /* not found, or no changes, return empty result */
      return (c);
    }
  n1 = g_list_length (elog->edits);
  n2 = g_list_length (elog->deletes);
  /* if no changes then return empty result */
  if (n1 == 0 && n2 == 0)
    {
      return (c);
    }
  l = elog->edits;
  c->edited._length = c->edited._maximum = n1;
  c->edited._buffer = CORBA_sequence_GEAS_FieldEdit_allocbuf (n1);
  count = 1;
  while (l)
    {
      evt = l->data;
      /* debug */
      printf (" evt %2d : edit field %15s in %s/%s at %s\n", count,
              evt->fieldname, evt->classname, evt->oid, evt->when);
      /* return data... */
      c->edited._buffer[count - 1].classname =
        CORBA_string_dup (evt->classname);
      c->edited._buffer[count - 1].objectid = CORBA_string_dup (evt->oid);
      c->edited._buffer[count - 1].fieldname =
        CORBA_string_dup (evt->fieldname);
      evt->usagecount--;
      if (evt->usagecount <= 0)
        {
          free_event_entry (evt);
        }
      /* next... */
      l = g_list_next (l);
      count++;
    }
  g_list_free (elog->edits);
  elog->edits = NULL;
  l = elog->deletes;
  c->deleted._length = c->deleted._maximum = n2;
  c->deleted._buffer = CORBA_sequence_GEAS_ObjectDeleted_allocbuf (n2);
  count = 1;
  while (l)
    {
      evt = l->data;
      /* debug */
      printf (" evt %2d : delete object %s/%s at %s\n", count,
              evt->classname, evt->oid, evt->when);
      /* return data... */
      c->deleted._buffer[count - 1].classname =
        CORBA_string_dup (evt->classname);
      c->deleted._buffer[count - 1].objectid = CORBA_string_dup (evt->oid);
      evt->usagecount--;
      if (evt->usagecount <= 0)
        {
          free_event_entry (evt);
        }
      /* next... */
      l = g_list_next (l);
      count++;
    }
  g_list_free (elog->deletes);
  elog->deletes = NULL;
  return (c);
}

void
Connection_logEdit (const char *classname, const char *oid,
                    const char *fieldname, GEAS_object_reference * id)
{
  LoggableEvent *evt = NULL;
  EventLog *e;
  GList *l = event_logs;

  if( !l ) return;

  /* on demand initialisation */
  if (edits == NULL)
    {
      edits = g_hash_table_new (logevt_hash_func, logevt_cmp_func);
    }
  if (deletes == NULL)
    {
      deletes = g_hash_table_new (logevt_hash_func, logevt_cmp_func);
    }
  /* don't log GEAS internal classes - they're irrelevant */
  if (strstr (classname, "geas::") != NULL)
    {
      return;
    }
  /* find old edits of this object, and decrement their usage count */

  if (l)
    {
      if (strncmp (classname, "root::", 6) == 0)
        {
          evt = log_edit_event (&classname[6], oid, fieldname);
        }
      else
        {
          evt = log_edit_event (classname, oid, fieldname);
        }
      if (!evt)
        {
          errormsg ("out of memory - please restart server and all clients.");
          abort ();
        }
    }
  while (l)
    {
      e = l->data;
      e->edits = g_list_prepend (e->edits, evt);
      evt->usagecount++;
      l = g_list_next (l);
    }
}



void
Connection_logDelete (const char *classname, const char *oid,
                      GEAS_object_reference * id)
{
  LoggableEvent *evt = NULL;
  EventLog *e;
  GList *l = event_logs;

  if( !l ) return;

  /* on demand initialisation */
  if (edits == NULL)
    {
      edits = g_hash_table_new (logevt_hash_func, logevt_cmp_func);
    }
  if (deletes == NULL)
    {
      deletes = g_hash_table_new (logevt_hash_func, logevt_cmp_func);
    }
  /* don't log GEAS internal classes - they're irrelevant */
  if (strstr (classname, "geas::") != NULL)
    {
      return;
    }
  if (l)
    {
      if (strncmp (classname, "root::", 6) == 0)
        {
          evt = log_delete_event (&classname[6], oid);
        }
      else
        {
          evt = log_delete_event (classname, oid);
        }
      if (!evt)
        {
          errormsg ("out of memory - please restart server and all clients.");
          abort ();
        }
    }
  while (l)
    {
      e = l->data;
      e->deletes = g_list_prepend (e->deletes, evt);
      evt->usagecount++;

      l = g_list_next (l);
    }
}
