/*
   connection.c - 
   
   Part of GNU Enterprise Application Server (GEAS)
 
   Copyright (C) 2000-2001 Free Software Foundation
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
   
   $Id: connection.c,v 1.65 2001/07/02 08:20:14 treshna Exp $
 
*/

/** \file connection.c
 *  \brief GEAS::Connection implementation
 */

#include <ctype.h>
#include <stdlib.h>

#include "config.h"

#include "geas.h"
#include "geas-server.h"
#include "connection.h"
#include "exceptions.h"
#include "collectiondata.h"
#include "oql/oql.h"
#include "objectcache/objectcache.h"
#include "datamonitor/datamonitor.h"
#include "config/configuration.h"
#include "objectstore/objectstore.h"
#include "transaction.h"
#include "schema.h"

/* raises: */
CORBA_char *
Connection__get_currentUserID (GEAS_object_reference * id,
                               CORBA_Environment * ev)
{
  CORBA_char *retval;

  retval = CORBA_string_dup (id->username);
  return retval;
}

/* raises: */
GEAS_classnames *
Connection__get_classes (GEAS_object_reference * id, CORBA_Environment * ev)
{
  return (schema_list_classes (id, ev));
}

/* raises: */
GEAS_classlist *
Connection__get_schema (GEAS_object_reference * id, CORBA_Environment * ev)
{
  return (schema_get_all (id, ev));
}

/* raises: UnknownClass, ServerError */
GEAS_ClassDefinition *
Connection_getClassDefinition (GEAS_object_reference * id,
                               CORBA_char * name, CORBA_Environment * ev)
{
  return (schema_get_class (id, name, FALSE, ev));
}

/* raises: UnknownClass, ServerError */
GEAS_ClassDefinition *
Connection_getFullClassDefinition (GEAS_object_reference * id,
                                   CORBA_char * name, CORBA_Environment * ev)
{
  return (schema_get_class (id, name, TRUE, ev));
}

/* raises: UnknownClass, UnknownField, NotLookupField, NotReadable */
GEAS_LookupOptions *
Connection_getLookupFieldOptions (GEAS_object_reference * id,
                                  CORBA_char * classname,
                                  CORBA_char * fieldname,
                                  CORBA_Environment * ev)
{
  GEAS_LookupOptions *retval = NULL;
  odl_class *cl;
  odl_field *f;
  int item_count = 0, i, j;
  GEAS_ObjectList list;

  /* find and validate lookup field */
  cl = odl_find_class (all_classes, classname, NULL);
  if (!cl)
    {
      make_ServerError_exception (ev, "Class '%s' went awol.", classname);
      return (NULL);
    }
  f = odl_class_get_field (cl, fieldname);
  if (!f)
    {
      make_UnknownField_exception (ev,
                                   "Field '%s' is not defined in class '%s'",
                                   fieldname, classname);
      return (NULL);
    }
  if (odl_field_get_type (f) != FT_lookup)
    {
      make_NotLookupField_exception (ev, "%s.%s is not a lookup field",
                                     classname, fieldname);
      return (NULL);
    }

  /* get list of objects eligible to be in this lookup field */
  list = Connection_loadAll (id, (char *) odl_field_get_sourceclass (f), ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      return (NULL);
    }
  item_count = GEAS_ObjectList__get_length (list, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      /* ..clean up.. */
      return (NULL);
    }

  /* make useable list to return */
  retval = (GEAS_LookupOptions *) GEAS_LookupOptions__alloc ();
  if (!retval)
    {
      /* ..clean up.. */
      make_ServerError_exception (ev,
                                  "Error creating return data structure.");
      return (NULL);
    }
  retval->_length = item_count /* number of entries */ ;
  retval->_buffer =
    CORBA_sequence_GEAS_LookupOption_allocbuf (retval->_length);

  for (i = 0; i < item_count; i++)
    {
      char *value;
      GEAS_setlookuplist *setd;
      GEAS_DataObject obj;
      GList *s, *t;

      /* value of lookup field if this object is selected */
      obj = GEAS_ObjectList_getEntry (list, i, ev);
      if (ev->_major != CORBA_NO_EXCEPTION)
        {
          /* ..clean up.. */
          return (NULL);
        }
      value =
        GEAS_DataObject_getField (obj, (char *) odl_field_get_sourcefield (f),
                                  ev);
      if (ev->_major != CORBA_NO_EXCEPTION)
        {
          /* ..clean up.. */
          return (NULL);
        }
      /* message( "%s.%s = %s WHEN" , classname , fieldname , value ); */

      retval->_buffer[i].value = CORBA_string_dup (value);

      s = odl_field_get_source_fields (f);
      t = odl_field_get_this_fields (f);

      /* record how to set this object: not done yet */
      setd = &retval->_buffer[i].setdata;
      setd->_length = g_list_length (s);        /* number of fields to match */
      setd->_buffer = CORBA_sequence_GEAS_setlookup_allocbuf (setd->_length);
      j = 0;
      while (s && t)
        {
          char *fieldval;

          fieldval = GEAS_DataObject_getField (obj, s->data, ev);

          if (ev->_major != CORBA_NO_EXCEPTION)
            {
              /* ..clean up.. */
              return (NULL);
            }
          setd->_buffer[j].field = CORBA_string_dup (t->data);
          setd->_buffer[j].value = CORBA_string_dup (fieldval);
          /* message( " %s.%s = %s" , classname , setd->_buffer[j].field ,
             setd->_buffer[j].value ); */

          s = g_list_next (s);
          t = g_list_next (t);
          j++;
        }

      CORBA_Object_release (obj, ev);
      if (ev->_major != CORBA_NO_EXCEPTION)
        {
          CORBA_free (value);
          /* ..clean up.. */
          return (NULL);
        }
    }

  /* done */
  return (retval);

}

/* raises: UnknownFeature */
CORBA_boolean
Connection_supportsFeature (GEAS_object_reference * id,
                            GEAS_Feature featureID, CORBA_Environment * ev)
{
  switch (featureID)
    {
    case GEAS_Transactions:
      /* not supported yet */
      return (CORBA_FALSE);

    case GEAS_Security:
      /* not supported yet */
      return (CORBA_FALSE);
    }

  /* by default, features are not supported */
  return (CORBA_FALSE);
}

/* raises: UnknownClass, ServerError, TransactionNotInProgress */
GEAS_DataObject
Connection_newObject (GEAS_object_reference * id,
                      CORBA_char * classname, CORBA_Environment * ev)
{
  GEAS_DataObject retval = CORBA_OBJECT_NIL;
  char *keystr = NULL;
  ObjectKey key = NULL;
  odl_class *cl;

  cl = odl_find_class (all_classes, classname, NULL);
  if (!cl)
    {
      make_UnknownClass_exception (ev, "Class '%s' not known", classname);
      return (CORBA_OBJECT_NIL);
    }

  key = create_new_object_key ();
  if (!key)
    {
      make_ServerError_exception (ev, "Error creating new key");
      return (CORBA_OBJECT_NIL);
    }

  keystr = (char *) object_key_as_string (key);
  if (!keystr)
    {
      free_object_key (key);
      make_ServerError_exception (ev, "Error creating new key");
      return (CORBA_OBJECT_NIL);
    }

  /* don't log GEAS classes */
  if (allow_log_class_event (classname))
    {
      dm_event (id->username, id->currenttransaction, DM_EVENT_OBJECT_NEW,
                odl_class_get_full_name (cl), keystr);
    }
  retval =
    make_dataobject_reference (classname, keystr, id->username, id->sessionid,
                               ev);
  return (retval);
}

/*
 * this is just another nasty hack. it's not my fault, and I did not want
 * this.  go bother someone else if it doesn't make sense.  it does definately
 * cause problems, though, not that I expect anyone to pay any attention to me - I
 * only wrote GEAS, I wouldn't know how to use it, or what problems it has.
 * 
 */

void
Connection_realNewObject (GEAS_object_reference * id)
{
  odl_class *cl;
  GList *fields, *fl;
  ObjectData *ob;
  GList *parents, *l;
  /* CORBA_Environment ev; */

  cl = odl_find_class (all_classes, id->classname, NULL);
  if (!cl)
    {
      return;
    }
  parents = odl_class_get_parentnames (cl);

  /* requested class to cache */
  ob = oc_add_empty_object (odl_class_get_full_name (cl), id->objectid);
  oc_set_object_flag (ob, of_indatabase, FLAG_ON);
  fields = odl_class_get_fields (cl, FT_basic);
  fl = fields;
  while (fl)
    {
      odl_field *f = fl->data;

      if (odl_field_get_default (f))
        {
          oc_set_object_field (ob, odl_field_get_name (f),
                               odl_field_get_default (f), FALSE);
        }
      fl = g_list_next (fl);
    }
  odl_fieldlist_free (fields);

  /* handle parent classes */
  l = parents;
  while (l)
    {
      odl_class *c = odl_find_class (all_classes, l->data, NULL);

      if (!c)
        {
          g_list_free (parents);
          return;
        }

      ob = oc_add_empty_object (l->data, id->objectid);
      oc_set_object_flag (ob, of_indatabase, FLAG_ON);
      fields = odl_class_get_fields (c, FT_basic);
      fl = fields;
      while (fl)
        {
          odl_field *f = fl->data;

          if (odl_field_get_default (f))
            {
              oc_set_object_field (ob, odl_field_get_name (f),
                                   odl_field_get_default (f), FALSE);
            }
          fl = g_list_next (fl);
        }
      g_list_free (fields);
      l = g_list_next (l);
    }
  if (parents)
    {
      odl_namelist_free (parents);
    }
  oc_set_object_flag (ob, of_indatabase, FLAG_OFF);
}

GEAS_ObjectList
Connection_newEmptyList (GEAS_object_reference * id,
                         CORBA_char * classname, CORBA_Environment * ev)
{
  GEAS_ObjectList retval;
  char *listid;
  GEAS_DataObject lst;

  lst = GEAS_Connection_newObject (id->server, "geas::listholder", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      return (CORBA_OBJECT_NIL);
    }
  listid = GEAS_DataObject_getField (lst, "objectID", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      GEAS_DataObject_delete (lst, ev);
      CORBA_Object_release (lst, ev);
      return (CORBA_OBJECT_NIL);
    }
  GEAS_DataObject_setField (lst, "length", "0", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      GEAS_DataObject_delete (lst, ev);
      CORBA_Object_release (lst, ev);
      return (CORBA_OBJECT_NIL);
    }
  GEAS_DataObject_setField (lst, "classname", classname, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      GEAS_DataObject_delete (lst, ev);
      CORBA_Object_release (lst, ev);
      return (CORBA_OBJECT_NIL);
    }
  retval = make_list_reference (listid, id->username, id->sessionid, ev);
  CORBA_free (listid);
  return (retval);
}

/* raises: UnknownClass, UnknownField, ServerError */
GEAS_ObjectList
Connection_loadObjects (GEAS_object_reference * id,
                        CORBA_char * classname,
                        CORBA_char * fieldname,
                        CORBA_char * searchvalue, CORBA_Environment * ev)
{
  GEAS_ObjectList retval;
  QueryData *q;
  char *keystr;
  /* char tmp[256]; */

  /* create query */
  if (!odl_find_class (all_classes, classname, NULL))
    {
      make_ServerError_exception (ev, "Class '%s' not found", classname);
      return (CORBA_OBJECT_NIL);
    }

  q =
    oql_objects_field_search (classname, fieldname, searchvalue);
  if (!q)
    {
      message (__PRETTY_FUNCTION__ " 1");
      make_ServerError_exception (ev, "Could not create query");
      return (CORBA_OBJECT_NIL);
    }

  /* get data */
  keystr = new_object_collection_from_query (q, id, ev);
  oql_free_query (q);
  if (!keystr)
    {
      make_ServerError_exception (ev,
                                  "1 Could not create list storage structure");
      return (CORBA_OBJECT_NIL);
    }
  /* create ObjectList reference */
  retval =
    (GEAS_ObjectList) make_list_reference (keystr, id->username,
                                           id->sessionid, ev);
  CORBA_free (keystr);
  return retval;
}

/* raises: UnknownClass, ServerError */
GEAS_ObjectList
Connection_loadAll (GEAS_object_reference * id,
                    CORBA_char * classname, CORBA_Environment * ev)
{
  GEAS_ObjectList retval;
  QueryData *q;
  char *keystr;

  /* create query */
  if (!odl_find_class (all_classes, classname, NULL))
    {
      make_ServerError_exception (ev, "Class '%s' not found", classname);
      return (CORBA_OBJECT_NIL);
    }

  q = oql_find_all_objects (classname);
  if (!q)
    {
      make_ServerError_exception (ev, "Could not create query");
      return (CORBA_OBJECT_NIL);
    }
  keystr = new_object_collection_from_query (q, id, ev);
  oql_free_query (q);
  if (!keystr)
    {
      make_ServerError_exception (ev,
                                  "2 Could not create list storage structure");
      return (CORBA_OBJECT_NIL);
    }
  /* create ObjectList reference */
  retval =
    (GEAS_ObjectList) make_list_reference (keystr, id->username,
                                           id->sessionid, ev);
  CORBA_free (keystr);
  return retval;
}

/* raises: UnknownClass,UnknownField,ServerError */
GEAS_DataObject
Connection_loadSingleObject (GEAS_object_reference * id,
                             CORBA_char * classname,
                             CORBA_char * fieldname,
                             CORBA_char * searchvalue, CORBA_Environment * ev)
{
  GEAS_DataObject retval;
  QueryData *q;
  /* char *keystr; */
  /* char tmp[256]; */
  struct query_result *result;
  int err;
  char *errmsg;
  int idx;
  _ObjectData *obj;

  /* create query */
  if (!odl_find_class (all_classes, classname, NULL))
    {
      make_ServerError_exception (ev, "Class '%s' not found", classname);
      return (CORBA_OBJECT_NIL);
    }

  /* printf( "load %s where %s.%s = %s\n" , classname , classname , fieldname
     , searchvalue ); */
  q =
    oql_objects_field_search (classname, fieldname, searchvalue);
  if (!q)
    {
      message (__PRETTY_FUNCTION__ " 3");
      make_ServerError_exception (ev, "Could not create query");
      return (CORBA_OBJECT_NIL);
    }
  oql_limit_results (q, 1);

  obj = oc_search_for_single_object (q);
  if (obj)
    {
      retval = make_dataobject_reference (oc_get_object_class (obj),
                                          oc_get_object_key (obj),
                                          id->username, id->sessionid, ev);
      return (retval);
    }

  /* perform the query */
  result = query_objectstore (q, &err, &errmsg);
  if (errmsg || !result)
    {
      if (errmsg)
        {
          make_ServerError_exception (ev, errmsg);
        }
      else
        {
          make_ServerError_exception (ev, "Query failed (unknown reason)");
        }
      if (errmsg)
        {
          errormsg (errmsg);
          g_free (errmsg);
        }
      oql_free_query (q);
      if (result)
        {
          free_query_result (result);
        }
      return (CORBA_OBJECT_NIL);
    }

  /* use first object returned (allows for searching on non-unique fields) */
  /* although that's a bad thing to do, as the object found is not */
  /* guaranteed to be the same for each call to this function */
  if (result->rows_affected > 0)
    {
      idx = oql_query_get_field_position (q, "objectid");
      retval = make_dataobject_reference (classname,
                                          get_result_field (result, 0, idx),
                                          id->username, id->sessionid, ev);
    }
  else
    {
      retval = CORBA_OBJECT_NIL;
    }

  /* done */
  free_query_result (result);
  oql_free_query (q);

  return retval;
}

/* raises: */
void
Connection_flush (GEAS_object_reference * id, CORBA_Environment * ev)
{
  /* Not required : leave blank until further notice */
}

/* raises: */
void
Connection_logout (GEAS_object_reference * id, CORBA_Environment * ev)
{
  /* terminate any transactions etc, free any other resources */
}

GEAS_Query_Query
Connection_newQuery (GEAS_object_reference * id, CORBA_Environment * ev)
{
  GEAS_Query_Query retval;
  GEAS_DataObject q = CORBA_OBJECT_NIL;
  char *uniqueid;

  /* make the object */
  q = Connection_newObject (id, "geas::searchcriteria", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      return (q);
    }
  /* fill in default values */
  GEAS_DataObject_setField (q, "parent", "", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      goto error;
    }
  GEAS_DataObject_setField (q, "classname", "unknown", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      goto error;
    }
  GEAS_DataObject_setField (q, "orderby", "", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      goto error;
    }
  GEAS_DataObject_setField (q, "reverse", "FALSE", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      goto error;
    }
  /* done */
  uniqueid = GEAS_DataObject__get_objectID (q, ev);
  retval = make_query_reference (uniqueid, id->username, id->sessionid, ev);
  CORBA_free (uniqueid);
  CORBA_Object_release (q, ev);
  return (retval);

error:
  /* ignore details */
  CORBA_exception_free (ev);
  CORBA_exception_init (ev);

  GEAS_DataObject_delete (q, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      /* ignore this exception */
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
    }
  CORBA_Object_release (q, ev);
  make_ServerError_exception (ev, "Error allocating new query.");
  return (CORBA_OBJECT_NIL);
}

static void
recursive_add_constraints_to_query (GEAS_object_reference * id,
                                    oql_constraint * c,
                                    const char *queryid,
                                    CORBA_Environment * ev)
{
  oql_constraint *parent = c;
  /* enum oql_logic logic = oql_and; */
  GEAS_ObjectList list = CORBA_OBJECT_NIL;
  GEAS_DataObject obj = CORBA_OBJECT_NIL;
  int len = 0, ii = 0;
  char *subid = NULL, *tmp = NULL;
  oql_field *f = NULL;
  char *field = NULL, *value = NULL, *teststr, *casestr = NULL, *invertstr =
    NULL;
  enum querytest test = q_equals;
  gboolean casesensitive = TRUE, invert = FALSE;

  /* get all fields with 'constraint' == queryid */
  list =
    Connection_loadObjects (id, "geas::searchfield", "constraint",
                            (char *) queryid, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      return;
    }
  len = GEAS_ObjectList__get_length (list, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      goto error;
    }
  for (ii = 0; ii < len; ii++)
    {
      obj = GEAS_ObjectList_getEntry (list, ii, ev);
      if (ev->_major != CORBA_NO_EXCEPTION)
        goto error;

      /* make the oql_field */
      field = GEAS_DataObject_getField (obj, "field", ev);
      if (ev->_major != CORBA_NO_EXCEPTION)
        goto error;
      value = GEAS_DataObject_getField (obj, "value", ev);
      if (ev->_major != CORBA_NO_EXCEPTION)
        goto error;
      teststr = GEAS_DataObject_getField (obj, "test", ev);
      if (ev->_major != CORBA_NO_EXCEPTION)
        goto error;
      casestr = GEAS_DataObject_getField (obj, "casesensitive", ev);
      if (ev->_major != CORBA_NO_EXCEPTION)
        goto error;
      invertstr = GEAS_DataObject_getField (obj, "invert", ev);
      if (ev->_major != CORBA_NO_EXCEPTION)
        goto error;

      if (casestr[0] == '1')
        casesensitive = TRUE;
      else
        casesensitive = FALSE;
      if (invertstr[0] == '1')
        invert = TRUE;
      else
        invert = FALSE;

      test = q_equals;
      if (teststr[0] == '1' && !invert)
        test = q_equals;
      else if (teststr[0] == '2' && !invert)
        test = q_lessthan;
      else if (teststr[0] == '3' && !invert)
        test = q_greaterthan;
      else if (teststr[0] == '4' && !invert)
        test = q_contains;
      else if (teststr[0] == '5' && !invert)
        test = q_startswith;
      else if (teststr[0] == '6' && !invert)
        test = q_endswith;

      f = oql_make_field (field, test, value, casesensitive, invert);

      CORBA_free (field);
      CORBA_free (value);
      CORBA_free (invertstr);
      CORBA_free (casestr);
      CORBA_free (teststr);
      field = NULL;
      value = NULL;
      invertstr = NULL;
      casestr = NULL;
      teststr = NULL;

      /* add it to the constraint */
      oql_add_field (c, f);
    }

  GEAS_ObjectList_release (list, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      list = CORBA_OBJECT_NIL;
      goto cleanup;
    }
  CORBA_Object_release (list, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      list = CORBA_OBJECT_NIL;
      goto cleanup;
    }
  list = CORBA_OBJECT_NIL;

  /* get objectid of all geas::searchcriteria instances with parent == queryid */

  list =
    Connection_loadObjects (id, "geas::searchcriteria", "parent",
                            (char *) queryid, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      return;
    }
  len = GEAS_ObjectList__get_length (list, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      goto error;
    }
  /* for each of those, call this function again :) */
  for (ii = 0; ii < len; ii++)
    {
      enum oql_logic logic;
      oql_constraint *c = NULL;

      obj = GEAS_ObjectList_getEntry (list, ii, ev);
      if (ev->_major != CORBA_NO_EXCEPTION)
        goto error;

      subid = GEAS_DataObject_getField (obj, "objectid", ev);
      if (ev->_major != CORBA_NO_EXCEPTION)
        goto error;

      tmp = GEAS_DataObject_getField (obj, "logic", ev);
      if (ev->_major != CORBA_NO_EXCEPTION)
        goto error;
      if (tmp[0] == '1')
        logic = oql_and;
      else if (tmp[0] == '2')
        logic = oql_or;
      else
        logic = oql_and;
      CORBA_free (tmp);
      tmp = NULL;

      c = oql_make_constraint (logic);
      if (!c)
        {
          fatal_error ("This ought to be impossible...\n");
        }
      oql_add_constraint (parent, c);
      recursive_add_constraints_to_query (id, c, subid, ev);

      CORBA_free (subid);
      subid = NULL;
    }

  /* done */
  goto cleanup;

error:

cleanup:
  if (subid)
    {
      CORBA_free (subid);
    }
  if (list != CORBA_OBJECT_NIL)
    {
      GEAS_ObjectList_release (list, ev);
      if (ev->_major != CORBA_NO_EXCEPTION)
        {
          list = CORBA_OBJECT_NIL;
        }
      CORBA_Object_release (list, ev);
      if (ev->_major != CORBA_NO_EXCEPTION)
        {
          list = CORBA_OBJECT_NIL;
        }
    }
}

GEAS_ObjectList
Connection_executeQuery (GEAS_object_reference * id,
                         GEAS_Query_Query querytoperform,
                         CORBA_Environment * ev)
{
  char *keystr = NULL;
  char *queryid = NULL, *classname = NULL, *orderby = NULL;
  char *temp = NULL;
  ObjectData *ob = NULL;
  odl_class *c;
  gboolean reverse = FALSE;
  enum oql_logic logic = oql_and;
  QueryData *q = NULL;
  GEAS_ObjectList retval = CORBA_OBJECT_NIL;
  oql_constraint *constraint = NULL;
  char *limit = NULL;

  /* find root of query */
  queryid = GEAS_Query_Query__get_id (querytoperform, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    return (CORBA_OBJECT_NIL);
  ob = oc_find_object_by_key ("geas::searchcriteria", queryid);

  /* find root details */
  classname = oc_get_object_field (ob, "classname");
  c = odl_find_class (all_classes, classname, NULL);
  if (!c)
    {
      make_ServerError_exception (ev, "Class %s was not found", classname);
      return (CORBA_OBJECT_NIL);
    }
  orderby = oc_get_object_field (ob, "orderby");

  temp = oc_get_object_field (ob, "reverse");
  if (temp[0] == '1')
    reverse = TRUE;
  else
    reverse = FALSE;
  g_free (temp);

  /* nasty, but hey, it should work */
/*  if( c->desc ) reverse = !reverse;
*/
  temp = oc_get_object_field (ob, "logic");
  if (temp[0] == '1')
    logic = oql_and;
  if (temp[0] == '2')
    logic = oql_or;
  g_free (temp);
  temp = NULL;

  /* make basic query */
  q = oql_load_object (odl_class_get_full_name (c));
  if (!q)
    goto error;

  if (limit)
    {
      if (strlen (limit) > 0 && isdigit (limit[0]))
        oql_limit_results (q, atoi (limit));
      CORBA_free (limit);
    }

  /* handle 'order by' */
  if (orderby)
    {
      add_order_by (q, orderby, reverse);
    }
  /* handle constraints */
  constraint = oql_make_constraint (logic);
  if (!constraint)
    {
      fatal_error ("This ought to be impossible...\n");
    }
  recursive_add_constraints_to_query (id, constraint, queryid, ev);
  oql_add_complex_constraint (q, constraint);

  /* execute query */
  debug_output (DEBUGLEVEL_8, "Complex query: '%s'",
                oql_query_as_sql (q, OQL_DBTYPE_CACHEONLY));
  keystr = new_object_collection_from_query (q, id, ev);
  if (!keystr)
    {
      make_ServerError_exception (ev,
                                  "1 Could not create list storage structure");
      goto error;
    }
  retval =
    (GEAS_ObjectList) make_list_reference (keystr, id->username,
                                           id->sessionid, ev);
  CORBA_free (keystr);

  /* done */
  return (retval);

  /* cleanup after error */
error:
  if (queryid)
    {
      g_free (queryid);
    }
  if (classname)
    {
      g_free (classname);
    }
  if (orderby)
    {
      g_free (orderby);
    }
  return (CORBA_OBJECT_NIL);
}

#if 0
static GEAS_ObjectList
old_Connection_executeQuery (GEAS_object_reference * id,
                             GEAS_Query_Query querytoperform,
                             CORBA_Environment * ev)
{
  char *keystr = NULL;
  odl_class *c = NULL;
  GEAS_ObjectList list = CORBA_OBJECT_NIL;
  CORBA_char *tmp = NULL;
  CORBA_char *classname = NULL, *orderby = NULL;
  gboolean reverse;
  GEAS_Query_logictype logic;
  CORBA_char *queryid = NULL;
  int i, length;
  GEAS_Query_Field *field = NULL;
  char *buf = NULL;
  GList *l = NULL, *tl;
  GEAS_ObjectList retval = CORBA_OBJECT_NIL;
  QueryData *query = NULL;
  QueryData *query2 = NULL;
  ObjectData *ob = NULL;
  CORBA_char *fieldid = NULL;
  struct query_result *result = NULL;
  DatabaseResultRow row;
  int fieldidcol;

  /* printf( "executing...\n" ); */
  /* find the query object */
  queryid = GEAS_Query_Query__get_id (querytoperform, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      return (CORBA_OBJECT_NIL);
    }
  /* printf( "done query\n" ); */
  ob = oc_find_object_by_key ("geas::searchcriteria", queryid);

  /* get classname, orderby, reverse, logic */
  classname = oc_get_object_field (ob, "classname");
  c = odl_find_class (all_classes, classname, NULL);
  if (!c)
    {
      make_ServerError_exception (ev, "Class %s was not found", classname);
      return (CORBA_OBJECT_NIL);
    }

  orderby = oc_get_object_field (ob, "orderby");

  tmp = oc_get_object_field (ob, "reverse");
  if (tmp[0] == '0')
    {
      reverse = FALSE;
    }
  else
    {
      reverse = TRUE;
    }
  g_free (tmp);

  tmp = oc_get_object_field (ob, "reverse");
  if (strcmp (tmp, "0") == 0)
    logic = GEAS_Query_UNKNOWN;
  if (strcmp (tmp, "1") == 0)
    logic = GEAS_Query_AND;
  if (strcmp (tmp, "2") == 0)
    logic = GEAS_Query_OR;
  g_free (tmp);

  /* read fields */
  query2 = oql_load_object ("geas::searchfield");
  do_oql_add_query_constraint (query2, "geas::searchfield", "constraint", "=",
                               queryid, q_hackish, TRUE);
  result = query_objectstore (query2, NULL, NULL);
  if (!result)
    {
      goto error;
    }
  if (!result->success)
    {
      goto error;
    }

  length = result->rows_affected;
  fieldidcol = oql_query_get_field_position (query2, "objectid");
  for (i = 0; i < length; i++)
    {
      row = get_result_row (result, i);
      field = g_new (GEAS_Query_Field, 1);

      fieldid = get_field_in_row (row, fieldidcol);
      ob = oc_find_object_by_key ("geas::searchfield", fieldid);

      field->field = oc_get_object_field (ob, "field");
      field->value = oc_get_object_field (ob, "value");

      buf = oc_get_object_field (ob, "test");
      if (strcmp (buf, "1") == 0)
        field->test = GEAS_Query_equals;
      if (strcmp (buf, "2") == 0)
        field->test = GEAS_Query_lessthan;
      if (strcmp (buf, "3") == 0)
        field->test = GEAS_Query_greaterthan;
      if (strcmp (buf, "4") == 0)
        field->test = GEAS_Query_contains;
      if (strcmp (buf, "5") == 0)
        field->test = GEAS_Query_startswith;
      if (strcmp (buf, "6") == 0)
        field->test = GEAS_Query_endswith;
      g_free (buf);

      buf = oc_get_object_field (ob, "invert");
      if (strcmp (buf, "1") == 0)
        field->invert = CORBA_TRUE;
      if (strcmp (buf, "0") == 0)
        field->invert = CORBA_FALSE;
      g_free (buf);

      buf = oc_get_object_field (ob, "casesensitive");
      if (strcmp (buf, "1") == 0)
        field->casesensitive = CORBA_TRUE;
      if (strcmp (buf, "0") == 0)
        field->casesensitive = CORBA_FALSE;

      l = g_list_append (l, field);
    }

  /* display */
  /* printf( "Load %s WHERE \n" , odl_class_get_full_name(c) ); */
  query = oql_load_object (odl_class_get_full_name (c));
  if (!query)
    goto error;

  tl = l;
  while (tl)
    {
      enum querytest q_test;
      char *test, *casesensitive;
      field = tl->data;

      test = "(test)";
      switch (field->test)
        {
        case GEAS_Query_equals:
          if (!field->invert)
            q_test = q_equals;
          else
            q_test = q_notequals;
          break;
        case GEAS_Query_lessthan:
          if (!field->invert)
            q_test = q_lessthan;
          else
            q_test = q_notlessthan;
          break;
        case GEAS_Query_greaterthan:
          if (!field->invert)
            q_test = q_greaterthan;
          else
            q_test = q_notgreaterthan;
          break;
        case GEAS_Query_contains:
          if (!field->invert)
            q_test = q_contains;
          else
            q_test = q_notcontains;
          break;
        case GEAS_Query_startswith:
          if (!field->invert)
            q_test = q_startswith;
          else
            q_test = q_notstartswith;
          break;
        case GEAS_Query_endswith:
          if (!field->invert)
            q_test = q_endswith;
          else
            q_test = q_notendswith;
          break;
        }
      if (field->casesensitive)
        {
          casesensitive = " (case sensitive)";
        }
      else
        {
          casesensitive = "";
        }
      /* printf( "  %s %s %s%s" , field->field , test , field->value,
         casesensitive ); */

      do_oql_add_query_constraint (query, classname, field->field, "",
                                   field->value, q_test,
                                   field->casesensitive);

      if (tl->next)
        {
          if (logic == GEAS_Query_AND)
            printf (" AND");
          if (logic == GEAS_Query_OR)
            printf (" OR");
        }
      /* printf( "\n" ); */
      tl = tl->next;
    }
#if 0
  if (strlen (orderby) > 0)
    {
      printf ("ORDER BY %s (%s)\n\n", orderby,
              reverse ? "descending" : "ascending");
      add_order_by (query, orderby, reverse);
    }
#endif

  /*      printf( "QUERY: '%s'\n" , oql_query_as_sql(query,OQL_DBTYPE_CACHEONLY) ); */
  /* get data */
  keystr = new_object_collection_from_query (query, id, ev);
  if (!keystr)
    {
      make_ServerError_exception (ev,
                                  "1 Could not create list storage structure");
      goto error;
    }
  /* create ObjectList reference */
  retval =
    (GEAS_ObjectList) make_list_reference (keystr, id->username,
                                           id->sessionid, ev);
  CORBA_free (keystr);

  /* done */
  goto done;

error:
  errormsg ("CORBA exception: '%s'", CORBA_exception_id (ev));

done:

  if (query)
    oql_free_query (query);
  if (query2)
    oql_free_query (query2);
  if (queryid)
    CORBA_free (queryid);
  if (classname)
    g_free (classname);
  if (orderby)
    g_free (orderby);
  if (result)
    free_query_result (result);

  if (field)
    {
      if (field->field)
        g_free (field->field);
      if (field->value)
        g_free (field->value);
      g_free (field);
    }

  if (list)
    CORBA_Object_release (list, ev);
  return (retval);
}
#endif

GEAS_LoadDefinition *
Connection_getReferenceDefinition (GEAS_object_reference * id,
                                   CORBA_char * classname,
                                   CORBA_char * fieldname,
                                   CORBA_Environment * ev)
{
  odl_class *c;
  odl_field *f;
  GEAS_LoadDefinition *d;
  GList *a, *b;
  int i;

  c = odl_find_class (all_classes, classname, NULL);
  if (!c)
    {
      make_ServerError_exception (ev, "Could not find class '%s'", classname);
      return (NULL);
    }
  f = odl_class_get_field (c, fieldname);
  if (!c)
    {
      make_ServerError_exception (ev,
                                  "Could not find field '%s' in class '%s'",
                                  fieldname, classname);
      return (NULL);
    }

  if (!(f->fieldtype == FT_list ||
        f->fieldtype == FT_reference || f->fieldtype == FT_lookup))
    {
      make_ServerError_exception (ev, "eC");
      return (NULL);
    }
  d = GEAS_LoadDefinition__alloc ();

  if (f->fieldtype == FT_list)
    d->type = GEAS_list;
  if (f->fieldtype == FT_reference)
    d->type = GEAS_reference;
  if (f->fieldtype == FT_lookup)
    d->type = GEAS_lookup;

  if (f->sourceclass)
    d->classname = CORBA_string_dup (f->sourceclass);
  else
    f->sourceclass = CORBA_string_dup ("unknown");

  if (f->sourcefield)
    d->loadfield = CORBA_string_dup (f->sourcefield);
  else
    d->loadfield = CORBA_string_dup ("");

  d->fieldname = CORBA_string_dup (fieldname);


  d->srcfields._maximum = d->srcfields._length =
    g_list_length (f->source_fields);
  d->srcfields._buffer =
    CORBA_sequence_CORBA_string_allocbuf (d->srcfields._maximum);

  d->thisfields._maximum = d->thisfields._length =
    g_list_length (f->this_fields);
  d->thisfields._buffer =
    CORBA_sequence_CORBA_string_allocbuf (d->thisfields._maximum);

  a = f->source_fields;
  i = 0;
  while (a)
    {
      d->srcfields._buffer[i] = CORBA_string_dup (a->data);
      a = a->next;
      i++;
    }

  for (i = 0, b = f->this_fields; b != NULL; i++, b = g_list_next (b))
    {
      d->thisfields._buffer[i] = CORBA_string_dup (b->data);
    }
  return (d);
}


/* for the last time, this should not be used. ever. */
void
Connection_clearCache (GEAS_object_reference * id, CORBA_Environment * ev)
{
  /* this is _supposed_ to be used to clear the cache on server */
  /* shutdown, not whenever any user wants to. using this will cause */
  /* a performance hit for all clients */
  oc_empty_cache ();
}

CORBA_unsigned_long
Connection_countAllInstances (GEAS_object_reference * id,
                              CORBA_char * classname, CORBA_Environment * ev)
{
  QueryData *q = NULL;
  struct query_result *result = NULL;
  char *buf;
  CORBA_unsigned_long retval = 0;

  /* is this database independant? */
  buf = g_strdup_printf ("SELECT count(objectid) from %s",
                         oql_quote_column (classname, NULL,
                                           OQL_DBTYPE_POSTGRESQL));
  q = oql_make_sql_query (buf);
  g_free (buf);
  result = query_objectstore (q, NULL, NULL);
  if (result)
    {
      if (result->success)
        {
          const char *res = get_result_field (result, 0, 0);
          /* database independance?  who needs that when you have grep */
          retval = atoi (res);
        }
      free_query_result (result);
    }
  /* everything here seems fine to someone */
  return retval;
}

#include "event_log_code.h"
