
%option    never-interactive

%{

/* 
   Source file: lparser.l
   
   Part of GNU Enterprise Application Server (GEAS)

   Copyright (C) 2001 Free Software Foundation

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
   
   $Id: lparser.l,v 1.28 2001/08/24 21:30:58 ntiffin Exp $

*/
#include "config.h"

#include <glib.h>
#include <string.h>
#include <stdlib.h>

#include "gcdparser.h"
#include "classdata.h"
#include "yparser.h"
#include "lparser.h"

unsigned int expecting_datatype  = 1;
unsigned int expecting_fieldprop = 0;
unsigned int expect_triggertype  = 0;

int yy_show_file_error = 1;

int current_pass = 0;

int yywarn( char *msg , ... );
int yyerror( char *msg , ... );
int yymessage( char *msg , ... );

/* error counting */
unsigned long int error_count = 0;
unsigned long int warn_count  = 0;

/* recording lines for error messages */
#define LINEBUF_LENGTH 1024
static char     linebuf[LINEBUF_LENGTH];
static long int token_pos = 1;
static int first_line = 1;
static int lastyyleng = 0;
#define ST lastyyleng=yyleng

/* tracing position in file(s) */
#define FILEDATA_FILENAME_LENGTH 512
struct FileData
{
   /* stack link */
   struct FileData *next;

   /* file line info */
   char name[FILEDATA_FILENAME_LENGTH];
   unsigned long int line;
   
   /* current module info */
   char modulename[256];
   char moduleprefix[128];
   
   /* store lexer state */
   YY_BUFFER_STATE yybuffer;
   
   int dummy[4096];
   /* int dummy[YY_BUF_SIZE]; */
};

                              /* current file being processed */
struct FileData currentfile = { NULL , "(reading from stdin)" , 1 , "" , "" , 0 };
                              /* pointer to most recent file pushed */
static struct FileData *file_stack = NULL; 

static int push_file( char *filename );
static int pop_file(void);           /* return 0 if nothing to pop else return 1 */

static char         **processed_files = NULL;
static unsigned int   processed_count = 0;

static int file_processed( const char *filename );
static int record_file_processed( const char *filename );

#ifdef DEBUG
static void dump_filedata_struct(struct FileData * filedata_ptr);
static void dump_file_stack(char * note) __attribute__ ((unused));
#endif

typedef char *charptr;

%}

/*
 * set up exclusive start conditions
 * exclusive = rules with no start conditions are not
 * active when these conditions are active
 */
%x incl
%x endincl
%x failinclude
%x FULLLINE

qstring    \"[^"\n]*\"
scomment   #.*
name       [a-zA-Z][a-zA-Z0-9]*
warnname   [a-zA-Z][a-zA-Z0-9_]*
whitespace [ \t\r]*
newline    \n

%%

%{ /* NO RULES BEFORE THIS LINE */
    /* makes sure the first line of the file gets recorded */
    if( first_line == 1 )
    {
        /* setup to read the first line of the primary file */
        first_line = 0; BEGIN(FULLLINE);
    }
    /* update token pos by the length of the last token */
    token_pos += lastyyleng;
%}

include            { token_pos += yyleng; BEGIN(incl); }

<*>\/\/.*          { /* ignore rest of line */ }

<incl>{whitespace} { /* ignore */ token_pos += yyleng; }
<incl>{qstring}    { 
                     char *p,*q;
                     p = yytext + 1;
                     q = strchr( p , '"' );
                     *q = '\0';
                     #ifdef DEBUG
                     printf("     Found include: %s,", p);
                     #endif
                     if( file_processed(p) )
                       {
                         #ifdef DEBUG
                         printf(" ALREADY processed.\n");
                         #endif
                         BEGIN(endincl);
                       }
                     else
                       {
                         /* not already processed, so start it now */
                         #ifdef DEBUG
                         printf(" processing now.\n");
                         #endif
                         record_file_processed(p);
                         if( push_file(p) )
                           {
                             BEGIN(FULLLINE);
                           }
                         else
                           {
                             BEGIN(failinclude);
                           }
                       }
                   }

<incl>{whitespace}{newline}    {
                                 yywarn("No filename given for include directive");
                                 BEGIN(failinclude);
                               }

<incl>{whitespace}[^\"\n]+\n |
<incl>{whitespace}[^\"\n]*     {
                                 yywarn("Include filename must be quoted.\n");
                                 BEGIN(failinclude);
                               }

<endincl>{whitespace}*  /* ignore whitespace */

<endincl>\n             {
                          currentfile.line++;
                          /* return to normal processing */
                          BEGIN(FULLLINE); 
                        }

<failinclude>.*\n   {
                      currentfile.line++;
                      /* return to normal processing after skipping failed include */
                      BEGIN(FULLLINE);
                    }

<<EOF>>             {
                      if(pop_file() == 0 )
                        {
                          yy_delete_buffer(YY_CURRENT_BUFFER);
                          YY_CURRENT_BUFFER = NULL;
                          /*printf( "fclose : %08lx\n" , yyin );*/
                          fclose(yyin);
                          yyterminate();
                        }
                      else
                        {
                          BEGIN(endincl);
                        }
                    }

<FULLLINE>\n      |
<FULLLINE>[^\n]*    { /* record the current line, truncating if necessary */
                      if (strlen( yytext) > LINEBUF_LENGTH)
                        {
                          yyerror( "Current line too long." );
                          yyerror( yytext );
                          g_assert_not_reached();
                        }
                      strncpy( linebuf , yytext , LINEBUF_LENGTH-1 );
                      linebuf[ strlen(linebuf) ] = '\0';
                      token_pos = 1;
                      BEGIN(INITIAL);
                      yyless(0);
                    }

^{whitespace}{newline}           |
^{whitespace}{scomment}{newline} |
{scomment}{newline}                { /* blank lines */
                                     currentfile.line++;
                                     BEGIN(FULLLINE);
                                   }

{newline}           {
                      currentfile.line++;
                      BEGIN(FULLLINE);
                    }

{whitespace}        { /* ignore whitespace*/
                      token_pos += yyleng;
                    }

{qstring}           {
                      ST;
                      while( yytext[strlen(yytext)-1] == '"' )
                        {
                          yytext[strlen(yytext)-1] = '\0';
                        }
                      yylval.string = g_strdup(&yytext[1]);
                      return(STRING);
                    }

[+-]?[0-9]+         { ST; yylval.integer  = atol(yytext); return(INTEGER);   }

[+-]?[0-9]+.[0-9]+  { ST; yylval.floating = atof(yytext); return(DOUBLE);    }

                    /* this set can be followed by more datatype fields,    */
                    /*  and the parser must reset expecting_datatype        */
public              { ST; if( expecting_datatype == 1 ) return(PUBLIC);    else return(SYMBOL); }
protected           { ST; if( expecting_datatype == 1 ) return(PROTECTED); else return(SYMBOL); }
private             { ST; if( expecting_datatype == 1 ) return(PRIVATE);   else return(SYMBOL); }
final               { ST; if( expecting_datatype == 1 ) return(FINAL);     else return(SYMBOL); }
abstract            { ST; if( expecting_datatype == 1 ) return(ABSTRACT);  else return(SYMBOL); }
unsigned            { ST; if( expecting_datatype == 1 ) return(UNSIGNED);  else return(SYMBOL); }
singleton           { ST; if( expecting_datatype == 1 ) return(SINGLETON); else return(SYMBOL); }
readonly            { ST; if( expecting_datatype == 1 ) return(READONLY);  else return(SYMBOL); }

                    /* the same, except using expecting_fieldprop */
not                 { ST; if( expecting_fieldprop == 1 ) return(NOT); else return(SYMBOL); }
null                { ST; if( expecting_fieldprop == 1 ) return(NUL); else return(SYMBOL); }

                    /* this set is the final entry in a datatype,    */
                    /*  so can reset expecting_datatype here         */
module              {
                      ST;
                      if( expecting_datatype == 1 )
                        { 
                          expecting_datatype = 0;
                          return(MODULE);
                        } 
                      else
                        {
                          yylval.string=g_strdup(yytext);
                          return(SYMBOL);
                        }
                    }
class               { ST; if( expecting_datatype == 1 ) { expecting_datatype = 0; return(CLASS);      } else { yylval.string=g_strdup(yytext); return(SYMBOL); } }
type                { ST; if( expecting_datatype == 1 ) { expecting_datatype = 0; return(CTYPE);      } else { yylval.string=g_strdup(yytext); return(SYMBOL); } }
extend              { ST; if( expecting_datatype == 1 ) { expecting_datatype = 0; return(EXTEND);     } else { yylval.string=g_strdup(yytext); return(SYMBOL); } }
unique              { ST; if( expecting_datatype == 1 ) { expecting_datatype = 0; return(UNIQUE);     } else { yylval.string=g_strdup(yytext); return(SYMBOL); } }
index               { ST; if( expecting_datatype == 1 ) { expecting_datatype = 0; return(INDEX);      } else { yylval.string=g_strdup(yytext); return(SYMBOL); } }
list                { ST; if( expecting_datatype == 1 ) { expecting_datatype = 0; return(LIST);       } else { yylval.string=g_strdup(yytext); return(SYMBOL); } }
lookup              { ST; if( expecting_datatype == 1 ) { expecting_datatype = 0; return(LOOKUP);     } else { yylval.string=g_strdup(yytext); return(SYMBOL); } }
reference           { ST; if( expecting_datatype == 1 ) { expecting_datatype = 0; return(REFERENCE);  } else { yylval.string=g_strdup(yytext); return(SYMBOL); } }
calculated          { ST; if( expecting_datatype == 1 ) { expecting_datatype = 0; return(CALCULATED); } else { yylval.string=g_strdup(yytext); return(SYMBOL); } }
trigger             { ST; if( expecting_datatype == 1 ) { expecting_datatype = 0; return(TRIGGER);    } else { yylval.string=g_strdup(yytext); return(SYMBOL); } }
enum                { ST; if( expecting_datatype == 1 ) { expecting_datatype = 0; return(ENUM);       } else { yylval.string=g_strdup(yytext); return(SYMBOL); } }
order               { ST; if( expecting_datatype == 1 ) { expecting_datatype = 0; return(ORDER);      } else { yylval.string=g_strdup(yytext); return(SYMBOL); } }
by                  { ST; if( expecting_datatype == 1 ) { expecting_datatype = 0; return(BY);         } else { yylval.string=g_strdup(yytext); return(SYMBOL); } }
desc                { ST; if( expecting_datatype == 1 ) { expecting_datatype = 0; return(DESC);       } else { yylval.string=g_strdup(yytext); return(SYMBOL); } }

[a-zA-Z][a-zA-Z0-9]*      { enum odl_datatype tmp;
                            ST;
                            if( expecting_datatype == 1 )
                              {
                                expecting_datatype = 0;
                                tmp = odl_datatype_id( yytext );
                                yylval.string = g_strdup( yytext );
                                if( tmp != DT_unknown ) return(DATATYPE);
                              }
                            else if(expect_triggertype == 1)
                              {
                                // yylval.integer = find_triggertype_id( yytext );
                                // if( yylval.integer != TT_UNKNOWN )
                                //     return(TRIGGERTYPE);
                              }
                           /* not expecting a datatype or trigger type, or didn't get one */ 
                           yylval.string = g_strdup( yytext );
                           return(SYMBOL);
                         }
                        
[_a-zA-Z][_a-zA-Z0-9]*  {   enum odl_datatype tmp;
                            ST;
                            if( expecting_datatype == 1 )
                              {
                                expecting_datatype = 0;
                                tmp = odl_datatype_id( yytext );
                                yylval.string = g_strdup( yytext );
                                if( tmp != DT_unknown ) return(DATATYPE);
                              }
                            else if(expect_triggertype == 1)
                              {
                                /* not valid symbol, but not expecting a symbol */
                                /* so the more useful syntax error will be generated */
                                return(SYMBOL);
                              }
                            /* not expecting a datatype or trigger type, or didn't get one */ 
                           if( current_pass == 1 )
                             {
                               if( strstr(yytext,"__") != NULL )
                                   yyerror( "Double underscores are not allowed in identifiers" );
                               else if( yytext[0] == '_' )
                                   yyerror( "Leading underscores are not allowed in identifiers" );
                               else if( yytext[ strlen(yytext)-1 ] == '_' )
                                   yyerror( "Trailing underscores are not allowed in identifiers" );
                             }
                           yylval.string = g_strdup( yytext );
                           return(SYMBOL);
                        }

.                   { ST; return( yytext[0] ); }

%%

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
int
yyerror (char *format, ...)
{
  int i;
  va_list ap;

  yyerrorcount++;
  fprintf (stderr, "error: ");
  if (yy_show_file_error == 1)
    {
      fprintf (stderr, "file %s, line %lu:\n", currentfile.name,
               currentfile.line);
    }
  va_start (ap, format);
  vfprintf (stderr, format, ap);
  va_end (ap);
  fprintf (stderr, "\n");
  if (yy_show_file_error == 1)
    {
      fprintf (stderr, "%s\n", linebuf);
      for (i = 1; i < token_pos; i++)
        {
          if (linebuf[i] == '\t')
            {
              fprintf (stderr, "\t");
            }
          else
            {
              fprintf (stderr, " ");
            }
        }
      fprintf (stderr, "^\n");
    }
  fprintf (stderr, "\n");
  return (0);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
int
yywarn (char *format, ...)
{
  int i;
  va_list ap;

  yywarncount++;
  fprintf (stderr, "warning: ");
  if (yy_show_file_error == 1)
    {
      fprintf (stderr, "file %s, line %lu:\n", currentfile.name,
               currentfile.line);
    }
  va_start (ap, format);
  vfprintf (stderr, format, ap);
  va_end (ap);
  fprintf (stderr, "\n");
  if (yy_show_file_error == 1)
    {
      fprintf (stderr, "%s\n", linebuf);
      for (i = 1; i < token_pos; i++)
        {
          if (linebuf[i] == '\t')
            {
              fprintf (stderr, "\t");
            }
          else
            {
              fprintf (stderr, " ");
            }
        }

      fprintf (stderr, "^\n");
    }
  fprintf (stderr, "\n");
  return (0);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
int
yymessage (char *format, ...)
{
  /*int i;*/
  va_list ap;

  fprintf (stderr, "message: ");
  if (yy_show_file_error == 1)
    {
      fprintf (stderr, "file %s, line %lu:\n", currentfile.name,
               currentfile.line);
    }
  va_start (ap, format);
  vfprintf (stderr, format, ap);
  va_end (ap);
  fprintf (stderr, "\n");
  return (0);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
void
yy_setfirstfilename (char *filename)
{
  strcpy (currentfile.name, filename);
}

#ifdef DEBUG
/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
static void
dump_file_stack(char * note)
{
  struct FileData * tmp = NULL;
  
  printf("\n**** Dump File Stack: %s *****\n", note);
  dump_filedata_struct(&currentfile);
  tmp = file_stack;
  while (tmp != NULL)
    {
      printf("\n      FileData: %p\n", tmp);
      dump_filedata_struct(tmp);
      tmp = tmp->next;
    }
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
static void
dump_filedata_struct(struct FileData * filedata_ptr)
{
  printf("          next: %p\n", filedata_ptr->next);
  printf("          name: %s\n", filedata_ptr->name);
  printf("          line: %lu\n", filedata_ptr->line);
  printf("   module name: %s\n", filedata_ptr->modulename);
  printf(" module prefix: %s\n", filedata_ptr->moduleprefix);
  printf("      yybuffer: %p\n", filedata_ptr->yybuffer);
  printf("         dummy: %s\n", (char *)filedata_ptr->dummy);
}
#endif

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
static int
push_file (char *fn)
{
  FILE *newfile;
  struct FileData *tmp;
  char *filename;
  char *p;
  /*
  printf ("\npush_file(%s)\n", fn);
  printf ("push_file() current.name: %s\n", currentfile.name);
  dump_file_stack("Before Push");
  */
  filename = g_strdup (currentfile.name);
  p = &filename[strlen (filename) - 1];
  /* strip filename from path/filename */
  while (p != filename)
    {
      if (*p == '/')
        {
          *p = '\0';
          break;
        }
      p--;
    }
  /*
  printf ("push_file() (after strip filename) filename: %s\n", filename);
  */
  /* if necessary add new path/file to old path */
  /* include file name is relative to current file location */
  if (strcmp (filename, "(reading from stdin)") == 0)
    {
      g_free (filename);
      filename = g_strdup (fn);
    }
  else if (p != filename)
    {
      char *tmp = g_strdup_printf ("%s/%s", filename, fn);
      g_free (filename);
      filename = tmp;
    }
  else
    {
      g_free (filename);
      filename = g_strdup (fn);
    }
  newfile = fopen (filename, "r");
  if (!newfile)
    {
      yywarn ("Could not open include file %s", filename);
      return (0);
    }
  /* printf( "fopen : %08lx\n" , newfile ); */
  /*
  printf ("push_file() open filename: %s\n", filename);
  */
  tmp = (struct FileData *) g_malloc (sizeof (struct FileData));
  g_assert (tmp != NULL);
  if (tmp)
    {
      memcpy (tmp, &currentfile, sizeof (struct FileData));
      tmp->next = file_stack;  /* insert the new one */
      file_stack = tmp;
                              /* copy currentfile -> tmp */
      tmp->yybuffer = YY_CURRENT_BUFFER;
      yyin = newfile;
      yy_switch_to_buffer (yy_create_buffer (yyin, YY_BUF_SIZE));
      g_assert( strlen(filename) < FILEDATA_FILENAME_LENGTH);
      if ( strlen(filename) < FILEDATA_FILENAME_LENGTH )
        {
          strcpy (currentfile.name, filename);
          currentfile.line = 1;
          strcpy (currentfile.modulename, "");
          strcpy (currentfile.moduleprefix, "");
          /*
          dump_file_stack("After Push");
          */
          return (1);
        }
    }
  /* out of memory */
  /* printf( "fclose : %08lx\n" , newfile ); */
  fclose (newfile);
  return (0);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
static int
pop_file (void)
{
  struct FileData *tmp;

  if (file_stack == NULL)
    {
      /*
      printf ("\npop_file() - NULL stack.\n");
      */
      return (0);
    }
  tmp = file_stack;
  /*
  printf ("\npop_file() - Done with file named: %s\n", currentfile.name);
  printf ("pop_file() - Make %s currentfile.\n", tmp->name);
  dump_file_stack("Before Pop");
  */
  file_stack = file_stack->next;
  memcpy (&currentfile, tmp, sizeof (struct FileData));
  yy_delete_buffer (YY_CURRENT_BUFFER);
  yy_switch_to_buffer (tmp->yybuffer);
  g_free (tmp);
  currentfile.next = NULL;
  /*
  dump_file_stack("After Pop");
  */
  return (1);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
int
yystartfile (char *filename)
{
  FILE *fp;

  /* if file has been processed, skip it */
  if (file_processed (filename))
    {
      return (0);
    }
  record_file_processed (filename);

  /* open it */
  fp = fopen (filename, "r");
  if (!fp)
    {
      return (-1);
    }

  /* printf( "fopen : %08lx\n" , fp ); */

/* ntiffin *//* reset lexer state */
  expecting_datatype = 1;
  expecting_fieldprop = 0;
  expect_triggertype = 0;

  /* setup lexer */
  yyrestart (fp);
  yy_setfirstfilename (filename);

  /* clear data */
  first_line = 1;
  currentfile.line = 1;
  strcpy (currentfile.modulename, "");
  strcpy (currentfile.moduleprefix, "");
  currentfile.next = NULL;

  /* done */
  return (1);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
static int
record_file_processed (const char *filename)
{
  g_assert( filename != NULL);
  if ((file_processed (filename) == 0) && (filename != NULL))
    {
      processed_count++;
      processed_files =
        (char **) realloc (processed_files,
                          (processed_count) * (sizeof (char *)));
      processed_files[processed_count - 1] = g_strdup (filename);
    }
  return (1);
}

/* ------------------------------------------------------------------------- *\
 * compare the name only and ignore path in filename
\* ------------------------------------------------------------------------- */
static gboolean
compare_file_name_only( const char * name1, const char *name2)
{
  char * ptr1;
  char * ptr2;
  
  g_assert( name1 );
  g_assert( name2 );

  ptr1 = g_basename( name1);
  ptr2 = g_basename( name2);
  g_assert( ptr1 );
  g_assert( ptr2 );
/*
  printf( "compare_file_name_only() - file 1: %s\n", name1);
  printf( "compare_file_name_only() - file 2: %s\n", name2);
  printf( "compare_file_name_only() - base 1: %s\n", ptr1);
  printf( "compare_file_name_only() - base 2: %s\n", ptr2);
*/
  if ( strcasecmp( ptr1, ptr2) == 0)
    {
      /* printf("compare_file_name_only() - true\n"); */
      return TRUE;
     }
  /* printf("compare_file_name_only() - false\n"); */
  return FALSE;
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
static int
file_processed (const char *filename)
{
  unsigned int i;
  
  /* empty list? */
  if (processed_files == NULL)
    {
      return (0);
    }
  for (i = 0; i < processed_count; i++)
    {
      if (compare_file_name_only (processed_files[i], filename) == TRUE)
        {
          /* printf( "file_processed() - true\n"); */
          return (1);
        }
    }
  return (0);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
void
clear_file_history (void)
{
  unsigned int i;
  g_assert (processed_files);
  if (processed_files == NULL)
    {
      return;
    }
  /* printf( "freeing elemts: 0 - %d\n" , processed_files->len ); */
  /* printf( "freeing elements: 0 - %d\n" , processed_count ); */

  for (i = 0; i < processed_count; i++)
    {
      if (processed_files[i])
        {
          free (processed_files[i]);
        }
    }
  free (processed_files);
  processed_count = 0;

  processed_files = NULL;
  /* printf( "freed processed files array\n" ); */
}

/* ------------------------------------------------------------------------- *\
 * Return address of the current file name for use by the lexer
\* ------------------------------------------------------------------------- */
const char * yyget_current_filename(void)
{
  return (&currentfile.name[0]);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
int
yywrap (void)
{
  return (1);                   /* stop scanning */
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
void
errors_show_file (int state)
{
  yy_show_file_error = state;
}
