/*
Copyright (C) 1993 by David H. Gay and Andrew L. Rohl 
 
dgay@ricx.royal-institution.ac.uk
andrew@ricx.royal-institution.ac.uk

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include "config.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

#include "gdis.h"
#include "vector.h"
#include "table.h"

TABLE_TYPEDEF(vector) trans_coord_table;
TABLE_TYPEDEF(vector) vector_table;
typedef gchar boolean;

#define EPSILON 1e-6
#define FILL_EPS 0.00025

#define DEBUG_GENSURF 0
gint generate_surface(struct model_pak *src, struct model_pak *dest)
{
vector normal;
vector lattice_vector[3];    /* lattice vectors */
vector t_mat[3];             /* transformation matrix */
vector work_lat[3];          /* transformed lattice vectors */
vector rec_work_lat[3];      /* reciprocal transformed lattice vectors */
vector s[3];                 /* space we are trying to fill */
vector rec_s[2];             /* reciprocal of s[0] and s[1] */
vector tempvec;
gfloat vec[3];
vector marvec;
vector trans_coord;
vector trans_shell;
vector trans_centroid;
trans_coord_table trans_coords;
trans_coord_table trans_shells;
trans_coord_table trans_centroids;
gfloat inv_denom;
gfloat cosa, cosb, cosg, sing, trm1;
gfloat shift, depth, depth_1, depth_2;
gint i, j;
gint h, k, l;
vector_table surf_vecs;
vector a, *v_a, *v_b;
gfloat cd, x;
gint c;
gint n;
boolean s2_found = FALSE;
gint la, ma, lb, mb, lc, mc; /* limits of lattice to fill piped */
gint ia, ib, ic;
gfloat dp, z;
gfloat z1_max, z1_min, z2_max, z2_min;
gfloat tempfloat;
struct atom_pak *src_atom;
gint curr_atom_no, curr_shell_no, items;
vector t, v;

gint GCD(gint, gint);
void *our_alloc (void *, size_t, char *);
gint vector_compare (const void *, const void *);
  
h = dest->surface.miller[0];
k = dest->surface.miller[1];
l = dest->surface.miller[2];
  
/* find surface normal from the miller index and lattice vectors*/
V_ZERO(normal);
  
cosa = cos(src->pbc[3]);
cosb = cos(src->pbc[4]);
cosg = cos(src->pbc[5]);
sing = sin(src->pbc[5]);
  
V_Y(lattice_vector[0]) = 0.0;
V_Z(lattice_vector[0]) = 0.0;
V_Z(lattice_vector[1]) = 0.0;
V_X(lattice_vector[0]) = src->pbc[0];
V_X(lattice_vector[1]) = src->pbc[1]*cosg;
V_Y(lattice_vector[1]) = src->pbc[1]*sing;
V_X(lattice_vector[2]) = src->pbc[2]*cosb;
V_Y(lattice_vector[2]) = src->pbc[2]*(cosa - cosg*cosb)/sing;
trm1 = V_Y(lattice_vector[2])/src->pbc[2];
V_Z(lattice_vector[2]) = src->pbc[2]*sqrt(1.0 - cosb*cosb - trm1*trm1);

V_CROSS(tempvec, lattice_vector[1], lattice_vector[2]);
V_QADD(normal, normal, +h*, tempvec);
V_CROSS(tempvec, lattice_vector[2], lattice_vector[0]);
V_QADD(normal, normal, +k*, tempvec);
V_CROSS(tempvec, lattice_vector[0], lattice_vector[1]);
V_QADD(normal, normal, +l*, tempvec);

TABLE_INIT(surf_vecs);
  
c = (float) GCD(h, k);
cd = c ? c : 1.0;
V_2_ASSIGN(a, = k/cd * ,lattice_vector[0], - h/cd *, lattice_vector[1]);
if (V_MAGSQ(a) > EPSILON) 
  {
  TABLE_ADD_RECORD(surf_vecs, a);
  }
  
c = (float) GCD(h, l);
cd = c ? c : 1.0;
V_2_ASSIGN(a, = l/cd * ,lattice_vector[0], - h/cd *, lattice_vector[2]);
if (V_MAGSQ(a) > EPSILON) 
  {
  TABLE_ADD_RECORD(surf_vecs, a);
  }
  
c = (float) GCD(k, l);
cd = c ? c : 1.0;
V_2_ASSIGN(a, = l/cd * ,lattice_vector[1], - k/cd *, lattice_vector[2]);
if (V_MAGSQ(a) > EPSILON) 
  {
  TABLE_ADD_RECORD(surf_vecs, a);
  }
  
n = TABLE_N(surf_vecs);

v_a = TABLE_R_ADRS(surf_vecs, 0);
TABLE_WHILE(surf_vecs, TABLE_INDEX(surf_vecs,v_a) < n-1, v_a) 
  {
  v_b = v_a + 1;
  TABLE_WHILE(surf_vecs, TABLE_INDEX(surf_vecs,v_b) < n, v_b) 
    {
    V_2_ASSIGN(a, = , *v_a, + , *v_b);
    if (V_MAGSQ(a) > EPSILON)
      TABLE_ADD_RECORD(surf_vecs,a);
    V_2_ASSIGN(a, = , *v_a, - , *v_b);
    if (V_MAGSQ(a) > EPSILON)
      TABLE_ADD_RECORD(surf_vecs,a);
    }
  }
TABLE_SORT(surf_vecs,vector_compare);

/* debug */
/*
TABLE_FOREACH(surf_vecs, v_a)
  {
  printf("vec %f %f %f\n",V_X(*v_a),V_Y(*v_a),V_Z(*v_a));
  }
*/

v_a = TABLE_R_ADRS(surf_vecs,0);

v_b = v_a + 1;
TABLE_WHILEEACH(surf_vecs, v_b) 
  {
  V_CROSS(a,*v_a,*v_b);
  x = V_MAGSQ(a);
  if (x > EPSILON) 
    {
    s2_found = TRUE;
    break;
    }
  }

if (!s2_found) 
  {
  printf ("can't find surface vectors\n");
  exit (1);
  }

/* calculate transformation matrix */
V_SCALER(t_mat[2], (1.0/V_MAG(normal)), normal);
V_SCALER(t_mat[0], (1.0/V_MAG(*v_a)), *v_a);
V_CROSS(t_mat[1], t_mat[2], t_mat[0]);

/* calculate transformed lattice vectors */
for (j = 0; j < 3; j++)
  for (i = 0; i < 3; i++)
    V_E(work_lat[j], i) = V_DOT(t_mat[i], lattice_vector[j]);

/* calculate reciprocal transformed lattice vectors */
V_CROSS(tempvec, work_lat[1], work_lat[2]);
inv_denom = 1.0 / V_DOT(tempvec, work_lat[0]);
V_CROSS(tempvec, work_lat[1], work_lat[2]);
V_SCALER(rec_work_lat[0], inv_denom, tempvec);
V_CROSS(tempvec, work_lat[2], work_lat[0]);
V_SCALER(rec_work_lat[1], inv_denom, tempvec);
V_CROSS(tempvec, work_lat[0], work_lat[1]);
V_SCALER(rec_work_lat[2], inv_denom, tempvec);
 
/* calculate transformed coordinates, shells and centroids */
  TABLE_INIT(trans_coords);
  TABLE_INIT(trans_shells);
  TABLE_INIT(trans_centroids);
 
  for (j = 0 ; j < src->num_atoms ; j++) {
    VEC3SET(vec, (src->atoms+j)->x, (src->atoms+j)->y, (src->atoms+j)->z);
    vecmat(src->latmat, vec);
    V_X(marvec) = vec[0];
    V_Y(marvec) = vec[1];
    V_Z(marvec) = vec[2];

    for (i = 0; i < 3; i++)
      trans_coord.element[i] = V_DOT(t_mat[i], marvec);
   TABLE_ADD_RECORD(trans_coords, trans_coord);
  }
  
  for (j = 0 ; j < src->num_shells ; j++) {
    VEC3SET(vec, (src->shells+j)->x, (src->shells+j)->y, (src->shells+j)->z);
    vecmat(src->latmat, vec);
    V_X(marvec) = vec[0];
    V_Y(marvec) = vec[1];
    V_Z(marvec) = vec[2];


    for (i = 0; i < 3; i++)
      trans_shell.element[i] = V_DOT(t_mat[i], marvec);
   TABLE_ADD_RECORD(trans_shells, trans_shell);
  }
  
  for (j = 0 ; j < src->num_mols ; j++) {
    ARR3SET(vec,(src->mols+j)->centroid);
    vecmat(src->latmat, vec);
    V_X(marvec) = vec[0];
    V_Y(marvec) = vec[1];
    V_Z(marvec) = vec[2];

    for (i = 0; i < 3; i++)
      trans_centroid.element[i] = V_DOT(t_mat[i], marvec);
   TABLE_ADD_RECORD(trans_centroids, trans_centroid);
  }

  /* calculate transformed surface vectors */
  for (i = 0; i < 3; i++)
    V_E(s[0], i) = V_DOT(t_mat[i], *v_a);
  for (i = 0; i < 3; i++)
    V_E(s[1], i) = V_DOT(t_mat[i], *v_b);

/* return transformed surface vectors data in dest->latmat */
  /* NB: latmat[0..8] is a 3x3 matrix so, */
  /*
    | a b 0 |
    | c d 0 |
    | 0 0 0 |

    goes into latmat rows first ie a,b,0,c,d,0,0,0,0
  */
  
/* set s[2] = to depths */
depth = 1.0e10;
if (fabs(V_Z(work_lat[0])) > EPSILON
    && fabs(V_Z(work_lat[0])) < depth)
  depth = fabs(V_Z(work_lat[0]));
if (fabs(V_Z(work_lat[1])) > EPSILON
    && fabs(V_Z(work_lat[1])) < depth)
  depth = fabs(V_Z(work_lat[1]));
if (fabs(V_Z(work_lat[2])) > EPSILON
    && fabs(V_Z(work_lat[2])) < depth)
  depth = fabs(V_Z(work_lat[2]));
  
/* MYMOD - don't alter the surface specification */
shift = dest->surface.shift * depth;
dest->surface.dspacing = depth/GCD(GCD(h, k), GCD(k, l));
#if DEBUG_GENSURF
printf("interplanar spacing is %f\n", dest->surface.dspacing);
#endif

depth_1 = dest->surface.region[0] * depth;
/* MYMOD - region[1] is NOT the sum of sizes */
/*
depth_2 = dest->surface.region[1] * depth;
*/
depth_2 = depth_1 + dest->surface.region[1] * depth;

V_ZERO(s[2]);
V_Z(s[2]) = -(depth_2);

/* transfer surface vectors */
/*
printf("%f %f %f\n", V_X(s[0]), V_Y(s[0]), V_Z(s[0]));
printf("%f %f %f\n", V_X(s[1]), V_Y(s[1]), V_Z(s[1]));
*/
VEC3SET(&dest->latmat[0], V_X(s[0]), V_X(s[1]), 0.0);
VEC3SET(&dest->latmat[3], V_Y(s[0]), V_Y(s[1]), 0.0);
VEC3SET(&dest->latmat[6], 0.0, 0.0, 1.0);
/*
transpose(dest->latmat);
*/
dest->construct_pbc = TRUE;

/* calculate reciprocal of two surface vectors */
inv_denom = 1.0 / (V_X(s[0])*V_Y(s[1]) - V_Y(s[0])*V_X(s[1]));
V_X(rec_s[0]) =  V_Y(s[1])*inv_denom;
V_Y(rec_s[0]) = -V_X(s[1])*inv_denom;
V_Z(rec_s[0]) =  0.0;
V_X(rec_s[1]) = -V_Y(s[0])*inv_denom;
V_Y(rec_s[1]) =  V_X(s[0])*inv_denom;
V_Z(rec_s[1]) =  0.0;

/* calculate extents of translations of transformed cell */
la = ma = lb = mb = lc = mc = 0.0;

#define INT_EPS 0.0001
x = V_DOT(rec_work_lat[0], s[0]);
if (x > ma) ma = x + INT_EPS;
if (x < la) la = x - INT_EPS;
x = V_DOT(rec_work_lat[1], s[0]);
if (x > mb) mb = x + INT_EPS;
if (x < lb) lb = x - INT_EPS;
x = V_DOT(rec_work_lat[2], s[0]);
if (x > mc) mc = x + INT_EPS;
if (x < lc) lc = x - INT_EPS;

x = V_DOT(rec_work_lat[0], s[1]);
if (x > ma) ma = x + INT_EPS;
if (x < la) la = x - INT_EPS;
x = V_DOT(rec_work_lat[1], s[1]);
if (x > mb) mb = x + INT_EPS;
if (x < lb) lb = x - INT_EPS;
x = V_DOT(rec_work_lat[2], s[1]);
if (x > mc) mc = x + INT_EPS;
if (x < lc) lc = x - INT_EPS;

x = V_DOT(rec_work_lat[0], s[2]);
if (x > ma) ma = x + INT_EPS;
if (x < la) la = x - INT_EPS;
x = V_DOT(rec_work_lat[1], s[2]);
if (x > mb) mb = x + INT_EPS;
if (x < lb) lb = x - INT_EPS;
x = V_DOT(rec_work_lat[2], s[2]);
if (x > mc) mc = x + INT_EPS;
if (x < lc) lc = x - INT_EPS;

V_QADD(tempvec, s[0],+, s[1]);
x = V_DOT(rec_work_lat[0], tempvec);
if (x > ma) ma = x + INT_EPS;
if (x < la) la = x - INT_EPS;
x = V_DOT(rec_work_lat[1], tempvec);
if (x > mb) mb = x + INT_EPS;
if (x < lb) lb = x - INT_EPS;
x = V_DOT(rec_work_lat[2], tempvec);
if (x > mc) mc = x + INT_EPS;
if (x < lc) lc = x - INT_EPS;

V_QADD(tempvec, s[0],+, s[2]);
x = V_DOT(rec_work_lat[0], tempvec);
if (x > ma) ma = x + INT_EPS;
if (x < la) la = x - INT_EPS;
x = V_DOT(rec_work_lat[1], tempvec);
if (x > mb) mb = x + INT_EPS;
if (x < lb) lb = x;
x = V_DOT(rec_work_lat[2], tempvec);
if (x > mc) mc = x + INT_EPS;
if (x < lc) lc = x - INT_EPS;

V_QADD(tempvec, s[1],+, s[2]);
x = V_DOT(rec_work_lat[0], tempvec);
if (x > ma) ma = x + INT_EPS;
if (x < la) la = x - INT_EPS;
x = V_DOT(rec_work_lat[1], tempvec);
if (x > mb) mb = x + INT_EPS;
if (x < lb) lb = x - INT_EPS;
x = V_DOT(rec_work_lat[2], tempvec);
if (x > mc) mc = x + INT_EPS;
if (x < lc) lc = x - INT_EPS;

V_QADD(tempvec, s[0],+, s[1]);
V_QADD(tempvec, tempvec,+, s[2]);
x = V_DOT(rec_work_lat[0], tempvec);
if (x > ma) ma = x + INT_EPS;
if (x < la) la = x - INT_EPS;
x = V_DOT(rec_work_lat[1], tempvec);
if (x > mb) mb = x + INT_EPS;
if (x < lb) lb = x - INT_EPS;
x = V_DOT(rec_work_lat[2], tempvec);
if (x > mc) mc = x + INT_EPS;
if (x < lc) lc = x - INT_EPS;
#undef INT_EPS  

#define LATBUF 2
ma += LATBUF;
mb += LATBUF;
mc += LATBUF;
la -= LATBUF;
lb -= LATBUF;
lc -= LATBUF;

z2_min = z1_min = 0.00;
z2_max = V_Z(s[2]);
z1_max = depth_1 * (z2_max < 0 ? -1.0: +1.0);

if (z2_max < z2_min) 
  {
  tempfloat = z2_max;
  z2_max = z2_min;
  z2_min = tempfloat;
  }
if (z1_max < z1_min) 
  {
  tempfloat = z1_max;
  z1_max = z1_min;
  z1_min = tempfloat;
  }

/* display name */
g_free(dest->basename);
dest->basename = g_strdup_printf("%s_%-1d%-1d%-1d_%6.4f",
                 g_strstrip(src->basename), h, k, l, dest->surface.shift);

/* set region flags */
if (dest->surface.region[0] > 0)
  dest->region_empty[REGION1A] = FALSE;
if (dest->surface.region[1] > dest->surface.region[0])
  dest->region_empty[REGION2A] = FALSE;

/* MARVIN - region highlighting */
if (dest->id == MARVIN) 
  {
  j=0;
  for (i=REGION1A ; i<REGION2B ; i++) 
    {
    if (!dest->region_empty[i] && !j) 
      {
      dest->region[i] = TRUE;
      j++;
      }
    else
      dest->region[i] = FALSE;
    }
  }
dest->axes_type = OTHER;
dest->sginfo.spacenum = 1;

if (genpos(dest))
  printf("Error in Space Group lookup.\n");
/* begin adding atoms */
dest->num_shells = dest->num_atoms = curr_atom_no = curr_shell_no = 0;
for (ia = la; ia <= ma; ia++) 
  {
  for (ib = lb; ib <= mb; ib++) 
    {
    for (ic = lc; ic <= mc; ic++) 
      {
/* calculate translation vector */
      for (i = 0; i < 3; i++) 
        {
        V_E(v,i)  = 0.0;
        V_E(v,i) += ia * V_E(work_lat[0], i);
        V_E(v,i) += ib * V_E(work_lat[1], i);
        V_E(v,i) += ic * V_E(work_lat[2], i);
        }
      V_Z(v) = V_Z(v) + -1 * shift;
/* loop over all molecules */
      for (i = 0; i < src->num_mols; i++) 
        {
/* add translation vector to transformed coordinates */
        V_QADD(t, v , + , TABLE_RECORD(trans_centroids, i));
/* test for centroid being inside Z boundary since simple technique */
/* since Z is perpendicular to the surface by definition */
        z = V_Z(t);
/* Borders have to adjusted to that block 1 does not include z=0, */
/* but block 2 does */
        if (z > z2_min - FILL_EPS && z <= z2_max - FILL_EPS) 
          {
/*      test for atom being under surface defined by a parallelegram
        defined by the vectors s[0] and s[1] using non-orthogonal projections
        as suggested by Dr. Susan Hill which is the same as using reciprocal
        lattice vectors */
          
/* projection of T onto S[0] || to S[1] */
          dp = V_DOT(t, rec_s[0]);
          if (dp < 1.0 - FILL_EPS && (dp >= -FILL_EPS)) 
            {
/* projection of T onto S[1] || to S[0] */
            dp = V_DOT(t, rec_s[1]);
            if ((dp >= -FILL_EPS) && dp < 1.0 - FILL_EPS) 
              {
/* inside is TRUE */
/* loop over all atoms in molecule */
              for (j = 0; j < (src->mols+i)->num_atoms; j++) 
                {

/* SEAN can do multiple faces of a given crystal, but if close box and reopen
 fails as src is now surface - perhaps specify model in dialog...*/
/* TODO - another global var job */

/* SEAN why not set idx_shell to -1 if no shell and remove has_shell */
/* probably should work, but if(has_shell) is more obvious than if(<0) */
                src_atom = src->atoms+(src->mols+i)->atom[j];
/* NEW - copy_core() takes care of core/shell init & mem alloc */
                items = copy_core(dest, src, (src->mols+i)->atom[j]);
                switch (items)
                  {
/* shell */
                  case 2:
/* NEW - all atoms are orig & primary */
                    (dest->shells+dest->num_shells-1)->orig = TRUE;
                    (dest->shells+dest->num_shells-1)->primary = TRUE;
/* add translation vector to transformed coordinates */
                    V_QADD(t, v , + , TABLE_RECORD(trans_shells,
                                          src_atom->idx_shell));
                    (dest->shells+dest->num_shells-1)->x = V_X(t);
                    (dest->shells+dest->num_shells-1)->y = V_Y(t);
                    (dest->shells+dest->num_shells-1)->z = V_Z(t);
                    if (z > z1_min - FILL_EPS && z <= z1_max - FILL_EPS)
                      (dest->shells+dest->num_shells-1)->region = REGION1A;
                    else
                      (dest->shells+dest->num_shells-1)->region = REGION2A;
/* add translation vector to transformed coordinates */
#if DEBUG_GENSURF
printf("added shell [%s] at %f %f %f\n", 
      (dest->atoms+dest->num_shells-1)->element, V_X(t), V_Y(t), V_Z(t));
#endif
/* core */
                  case 1:
/* NEW - all cores are orig & primary */
                    (dest->atoms+dest->num_atoms-1)->orig = TRUE;
                    (dest->atoms+dest->num_atoms-1)->primary = TRUE;
/* add translation vector to transformed coordinates */
                    V_QADD(t, v , + , TABLE_RECORD(trans_coords, 
                                       (src->mols+i)->atom[j]));
                    (dest->atoms+dest->num_atoms-1)->x = V_X(t);
                    (dest->atoms+dest->num_atoms-1)->y = V_Y(t);
                    (dest->atoms+dest->num_atoms-1)->z = V_Z(t);
                    if (z > z1_min - FILL_EPS && z <= z1_max - FILL_EPS)
                      (dest->atoms+dest->num_atoms-1)->region = REGION1A;
                    else
                      (dest->atoms+dest->num_atoms-1)->region = REGION2A;
#if DEBUG_GENSURF
printf("added atom [%s] at %f %f %f\n", 
      (dest->atoms+dest->num_atoms-1)->element, V_X(t), V_Y(t), V_Z(t));
#endif
                    break;
/* nothing copied */
                  default:
                    printf("copy_core() failed.\n");
                  }  /* copy_core() return val */
                }    /* loop j - all atoms in molecule */
              }
            }
          }
        }            /* loop i - all molecules */
      }              /* loop ia - space fill */
    }                /* loop ib - space fill */
  }                  /* loop ic - space fill */
TABLE_FREE(surf_vecs);
TABLE_FREE(trans_coords);
TABLE_FREE(trans_shells);
TABLE_FREE(trans_centroids);
return(1);
}

/******************************************************************************
 * GCD
 *      Greatest common denominator
 ******************************************************************************/
gint GCD(gint p, gint q)
{
#if defined(__MWERKS__)  /* fix for compiler bug! */
  gint  i;  
  
  if (q == 0) {
    i = abs(p);
    return(i);
  }
#else
  if (q == 0)
    return(abs(p));
#endif
  else
    return(GCD(q, p%q));
}

/******************************************************************************
 * vector_compare
 *      compare two vectors returning their difference as an integer.  This
 *      routine is to be used in conjuction with SORT().
 ******************************************************************************/
int vector_compare (const void *x, const void *y)
{
int i;
vector  *a, *b;
double  diff;
  
a = (vector *) x;
b = (vector *) y;
diff = V_MAGSQ(*a) - V_MAGSQ(*b);
if (diff < -EPSILON)
  return(-1);
else if (diff > EPSILON)
  return(1);
else
  {
/* magnitude of a & b is the same, so sort based on element magnitudes */ 
  for (i=0 ; i<3 ; i++)
    {
    diff = V_E(*a,i) - V_E(*b,i);
    if (diff < -EPSILON)
      return(1);
    else if (diff > EPSILON)
      return(-1);
    }
  }
/* a & b are identical */
return(0);
}

void *our_alloc (void *ptr, size_t size, gchar *err_msg)
{
void  *space;
 
if (size == 0)                    /* asks for ZERO elements */
  return NULL;
 
space = NULL;
  
if (ptr == NULL)
  space = g_malloc(size);
else
  space = g_realloc(ptr, size);
if (space == NULL) 
  {
  fprintf(stderr, "Cannot malloc memory %s\n", err_msg);
  exit(1);
  }
return space;
}

