/*
Copyright (C) 2000 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include "config.h"
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/param.h>
#ifdef __sun
#include <sys/dirent.h>
#else
#include <sys/dir.h>
#endif

#include "gdis.h"

#define DEBUG_MORE 0
#define MAX_KEYS 15

/* main structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/* TODO - marvin input file parsing routine (adapt load_data) */

/*******************************/
/* MARVIN specific file saving */
/*******************************/
gint write_marvin(gchar *filename, struct model_pak *data)
{
gint i, j;
gint mol, region;
gfloat vec[3];
FILE *fp;

/* open the file */
fp = fopen(filename,"w");
if (!fp)
  {
  show_text("Error, bad filename!");
  return(1);
  }

show_text("Saving file in MARVIN restart format!");

/* save via the number of molecules in the */
/* model - since this allows mol numbering */
mol=0;

/* four region types printing */
for (region=REGION1A ; region<REGION2B ; region++)
  {
  if (data->region_empty[region])
    continue;
  switch(region)
    {
    case REGION1A:
      fprintf(fp,"coordinates 1 A\n");
      break;
    case REGION2A:
      fprintf(fp,"coordinates 2 A\n");
      break;
    case REGION1B:
      fprintf(fp,"coordinates 1 B\n");
      break;
    case REGION2B:
      fprintf(fp,"coordinates 2 B\n");
      break;
    }

/* do atoms */
  for (i=0 ; i<data->num_atoms ; i++)
    {
/* don't print if not in current region */
    if ((data->atoms+i)->region != region)
      continue;
/* NB: y coord no longer inverted */
    if ((data->atoms+i)->status & DELETED)
      continue;
/* NB: x,y,z have been fractionalized */
    VEC3SET(vec,(data->atoms+i)->x,(data->atoms+i)->y,(data->atoms+i)->z);
    vecmat(data->latmat, vec);
    fprintf(fp,"%-7s   core   %14.9f  %14.9f  %14.9f\n",
               (data->atoms+i)->label, vec[0], vec[1], vec[2]);
/* FIXME = how to get proper rotated coords? */
/* shell? */
    if ((data->atoms+i)->has_shell)
      {
/* get the shell's index */
      j = (data->atoms+i)->idx_shell;
VEC3SET(vec,(data->shells+j)->x,(data->shells+j)->y,(data->shells+j)->z);
      vecmat(data->latmat, vec);
      fprintf(fp,"%-7s   shel   %14.9f  %14.9f  %14.9f\n",
             (data->shells+j)->label, vec[0], vec[1], vec[2]);
      }
    }
/* done region */
  fprintf(fp,"end\n");
  }

/* done */
fclose(fp);
return(0);
}

/****************************/
/* marvin output file parse */
/****************************/
#define DEBUG_LOAD_MVNOUT 0
gint load_mvnout(FILE *fp, gint model, gchar *inp_file)
{
gchar line[LINELEN], **key, **buff;
gint i, j, k, current_block, region, status, offset;
gint block[4];
gfloat veca[2], vecb[2];
gfloat version;
struct model_pak *data;

data = model_ptr(model, ASSIGN);
if (data == NULL)
  return(1);

/* setup */
data->id = MVNOUT;
data->periodic = 2;
strcpy(data->filename, inp_file);
g_free(data->basename);
data->basename = strdup_no_extension(g_basename(inp_file));

/* starting locations */
block[0] = -1;
block[1] = -1;
block[2] = -1;
block[3] = -1;
current_block = 0;

/*****************************************/
/* parse 1 - get location of data blocks */
/*****************************************/
/* scheme - ignore all but the last block of a particular region */
/* read until EOF all coordinate BLOCK data */
status=offset=0;
while (!fgetline(fp,line))
  {
  key = get_tokens(line, 7);
/* YAMH - get version to determine a parsing parameter */
  if (g_strncasecmp(line,"                    Version",27) == 0)
    {
    sscanf(*(key+1),"%f",&version);
    offset = (version < 1.99) ? 0 : 1;
#if DEBUG_LOAD_MVNOUT
printf("Marvin version %f, offset = %d\n",version,offset);
#endif 
    }
/* get surface vectors */
  if (g_strncasecmp(line,"Surface Vectors:",16) == 0)
    {
/* read them in */
    sscanf(*(key+2),"%f",&veca[0]);
    sscanf(*(key+3),"%f",&veca[1]);

    if (fgetline(fp,line))
      return(3);
    buff = get_tokens(line, 2);
    sscanf(*buff,"%f",&vecb[0]);
    sscanf(*(buff+1),"%f",&vecb[1]);
    g_strfreev(buff);
/* NB: gdis stores vectors in reverse order */
    VEC3SET(&(data->latmat[0]), veca[0], vecb[0], 0.0);
    VEC3SET(&(data->latmat[3]), veca[1], vecb[1], 0.0);
    VEC3SET(&(data->latmat[6]), 0.0, 0.0, 1.0);
    data->construct_pbc = TRUE;
    data->periodic = 2;
    make_xlat(data);

    status++;
    continue;
    }
/* NB: disallow image creation in this direction */
data->pbc[2] = 0.0;
/*
  if (g_strncasecmp(line,"Layer Thickness:",16) == 0)
    {
    copy_items(buff,line,3,SINGLE);
    sscanf(buff,"%f",&(data->pbc[2]));
    continue;
    }
*/

/* else, BLOCK (data) keyword check */
/* NB: marvin has block and BLOCK littered about, want BLOCK only */
  if (strncmp(*key,"BLOCK",5) == 0)
    {
    current_block++;
/* is it region data? */
    if (g_strncasecmp(*(key+2),"region",5) != 0)
      continue;
/* get region type */
    if (g_strncasecmp(*(key+5),"1a",2) == 0)
      region = REGION1A;
    else if (g_strncasecmp(*(key+5),"1b",2) == 0)
      region = REGION1B;
    else if (g_strncasecmp(*(key+5),"2a",2) == 0)
      region = REGION2A;
    else if (g_strncasecmp(*(key+5),"2b",2) == 0)
      region = REGION2B;
    else
      continue;
/* is it coordinate data? (don't want gradients) */
    fgetline(fp,line);
    fgetline(fp,line);

    buff = get_tokens(line, 4);
    if (g_strncasecmp(*(buff+3),"coordinates",11) != 0)
      continue;
    block[region] = current_block;
    g_strfreev(buff);

#if DEBUG_LOAD_MVNOUT
printf("Found region type: [%d], block id: %d\n",region,current_block);
#endif
    data->region_empty[region] = FALSE;
    status++;
    }
  g_strfreev(key);
  }

/* checks, return bad status if a key item was missing */
if (status < 2)
  return(status);

/**************************/
/* parse 2 - get the data */
/**************************/
rewind(fp);
current_block = 0;
/* core/shell counters */
i=j=0;
while (!fgetline(fp,line))
  {
/* go through the regions and see which block match should be read */
  if (strncmp(line,"BLOCK",5) == 0)
    {
    current_block++;
    region = -1;
    for (k=REGION1A ; k<REGION2B ; k++)
      if (block[k] == current_block)
        region = k;
    if (region == -1)
      continue;
/* skip straight to data */
    fgetline(fp,line);
    fgetline(fp,line);
    fgetline(fp,line);
    fgetline(fp,line);
/* read it in */
#if DEBUG_LOAD_MVNOUT
printf("Block id: %d, reading region type: %d\n",current_block,region);
#endif
    while(g_strncasecmp(line,"-------",7) != 0)
      {
      buff = get_tokens(line, 7);
/* NB: marvin 1.8 & 2.0 have this at different positions! */
      if (g_strncasecmp(*(buff+offset+2),"cor",3) == 0)
        {
/* new core */
        if (i >= data->atom_limit)
          mem_grow(data, ATOM, 10);
/* setup */
        (data->atoms+i)->status = NORMAL;
        (data->atoms+i)->region = region;
        (data->atoms+i)->primary = TRUE;
        (data->atoms+i)->orig = TRUE;
        (data->atoms+i)->offx = 0;
        (data->atoms+i)->offy = 0;
/* read the atom type */
        sscanf(*(buff+offset+1),"%s",(data->atoms+i)->element);
        sscanf(*(buff+offset+1),"%s",(data->atoms+i)->label);
/* get coords */
        (data->atoms+i)->x = str_to_float(*(buff+offset+3));
        (data->atoms+i)->y = str_to_float(*(buff+offset+4));
        (data->atoms+i)->z = str_to_float(*(buff+offset+5));
/* occupancy */
        (data->atoms+i)->sof = 1.0;
        *(data->atoms+i)->tail = '\0';
/* do element setup */
        elem_seek(i, ATOM, data);
        i++;
        }
      else if (g_strncasecmp(*(buff+offset+2),"she",3) == 0)
        {
/* new shell */
        if (j >= data->shell_limit)
          mem_grow(data, SHELL, 10);
/* setup the shell */
        (data->shells+j)->status = NORMAL;
        (data->shells+j)->region = region;
        (data->shells+j)->primary = TRUE;
        (data->shells+j)->orig = TRUE;
        (data->shells+j)->offx = 0;
        (data->shells+j)->offy = 0;
/* read the atom type */
        sscanf(*(buff+offset+1),"%s",(data->shells+j)->element);
        sscanf(*(buff+offset+1),"%s",(data->shells+j)->label);
/* get coords */
        (data->shells+j)->x = str_to_float(*(buff+offset+3));
        (data->shells+j)->y = str_to_float(*(buff+offset+4));
        (data->shells+j)->z = str_to_float(*(buff+offset+5));
        *(data->shells+j)->tail = '\0';
/* do element setup */
        elem_seek(j, SHELL, data);
        j++;
        }
      else
        {
        printf("load_mvnout(): parse error.\n");
printf("[%s]\n",line);
        break;
        }
/* get next line, terminate on EOF */
      if (fgetline(fp,line))
        {
        printf("Warning: EOF in data block.\n");
        break;
        }
      g_strfreev(buff);
      }
    }

/* energy searches */
  if (strncmp(line,"Attachment Energy",17) == 0)
    {
    buff = get_tokens(line, 5);
    data->gulp.eatt = str_to_float(*(buff+3));
    g_free(data->gulp.eatt_units);
    data->gulp.eatt_units = g_strdup(*(buff+4));
    g_strfreev(buff);
    }
  if (strncmp(line,"Surface Energy",14) == 0)
    {
    buff = get_tokens(line, 6);
    data->gulp.esurf = str_to_float(*(buff+3));
    g_free(data->gulp.esurf_units);
    data->gulp.esurf_units = g_strdup(*(buff+5));
    g_strfreev(buff);
    }
  }

/* got them all */
data->num_asym = data->num_orig = data->num_atoms = i;
data->num_shells = j;

/* init lighting */
j=0;
for (i=REGION1A ; i<REGION2B ; i++)
  {
  if (!data->region_empty[i] && !j)
    {
    data->region[i] = TRUE;
    j++;
    }
  else
    data->region[i] = FALSE;
  }
   
#if DEBUG_LOAD_MVNOUT
printf("model loaded: found %d atom(s) ",data->num_atoms);
printf("and %d shell(s).\n",data->num_shells);
printf("Surface vectors: %f x %f\n",data->pbc[0],data->pbc[1]);
#endif

/* FIXME - force non periodic in z (marvin is 2D) */
/* FIXME - update image replication code accordingly */
data->pbc[2] = 0;

/* post read init */
data->mode = FREE;
sysenv.active = model;
if (data->periodic)
  {
  data->cell_on = TRUE;
  data->axes_type = OTHER;
/* space group init here */
  update_shells(data);
  if (!data->sginfo.spacenum)
    data->sginfo.spacenum=1;
  if (genpos(data))
    printf("Error in Space Group lookup.\n");
  }
/* generate latmat */
init_objs(INIT_COORDS, data);
/* store original cartesian coords */
save_cart(data);
/* setup latmat */
latmat_fudge(data);

data->axes_on = TRUE;
init_objs(INIT_COORDS, data);
calc_bonds(data);
calc_mols(data);

new_tree_item(model, APPEND);
return(0);
}

