/*
** Copyright (C) 10 Feb 1999 Jonas Munsin <jmunsin@iki.fi>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <stdlib.h>
#include <string.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>

#include "common_gtk.h"
#include "config_files.h"
#include "vector_commands.h"
#include "cdrecord_options.h"
#include "modify_file_set.h"
#include "command.h"
#include "rip_audio.h"
#include "cdlabel.h"
#include "linebuffer.h"
#include "preferences.h"
#include "selection_IO.h"
#include "mainwindow.h"
#include "globals.h"

/* I expect to need to write to several config files, might as well use
 * a whole dir for gcombust */

#define GCOMBUST_CONFIG_DIR ".gcombust"
#define GCOMBUST_CONFIG_FILE_NAME "gcombustrc"

static void read_env(void) {
	char *s, *end;
	long int l;
	char *tmp;

	if ((s = getenv("CDR_SPEED")) != NULL) {
		l = strtol(s, (char **)NULL, 10);
		gtk_spin_button_set_value(GTK_SPIN_BUTTON(speed_spin), l);
	}
	if ((s = getenv("CDR_FIFOSIZE")) != NULL) {
		gtk_entry_set_text(GTK_ENTRY(fsentry), s);
	}
/* only handles the dev=(x,),y,z format, in a kind of ugly way */
	if ((s = getenv("CDR_DEVICE")) !=NULL) {
		l = strtol(s, &end, 10);
		if (strchr(s, ',') == strrchr(s, ',')) {  /* 1,0 */
			gtk_spin_button_set_value(
					GTK_SPIN_BUTTON(mainptr->cdr_option_info->scsibus), 0);
			end = s - 1;
		} else /* 0,1,0 */
			gtk_spin_button_set_value(
					GTK_SPIN_BUTTON(mainptr->cdr_option_info->scsibus), l);
		if ('\0' != *end) {
			s = end;
			l = strtol(s + 1, &end, 10);
			gtk_spin_button_set_value(
					GTK_SPIN_BUTTON(mainptr->cdr_option_info->target), l);
		}
		if ('\0' != *end) {
			s = end;
			l = strtol(s + 1, &end, 10);
			gtk_spin_button_set_value(
					GTK_SPIN_BUTTON(mainptr->cdr_option_info->lun), l);
		}
	}

	tmp = getenv("CDDA_DEVICE");
	if (NULL == tmp)
		cdda_device_path = g_strdup("");
	else
		cdda_device_path = g_strdup(tmp);
}

static char *get_config_dir(void) {
	char *config_dir, *config;

	if (NULL == (config_dir = getenv("HOME")))
		config_dir = ".";

	config = string_append(config_dir, NULL);
	config = string_append(config, "/");
	config = string_append(config, GCOMBUST_CONFIG_DIR);

	return config;
}

static char *get_config_file(void) {
	char *config;

	config = get_config_dir();
	config = string_append(config, "/");
	config = string_append(config, GCOMBUST_CONFIG_FILE_NAME);

	return config;
}

static void set_mkisofs_path(char *path) {
	g_free(mkisofs_path);
	mkisofs_path = g_strdup(path);
}
static void set_cdrecord_path(char *path) {
	g_free(cdrecord_path);
	cdrecord_path = g_strdup(path);
}
static void set_cdlabelgen_path(char *path) {
	g_free(cdlabelgen_path);
	cdlabelgen_path = g_strdup(path);
}
static void set_diff_path(char *path) {
	g_free(diff_path);
	diff_path = g_strdup(path);
}
static void set_cdda2wav_path(char *path) {
	g_free(cdda2wav_path);
	cdda2wav_path = g_strdup(path);
}
static void set_cdparanoia_path(char *path) {
	g_free(cdparanoia_path);
	cdparanoia_path = g_strdup(path);
}
static void set_lpr_path(char *path) {
	g_free(lpr_path);
	lpr_path = g_strdup(path);
}
static void set_iso_location(char *path) {
	g_free(def_image_path);
	def_image_path = g_strdup(path);
}
static void set_rip_location(char *path) {
	g_free(def_rip_path);
	def_rip_path = g_strdup(path);
}
static void set_selection_location(char *path) {
	g_free(def_selection_path);
	def_selection_path = g_strdup(path);
}
static void set_enable_tooltips(char *path) {
	if (*(path) == '1')
		misc_prefs.enable_tooltips = TRUE;
	else
		misc_prefs.enable_tooltips = FALSE;
}
static void set_own_progresswin(char *path) {
	if (*(path) == '1')
		misc_prefs.own_progresswin = TRUE;
	else
		misc_prefs.own_progresswin = FALSE;
}
static void set_scsibus(char *path) {
	gtk_spin_button_set_value(
			GTK_SPIN_BUTTON(mainptr->cdr_option_info->scsibus), atoi(path));
}
static void set_target(char *path) {
	gtk_spin_button_set_value(
			GTK_SPIN_BUTTON(mainptr->cdr_option_info->target), atoi(path));
}
static void set_lun(char *path) {
	gtk_spin_button_set_value(
			GTK_SPIN_BUTTON(mainptr->cdr_option_info->lun), atoi(path));
}
static void set_driver(char *path) {
	gtk_entry_set_text(GTK_ENTRY(mainptr->cdr_option_info->driver), path);
}
static void set_device(char *path) {
	gtk_entry_set_text(GTK_ENTRY(mainptr->cdr_option_info->device), path);
}
static void set_fifosize(char *path) {
	gtk_entry_set_text(GTK_ENTRY(mainptr->cdr_option_info->fifo), path);
}
static void set_cddadevice(char *path) {
	g_free(cdda_device_path);
	cdda_device_path = g_strdup(path);
}
static void set_isodevice(char *path) {
	g_free(isocdrom_device_path);
	isocdrom_device_path = g_strdup(path);
}
static void set_swab(char *path) {
	if (*(path) == '1')
		gtk_toggle_button_set_active(
				GTK_TOGGLE_BUTTON(mainptr->cdr_option_info->swab_audio),
				TRUE);
	else
		gtk_toggle_button_set_active(
				GTK_TOGGLE_BUTTON(mainptr->cdr_option_info->swab_audio),
				FALSE);
}
static void set_speed(char *s) {
	int l;
	l = strtol(s, (char **)NULL, 10);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(speed_spin), l);
}
static void set_grace_time(char *s) {
	int l;
	l = strtol(s, (char **)NULL, 10);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(grace_time_spin), l);
}
static void set_dummy(char *path) {
	if (*(path) == '1')
		misc_prefs.dummy = TRUE;
	else
		misc_prefs.dummy = FALSE;
}
static void set_get_size(char *path) {
	if (*(path) == '1')
		misc_prefs.get_size = TRUE;
	else
		misc_prefs.get_size = FALSE;
}
static void set_eject(char *path) {
	if (*(path) == '1')
		misc_prefs.eject = TRUE;
	else
		misc_prefs.eject = FALSE;
}
static void set_multisession(char *path) {
	if (*(path) == '1')
		misc_prefs.set_multisession = TRUE;
	else
		misc_prefs.set_multisession = FALSE;
}
static void set_do_ms_check(char *path) {
	if (*(path) == '1')
		misc_prefs.do_ms_check = TRUE;
	else
		misc_prefs.do_ms_check = FALSE;
}
static void set_beep(char *path) {
	if (*(path) == '1')
		misc_prefs.beep = TRUE;
	else
		misc_prefs.beep = FALSE;
}
static void set_burnproof(char *path) {
	if (*(path) == '1')
		misc_prefs.burnproof = TRUE;
	else
		misc_prefs.burnproof = FALSE;
}
static void set_disable_fixate(char *path) {
	if (*(path) == '1')
		misc_prefs.disable_fixate = TRUE;
	else
		misc_prefs.disable_fixate = FALSE;
}
static void set_old_session(char *path) {
	if (NULL != misc_prefs.old_session)
		g_free(misc_prefs.old_session);
	misc_prefs.old_session = g_strdup(path);
}

static void set_completion_msg(char *path) {
	misc_prefs.completion_msg = (*(path) == '1');
}
static void set_overburn(char *path) {
	misc_prefs.overburn = (*(path) == '1');
}

static struct {
	char *string;
	void (*function)(char *arg);
} string_function[] = {
	{"mkisofs_path=", set_mkisofs_path},
	{"cdrecord_path=", set_cdrecord_path},
	{"cdlabelgen_path=", set_cdlabelgen_path},
	{"gdiff_path=", set_diff_path},
	{"cdda2wav_path=", set_cdda2wav_path},
	{"cdparanoia_path=", set_cdparanoia_path},
	{"lpr_path=", set_lpr_path},
	{"iso_location=", set_iso_location},
	{"rip_location=", set_rip_location},
	{"selection_location=", set_selection_location},
	{"enable_tooltips=", set_enable_tooltips},
	{"own_progresswin=", set_own_progresswin},
	{"dev_scsibus=", set_scsibus},
	{"dev_target=", set_target},
	{"dev_lun=", set_lun},
	{"dev_driver=", set_driver},
	{"dev_device=", set_device},
	{"dev_fifosize=", set_fifosize},
	{"dev_cddadevice=", set_cddadevice},
	{"dev_isodevice=", set_isodevice},
	{"dev_swab=", set_swab},
	{"dev_speed=", set_speed},
	{"dev_grace_time=", set_grace_time},
	{"dummy=", set_dummy},
	{"get_size=", set_get_size},
	{"eject=", set_eject},
	{"set_multisession=", set_multisession},
	{"do_ms_check=", set_do_ms_check},
	{"beep=", set_beep},
	{"burnproof=", set_burnproof},
	{"disable_fixate=", set_disable_fixate},
	{"completion_msg=", set_completion_msg},
	{"overburn=", set_overburn},
	{"set_old_session=", set_old_session}
};

/* This is loosely based on an old which implementation by Paul Vixie
 * - license unclear, but it's not a plain copy so there shouldn't be
 * much of a problem anyway */
static char *which(char* name) {
	char    test[256], *pc, save;
	int     len;

	pc = getenv("PATH");

	if (pc == NULL) {
		return g_strdup("NOT_FOUND");
	}

	while (*pc != '\0') {
		len = 0;
		while (*pc != ':' && *pc != '\0') {
			len++;
			pc++;
		}

		save = *pc;
		*pc = '\0';
		g_snprintf(test, 256, "%s/%s", pc-len, name);
		*pc = save;
		if (*pc)
			pc++;

		if (0 == access(test, X_OK))
			return (g_strdup(test));
	}

	return g_strdup("NOT_FOUND");
}


void read_config_file(void) {
	FILE *fp;
	char line[BUF_S];
	linebuffer file;
	char *config;

	read_env();

	if (NULL == mkisofs_path)
		mkisofs_path = which("mkisofs");
	if (NULL == cdrecord_path)
		cdrecord_path = which("cdrecord");
	if (NULL == cdlabelgen_path)
		cdlabelgen_path = which("cdlabelgen");
	if (NULL == diff_path)
		diff_path = which("diff");
	if (NULL == cdda2wav_path)
		cdda2wav_path = which("cdda2wav");
	if (NULL == cdparanoia_path)
		cdparanoia_path = which("cdparanoia");
	if (NULL == lpr_path) {
		lpr_path = which("lpr");
		if (strcmp(lpr_path, "NOT_FOUND") == 0) {
			g_free(lpr_path);
			lpr_path = which("lp");
		}
	}

	isocdrom_device_path = g_strdup("");

	config = get_config_file();

	if (NULL == (fp = fopen(config, "r"))) {
		alert_user_of_error(_(" Could not open config file.\n"
					" Please configure gcombust in the File/Preferences menu \n"
					" (Preferences window opened, at least the path to \n"
					" mkisofs need to be filled in) "));
		preferences_cmd_callback(NULL, NULL);
		free(config);
		return;
	}
	free(config);

	init_buffer(&file);

	while (NULL != fgets(line, BUF_S, fp)) {
		add_to_buffer(&file, line);
	}

	fclose(fp);
	while (TRUE == extract_line(&file, "\n")) {
		unsigned int i;
		file.newline[strlen(file.newline)-1] = '\0'; /* \n */
		for (i = 0; i < sizeof(string_function)/sizeof(string_function[0]); i++) {
			if (strlen(file.newline) > strlen(string_function[i].string)) {
				if (!(strncmp(string_function[i].string, file.newline,
								strlen(string_function[i].string)))) {
					string_function[i].function(
							file.newline + strlen(string_function[i].string));
				}
			}
		}
	}
	apply_misc_changes();
	apply_storage_changes();
	gtk_entry_set_text(GTK_ENTRY(audio_device_path), cdda_device_path);
	gtk_entry_set_text(GTK_ENTRY(iso_device_path), isocdrom_device_path);

	destroy_buffer(&file);
}

void save_config_file(config_cdr_data *cdr_info) {
	FILE *fp;
	char *config;

	config = get_config_dir();
	mkdir(config, S_IRWXU | S_IRWXG);
	free(config);

	config = get_config_file();
	if (NULL == (fp = fopen(config, "w"))) {
		alert_user_of_error(_(" Could not open config file for writing! \n"));
		free(config);
		return;
	}
	free(config);

	fprintf(fp, "mkisofs_path=%s\n", mkisofs_path);
	fprintf(fp, "cdrecord_path=%s\n", cdrecord_path);
	fprintf(fp, "cdlabelgen_path=%s\n", cdlabelgen_path);
	fprintf(fp, "gdiff_path=%s\n", diff_path);
	fprintf(fp, "cdda2wav_path=%s\n", cdda2wav_path);
	fprintf(fp, "cdparanoia_path=%s\n", cdparanoia_path);
	fprintf(fp, "lpr_path=%s\n", lpr_path);
	fprintf(fp, "iso_location=%s\n", def_image_path);
	fprintf(fp, "rip_location=%s\n", def_rip_path);
	fprintf(fp, "selection_location=%s\n", def_selection_path);

	fprintf(fp, "enable_tooltips=%d\n", misc_prefs.enable_tooltips);
	fprintf(fp, "own_progresswin=%d\n", misc_prefs.own_progresswin);
	fprintf(fp, "dummy=%d\n", misc_prefs.dummy);
	fprintf(fp, "get_size=%d\n", misc_prefs.get_size);
	fprintf(fp, "eject=%d\n", misc_prefs.eject);
	fprintf(fp, "set_multisession=%d\n", misc_prefs.set_multisession);
	fprintf(fp, "do_ms_check=%d\n", misc_prefs.do_ms_check);
	fprintf(fp, "beep=%d\n", misc_prefs.beep);
	fprintf(fp, "burnproof=%d\n", misc_prefs.burnproof);
	fprintf(fp, "disable_fixate=%d\n", misc_prefs.disable_fixate);
	fprintf(fp, "completion_msg=%d\n", misc_prefs.completion_msg);
	fprintf(fp, "overburn=%d\n", misc_prefs.overburn);

	fprintf (fp, "set_old_session=%s\n",misc_prefs.old_session);

	fprintf(fp, "dev_scsibus=%i\n", gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cdr_info->scsibus)));
	fprintf(fp, "dev_target=%i\n", gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cdr_info->target)));
	fprintf(fp, "dev_lun=%i\n", gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cdr_info->lun)));
	fprintf(fp, "dev_driver=%s\n", gtk_entry_get_text(GTK_ENTRY(cdr_info->driver)));
	fprintf(fp, "dev_device=%s\n", gtk_entry_get_text(GTK_ENTRY(cdr_info->device)));
	fprintf(fp, "dev_fifosize=%s\n", gtk_entry_get_text(GTK_ENTRY(cdr_info->fifo)));
	fprintf(fp, "dev_cddadevice=%s\n", cdda_device_path);
	fprintf(fp, "dev_isodevice=%s\n", isocdrom_device_path);

	if (GTK_TOGGLE_BUTTON(cdr_info->swab_audio)->active)
		fprintf(fp, "dev_swab=1\n");
	else
		fprintf(fp, "dev_swab=0\n");

	fprintf(fp, "dev_speed=%i\n", gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(speed_spin)));
	fprintf(fp, "dev_grace_time=%i\n", gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(grace_time_spin)));

	fclose(fp);
}

