/*
** Copyright (C) 10 Feb 1999 Jonas Munsin <jmunsin@iki.fi>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>

#include "multisession.h"
#include "vector_commands.h"
#include "common_gtk.h"
#include "modify_file_set.h"
#include "filepicker.h"
#include "command.h"
#include "linebuffer.h"
#include "optimize_usage.h"
#include "contractions.h"
#include "locks.h"
#include "mainwindow.h"
#include "globals.h"

/* static gint sel_row_path = -1; */
static long int lastsize = 0;
static char *dir_base = NULL;

static void update_totalsize_label(void) {
	char *sizeinfolabel;
	char tmp_buf[SIZE_BUF_SIZE];

	if (sectorsize < 0)
		sectorsize = 150;
	if (totalsize < 0)
		totalsize = 0;

	sizeinfolabel = string_append(_("Sectors: "), NULL);
	if (g_snprintf(tmp_buf, SIZE_BUF_SIZE, "%6.0f", (double)sectorsize) == -1)
		g_warning("%s::%i: sector size too large? Sector-result is probably wrong",
				__FILE__, __LINE__);
	sizeinfolabel = string_append(sizeinfolabel, tmp_buf);
	sizeinfolabel = string_append(sizeinfolabel, _("\nSize: "));
	if (g_snprintf(tmp_buf, SIZE_BUF_SIZE, "%5.1f", (double)totalsize/(1024*1024)) == -1)
		g_warning("%s::%i: file size too large? Total size-result is probably wrong",
				__FILE__, __LINE__);
	sizeinfolabel = string_append(sizeinfolabel, tmp_buf);
	sizeinfolabel = string_append(sizeinfolabel, " MB");

	if (opt_cdsize >= sectorsize) {
		GtkRcStyle *rc_style;
		rc_style = gtk_rc_style_new();
		gtk_widget_modify_style(GTK_WIDGET(total_size), rc_style);
		gtk_rc_style_unref(rc_style);
	} else {
		GtkRcStyle *rc_style;
		GdkColor color;

		gdk_color_parse("red", &color);

		rc_style = gtk_rc_style_new();

		rc_style->fg[GTK_STATE_NORMAL] = color;
		rc_style->fg[GTK_STATE_INSENSITIVE] = color;
		rc_style->color_flags[GTK_STATE_NORMAL] |= GTK_RC_FG;
		rc_style->color_flags[GTK_STATE_INSENSITIVE] |= GTK_RC_FG;

		gtk_widget_modify_style(GTK_WIDGET(total_size), rc_style);

		gtk_rc_style_unref(rc_style);
	}

	gtk_label_set(GTK_LABEL(total_size), sizeinfolabel);
	free(sizeinfolabel);
}

static void error_when_executing(cmd_v *command, const char *error_details, char *output) {
	char *cmd_string, *error_msg;

	cmd_string = cmd_to_string(command);
	error_msg = g_strdup_printf(_("%s\n\n The following command was executed when the error occured: \n%s"), error_details, cmd_string);

	if (NULL == output)
		alert_user_of_error(error_msg);
	else
		alert_user_of_error_msg(error_msg, output);

	free(cmd_string);
	free(error_msg);
}

static struct {
	const char *mkisofs_error;
	const char *msg_to_user;
} mkisofs_size_errors[] = {
	{"Usage: mkisofs", N_(" mkisofs received arguments it did "
			"not understand, \n maybe you are using "
			"too old a version of mkisofs? ")},
	{"Invalid node -", N_(" mkisofs detected an invalid filename "
			"in the list ")},
	{"volume size must be >= 800K", N_(" HFS volume size must be >= 800K "
			"- file/dir size too small (the size \n"
			" estimate might be a little bit wrong), "
			"add more files before burning ")},
	{"Unable to sort directory ", N_(" mkisofs failed to sort directory, "
			"(maybe a problem with multisession, \n try "
			"enabeling the No RR option if multisession "
			"is used). It it also possible \n that there "
			"is a duplicate filename in the list, if so, "
			"please rename it. \n It is also quite "
			"possible that there is a bug in mkisofs. ")},
	{"Joliet tree sort failed.", N_(" Joliet tree sort failed - maybe "
			"the image contains \n a filename-component "
			"longer than 64 characters? ")}

};

static const char *search_for_mkisofs_error(const char *msgbuf) {
	unsigned int i;

	for (i = 0; i < sizeof(mkisofs_size_errors)/sizeof(mkisofs_size_errors[0]); i++) {
		if (strstr(msgbuf, mkisofs_size_errors[i].mkisofs_error))
			return mkisofs_size_errors[i].msg_to_user;
	}
	return NULL;

}

void execute_recalc_size(cmd_v *command) {
	int done = 0;
	FILE *output;
	char in_buf[BUF_S];
	char *saved_output;
	const char *error_msg = NULL;

	lastsize = 0;

	if (NULL == (output = popen_r_stderr(command))) {
		error_when_executing(command, _(" The command could not be executed. "), NULL);
		return;
	}

	saved_output = g_strdup("");
	while (!(done)) {
		if (fgets(in_buf, BUF_S, output) != NULL) {
			char *s, *start;
			s = g_strconcat(saved_output, in_buf, NULL);
			g_free(saved_output);
			saved_output = s;

			if (feof(output)) {
				error_msg = _(" EOF reached reading "
						"mkisofs size output ");
				done = 1;
			} else if (ferror(output)) {
				error_msg = _(" error reading mkisofs "
						"size output ");
				done = 1;
			} else if ((start = strstr(in_buf, "Total extents scheduled to be written"))) {
				int add = 40;
				if (strstr(start, "(inc HFS)"))
					add = 50;
				g_assert((start-in_buf)+add < BUF_S);
				lastsize = atol(start + add);
				sectorsize += lastsize;
				totalsize = 2048 * sectorsize;
				done = 1;
			} else if (NULL != (error_msg = search_for_mkisofs_error(in_buf))) {
				done = 1;
			}
		} else {
			error_msg = _(" There were and unrecognized error "
					"with mkisofs when calculating the "
					"size; \n perhaps your mkisofs "
					"is too old for gcombust? ");
			done = 1;
		}
	}
	if (NULL != error_msg)
		error_when_executing(command, error_msg, saved_output);
	g_free(saved_output);
	pclose(output);
}

long int get_old_session_size(config_cdr_data *cdr_info) {
	long int size = 0;
	if ((GTK_TOGGLE_BUTTON(enable_multisession)->active) &&
			(0 != strlen(gtk_entry_get_text(GTK_ENTRY((old_session)))))) {
		char *lastsize = get_last_session_info(cdr_info);
		if (NULL != lastsize) {
			char *comma = strchr(lastsize, ',');
			if (NULL != comma) {
				comma++;
				size = atoi(comma);
			}
			free(lastsize);
		}
	}
	return size;
}

/* TODO: the size for each entry should be recalculated to reflect changes
 * 	 on follow symlink flags etc. (at least if auto bin packing is
 *	 implemented someday)
 */
void recalc_size(GtkWidget *widget, gpointer data) {
	cmd_v *mkisofs_size_cmd;
	config_cdr_data *cdr_info = (config_cdr_data *) data;

	if ((mkisofs_size_cmd = make_mkisofs_command(TRUE, NULL, cdr_info)) == NULL)
		return;

	sectorsize = 150; /* for 74m CDs, see comment elsewhere */

	sectorsize += get_old_session_size(cdr_info);

	if (!(is_running())) {
		execute_recalc_size(mkisofs_size_cmd);
		not_running();
	}

	destroy_cmd(mkisofs_size_cmd);

	update_totalsize_label();
	make_estimate_inaccurate();
	gtk_widget_set_sensitive(total_size, 1);
}

/* Checks if a path is in the file-list
 * returns TRUE if it's already there, FALSE if it's not */
static int path_is_in_iso(char *path) {
	int i;
	file_data *info;

	for (i = 0; i < GTK_CLIST(clist)->rows; i ++) {
		info = gtk_clist_get_row_data(GTK_CLIST(clist), i);
		g_assert(NULL != info);
		if (strlen(path) == strlen(info->realpath) && (!(strncmp(path, info->realpath, strlen(path)))))
			return TRUE;
	}
	return FALSE;
}

static void add_path_to_iso_files(gpointer data, gpointer user_data) {
	cmd_v *exec_string;
	char tmp_buf[SIZE_BUF_SIZE];
	char *text[2];
	int newrow;
	file_data *info;
	char *path;

	/* extract path */
	if (NULL == user_data)
		path = strdup(data);
	else {
		char *tmp;
		gtk_clist_get_text(user_data, GPOINTER_TO_INT(data), 0, &tmp);
		path = string_append(dir_base, NULL);
		path = string_append(path, tmp);
	}
	if (path_is_in_iso(path)) {
		alert_user_of_error(_("You tried to add a duplicate file/dir to the file list!"));
		free(path);
		return;
	}

	if ((exec_string = make_mkisofs_command(TRUE, path, mainptr->cdr_option_info)) == NULL) {
		free(path);
		return;
	}
	if (!(is_running())) {
		if (0 == GTK_CLIST(clist)->rows) {
			/* as reading the multisession info takes quite some time,
			 * we only read it for the first file added (and also on
			 * recalc/optimize actions) */
			sectorsize = 150;
			sectorsize += get_old_session_size(mainptr->cdr_option_info);
		}
		execute_recalc_size(exec_string);
		not_running();
	}
	destroy_cmd(exec_string);

	if (g_snprintf(tmp_buf, SIZE_BUF_SIZE, "%.1f",
			(double)lastsize*2048/(1024*1024)) == -1)
		g_warning("%s::%i: file size too large? Size-result is probably wrong", __FILE__, __LINE__);
	text[1] = tmp_buf;

	info = malloc(sizeof(file_data));
	info->size = lastsize;

	/* this might seem a bit too compact, but attempts to make it more readable
	 * resulted in longer and harder to trace trough code */
	if (path[strlen(path)-1] == '/') {
		my_strip_trailing_slashes(path);
		text[0] = my_basename(path);
		info->realpath = g_strdup_printf("%s/", path);
		info->userpath = g_strdup_printf("%s/", my_basename(path));
		path[strlen(path)] = '/';
	} else {
		text[0] = my_basename(path);
		info->realpath = g_strdup(path);
		info->userpath = g_strdup(my_basename(path));
	}

	newrow = gtk_clist_append(GTK_CLIST(clist), text);
	gtk_clist_set_row_data(GTK_CLIST(clist), newrow, info); 

	free(path);
}

/* Add file to the (iso) file-list
 * if widget != 0, add the filepath contained in the file_selection data to
 *	the isofile clist
 * if widget == 0, add the path data to the isofile clist
 * TODO: go trough the clist and check if adding a duplicate entry */
void selected_file(GtkWidget *widget, gpointer data) {
	if (widget != 0) {
		char *tmp;
		GList *list;

		dir_base = g_strdup(gtk_file_selection_get_filename(data));
		if (NULL == (tmp = strrchr(dir_base, '/'))) {
			g_warning("%s::%i: no / in path, this is a bug", __FILE__, __LINE__);
			return;
		}
		tmp++;
		*tmp = '\0';

		cursor_wait(NULL);

		list = (GTK_CLIST(GTK_FILE_SELECTION(data)->file_list))->selection;
		if (NULL != list) {
			g_list_foreach(list, add_path_to_iso_files,
					(GTK_CLIST(GTK_FILE_SELECTION(data)->file_list)));
		} else {
			tmp--;
			*tmp = '\0';
			if (NULL == (tmp = strrchr(dir_base, '/'))) {
				/* FIXME: adding /asdasd causes this to occur*/
				g_warning("%s::%i: no / in path, this is probably a bug "
						"(did you maybe add a non existant root dir?)",
						__FILE__, __LINE__);
				cursor_reset(NULL);
				return;
			}
			tmp++;
			*tmp = '\0';
		}

		list = (GTK_CLIST(GTK_FILE_SELECTION(data)->dir_list))->selection;
		if (NULL != list)
			g_list_foreach(list, add_path_to_iso_files,
					(GTK_CLIST(GTK_FILE_SELECTION(data)->dir_list)));

		free(dir_base);
		dir_base = NULL;
		gtk_clist_unselect_all(GTK_CLIST(GTK_FILE_SELECTION(data)->file_list));
		gtk_clist_unselect_all(GTK_CLIST(GTK_FILE_SELECTION(data)->dir_list));
		cursor_reset(NULL);
	} else {
		add_path_to_iso_files(data, NULL);
	}

	update_totalsize_label();
	make_estimate_inaccurate();

	resize_clist_col_ratio(NULL, NULL, clist);

	mark_size_inaccurate();
}

void del_selected_rows(GtkWidget *widget, gpointer data) {
	file_data *info;
	int i;

	for (i = MAX_ROWS-1; i >= 0; i--) {
		if (1 == selected_files[i]) {
			info = gtk_clist_get_row_data(GTK_CLIST(clist), i);
			if (info == NULL) {
				g_warning("modify_file_set.c::del_selected_rows: "
						"trying to access non-existant clist row!");
				break;
			}
			totalsize -= info->size*2048;
			sectorsize -= info->size;
			update_totalsize_label();
			make_estimate_inaccurate();
			mark_size_inaccurate();
			gtk_clist_remove(GTK_CLIST(clist), i);
			g_free(info->realpath);
			g_free(info->userpath);
			free(info);
		}
	}
}

/*
void selected_dir(GtkWidget *widget, gpointer data) {
	gchar *text[1];

	text[0] = g_strdup(gtk_file_selection_get_filename(data));

	gtk_clist_append(GTK_CLIST(exclude_dir), text);
	totalrows_path++;
	mark_size_inaccurate();
}

void remove_dir(GtkWidget *widget, gpointer data) {
	if (sel_row_path != -1) {
		gtk_clist_remove(GTK_CLIST(exclude_dir), sel_row_path);
		totalrows_path--;
		mark_size_inaccurate();
	}
}
*/
