// -*- C++ -*-

/* 
 * GChemPaint library
 * text.cc 
 *
 * Copyright (C) 2002-2005 Jean Bréfort <jean.brefort@normalesup.org>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301
 * USA
 */

#include "gchempaint-config.h"
#include "text.h"
#include "widgetdata.h"
#include "view.h"
#include "document.h"
#include  "application.h"
#include "settings.h"
#include "libgcpcanvas/gcp-canvas-group.h"
#include "libgcpcanvas/gcp-canvas-rect-ellipse.h"
#include "libgcpcanvas/gprintable.h"

GtkTextTagTable* TextTagTable;

void gcpText::InitTags(const char* FontName)
{
	TextTagTable = gtk_text_tag_table_new();
	GtkTextTag* tag = gtk_text_tag_new("bold");
	g_object_set(G_OBJECT(tag), "weight", PANGO_WEIGHT_BOLD, NULL);
	gtk_text_tag_table_add(TextTagTable, tag);
	g_object_unref((GObject*)tag);
	tag = gtk_text_tag_new("italic");
	g_object_set(G_OBJECT(tag), "style", PANGO_STYLE_ITALIC, NULL);
	gtk_text_tag_table_add(TextTagTable, tag);
	g_object_unref((GObject*)tag);
	tag = gtk_text_tag_new("underline");
	g_object_set(G_OBJECT(tag), "underline", PANGO_UNDERLINE_SINGLE, NULL);
	gtk_text_tag_table_add(TextTagTable, tag);
	g_object_unref((GObject*)tag);
	tag = gtk_text_tag_new("strikethrough");
	g_object_set(G_OBJECT(tag), "strikethrough", true, NULL);
	gtk_text_tag_table_add(TextTagTable, tag);
	g_object_unref((GObject*)tag);
	tag = gtk_text_tag_new("subscript");
	g_object_set(G_OBJECT(tag), "rise", -2 * PANGO_SCALE, "size", 8 * PANGO_SCALE, NULL);
	gtk_text_tag_table_add(TextTagTable, tag);
	g_object_unref((GObject*)tag);
	tag = gtk_text_tag_new("superscript");
	g_object_set(G_OBJECT(tag), "rise", 4 * PANGO_SCALE, "size", 8 * PANGO_SCALE, NULL);
	gtk_text_tag_table_add(TextTagTable, tag);
	g_object_unref((GObject*)tag);
	PangoFontDescription* pfd =pango_font_description_from_string(FontName);
	tag = gtk_text_tag_new(FontName);
	g_object_set(G_OBJECT(tag),
						"family", pango_font_description_get_family(pfd),
						"size", pango_font_description_get_size(pfd),
						NULL);
	pango_font_description_free(pfd);
	gtk_text_tag_table_add(TextTagTable, tag);
	g_object_unref((GObject*)tag);
}

enum {
  UPDATE_BOUNDS,
  LAST_SIGNAL
};

/*
Derivation of a new widget from GnomeCanvasRichTextExt with an event for setting canvas size on realize
*/
static void gnome_canvas_rich_text_gcp_class_init(GnomeCanvasRichTextGCPClass *Class);
static void gnome_canvas_rich_text_gcp_init(GnomeCanvasRichTextGCP *canvas);

static void gnome_canvas_rich_text_gcp_realize(GnomeCanvasItem *item);
static gint gnome_canvas_rich_text_gcp_event(GnomeCanvasItem *item, GdkEvent *event);
static GnomeCanvasItemClass *parent_class;

GType
gnome_canvas_rich_text_gcp_get_type (void)
{
	static GType canvas_rich_text_gcp_type;

	if (!canvas_rich_text_gcp_type) {
		static const GTypeInfo object_info = {
			sizeof(GnomeCanvasRichTextGCPClass),
			(GBaseInitFunc)NULL,
			(GBaseFinalizeFunc)NULL,
			(GClassInitFunc)gnome_canvas_rich_text_gcp_class_init,
			(GClassFinalizeFunc)NULL,
			NULL,			/* class_data */
			sizeof(GnomeCanvasRichTextGCP),
			0,			/* n_preallocs */
			(GInstanceInitFunc)gnome_canvas_rich_text_gcp_init,
			NULL			/* value_table */
		};

		canvas_rich_text_gcp_type = g_type_register_static(GNOME_TYPE_CANVAS_RICH_TEXT_EXT, "GnomeCanvasRichTextGCP",
						      &object_info, (GTypeFlags)0);
	}

	return canvas_rich_text_gcp_type;
}

void gnome_canvas_rich_text_gcp_class_init(GnomeCanvasRichTextGCPClass *Class)
{
	GnomeCanvasItemClass *item_class = GNOME_CANVAS_ITEM_CLASS(Class);
	parent_class = (GnomeCanvasItemClass*)g_type_class_peek_parent (Class);
	item_class->realize = (void (*)(GnomeCanvasItem*))gnome_canvas_rich_text_gcp_realize;
	item_class->event = (gint (*)(GnomeCanvasItem*, GdkEvent*))gnome_canvas_rich_text_gcp_event;
}

void gnome_canvas_rich_text_gcp_init(GnomeCanvasRichTextGCP *canvas)
{
}

static void gnome_canvas_rich_text_gcp_realize(GnomeCanvasItem *item)
{
	GnomeCanvasRichTextExt *text = GNOME_CANVAS_RICH_TEXT_EXT(item);
	(* GNOME_CANVAS_ITEM_CLASS(parent_class)->realize)(item);
	gcpText* pText = (gcpText*)g_object_get_data(G_OBJECT(item), "object");
	g_object_set_data(G_OBJECT(item), "realized", (void*)1);
	if (pText) pText->OnChanged(gnome_canvas_rich_text_ext_get_buffer(text));
}

static gint gnome_canvas_rich_text_gcp_event(GnomeCanvasItem *item, GdkEvent *event)
{
	gcpText* pText = (gcpText*)g_object_get_data(G_OBJECT(item), "object");
	if (pText)
		if (!pText->OnEvent(event))
			return (* GNOME_CANVAS_ITEM_CLASS(parent_class)->event)(item, event);
	return true;
}

gcpText::gcpText(): gcpTextObject(TextType)
{
}

gcpText::gcpText(double x, double y): gcpTextObject(x, y, TextType)
{
}

gcpText::~gcpText()
{
}

void gcpText::GetCoords(double *x, double *y)
{
	*x = m_x;
	*y = m_y;
}

void gcpText::SetCoords(double x, double y)
{
	m_x = x;
	m_y = y;
}

xmlNodePtr gcpText::Save(xmlDocPtr xml)
{
	GtkTextIter start, end;
	gtk_text_buffer_get_start_iter(m_buf, &start);
	gtk_text_buffer_get_end_iter(m_buf, &end);
	xmlNodePtr node = xmlNewDocNode(xml, NULL, (xmlChar*)"text", NULL);
	if (!node) return NULL;
	GSList* TagList = gtk_text_iter_get_tags(&start), *OtherTagList = NULL;
	if (! SaveNode(xml, node, &start, &end, TagList, OtherTagList))
	{
		xmlFree(node);
		return NULL;
	}
	g_slist_free(TagList);
	return (gcpTextObject::SaveNode(xml, node))? node: NULL;
}

xmlNodePtr gcpText::SaveSelection(xmlDocPtr xml)
{
	GtkTextIter start, end;
	gtk_text_buffer_get_selection_bounds(m_buf, &start, &end);
	xmlNodePtr node = xmlNewDocNode(xml, NULL, (xmlChar*)"text", NULL);
	if (!node) return NULL;
	GSList* TagList = gtk_text_iter_get_tags(&start), *OtherTagList = NULL;
	if (! SaveNode(xml, node, &start, &end, TagList, OtherTagList))
	{
		xmlFree(node);
		return NULL;
	}
	g_slist_free(TagList);
	return (gcpTextObject::SaveNode(xml, node))? node: NULL;
}

bool gcpText::SaveNode(xmlDocPtr xml, xmlNodePtr node, GtkTextIter *iter, GtkTextIter *end, GSList*& TagList, GSList*& OtherTagList)
{
	GtkTextIter start = *iter, next;
	GSList* TagList0, *TagList1;
	if (g_slist_length(TagList) == 0)
	{
		gtk_text_iter_forward_to_tag_toggle(iter, NULL);
		if (gtk_text_iter_compare(end, iter) <= 0)
		{
			*iter = *end;
		}
		char *rn, *text = gtk_text_buffer_get_text(m_buf, &start, iter, true);
		while ((rn = strchr(text, '\n')))
		{
			xmlNodeAddContentLen(node, (xmlChar*)text, rn - text);
			text = rn + 1;
			xmlNodePtr child = xmlNewDocNode(xml, NULL, (xmlChar*)"br", NULL);
			xmlAddChild(node, child);
		}
		xmlNodeAddContent(node, (xmlChar*)text);
	}
	else
	{
		//find the longest included tag
		TagList0 = TagList;
		GtkTextTag *tag = NULL;
		while (TagList0)
		{
			next = start;
			gtk_text_iter_forward_to_tag_toggle(&next, GTK_TEXT_TAG(TagList0->data));
			if (gtk_text_iter_compare(iter, &next) < 0)
			{
				*iter = next;
				tag = GTK_TEXT_TAG(TagList0->data);
			}
			if (gtk_text_iter_compare(end, &next) <= 0)
			{
				*iter = *end;
				break;
			}
			TagList0 = TagList0->next;
		}
		GValue val;
		memset(&val, 0, sizeof(GValue));
		gchar* tagname;
		g_value_init(&val, G_TYPE_STRING);
		g_object_get_property((GObject*) tag, "name", &val);
		const gchar* name = g_value_get_string(&val);
		if (!strcmp(name, "bold")) tagname = "b";
		else if  (!strcmp(name, "italic")) tagname = "i";
		else if  (!strcmp(name, "underline")) tagname = "u";
		else if  (!strcmp(name, "strikethrough")) tagname = "s";
		else if  (!strcmp(name, "subscript")) tagname = "sub";
		else if  (!strcmp(name, "superscript")) tagname = "sup";
		else tagname = "font";
		OtherTagList = g_slist_append(OtherTagList, tag);
		TagList = g_slist_remove(TagList, tag);
		xmlNodePtr child = xmlNewDocNode(xml, NULL, (xmlChar*)tagname, NULL);
		if (!child) return false;
		if (!strcmp(tagname, "font"))
			xmlNewProp(child, (xmlChar*)"name", (xmlChar*)name);
		SaveNode(xml, child, &start, iter, TagList, OtherTagList);
		xmlAddChild(node, child);
		OtherTagList = g_slist_remove(OtherTagList, tag);
		TagList = g_slist_append(TagList, tag);
	}
	if (!gtk_text_iter_equal(iter, end))
	{
		TagList1 = TagList0 = gtk_text_iter_get_tags(iter);
		if (OtherTagList) while (TagList1)
		{
			if (g_slist_find(OtherTagList, TagList1->data)) TagList0 = g_slist_remove(TagList0, TagList1->data);
			TagList1 = TagList1->next;
		}
		SaveNode(xml, node, iter, end, TagList0, OtherTagList);
		g_slist_free(TagList0);
	}
	return true;
}

bool gcpText::Load (xmlNodePtr node)
{
	if (!gcpTextObject::Load (node))
		return false;
	xmlNodePtr child;
	m_bLoading = true;
	child = node->children;
	GtkTextIter iter;
	gtk_text_buffer_get_iter_at_offset (m_buf, &iter, 0);
	while (child)
	{
		if (!LoadNode (child, &iter))
			return false;
		child = child->next;
	}
	m_bLoading = false;
	OnChanged (m_buf);
	return true;
}

bool gcpText::LoadSelection (xmlNodePtr node, GtkTextIter *iter)
{
	xmlNodePtr child;
	m_bLoading = true;
	child = node->children;
	while (child)
	{
		if (!LoadNode (child, iter))
			return false;
		child = child->next;
	}
	m_bLoading = false;
	OnChanged (m_buf);
	return true;
}

bool gcpText::LoadNode(xmlNodePtr node, GtkTextIter* CurIter, int level)
{
	char* tmp;
	if (!strcmp((const char*)node->name, "text"))
	{
		if (!level)
			return true;
		tmp = (char*) xmlNodeGetContent(node);
		if (tmp) {
			gtk_text_buffer_insert(m_buf, CurIter, tmp, -1);
			xmlFree (tmp);
		}
		return true;
	}
	else if (!strcmp((const char*)node->name, "br"))
	{
		gtk_text_buffer_insert(m_buf, CurIter, "\n", -1);
		return true;
	}
	else if (!strcmp((const char*)node->name, "b")) return LoadTaggedNode(node, CurIter, "bold");
	else if (!strcmp((const char*)node->name, "i")) return LoadTaggedNode(node, CurIter, "italic");
	else if (!strcmp((const char*)node->name, "u")) return LoadTaggedNode(node, CurIter, "underline");
	else if (!strcmp((const char*)node->name, "s")) return LoadTaggedNode(node, CurIter, "strikethrough");
	else if (!strcmp((const char*)node->name, "sub")) return LoadTaggedNode(node, CurIter, "subscript");
	else if (!strcmp((const char*)node->name, "sup")) return LoadTaggedNode(node, CurIter, "superscript");
	else if (!strcmp((const char*)node->name, "font"))
	{
		char* TagName = (char*) xmlGetProp (node, (xmlChar*) "name");
		if (!TagName) return false;
		GtkTextTag* tag = gtk_text_tag_table_lookup (TextTagTable, TagName);
		if (!tag)
		{
			PangoFontDescription* pfd =pango_font_description_from_string (TagName);
			GtkTextTag* tag = gtk_text_tag_new (TagName);
			g_object_set (G_OBJECT (tag),
								"family", pango_font_description_get_family (pfd),
								"size", pango_font_description_get_size (pfd),
								NULL);
			pango_font_description_free (pfd);
			gtk_text_tag_table_add (TextTagTable, tag);
			g_object_unref ((GObject*) tag);
		}
		bool result = LoadTaggedNode (node, CurIter, TagName);
		xmlFree (TagName);
		return result;
	}
	else return true;
}

bool gcpText::LoadTaggedNode (xmlNodePtr node, GtkTextIter* CurIter, const char* TagName)
{
	GtkTextIter start, end;
	gint offset = gtk_text_iter_get_offset (CurIter);
	xmlNodePtr child = node->children;
	while (child) {
		if (!LoadNode (child, CurIter, 1))
			return false;
		child = child->next;
	}
	gtk_text_buffer_get_iter_at_offset (m_buf, &start, offset);
	gtk_text_buffer_get_selection_bounds (m_buf, NULL, &end);
	gtk_text_buffer_apply_tag (m_buf, gtk_text_tag_table_lookup(TextTagTable, TagName), &start, &end);
	return true;
}

void gcpText::Add(GtkWidget* w)
{
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	if (m_ascent <= 0)
	{
		PangoContext* pc = pData->View->GetPangoContext();
		PangoLayout *pl = pango_layout_new(pc);
		pango_layout_set_text(pl, "l", 1);
		PangoLayoutIter* iter = pango_layout_get_iter(pl);
		m_ascent = pango_layout_iter_get_baseline(iter) / PANGO_SCALE;
		pango_layout_iter_free(iter);
		g_object_unref(pl);
	}
	GnomeCanvasGroup* group = GNOME_CANVAS_GROUP(gnome_canvas_item_new(pData->Group, gnome_canvas_group_ext_get_type(), NULL));
	GnomeCanvasItem* item = gnome_canvas_item_new(
						group,
						gnome_canvas_rect_ext_get_type(),
						"x1", m_x * pData->ZoomFactor - pData->Padding,
						"y1", m_y * pData->ZoomFactor - pData->Padding - m_ascent,
						"x2", m_x * pData->ZoomFactor + m_length + pData->Padding,
						"y2", m_y * pData->ZoomFactor + m_height + pData->Padding - m_ascent,
						"fill_color", "white",
						"outline_color", "white",
						NULL);
	g_object_set_data(G_OBJECT(group), "rect", item);
	g_signal_connect(G_OBJECT(item), "event", G_CALLBACK(on_event), w);
	g_object_set_data(G_OBJECT(item), "object", this);
	item = gnome_canvas_item_new(
						group,
						gnome_canvas_rich_text_gcp_get_type(),
						"text", "",
						"x", m_x * pData->ZoomFactor,
						"y", m_y * pData->ZoomFactor - m_ascent,
						"width", m_length,
						"height", m_height,
						"grow_height", false,
						"editable", false,
						"cursor_visible", false,
						NULL);
	g_object_set_data(G_OBJECT(group), "text", item);
	gnome_canvas_rich_text_ext_set_buffer(GNOME_CANVAS_RICH_TEXT_EXT(item), m_buf);
	g_object_set_data(G_OBJECT(item), "object", this);
	g_signal_connect(G_OBJECT(item), "event", G_CALLBACK(on_event), w);
	pData->Items[this] = group;
	if (g_object_get_data(G_OBJECT(item), "realized")) OnChanged(m_buf);
}

bool gcpText::OnChanged (GtkTextBuffer *textbuffer)
{
	if (m_bLoading)
		return false;
	if (textbuffer != m_buf)
		return false;
	while (gtk_events_pending ())
		gtk_main_iteration();
	gcpDocument* pDoc = (gcpDocument*) GetDocument ();
	if (!pDoc)
		return false;
	gcpView* pView = pDoc->GetView ();
	GtkWidget* pWidget = pView->GetWidget ();
	gcpWidgetData* pData = (gcpWidgetData*) g_object_get_data (G_OBJECT (pWidget), "data");
	if (!pData->Items[this]) {
		pData->Items.erase (this);
		return false;
	}
	GnomeCanvasRichTextExt* text = (GnomeCanvasRichTextExt*) g_object_get_data (G_OBJECT (pData->Items[this]), "text");
	PangoLayout* layout = gnome_canvas_rich_text_ext_get_pango_layout (text, 0);
	if (layout) {
		PangoLayoutIter* iter = pango_layout_get_iter(layout);
		int base = pango_layout_iter_get_baseline(iter) / PANGO_SCALE;
		if (base)
			m_ascent = base;
		pango_layout_iter_free (iter);
	}
	GdkRectangle rect;
	GtkTextIter iter;
	gtk_text_buffer_get_start_iter (m_buf,&iter);
	int  y0 = 0, offset, left = 0;
	m_height = 1;
	bool bNotEnd = true;
	m_length = 0;
	if (!gtk_text_buffer_get_char_count (m_buf)) {
		gnome_canvas_rich_text_ext_get_iter_location (text, &iter, &rect);
		m_height = rect.height + 1;
		m_length = 1;
		pView->Update (this);
		EmitSignal (OnChangedSignal);
		return true;
	}
	while (bNotEnd) {
		gnome_canvas_rich_text_ext_get_iter_location (text, &iter, &rect);
		if (left > rect.x)
			left = rect.x;
		bNotEnd = gtk_text_iter_forward_to_line_end (&iter);
		gnome_canvas_rich_text_ext_get_iter_location (text, &iter, &rect);
		if (m_length <= rect.x + rect.width)
			m_length = rect.x + rect.width + 1;
		m_height += rect.height;
		if (m_height <= rect.height + y0)
			m_height = rect.height + y0 + 1;
		if (rect.y != y0) {
			while (rect.y != y0) {
				int y = rect.y;
				while (y == rect.y) {
					gtk_text_iter_backward_char (&iter);
					gnome_canvas_rich_text_ext_get_iter_location (text, &iter, &rect);
				}
				m_length += rect.x + rect.width;
			}
			gtk_text_iter_forward_to_line_end (&iter);
		}
		offset = gtk_text_iter_get_offset (&iter);
		m_length -= left;
		pView->Update (this);
		while (gtk_events_pending ())
			gtk_main_iteration ();
		gtk_text_buffer_get_iter_at_offset (m_buf, &iter, offset + 1);
		gnome_canvas_rich_text_ext_get_iter_location (text, &iter, &rect);
		y0 = rect.y;
	}
	EmitSignal (OnChangedSignal);
	return true;
}

bool gcpText::OnMarkSet(GtkTextBuffer *textbuffer, GtkTextIter *iter, GtkTextMark *mark)
{
	if (textbuffer != m_buf) return false;
	gcpDocument* pDoc = (gcpDocument*)GetDocument();
	gcpApplication* pApp = pDoc->GetApplication();
	gcpTool* TextTool = pApp->GetTool("Text");
	GtkToggleToolButton *button;
	m_bLoading = true;
	GtkTextIter start, end, next;
	gtk_text_buffer_get_selection_bounds(m_buf, &start, &end);
	next = start;
	GtkTextTag *tag = gtk_text_tag_table_lookup(TextTagTable, "bold");
	bool bTag = gtk_text_iter_has_tag(&start, tag);
	gtk_text_iter_forward_to_tag_toggle(&next, tag);
	button = (GtkToggleToolButton*)pApp->GetToolItem("bold");
	if (button) gtk_toggle_tool_button_set_active(button, bTag && (gtk_text_iter_compare(&next, &end) >= 0));
	next = start;
	tag = gtk_text_tag_table_lookup(TextTagTable, "italic");
	bTag = gtk_text_iter_has_tag(&start, tag);
	gtk_text_iter_forward_to_tag_toggle(&next, tag);
	button = (GtkToggleToolButton*)pApp->GetToolItem("italic");
	if (button) gtk_toggle_tool_button_set_active(button, bTag && (gtk_text_iter_compare(&next, &end) >= 0));
	next = start;
	tag = gtk_text_tag_table_lookup(TextTagTable, "underline");
	bTag = gtk_text_iter_has_tag(&start, tag);
	gtk_text_iter_forward_to_tag_toggle(&next, tag);
	button = (GtkToggleToolButton*)pApp->GetToolItem("underline");
	if (button) gtk_toggle_tool_button_set_active(button, bTag && (gtk_text_iter_compare(&next, &end) >= 0));
	next = start;
	tag = gtk_text_tag_table_lookup(TextTagTable, "strikethrough");
	bTag = gtk_text_iter_has_tag(&start, tag);
	gtk_text_iter_forward_to_tag_toggle(&next, tag);
	button = (GtkToggleToolButton*)pApp->GetToolItem("strikethrough");
	if (button) gtk_toggle_tool_button_set_active(button, bTag && (gtk_text_iter_compare(&next, &end) >= 0));
	next = start;
	tag = gtk_text_tag_table_lookup(TextTagTable, "subscript");
	bTag = gtk_text_iter_has_tag(&start, tag);
	gtk_text_iter_forward_to_tag_toggle(&next, tag);
	button = (GtkToggleToolButton*)pApp->GetToolItem("subscript");
	if (button) gtk_toggle_tool_button_set_active(button, bTag && (gtk_text_iter_compare(&next, &end) >= 0));
	next = start;
	tag = gtk_text_tag_table_lookup(TextTagTable, "superscript");
	bTag = gtk_text_iter_has_tag(&start, tag);
	gtk_text_iter_forward_to_tag_toggle(&next, tag);
	button = (GtkToggleToolButton*)pApp->GetToolItem("superscript");
	if (button) gtk_toggle_tool_button_set_active(button, bTag && (gtk_text_iter_compare(&next, &end) >= 0));
	//Search the font used at end iterator
	GSList *gsl = gtk_text_iter_get_tags(&end);
	while (gsl)
	{
		char* name;
		g_object_get(gsl->data, "name", &name, NULL);
		if (strcmp(name, "bold")
				&& strcmp(name,"italic")
				&& strcmp(name,"underline")
				&& strcmp(name,"strikethrough")
				&& strcmp(name,"subscript")
				&& strcmp(name,"superscipt"))
		{
			pApp->SetFontName(name);
//			g_free(FontName);
//			FontName = g_strdup(name);
		}
		gsl = gsl->next;
	}
	if (gtk_text_iter_compare(&start, &end))
	{
		GtkClipboard *clipboard = gtk_clipboard_get (GDK_SELECTION_PRIMARY);
		if (TextTool) TextTool->CopySelection(clipboard);
		pApp->ActivateMenu("Copy", true);
		pApp->ActivateMenu("Cut", true);
		pApp->ActivateMenu("Erase", true);
	}
	else
	{
		pApp->ActivateMenu("Copy", false);
		pApp->ActivateMenu("Cut", false);
		pApp->ActivateMenu("Erase", false);
	}
	m_bLoading = false;
	return true;
}

bool gcpText::OnInsertText(GtkTextBuffer *textbuffer, GtkTextIter *iter, gchar* newtext, gint length)
{
	if (m_InsertOffset == -2) m_InsertOffset = gtk_text_iter_get_offset(iter);
	return true;
}

bool gcpText::OnEndUserAction(GtkTextBuffer *textbuffer)
{
	GtkTextIter start, end;
	gcpDocument* pDoc = (gcpDocument*)GetDocument();
	gcpApplication* pApp = pDoc->GetApplication();
	gcpTool* TextTool = pApp->GetTool("Text");
	GtkToggleToolButton *button;
	gtk_text_buffer_get_selection_bounds(textbuffer, &start, &end);
	if (!m_bLoading && (m_InsertOffset >= 0))
	{
		gtk_text_iter_backward_chars (&start, gtk_text_iter_get_offset(&end) - m_InsertOffset);
		gtk_text_buffer_remove_all_tags(textbuffer, &start, &end);
		gtk_text_buffer_apply_tag_by_name(textbuffer, pApp->GetFontName(), &start, &end);
		button = (GtkToggleToolButton*)pApp->GetToolItem("bold");
		if (button && gtk_toggle_tool_button_get_active(button)) gtk_text_buffer_apply_tag_by_name(textbuffer, "bold", &start, &end);
		button = (GtkToggleToolButton*)pApp->GetToolItem("italic");
		if (button && gtk_toggle_tool_button_get_active(button)) gtk_text_buffer_apply_tag_by_name(textbuffer, "italic", &start, &end);
		button = (GtkToggleToolButton*)pApp->GetToolItem("underline");
		if (button && gtk_toggle_tool_button_get_active(button)) gtk_text_buffer_apply_tag_by_name(textbuffer, "underline", &start, &end);
		button = (GtkToggleToolButton*)pApp->GetToolItem("strikethrough");
		if (button && gtk_toggle_tool_button_get_active(button)) gtk_text_buffer_apply_tag_by_name(textbuffer, "strikethrough", &start, &end);
		button = (GtkToggleToolButton*)pApp->GetToolItem("subscript");
		if (button && gtk_toggle_tool_button_get_active(button)) gtk_text_buffer_apply_tag_by_name(textbuffer, "subscript", &start, &end);
		button = (GtkToggleToolButton*)pApp->GetToolItem("superscript");
		if (button && gtk_toggle_tool_button_get_active(button)) gtk_text_buffer_apply_tag_by_name(textbuffer, "superscript", &start, &end);
	}
	m_InsertOffset = -2;
	OnChanged(textbuffer);
	if (gtk_text_buffer_get_modified(m_buf) && !m_bLoading)
	{
		xmlNodePtr node = SaveSelected();
		if (node && TextTool) TextTool->PushNode(node);
		gtk_text_buffer_set_modified(m_buf, false);
	}
	return true;
}

void gcpText::Update(GtkWidget* w)
{
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	GnomeCanvasGroup* group = pData->Items[this];
	g_object_set(G_OBJECT(g_object_get_data(G_OBJECT(group), "text")),
						"x", m_x * pData->ZoomFactor,
						"y", m_y * pData->ZoomFactor - m_ascent,
						"width", m_length,
						"height", m_height,
						NULL);
	g_object_set(G_OBJECT(g_object_get_data(G_OBJECT(group), "rect")),
						"x1", m_x * pData->ZoomFactor - pData->Padding,
						"y1", m_y * pData->ZoomFactor - pData->Padding - m_ascent,
						"x2", m_x * pData->ZoomFactor + m_length + pData->Padding,
						"y2", m_y * pData->ZoomFactor + m_height + pData->Padding - m_ascent,
						NULL);
}

void gcpText::SetSelected(GtkWidget* w, int state)
{
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	GnomeCanvasGroup* group = pData->Items[this];
	gchar* color;
	switch (state) {	
		case SelStateUnselected:
			color = "white";
			break;
		case SelStateSelected:
			color = SelectColor;
			break;
		case SelStateUpdating:
			color = AddColor;
			break;
		case SelStateErasing:
			color = DeleteColor;
			break;
		default:
			color = "white";
			break;
	}
	g_object_set(G_OBJECT(g_object_get_data(G_OBJECT(group), "rect")), "fill_color", color, NULL);
}

bool gcpText::OnEvent(GdkEvent *event)
{
	if ((event->type == GDK_BUTTON_PRESS) && (event->button.button == 2))
	{
		return true;
	}
	else return false;
}

void gcpText::Transform2D(Matrix2D& m, double x, double y)
{
	m_x += m_length / 2 - x;
	m_y += m_height / 2 - m_ascent +  - y;
	m.Transform (m_x, m_y);
	m_x -= m_length / 2 - x;
	m_y -= m_height / 2 - m_ascent - y;
}
	
double gcpText::GetYAlign ()
{
	return m_y - ((gcpDocument*) GetDocument ())->GetView ()->GetBaseLineOffset ();
}
