
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <pwd.h>
#include <sys/time.h>
#include <dirent.h>
#include <errno.h>
#include <ctype.h>

#include "gbuffy.h"
#include "rfc2047.h"

int parse_mime_header (BOX_INFO *mbox, GList *headers, FILE *fp)
{
  static char *buffer = NULL;
  static size_t buflen = STRING;
  char from[STRING] = "";
  char subject[STRING] = "";
  char xface[STRING] = "";
  int status = FALSE;
  int is_new = FALSE;

  if (buffer == NULL)
    buffer = (char *) malloc (buflen);

  while (*(buffer = read_rfc822_line (fp, buffer, &buflen)) != 0) 
  {
    if (!strncmp (buffer, "Status:", 7)) 
    {
      status = TRUE;
      if (!strchr (buffer, 'R') && !strchr (buffer, 'O')) 
      {
	mbox->new_messages++;
	is_new = TRUE;
      }
    }
    else if (headers)
    {
      if (!strncasecmp (buffer, "From:", 5))
      {
	rfc2047_decode (from, buffer, sizeof (from));
      }
      else if (!strncasecmp (buffer, "Subject:", 8))
      {
	rfc2047_decode (subject, buffer, sizeof (subject));
      }
      else if (!strncasecmp (buffer, "X-Face:", 7))
      {
	char *s;
	s = buffer + 7;
	while (*s && ISSPACE (*s)) s++;
	strfcpy (xface, s, sizeof (xface));
	if (strlen (s) > sizeof (xface))
	  g_print ("-E- xface header is larger than buffer\n");
      }
    }
  }
  if (status == FALSE) 
  {
    mbox->new_messages++;
    is_new = TRUE;
  }
  else 
    status = FALSE;

  if (headers && is_new)
  {
    MESSAGE_INFO *m;

    m = (MESSAGE_INFO *) calloc (1, sizeof (MESSAGE_INFO));

    m->from = safe_strdup (from);
    m->subject = safe_strdup (subject);
    m->face = safe_strdup (xface);
    headers = g_list_append (headers, m);
  }

  return 0;
}

/*
 * Return 0 on no change/failure, 1 on change
 */
int mbox_folder_count (BOX_INFO *mbox, int force, GList *headers)
{
  FILE *fp;
  char buffer[STRING];
  char garbage[STRING];
  char path[_POSIX_PATH_MAX];
  struct stat s;
  struct timeval t[2];
  
  if (mbox->path == NULL)
    return 0;

  strfcpy (path, mbox->path, sizeof (path));
  gbuffy_expand_path (path, sizeof (path));
/*  g_print ("-I- Trying mbox %s\n", path); */

  if (stat (path, &s) != 0)
  {
    mbox->file_size = 0;
    mbox->file_mtime = 0;
    if (mbox->num_messages)
    {
      mbox->num_messages = 0;
      mbox->new_messages = 0;
      return 1;
    }
    return 0;
  }
  
  if (!force && (s.st_size == mbox->file_size) && (s.st_mtime == mbox->file_mtime))
    return 0;

  if ((s.st_size == 0) || (!S_ISREG(s.st_mode))) 
  {
    mbox->file_size = 0;
    mbox->file_mtime = 0;
    mbox->num_messages = 0;
    mbox->new_messages = 0;
    return 1;
  }


  if ((fp = fopen (path, "r")) == NULL)
    return 0;

  /* Check if a folder by checking for From as first thing in file */
  fgets(buffer, sizeof (buffer), fp);
  if (!is_from(buffer, garbage, sizeof (garbage))) 
    return 0;

  mbox->num_messages = 1;
  mbox->new_messages = 0;
  mbox->file_mtime = s.st_mtime;
  mbox->file_size = s.st_size;

  parse_mime_header (mbox, headers, fp);
  while (fgets (buffer, sizeof (buffer), fp) != 0)
  {
    if (is_from (buffer, garbage, sizeof (garbage)))
    {
      mbox->num_messages++;
      parse_mime_header (mbox, headers, fp);
    } 
  }
  fclose (fp);

  /* Restore the access time of the mailbox for other checking programs */
  t[0].tv_sec = s.st_atime;
  t[0].tv_usec = 0;
  t[1].tv_sec = s.st_mtime;
  t[1].tv_usec = 0;

  utimes (path, t);
  return 1;
}


int dir_folder_count (BOX_INFO *mbox, int force, GList *headers)
{
  DIR *dp = 0;
  FILE *fp = 0;
  char path[_POSIX_PATH_MAX];
  char file[_POSIX_PATH_MAX];
  int mailfile = TRUE;
  struct dirent *de;
  struct stat s;
  struct timeval t[2];

  if (mbox->path == NULL)
    return 0;

  if (mbox->type == GB_MAILDIR) 
  {
    snprintf (path, sizeof (path), "%s/new", mbox->path);
  } 
  else 
  {
    strfcpy (path, mbox->path, sizeof (path));
  }
  gbuffy_expand_path (path, sizeof (path));

  if (stat (path, &s) != 0)
  {
    mbox->file_size = 0;
    mbox->file_mtime = 0;
    if (mbox->num_messages)
    {
      mbox->num_messages = 0;
      mbox->new_messages = 0;
      return 1;
    }
    return 0;
  }
  
  if (!force && (s.st_size == mbox->file_size) && (s.st_mtime == mbox->file_mtime))
    return 0;

  mbox->num_messages = 0;
  mbox->new_messages = 0;

  dp = opendir (path);
  if (dp == NULL)
    return 0;

  while ((de = readdir (dp)) != NULL)
  {
    mailfile = TRUE;
    if (mbox->type == GB_MH) 
    {
      char *p;

      p = de->d_name;
      while (*p && mailfile)
      {
	if (!isdigit (*p)) 
	  mailfile = FALSE;
	p++;
      }
    } 
    else if (mbox->type == GB_MAILDIR) 
    {
      if (*de->d_name == '.') 
	mailfile = FALSE;
    }
    if (mailfile)
    {
      mbox->num_messages++;
      if (headers != NULL || mbox->type == GB_MH) 
      {
	/* Ok, we need to get the From: and Subject: lines */
	snprintf (file, sizeof (file), "%s/%s", path, de->d_name);
	fp = fopen (file, "r");
	if (fp != NULL) 
	{
	  parse_mime_header (mbox, headers, fp);
	  fclose(fp);
	}
      }
      else
      {
	/* For maildir, if we aren't getting headers, we just count
	 * everything in this directory as new 
	 */
	if (mbox->type == GB_MAILDIR) 
	  mbox->new_messages++;
      }
    }
  }
  closedir(dp);

  /* Restore the access time of the mailbox for other checking programs */
  t[0].tv_sec = s.st_atime;
  t[0].tv_usec = 0;
  t[1].tv_sec = s.st_mtime;
  t[1].tv_usec = 0;

  utimes (path, t);

  return 1;
}

int external_folder_count (BOX_INFO *mbox, int force, GList *headers)
{
  return 0;
}
