/* Copyright (C) 2000 Damir Zucic */

/*=============================================================================

				gen_hybonds.c

Purpose:
	Generate hydrogen bonds. Scan every macromolecular complex, check
	every atom.  This function  generates  internal bonds - it is not
	not enabled to generate bonds between two complexes. If bonds are
	available already, just change the bond drawing style. Only O and
	N atoms are taken into account here.

Input:
	(1) Pointer to MolComplexS structure,  with macromol. complexes.
	(2) The number of macromolecular complexes.
	(3) Pointer to ConfigS structure, with configuration data.
	(4) The bond drawing style index.

Output:
	(1) Hydrogen bonds generated.
	(2) Return value.

Return value:
	The number of hydrogen bonds (unsigned long).

Notes:
	(1) Indentation is exceptionally 4 spaces.

	(2) The chemical (element) symbol is right justified.

	(3) This function may be  pretty slow  if generating bonds  for a
	    very large macromolecular complex. Be patient!

========includes:============================================================*/

#include <stdio.h>

#include <math.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>

#include "defines.h"
#include "typedefs.h"

/*======function prototypes:=================================================*/

int		ChangeHybondStyle_ (MolComplexS *, int);
int		PrepareOCVector_ (VectorS *, AtomS *, MolComplexS *, int);
int		CheckCONAngle_ (VectorS *, AtomS *, AtomS *, ConfigS *);
short int	AddBond_ (AtomS *, short int, unsigned char,
			  int, size_t, double, short int);

/*======generate hydrogen bonds:=============================================*/

unsigned long GenerateHyBonds_ (MolComplexS *mol_complexSP,
				int mol_complexesN,
				ConfigS *configSP, int styleI)
{
unsigned long	hydrogen_bondsN = 0;
double		min_dist, max_dist;
double		min_distance_squared, max_distance_squared;
int		mol_complexI;
MolComplexS	*curr_mol_complexSP;
size_t		atomsN, atom1I, atom2I;
AtomS		*atom1SP, *atom2SP;
int		n11, n12, n21, n22;
VectorS		O_C_vectorS;
int		n;
double		delta_x, delta_y, delta_z;
double		distance, distance_squared;

/* Geometric parameters: */
min_dist = configSP->hydro_bond_length_min;
max_dist = configSP->hydro_bond_length_max;
min_distance_squared = configSP->hydro_min_squared;
max_distance_squared = configSP->hydro_max_squared;

/* Check every macromolecular complex: */
for (mol_complexI = 0; mol_complexI < mol_complexesN; mol_complexI++)
    {
    /* Pointer to the current macromolecular complex: */
    curr_mol_complexSP = mol_complexSP + mol_complexI;

    /* Check is the current macromolecular complex caught: */
    if (curr_mol_complexSP->catchF == 0) continue;

    /* Prepare and check the number of atoms in this complex: */
    atomsN = curr_mol_complexSP->atomsN;
    if (atomsN == 0) continue;

    /* Hydrogen bonds  may be available already;  in */
    /* that case just change the bond drawing style: */
    if (curr_mol_complexSP->hydrogen_bondsF == 1)
	{
	ChangeHybondStyle_ (curr_mol_complexSP, styleI);
	continue;
	}

    /* The outer atomic loop: */
    for (atom1I = 0; atom1I < atomsN; atom1I++)
	{
	/* Pointer to this atom: */
	atom1SP = curr_mol_complexSP->atomSP + atom1I;

	/* Check only oxygen atoms in this loop: */
	n11 = atom1SP->raw_atomS.chemical_symbolA[0];
	n12 = atom1SP->raw_atomS.chemical_symbolA[1];
	if (n12 != 'O') continue;
	if (n11 != ' ') continue;

	/* Prepare the C-O vector: */
	n = PrepareOCVector_ (&O_C_vectorS, atom1SP,
			      mol_complexSP, mol_complexI);
	if (n < 0) continue;

	/* The inner atomic loop: */
	for (atom2I = 0; atom2I < atomsN; atom2I++)
	    {
	    /* Pointer to another atom (possible bond partner): */
	    atom2SP = curr_mol_complexSP->atomSP + atom2I;

	    /* Check only nitrogen atoms in this loop: */
	    n21 = atom2SP->raw_atomS.chemical_symbolA[0];
	    n22 = atom2SP->raw_atomS.chemical_symbolA[1];
	    if (n22 != 'N') continue;
	    if (n21 != ' ') continue;

	    /* Calculate and check coordinate differences: */
	    delta_x = atom1SP->raw_atomS.x[0] - atom2SP->raw_atomS.x[0];
	    if (delta_x > max_dist) continue;
	    delta_y = atom1SP->raw_atomS.y - atom2SP->raw_atomS.y;
	    if (delta_y > max_dist) continue;
	    delta_z = atom1SP->raw_atomS.z[0] - atom2SP->raw_atomS.z[0];
	    if (delta_z > max_dist) continue;

	    /* Calculate and check the squared distance between two atoms: */
	    distance_squared = delta_x * delta_x +
			       delta_y * delta_y +
			       delta_z * delta_z;
	    if (distance_squared > max_distance_squared) continue;
	    if (distance_squared < min_distance_squared) continue;

	    /* Calculate and check C-O...N angle: */
	    n = CheckCONAngle_ (&O_C_vectorS, atom1SP, atom2SP, configSP);
	    if (n < 0) continue;

	    /* If this point is reached, add a new */
	    /* bond to both atoms  (O = 1, N = 2): */
	    distance = sqrt (distance_squared);
	    AddBond_ (atom1SP,
		      0, 0,
		      mol_complexI, atom2I,
		      distance, styleI);
	    AddBond_ (atom2SP,
		      0, 0,
		      mol_complexI, atom1I,
		      distance, styleI);

	    /* Update the counter: */
	    hydrogen_bondsN++;

	    }		/* End of atom2I loop */


	}		/* End of atom1I loop    */

    /* Set flag which says that hydrogen bonds are available: */
    curr_mol_complexSP->hydrogen_bondsF = 1;

    }			/* End of mol_complexI loop  */

/* Return the number of hydrogen bonds: */
return hydrogen_bondsN;
}

/*===========================================================================*/


