/***************************************

    This is part of frox: A simple transparent FTP proxy
    Copyright (C) 2000 James Hollingshead

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

ntp.c -- non transparent proxying stuff

Overview: 

ntp_changedest gets called just before the proxy connects to the
remote server. If the connection is destined for a remote machine then
we assume that is the destination and that no ntp support is required. 
Otherwise we send a welcome to the client and read a reply of the form
"USER username[@host[:port]]". If necessary we then change the
destination (in the global "info" structure), save the hostname, and
return. We also call set config.transdata=FALSE to stop NAT on data
connections for this session.

We also need get called(ntp_senduser) when the remote server sends us
a 220 welcome message. If we have already welcomed the client we send
the login name we got then to the server and return TRUE. Otherwise we
return FALSE and the main proxy code will forward the welcome on to
the client.

***************************************/

#include <sys/ioctl.h>
#include <netdb.h>

#include "common.h"
#include "control.h"
#include "ntp.h"
#include "os.h"

extern int extract_clientcmd(sstr * buf, sstr * cmd, sstr * arg);
void parse_client_cmd(sstr * cmd, sstr * arg);

void parseuser(sstr * arg);

static sstr *user = NULL;
static int working = FALSE;

/* ------------------------------------------------------------- **
**  Called before the proxy connects to the server.
**  
**  Send client "220 Send Login", read and parse reply.
**  ------------------------------------------------------------- */
void ntp_changedest(void)
{
	sstr *cmd, *arg;
	struct sockaddr_in tmp;

	if (!config.ntp)
		return;

	if(config.ntpdest.sin_addr.s_addr){
		/*Don't do ntp proxying unless the connection is to NTPDest. */
		get_orig_dest(info->client_control.fd, &tmp);
		if (tmp.sin_addr.s_addr != config.ntpdest.sin_addr.s_addr
		    || (config.ntpdest.sin_port &&
			tmp.sin_port !=config.ntpdest.sin_port) )
			return;
	}

	working = TRUE;
	if (!user)
		user = sstr_init(MAX_LINE_LEN);

	send_cmessage(220,
		      "Frox transparent ftp proxy. Login with username[@host[:port]]");
	do {
		get_command(&cmd, &arg);
		if (sstr_casecmp2(cmd, "USER"))
			send_cmessage(530, "Please login with USER first");
	} while (sstr_casecmp2(cmd, "USER"));

	parseuser(arg);
}

/* ------------------------------------------------------------- **
**  If we have a username send it to the server.
**  ------------------------------------------------------------- */
void ntp_senduser(void)
{
	int i;
	sstr *msg;

	if (!working)
		return;

	get_message(&i, &msg);
	if (i != 220) {
		send_cmessage(421, "Server Unable to accept connection");
		close(info->client_control.fd);
		quit(NO_MESSAGE);
	}

	working = FALSE;

	msg = sstr_dup2("USER");
	parse_client_cmd(msg, user);	/*Make sure we go back through
					   * cache code if necessary */
	sstr_free(msg);
	sstr_free(user);
	user = NULL;
}

/* ------------------------------------------------------------- **
**  Parse the user command, resolve the hostname if present, and do
**  security checks. If all ok alter info->server_control.address
**  ------------------------------------------------------------- */
void parseuser(sstr * arg)
{
	struct hostent *hostinfo;
	sstr *host = NULL;
	int sep, port = 0;
	sstr *tok;

	sep = sstr_token(arg, user, "@", 0);
	if (sep == -1) {
		sstr_cpy(user, arg);
		return;
	}

	tok = sstr_init(MAX_LINE_LEN);
	sep = sstr_token(arg, tok, ":", 0);
	host = (sep == -1 ? arg : tok);

	if (sep == ':')
		port = sstr_atoi(arg);
	else
		port = 21;

	debug2("  Host=%s", sstr_buf(host));
	debug2(", port=%d\n", port);

	/*Turn off data connection NAT for this connection! */
	config.transdata = FALSE;

	hostinfo = gethostbyname(sstr_buf(host));
	if (!hostinfo) {
		send_cmessage(501, "Can't find that host");
		quit(SERVER_CLOSE);	/*Not really true! */
	}
	if (hostinfo->h_addrtype != AF_INET) {
		send_cmessage(501, "Invalid host");
		quit(SERVER_CLOSE);	/*Not really true! */
	}


	info->server_control.address.sin_addr =
	    *((struct in_addr *) hostinfo->h_addr_list[0]);
	info->server_control.address.sin_port = htons(port);
	info->server_name = sstr_dup(host);

	sstr_free(tok);
}
