/***************************************

    This is part of frox: A simple transparent FTP proxy
    Copyright (C) 2000 James Hollingshead

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    misc.c - Miscellaneous stuff!
  ***************************************/


#include <stdarg.h>
#include <syslog.h>
#include <netdb.h>
#include <stdlib.h>
#include <sys/param.h>
#include <sys/uio.h>
#include <fcntl.h>
#include <sys/signal.h>

#include "common.h"
#include "control.h"
#include "transdata.h"
#include "misc.h"

/* ------------------------------------------------------------- **
** Listens on socket. If port==0 then picks a port from the range
** specified in the config file for the use given in use.
** ------------------------------------------------------------- */
int listen_on_socket(struct sockaddr_in *listen_address, socketuse use)
{
	int sockfd;

	sockfd = socket(AF_INET, SOCK_STREAM, 0);
	if (use == CTRL) {
		int i = 1;
		setsockopt(sockfd, SOL_SOCKET, SO_OOBINLINE, &i, sizeof(i));
		i = i;
		setsockopt(sockfd, SOL_SOCKET, SO_REUSEADDR, &i, sizeof(i));
	}

	if (listen_address->sin_port) {
		if (bind(sockfd, (struct sockaddr *) listen_address,
			 sizeof(*listen_address))) {
			debug_perr("bind");
			close(sockfd);
			return (-1);
		}
	} else {
		int lo, hi;

		lo = config_loport(use);
		hi = config_hiport(use);

		if (bind_me(sockfd, listen_address, lo, hi, hi - lo)) {
			debug_perr("bind_me");
			close(sockfd);
			return (-1);
		}
	}
	if (listen(sockfd, 5)) {
		close(sockfd);
		debug_perr("listen");
		return (-1);
	}

	return (sockfd);
}

/* ------------------------------------------------------------- **
** Connects to address. Local port is picked from the range given in
** the config file for the use specified in use. 
** ------------------------------------------------------------- */
int connect_to_socket(struct sockaddr_in connect_address, socketuse use)
{
	int sockfd;
	struct sockaddr_in address;
	int lo, hi;

	if ((sockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		debug_perr("socket");
		quit(SOCKET);
	}

	lo = config_loport(use);
	hi = config_hiport(use);

	memset(&address, 0, sizeof(address));
	address.sin_family = AF_INET;

	if (bind_me(sockfd, &address, lo, hi, hi - lo)) {
		debug_perr("bind_me");
		close(sockfd);
		return (-1);
	}

	connect_address.sin_family = AF_INET;
	if (connect(sockfd, (struct sockaddr *) &connect_address,
		    sizeof(address))) {
		debug_perr("connect");
		close(sockfd);
		return (-1);
	}

	return (sockfd);
}


/* ------------------------------------------------------------- **
** Try "attempts" times to bind socket "fd" to address "address", with
** a port picked at random from between "lo" and "hi".
** ------------------------------------------------------------- */
int bind_me(int fd, struct sockaddr_in *address, int lo, int hi, int attempts)
{
	int i, j;

	srand(time(NULL));

	for (i = 0; i <= attempts; i++) {
		j = (rand() % (hi - lo)) + lo;
		address->sin_port = htons(j);

		if (bind(fd, (struct sockaddr *) address,
			 sizeof(*address)) == 0)
			break;
		if (errno != EADDRINUSE) {
			return (-1);
		}
	}

	return (i > attempts ? -1 : 0);
}

/* ------------------------------------------------------------- **
** Convert a comma separted values address:port to a sockaddr_in 
** in network order
** ------------------------------------------------------------- */
struct sockaddr_in com2n(int a1, int a2, int a3, int a4, int p1, int p2)
{
	struct sockaddr_in ret;

	ret.sin_addr.s_addr = htonl(a4 + (a3 << 8) + (a2 << 16) + (a1 << 24));
	ret.sin_port = htons((p1 << 8) + p2);
	ret.sin_family = AF_INET;

	return (ret);
}


/* ------------------------------------------------------------- **
** Convert network order address to comma separated values
** ------------------------------------------------------------- */
void n2com(struct sockaddr_in address, int *a1, int *a2, int *a3, int *a4,
	   int *p1, int *p2)
{
	address.sin_addr.s_addr = ntohl(address.sin_addr.s_addr);
	address.sin_port = ntohs(address.sin_port);
	*a1 = (address.sin_addr.s_addr & 0xFF000000) >> 24;
	*a2 = (address.sin_addr.s_addr & 0x00FF0000) >> 16;
	*a3 = (address.sin_addr.s_addr & 0x0000FF00) >> 8;
	*a4 = (address.sin_addr.s_addr & 0x000000FF);
	*p1 = (address.sin_port & 0xFF00) >> 8;
	*p2 = (address.sin_port & 0x00FF);
}

/* ------------------------------------------------------------- **
** Extract a comma delimited address/port from buf. buf is NULL
** terminated, but otherwise unchecked user input --- make sure we
** don't buffer overflow here!
** ------------------------------------------------------------- */
struct sockaddr_in extract_address(const sstr * buf)
{
	int i, a[6];
	struct sockaddr_in tmp;
	sstr *p=sstr_dup(buf);

	tmp.sin_addr.s_addr = tmp.sin_port = 0;

	sstr_split(p, NULL, 0, sstr_pbrk2(p, "01234456789"));
	for (i = 0; i < 6; i++) {
		a[i] = sstr_atoi(p);
		if (i != 5 && sstr_token(p, NULL, ",", 0) == -1){
			sstr_free(p);
			return (tmp);
		}
	}

	tmp = com2n(a[0], a[1], a[2], a[3], a[4], a[5]);
	sstr_free(p);
	return (tmp);
}

int do_chroot(void)
{
	if (config.dontchroot) return(0);
	if (chroot(config.chroot) != 0 || chdir("/") != 0) {
		write_log(ERROR, "Failed to chroot.");
		return(-1);
	}

	write_log(IMPORT, "Chrooted to %s", config.chroot);

	strip_filenames();

	return (0);
}

int droppriv(void)
{
	if(config.uid==0) {
#ifdef ENFORCE_DROPPRIV
		write_log(ERROR, "Running frox as root is not allowed. " 
			  "Set \"User\" to another value in the config file");
		write_log(ERROR, "Alternatively you may recompile giving "
			  "--enable-run-as-root to ./configure");
		exit(-1);
#else
		write_log(IMPORT, "WARNING frox set to run as root");
#endif
	}

	if (config.gid != 0) {
		setgid(config.gid);
		setgid(config.gid);
	}
	if (config.uid != 0) {
		setuid(config.uid);
		setuid(config.uid);
		write_log(IMPORT, "Dropped privileges");
	}
	return (0);
}

void write_log(int priority, const char *msg, ...)
{
	char buf[MAX_LINE_LEN];
	va_list argptr;
	time_t tstamp;

	if (priority > config.loglevel)
		return;
	va_start(argptr, msg);
	vsnprintf(buf, MAX_LINE_LEN - 10, msg, argptr);
	va_end(argptr);

	if (priority <= ERROR) {
		syslog(LOG_ERR | LOG_DAEMON, "%s\n", buf);
	} else {
		syslog(LOG_NOTICE | LOG_DAEMON, "%s\n", buf);
	}

	if(config.logfile) {
		sstr *s;
		s = sstr_init(MAX_LINE_LEN);
		time(&tstamp);
		sstr_cpy2(s, ctime(&tstamp));
		sstr_setchar(s, sstr_chr(s, '\n'), ' ');
		sstr_apprintf(s, "frox[%d] %s\n", getpid(), buf);
		sstr_write(2, s, 0);
		sstr_free(s);
	}
}

/* ------------------------------------------------------------- **
** Return hostname of address, or failing that the IP as a string
** ------------------------------------------------------------- */
sstr *addr2name(struct in_addr address)
{
	struct hostent *hostinfo;
	static sstr *buf = NULL;

	if (!buf) buf = sstr_init(MAX_LINE_LEN);
	sstr_cpy2(buf, inet_ntoa(address));

	if ((hostinfo = gethostbyaddr((char *) &address,
				      sizeof(address), AF_INET)) != NULL)
		sstr_apprintf(buf, "(%s)", hostinfo->h_name);

	return (buf);
}

void set_server_name(void)
{
	struct hostent *hostinfo;
	struct in_addr *address = &info->server_control.address.sin_addr;

	if ((hostinfo = gethostbyaddr((char *) address,
				      sizeof(*address), AF_INET)) != NULL) {
		if(gethostbyname( (char *) hostinfo->h_name) != NULL) {
			info->server_name = sstr_dup2(hostinfo->h_name);
			return;
		}
	}

	info->server_name = sstr_dup2(inet_ntoa(*address));
}

/* ------------------------------------------------------------- **
** Deal with session quit conditions.
** ------------------------------------------------------------- */
void quit(exitreason reason)
{
	il_free();

	switch (reason) {
	case TIMEOUT:
		write_log(INFO, "Select timed out.");
		closeandexit(0);
	case SERVER_CLOSE:
		write_log(INFO, "Server closed the control connection.");
		closeandexit(0);
	case CLIENT_CLOSE:
		write_log(INFO, "Client closed the control connection.");
		closeandexit(0);

	case NO_SERVER_DATA:
		write_log(ERROR, "  Unable to connect to server data port");
		closeandexit(-1);
	case NO_CLIENT_DATA:
		write_log(ERROR, "  Unable to connect to client data port");
		closeandexit(-1);
	case MALLOC:
		write_log(ERROR, "  malloc failed");
		closeandexit(-1);
	case SOCKET:
		write_log(ERROR, "  socket command failed");
		closeandexit(-1);
	case CLIENT_RUBBISH:
		send_cmessage(421, "You are sending me rubbish. Goodbye.");
		write_log(ATTACK,
			  "Client is sending us a badly formed control stream.");
		closeandexit(-1);
	case SERVER_RUBBISH:
		send_cmessage(421,
			      "FTP server is sending you rubbish! Closing connection.");
		write_log(ATTACK,
			  "Server is sending us a badly formed control stream.");
		closeandexit(-1);
	case NO_MESSAGE:
		closeandexit(-1);
	}
}

void kill_procs(void)
{
	if(cmgrpid) kill(cmgrpid, SIGTERM);
	if(tdatapid) kill(tdatapid, SIGTERM);
}

void closeandexit(int status)
{
	close(info->client_control.fd);
	close(info->client_data.fd);
	close(info->server_control.fd);
	close(info->server_data.fd);
	close(info->listen);
	free(info);
	write_log(INFO, "Closing session");
	kill_procs();
	exit(status);
}

void sstrerr(void)
{
	write_log(ERROR, "sstr internal failure. Exiting");
	quit(NO_MESSAGE);
}

#if defined(USE_LCACHE) || defined(TRANS_DATA)
#if 1
/* Pinched from vsftpd. */
int send_fd(int sock_fd, int send_fd, char sendchar)
{
	int retval;
	struct msghdr msg;
	struct cmsghdr* p_cmsg;
	struct iovec vec;
	char cmsgbuf[CMSG_SPACE(sizeof(send_fd))];
	int* p_fds;
	msg.msg_control = cmsgbuf;
	msg.msg_controllen = sizeof(cmsgbuf);
	p_cmsg = CMSG_FIRSTHDR(&msg);
	p_cmsg->cmsg_level = SOL_SOCKET;
	p_cmsg->cmsg_type = SCM_RIGHTS;
	p_cmsg->cmsg_len = CMSG_LEN(sizeof(send_fd));
	p_fds = (int*)CMSG_DATA(p_cmsg);
	*p_fds = send_fd;
	msg.msg_controllen = p_cmsg->cmsg_len;

	msg.msg_name = NULL;
	msg.msg_namelen = 0;
	msg.msg_iov = &vec;
	msg.msg_iovlen = 1;
	msg.msg_flags = 0;
	/* "To pass file descriptors or credentials you need to send/read at
	 * least on byte" (man 7 unix)
	 */
	vec.iov_base = &sendchar;
	vec.iov_len = sizeof(sendchar);
	retval = sendmsg(sock_fd, &msg, 0);
	if (retval != 1) {
		debug_perr("sendmsg");
		return -1;
	}
	return 0;
}

char recv_fd(int sock_fd, int *recv_fd)
{
	int retval;
	struct msghdr msg;
	char recvchar=0;
	struct iovec vec;
	char cmsgbuf[CMSG_SPACE(sizeof(*recv_fd))];
	struct cmsghdr* p_cmsg;
	int* p_fd;
	vec.iov_base = &recvchar;
	vec.iov_len = sizeof(recvchar);
	msg.msg_name = NULL;
	msg.msg_namelen = 0;
	msg.msg_iov = &vec;
	msg.msg_iovlen = 1;
	msg.msg_control = cmsgbuf;
	msg.msg_controllen = sizeof(cmsgbuf);
	msg.msg_flags = 0;
	/* In case something goes wrong, set the fd to -1 before the syscall */
	p_fd = (int*)CMSG_DATA(CMSG_FIRSTHDR(&msg));
	*p_fd = -1;  
	retval = recvmsg(sock_fd, &msg, 0);
	if (retval != 1) {
		debug_perr("recvmsg");
		return(0);
	}

	p_cmsg = CMSG_FIRSTHDR(&msg);
	if (p_cmsg == NULL) {
		printf("no passed fd\n");
		return(recvchar);
	}
	/* We used to verify the returned cmsg_level, cmsg_type and
	 * cmsg_len here, but Linux 2.0 totally uselessly fails to
	 * fill these in.  */
	p_fd = (int*)CMSG_DATA(p_cmsg);
	*recv_fd = *p_fd;
	if (*recv_fd == -1) {
		debug("no passed fd\n");
	}
	return recvchar;
}
#else /*Old style send/recv _fd*/

int send_fd(int sock_fd, int send_fd, char sendchar)
{
	int retval;
	struct msghdr msg;
	struct iovec vec;
	vec.iov_base = &sendchar;
	vec.iov_len = 1;
	msg.msg_name = NULL;
	msg.msg_namelen = 0;
	msg.msg_iov = &vec;
	msg.msg_iovlen = 1;
	msg.msg_accrights = (caddr_t) &send_fd;
	msg.msg_accrightslen = sizeof(send_fd);
	retval = sendmsg(sock_fd, &msg, 0);
	if (retval != 1) {
		debug_perr("sendmsg");
		return -1;
	}
	return 0;
}

char recv_fd(int sock_fd, int *recv_fd)
{
	int retval;
	struct msghdr msg;
	struct iovec vec;
	char recv_char=0;
	*recv_fd = -1;
	vec.iov_base = &recv_char;
	vec.iov_len = 1;
	msg.msg_name = NULL;
	msg.msg_namelen = 0;
	msg.msg_iov = &vec;
	msg.msg_iovlen = 1;
	msg.msg_accrights = (caddr_t) recv_fd;
	msg.msg_accrightslen = sizeof(*recv_fd);
	retval = recvmsg(sock_fd, &msg, 0);
	if (retval != 1) {
		debug_perr("recvmsg");
		return 0;
	}
	if (*recv_fd == -1) {
		debug_perr("no passed fd");
	}
	return recv_char;
}
#endif  /*Old style send/recv _fd*/
#endif  /*LCACHE || TRANSDATA */

#ifdef USE_LCACHE

void set_write_lock(int fd)
{
	struct flock lck;
	int i;

	lck.l_type = F_WRLCK;
	lck.l_whence = SEEK_SET;
	lck.l_start = 0;
	lck.l_len = 0;
	i = fcntl(fd, F_SETLK, &lck);

	if(i==-1) {
		debug_perr("Setting file lock\n");
		quit(-1);
	}
}

int set_read_lock(int fd)
{
	struct flock lck;

	lck.l_type = F_RDLCK;
	lck.l_whence = SEEK_SET;
	lck.l_start = 0;
	lck.l_len = 0;
	return fcntl(fd, F_SETLK, &lck);
}

#endif /*USE_LCACHE*/
