// Copyright 2004 "Gilles Degottex"

// This file is part of "fmit"

// "fmit" is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// "fmit" is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "CustomInstrumentTunerForm.h"

#include <cassert>
#include <iostream>
using namespace std;
#include <qstring.h>
#include <qaction.h>
#include <qlabel.h>
#include <qgl.h>
#include <qlayout.h>
#include <qlcdnumber.h>
#include <qdial.h>
#include <qspinbox.h>
#include <qcombobox.h>
#include <qsplitter.h>
#include <qprogressbar.h>
#include <qmessagebox.h>
#include <qmenubar.h>
#include <qpushbutton.h>
#include <qcheckbox.h>
#include <qgroupbox.h>
#include <qlineedit.h>
#include <qstatusbar.h>
#include <qdatetime.h>
#include <qradiobutton.h>
#include <Music/Convolution.h>
using namespace Music;
#include "View.h"

CustomInstrumentTunerForm::CustomInstrumentTunerForm()
: m_capture_thread("fmit")
, m_selected_algo(NULL)
, m_algo_multidiff(NULL)
, m_algo_diff(NULL)
// , m_algo_bubble(NULL)
// , m_algo_combedfft(NULL)
, m_first_time(true)
, m_test_freq(0.0f)
{
	m_settings.setPath("gillesdegottex.ch", "FMIT");
	m_settings.beginGroup("/fmit1/");
	m_settings.beginGroup("default/");
	View::s_settings = &m_settings;

#ifdef CAPTURE_ALSA
#else
	m_config_form.ui_rdbtnALSA->hide();
	m_config_form.ui_grpALSA->hide();
#endif
#ifdef CAPTURE_JACK
#else
	m_config_form.ui_rdbtnJACK->hide();
	m_config_form.ui_grpJACK->hide();
#endif

	ui_lblA3Offset->hide();
	ui_spinA3Offset->hide();

	connect(&m_capture_thread, SIGNAL(samplingRateChanged(int)), this, SLOT(samplingRateChanged(int)));
	connect(&m_capture_thread, SIGNAL(errorRaised(const QString&)), this, SLOT(errorRaised(const QString&)));
	connect(&m_capture_thread, SIGNAL(transportChanged(const QString&)), this, SLOT(transportChanged(const QString&)));

	m_dialTune = new DialView(centralWidget());
	ui_dialTuneLayout->addWidget(m_dialTune);

	m_glGraph = new GLGraph("Graph", centralWidget());
	connect(m_glGraph->setting_show, SIGNAL(toggled(bool)), this, SLOT(update_views()));
	connect(m_glGraph->setting_spinMaxHeight, SIGNAL(valueChanged(int)), this, SLOT(update_views()));
	m_glGraph->setting_show->addTo(ui_tbViews);
	ui_graphLayout->addWidget(m_glGraph);

	m_glErrorHistory = new GLErrorHistory(centralWidget());
	connect(m_glErrorHistory->setting_show, SIGNAL(toggled(bool)), this, SLOT(update_views()));
	m_glErrorHistory->setting_show->addTo(ui_tbViews);
	ui_errorLayout->addWidget(m_glErrorHistory);

	// link scales
	connect(m_dialTune->setting_spinScale, SIGNAL(valueChanged(int)), m_glErrorHistory->setting_spinScale, SLOT(setValue(int)));
	connect(m_glErrorHistory->setting_spinScale, SIGNAL(valueChanged(int)), m_dialTune->setting_spinScale, SLOT(setValue(int)));
	connect(m_dialTune->setting_useCents, SIGNAL(toggled(bool)), m_glErrorHistory->setting_useCents, SLOT(setOn(bool)));
	connect(m_glErrorHistory->setting_useCents, SIGNAL(toggled(bool)), m_dialTune->setting_useCents, SLOT(setOn(bool)));

	m_glVolumeHistory = new GLVolumeHistory(centralWidget());
	connect(m_glVolumeHistory->setting_show, SIGNAL(toggled(bool)), this, SLOT(update_views()));
	m_glVolumeHistory->setting_show->addTo(ui_tbViews);
	ui_volumeLayout->addWidget(m_glVolumeHistory);

	// link keep settings
	connect(ui_btnKeepErrorHistory, SIGNAL(toggled(bool)), m_glErrorHistory->setting_keep, SLOT(setOn(bool)));
	connect(m_glErrorHistory->setting_keep, SIGNAL(toggled(bool)), m_glVolumeHistory->setting_keep, SLOT(setOn(bool)));
	connect(m_glErrorHistory->setting_keep, SIGNAL(toggled(bool)), ui_btnKeepErrorHistory, SLOT(setOn(bool)));

	m_glSample = new GLSample(centralWidget());
	connect(m_glSample->setting_show, SIGNAL(toggled(bool)), this, SLOT(update_views()));
	m_glSample->setting_show->addTo(ui_tbViews);
	ui_sampleLayout->addWidget(m_glSample);

	m_glFreqStruct = new GLFreqStruct(centralWidget());
	connect(m_glFreqStruct->setting_show, SIGNAL(toggled(bool)), this, SLOT(update_views()));
	m_glFreqStruct->setting_show->addTo(ui_tbViews);
	ui_formantsLayout->addWidget(m_glFreqStruct);

	m_microtonalView = new MicrotonalView(centralWidget());
	connect(m_microtonalView->setting_show, SIGNAL(toggled(bool)), this, SLOT(update_views()));
	connect(m_microtonalView, SIGNAL(tuningFreqChanged(float)), this, SLOT(tuningFreqChanged(float)));
	m_microtonalView->setting_show->addTo(ui_tbViews);
	ui_microtonalLayout->addWidget(m_microtonalView);

	connect(m_config_form.buttonOk, SIGNAL(clicked()), this, SLOT(configure_ok()));
	connect(m_config_form.ui_btnRestoreFactorySettings, SIGNAL(clicked()), this, SLOT(restoreFactorySettings()));
	connect(m_config_form.ui_spinMinHT, SIGNAL(valueChanged(int)), this, SLOT(noteRangeChanged()));
	connect(m_config_form.ui_spinMaxHT, SIGNAL(valueChanged(int)), this, SLOT(noteRangeChanged()));
	connect(m_config_form.ui_cbTonality, SIGNAL(highlighted(int)), this, SLOT(noteRangeChanged()));
	connect(m_config_form.ui_cbNotesName, SIGNAL(highlighted(int)), this, SLOT(noteRangeChanged()));
	connect(m_config_form.ui_btnAutoDetect, SIGNAL(clicked()), this, SLOT(autoDetectTransport()));
	connect(m_config_form.ui_rdbtnALSA, SIGNAL(clicked()), this, SLOT(alsaTransportSelected()));
	connect(m_config_form.ui_rdbtnJACK, SIGNAL(clicked()), this, SLOT(jackTransportSelected()));

	loadSettings();

	m_timer_refresh = new QTimer(this, "m_timer_refresh");
	connect((QObject*)m_timer_refresh, SIGNAL(timeout()), this, SLOT(refresh()));
	m_timer_refresh->start(m_config_form.ui_spinRefreshTime->value());

	if(m_config_form.ui_chkAutoDetect->isChecked())
		m_capture_thread.autoDetectTransport();

	configure_ok();

	if(m_algo_multidiff==NULL || m_algo_diff==NULL)
		init();

	m_time.start();
	m_last_refresh = 0.0;
	m_last_views_update = 0;

	if(m_config_form.ui_chkFullScreen->isChecked())
		toggleFullScreen();
}

void CustomInstrumentTunerForm::transportChanged(const QString& name)
{
	if(m_capture_thread.getCurrentTransport()=="ALSA")
		alsaTransportSelected();
	if(m_capture_thread.getCurrentTransport()=="JACK")
		jackTransportSelected();
}
void CustomInstrumentTunerForm::autoDetectTransport()
{
	m_capture_thread.autoDetectTransport();

	if(m_capture_thread.getCurrentTransport()=="ALSA")
		alsaTransportSelected();
	if(m_capture_thread.getCurrentTransport()=="JACK")
		jackTransportSelected();

	configure_ok();
}

void CustomInstrumentTunerForm::alsaTransportSelected()
{
	m_config_form.ui_rdbtnALSA->setChecked(true);
	m_config_form.ui_rdbtnJACK->setChecked(false);

	m_config_form.ui_grpALSA->setEnabled(m_config_form.ui_rdbtnALSA->isChecked());
	m_config_form.ui_grpJACK->setEnabled(m_config_form.ui_rdbtnJACK->isChecked());
}
void CustomInstrumentTunerForm::jackTransportSelected()
{
	m_config_form.ui_rdbtnJACK->setChecked(true);
	m_config_form.ui_rdbtnALSA->setChecked(false);

	m_config_form.ui_grpALSA->setEnabled(m_config_form.ui_rdbtnALSA->isChecked());
	m_config_form.ui_grpJACK->setEnabled(m_config_form.ui_rdbtnJACK->isChecked());
}

void CustomInstrumentTunerForm::toggleFullScreen()
{
	static bool fs = true;
	if(fs)
	{
		m_config_form.ui_chkFullScreen->setChecked(true);
		showFullScreen();
	}
	else
	{
		m_config_form.ui_chkFullScreen->setChecked(false);
		showNormal();
	}
	fs = !fs;
}

void CustomInstrumentTunerForm::noteRangeChanged()
{
	//	cerr << "CustomInstrumentTunerForm::noteRangeChanged" << endl;

	m_config_form.ui_txtMinHT->setText(h2n(m_config_form.ui_spinMinHT->value())+" = "+QString::number(h2f(m_config_form.ui_spinMinHT->value()))+" hz");
	m_config_form.ui_txtMaxHT->setText(h2n(m_config_form.ui_spinMaxHT->value())+" = "+QString::number(h2f(m_config_form.ui_spinMaxHT->value()))+" hz");
}

void CustomInstrumentTunerForm::errorRaised(const QString& error)
{
	//	cerr << "CustomInstrumentTunerForm::errorRaised " << error << endl;

	statusBar()->message(QString("ERROR: ")+error);

	//	QMessageBox::warning(this, "Sound Capture", QString("Bad source: ")+src, QMessageBox::Ok, QMessageBox::NoButton);	//TODO crash ! thread of this signal is not the same as the one used for the window.
}

void CustomInstrumentTunerForm::samplingRateChanged(int sampling_rate)
{
	Music::SetSamplingRate(sampling_rate);

	if(m_config_form.ui_rdbtnJACK->isChecked())
		m_config_form.ui_lblJACKSamplingRate->setText(QString::number(sampling_rate));

	init();
}

void CustomInstrumentTunerForm::ui_spinAFreq_valueChanged(int AFreq)
{
	double A = AFreq;
	if(m_config_form.ui_chkShowA3Offset->isOn())
		A = h2f(ui_spinA3Offset->value()*1/100.0f, A);
	Music::SetAFreq(A);
//	cerr << A << endl;
}
void CustomInstrumentTunerForm::ui_spinAOffset_valueChanged(int offset)
{
	double A = ui_spinAFreq->value();
	if(m_config_form.ui_chkShowA3Offset->isOn())
		A = h2f(offset*1/100.0f, ui_spinAFreq->value());
	Music::SetAFreq(A);
//	cerr << A << endl;
}

void CustomInstrumentTunerForm::tuningFreqChanged(float freq)
{
	//	cerr << "CustomInstrumentTunerForm::tuningFreqChanged " << freq << endl;

	if(freq==0.0f)
	{
		if(m_compared_freq!=0.0f)
		{
			ui_txtNoteFreq->display(m_compared_freq);
			ui_txtNote->setText(h2n(f2h(m_compared_freq)));
		}
	}
	else
	{
		m_compared_freq = freq;
		ui_txtNoteFreq->display(int(freq*100)/100.0f);
		ui_txtNote->setText(m_microtonalView->getTuningNoteName());
	}

	m_first_time = true;
//	m_note_timer_started = false;

	//	m_dialTune->setError(-10.0f);
}

void CustomInstrumentTunerForm::init()
{
	if(m_capture_thread.getSamplingRate()<=0)	return;

	if(m_algo_multidiff!=NULL)	delete m_algo_multidiff;
	m_algo_multidiff = new MultiCumulativeDiffAlgo(m_config_form.ui_spinLatencyFactor->value(), m_config_form.ui_spinMonoTestComplexity->value());
	m_algo_multidiff->setVolumeTreshold(m_config_form.ui_spinVolumeTreshold->value()/1000.0);
	m_algo_multidiff->setComponentsTreshold(m_config_form.ui_spinComponentsTreshold->value()/100.0);

	if(m_algo_diff!=NULL)	delete m_algo_diff;
	m_algo_diff = new CumulativeDiffAlgo(m_config_form.ui_spinNoiseThreshold->value()/1000.0);
	m_algo_diff->setNoiseThreshold(m_config_form.ui_spinNoiseThreshold->value()/1000.0);

	if(m_algo_combedfft!=NULL)	delete m_algo_combedfft;
	m_algo_combedfft = new CombedFFT();
	m_algo_combedfft->setVolumeTreshold(m_config_form.ui_spinVolumeTreshold->value()/1000.0);

	m_glGraph->m_treshold = m_config_form.ui_spinVolumeTreshold->value()/1000.0;
	m_microtonalView->setAFreq(Music::GetAFreq());
	//	m_dialTune->setError(-10.0f);
}

void CustomInstrumentTunerForm::pause(bool on)
{
	m_capture_thread.togglePause(on);

	if(on)
		m_timer_refresh->stop();
	else
		m_timer_refresh->start(m_config_form.ui_spinRefreshTime->value());
}

void CustomInstrumentTunerForm::refresh()
{
	QColor ok_color(83,165,105);
	QColor prb_color(180,74,74);
	QColor empty_color(128,128,128);
	bool incoming_data = false;

	int limit = int( m_capture_thread.getSamplingRate() * double(m_config_form.ui_spinRefreshTime->value())/1000.0 );

	m_capture_thread.lock();
	double new_data_amplitude = 0.0;
	int nb_new_data = 0;
	while(!m_capture_thread.m_values.empty() &&
			(int(m_capture_thread.m_values.size())>m_capture_thread.getPacketSize() || nb_new_data<limit))
	{
		incoming_data = true;

		new_data_amplitude = max(new_data_amplitude, fabs(m_capture_thread.m_values.back()));

		m_queue.push_front(m_capture_thread.m_values.back());
		if(m_glGraph!=NULL)
		{
			m_glGraph->m_pending_queue.push_front(m_capture_thread.m_values.back());
			m_glGraph->m_queue.push_front(m_capture_thread.m_values.back());
		}
		m_capture_thread.m_values.pop_back();

		nb_new_data++;
	}
	m_capture_thread.unlock();

	// refresh graph data
	m_glGraph->add_maxs();
	ui_pgbVolume->setProgress(int(new_data_amplitude*100));

	// drop unused data, do not keep more data than 1 second
	int max_size = m_capture_thread.getSamplingRate()*4;
	while(!m_queue.empty() && int(m_queue.size())>max_size)
		m_queue.pop_back();

	// ------- Analysis stage -------

	if(m_time.elapsed()-m_gap_time > m_config_form.ui_spinErrorLatency->value())
	{
		m_old_analysis.clear();
		m_first_time = true;
		m_note_start = m_time.elapsed();
	}

	if(m_microtonalView->setting_show->isOn())
		m_selected_algo = m_algo_diff;
	else
		m_selected_algo = m_algo_multidiff;

// 	m_selected_algo = m_algo_combedfft;	//DEV

	// if something goes wrong in the capture system
	// TODO critère de répartition des analyses
	//		&&	current_time-m_old_diffs.back().first>0.9*m_config_form.ui_spinErrorLatency->value())
	if(m_time.elapsed()-m_last_refresh>1.5*m_config_form.ui_spinRefreshTime->value() ||
			!incoming_data ||
			m_selected_algo==NULL)
	{
		m_test_freq = 0.0;
		ui_lblRecognitionStability->setBackgroundColor(prb_color);
	}
	else
	{
		ui_lblRecognitionStability->setBackgroundColor(ok_color);

		if(new_data_amplitude<m_config_form.ui_spinVolumeTreshold->value()/1000.0)
		{
			m_test_freq = 0.0;
			ui_lblSoundStability->setBackgroundColor(empty_color);
		}
		else
		{
			// ... let's start with the selected algorithm
			m_test_freq = 0.0;

			m_selected_algo->apply(m_queue);

//			refresh_data_harmonics();	//DEV

			if(m_selected_algo->hasNoteRecognized())
				m_test_freq = double(m_capture_thread.getSamplingRate())/m_selected_algo->getFondamentalWaveLength();

// 			cerr << m_capture_thread.getSamplingRate() << "\t" << m_test_freq << ":" << f2h(m_test_freq) << endl;

//			cerr << "1) test freq=" << m_test_freq <<endl;

			// get the real frequency
			if(m_test_freq!=0.0f)
			{
				if(m_microtonalView->setting_show->isOn() && m_microtonalView->hasTuningFreqSelected())
					m_test_freq = GetAverageWaveLengthFromApprox(m_queue, size_t(m_capture_thread.getSamplingRate()/m_test_freq), m_config_form.ui_spinFrequencyAverageSize->value());
				else
					m_test_freq = GetAverageWaveLengthFromApprox(m_queue, size_t(m_capture_thread.getSamplingRate()/m_test_freq), m_config_form.ui_spinFrequencyAverageSize->value(), Music::GetAFreq(), m_capture_thread.getSamplingRate());

				m_test_freq = m_capture_thread.getSamplingRate()/m_test_freq;
			}

//			cerr << "2) test freq=" << m_test_freq <<endl;

			if(m_test_freq!=0.0f)
			{
				double old_compared_freq = m_compared_freq;
				if(m_microtonalView->setting_show->isOn() && m_microtonalView->hasTuningFreqSelected())
					m_compared_freq = m_microtonalView->getTuningFreq();
				else
					m_compared_freq = h2f(f2h(m_test_freq));

				if(m_compared_freq!=old_compared_freq)
				{
					m_test_freq = 0.0f;
					m_first_time = true;
					m_note_start = m_time.elapsed();
					m_old_analysis.clear();
				}
			}

			// get error and his variance and test stability
			double error, error_var;
			if(m_test_freq!=0.0f)
				get_error_and_variance(error, error_var);

			// check if the note is long enough
			if(m_test_freq!=0.0)
			{
				if(m_time.elapsed()-m_note_start<m_config_form.ui_spinErrorLatency->value())
				{
					m_test_freq = 0.0f;
					m_gap_time = m_time.elapsed();
				}
			}

			if(m_test_freq==0.0)
			{
				ui_lblSoundStability->setBackgroundColor(prb_color);
			}
			else
			{
				m_gap_time = m_time.elapsed();
				ui_lblSoundStability->setBackgroundColor(ok_color);
				m_freq = m_test_freq;
				m_error = error;

				// if first time, a new note appears and should be stored in views
				if(m_first_time)
				{
					m_first_time = false;

					// so clean everything
					m_old_analysis.clear();

					// set the compared freq
					if(m_microtonalView->setting_show->isOn() && m_microtonalView->hasTuningFreqSelected())
					{
						ui_txtNoteFreq->display(int(m_microtonalView->getTuningFreq()*100)/100.0);
						ui_txtNote->setText(m_microtonalView->getTuningNoteName());
						if(m_microtonalView->m_selected_jivalue->is_ratio)
						{
							m_glErrorHistory->m_notes.push_back(GLErrorHistory::Note(m_microtonalView->setting_selectedRoot, m_microtonalView->m_selected_jivalue->num, m_microtonalView->m_selected_jivalue->den));
							m_glVolumeHistory->m_notes.push_back(GLVolumeHistory::Note(m_microtonalView->setting_selectedRoot, m_microtonalView->m_selected_jivalue->num, m_microtonalView->m_selected_jivalue->den));
						}
						else
						{
							m_glErrorHistory->m_notes.push_back(GLErrorHistory::Note(m_microtonalView->setting_selectedRoot, m_microtonalView->m_selected_jivalue->cents));
							m_glVolumeHistory->m_notes.push_back(GLVolumeHistory::Note(m_microtonalView->setting_selectedRoot, m_microtonalView->m_selected_jivalue->cents));
						}
					}
					else
					{
						ui_txtNoteFreq->display(m_compared_freq);
						ui_txtNote->setText(h2n(f2h(m_compared_freq)));
						m_glErrorHistory->m_notes.push_back(GLErrorHistory::Note(f2h(m_test_freq)));
						m_glVolumeHistory->m_notes.push_back(GLVolumeHistory::Note(f2h(m_test_freq)));
					}
				}

				// refresh error
				m_glErrorHistory->m_notes.back().addError(m_error);
				m_glVolumeHistory->m_notes.back().volumes.push_back(new_data_amplitude);
				m_dialTune->setError(m_error);

				m_dialTune->m_avg_error = m_glErrorHistory->m_notes.back().avg_err;
				m_dialTune->m_min_error = m_glErrorHistory->m_notes.back().min_err;
				m_dialTune->m_max_error = m_glErrorHistory->m_notes.back().max_err;
				ui_txtFreq->display(m_freq);

				// refresh intonal tuning cursor
				m_microtonalView->setAFreq(Music::GetAFreq());
				m_microtonalView->updateCursor(m_freq);

				// refresh sample data
				refresh_data_sample();

				// refresh formants data
				refresh_data_harmonics();
			}
		}

		if(m_time.elapsed()-m_last_views_update>40)	// 25 images/second
		{
			refresh_views();

			m_last_views_update = m_time.elapsed();
		}
	}

	m_last_refresh = m_time.elapsed();
}

void CustomInstrumentTunerForm::get_error_and_variance(double& error, double& error_var)
{
	error = 1000.0;
	error_var = 1000.0;

	double diff = f2hf(m_test_freq, m_compared_freq);
	double current_time = m_time.elapsed();
	m_old_analysis.push_front(Analysis(current_time, m_test_freq, diff));
	while(!m_old_analysis.empty() &&
			current_time-m_old_analysis.back().time>m_config_form.ui_spinErrorLatency->value())
		m_old_analysis.pop_back();

	// get an average freq
	m_test_freq = 0.0f;
	for(size_t i=0; i<m_old_analysis.size(); i++)
		m_test_freq += m_old_analysis[i].freq;
	m_test_freq /= m_old_analysis.size();

	error = 0.0;
	for(size_t i=0; i<m_old_analysis.size(); i++)
		error += m_old_analysis[i].diff;
	error /= m_old_analysis.size();

	error_var = 0.0;
	for(size_t i=0; i<m_old_analysis.size(); i++)
		error_var = max(error_var, double(abs(error-m_old_analysis[i].diff)));

	if(error_var>m_config_form.ui_spinErrorVarianceTolerance->value()/100.0 || m_old_analysis.empty())
		m_test_freq = 0.0f;
}

void CustomInstrumentTunerForm::refresh_data_sample()
{
	if(m_freq==0.0f || !m_glSample->setting_show->isOn())
	{
		m_glSample->clear();
		return;
	}

	deque<double> sample;
	GetWaveSample(m_queue, size_t(m_capture_thread.getSamplingRate()/m_freq), sample);
	m_glSample->add(m_time.elapsed(), sample);
}

void CustomInstrumentTunerForm::refresh_data_harmonics()
{
// 	if(m_freq==0.0f || !m_glFreqStruct->setting_show->isOn())	return;

	m_glFreqStruct->m_components_max = 0.0;
	for(int i=0; i<int(m_glFreqStruct->m_components.size()); i++)
	{
		double ht = f2h((i+1)*(m_freq));

		// TODO temp !
// 		Convolution conv(20, 2, ht);

// 		conv.apply(m_queue);

// 		m_glFreqStruct->m_components[i] = Math::mod(conv.m_value);
		m_glFreqStruct->m_components[i] = m_algo_combedfft->getComponents()[i];

		m_glFreqStruct->m_components_max = max(m_glFreqStruct->m_components_max, m_glFreqStruct->m_components[i]);
	}
}

void CustomInstrumentTunerForm::refresh_views()
{
//	m_dialTune->repaint();

	if(m_glGraph->setting_show->isOn())
		m_glGraph->updateGL();

	if(m_glErrorHistory->setting_show->isOn())
		m_glErrorHistory->updateGL();

	if(m_glVolumeHistory->setting_show->isOn())
		m_glVolumeHistory->updateGL();

	if(m_microtonalView->setting_show->isOn())
		m_microtonalView->update();

	if(m_glSample->setting_show->isOn())
		m_glSample->updateGL();

	if(m_glFreqStruct->setting_show->isOn())
		m_glFreqStruct->updateGL();
}

void CustomInstrumentTunerForm::keyPressEvent(QKeyEvent * e)
{
	if(e->key()==Key_F)
		toggleFullScreen();
}

void CustomInstrumentTunerForm::resizeEvent(QResizeEvent* e)
{
	update_views();

	InstrumentTunerForm::resizeEvent(e);
}

void CustomInstrumentTunerForm::update_views()
{
	//	m_dialTune->setMaximumWidth(size().width()/4);
	m_dialTune->setMinimumWidth(size().width()/4);

	if(m_glGraph->setting_show->isOn() &&
			!m_glErrorHistory->setting_show->isOn() &&
			!m_glVolumeHistory->setting_show->isOn() &&
			!m_glSample->setting_show->isOn() &&
			!m_glFreqStruct->setting_show->isOn())
		m_glGraph->setMaximumHeight(32767);		// TODO pas beau !
	else
		m_glGraph->setMaximumHeight(m_glGraph->setting_spinMaxHeight->value());

	if(!m_glErrorHistory->setting_show->isOn() && !m_glVolumeHistory->setting_show->isOn())
		ui_btnKeepErrorHistory->hide();
	else
		ui_btnKeepErrorHistory->show();
}

void CustomInstrumentTunerForm::configure()
{
	noteRangeChanged();
	m_config_form.ui_spinALSASamplingRate->setValue(m_capture_thread.getSamplingRate());
	m_config_form.ui_lblJACKSamplingRate->setText(QString::number(m_capture_thread.getSamplingRate()));

	m_config_form.show();
}
void CustomInstrumentTunerForm::configure_ok()
{
	if(m_config_form.ui_cbTonality->currentItem()==0)		SetTonality(0);
	else if(m_config_form.ui_cbTonality->currentItem()==1)	SetTonality(+2);
	else													SetTonality(-3);

	if(m_config_form.ui_cbNotesName->currentItem()==0)		SetNotesName(LOCAL_ANGLO);
	else													SetNotesName(LOCAL_LATIN);
	m_microtonalView->notesNameChanged();

	SetSemitoneBounds(m_config_form.ui_spinMinHT->value(), m_config_form.ui_spinMaxHT->value());

	ui_spinA3Offset->setShown(m_config_form.ui_chkShowA3Offset->isOn());
	ui_lblA3Offset->setShown(m_config_form.ui_chkShowA3Offset->isOn());

	//	if(m_note!=-1000)
	//		ui_txtNote->setText(h2n(m_note));

#ifdef CAPTURE_ALSA
	if(m_config_form.ui_rdbtnALSA->isChecked())
	{
		m_capture_thread.selectTransport("ALSA");
		m_config_form.ui_grpALSA->setEnabled(true);
		m_capture_thread.setSource(m_config_form.ui_txtALSAPCMName->text());
		if(m_config_form.ui_chkALSASamplingRateMax->isChecked())
			m_capture_thread.setSamplingRate(CaptureThread::SAMPLING_RATE_MAX);
		else
			m_capture_thread.setSamplingRate(m_config_form.ui_spinALSASamplingRate->value());
	}
	else
		m_config_form.ui_grpALSA->setDisabled(true);
#endif
#ifdef CAPTURE_JACK
	if(m_config_form.ui_rdbtnJACK->isChecked())
	{
		m_capture_thread.selectTransport("JACK");
		m_config_form.ui_grpJACK->setEnabled(true);
		if(m_config_form.ui_chkJACKAutoConnect->isChecked())
			m_capture_thread.setSource(m_config_form.ui_txtJACKSourcePort->text());
		else
			m_capture_thread.setSource("");
		m_config_form.ui_lblJACKSamplingRate->setText(QString::number(m_capture_thread.getSamplingRate()));
	}
	else
		m_config_form.ui_grpJACK->setDisabled(true);
#endif
	m_timer_refresh->changeInterval(m_config_form.ui_spinRefreshTime->value());

	m_glGraph->setMaximumHeight(m_glGraph->setting_spinMaxHeight->value());
	m_glGraph->m_queue.clear();
	m_glGraph->update_maxs();

	if(!pauseAction->isOn() && !m_capture_thread.isCapturing())
		m_capture_thread.startCapture();
}

void CustomInstrumentTunerForm::saveSettings()
{
	// views
	m_settings.writeEntry("width", width());
	m_settings.writeEntry("height", height());
	m_settings.writeEntry("ui_tbViews", ui_tbViews->isShown());
	m_settings.writeEntry("ui_tbButtons", ui_tbButtons->isShown());
	m_settings.writeEntry("ui_chkFullScreen", m_config_form.ui_chkFullScreen->isChecked());
	m_settings.writeEntry("ui_chkAutoSaveOnExit", m_config_form.ui_chkAutoSaveOnExit->isChecked());
	m_settings.writeEntry("ui_cbTonality", m_config_form.ui_cbTonality->currentItem());
	m_settings.writeEntry("ui_cbNotesName", m_config_form.ui_cbNotesName->currentItem());
	m_settings.writeEntry("ui_spinErrorLatency", m_config_form.ui_spinErrorLatency->value());

	// A frequency
	m_settings.writeEntry("ui_spinAFreq", ui_spinAFreq->value());
	m_settings.writeEntry("ui_spinA3Offset", ui_spinA3Offset->value());
	m_settings.writeEntry("ui_chkShowA3Offset", m_config_form.ui_chkShowA3Offset->isChecked());

	// Sound Capture
	m_settings.writeEntry("ui_chkAutoDetect", m_config_form.ui_chkAutoDetect->isChecked()?1:0);
	m_settings.writeEntry("ui_rdbtnALSA", m_config_form.ui_rdbtnALSA->isChecked()?1:0);
	m_settings.writeEntry("ui_rdbtnJACK", m_config_form.ui_rdbtnJACK->isChecked()?1:0);
#ifdef CAPTURE_ALSA
	m_settings.writeEntry("ui_txtALSAPCMName", m_config_form.ui_txtALSAPCMName->text());
	m_settings.writeEntry("ui_chkALSASamplingRateMax", (m_config_form.ui_chkALSASamplingRateMax->isChecked()?1:0));
	m_settings.writeEntry("ui_spinALSASamplingRate", m_config_form.ui_spinALSASamplingRate->value());
	m_settings.writeEntry(m_config_form.ui_chkALSAMixMultipleChannels->name(), m_config_form.ui_chkALSAMixMultipleChannels->isChecked()?1:0);
#endif
#ifdef CAPTURE_JACK
	m_settings.writeEntry("ui_chkJACKAutoConnect", (m_config_form.ui_chkJACKAutoConnect->isChecked()?1:0));
	m_settings.writeEntry("ui_txtJACKSourcePort", m_config_form.ui_txtJACKSourcePort->text());
#endif
	m_settings.writeEntry("ui_spinRefreshTime", m_config_form.ui_spinRefreshTime->value());

	// Sound Analysis
	m_settings.writeEntry("ui_spinMinHT", m_config_form.ui_spinMinHT->value());
	m_settings.writeEntry("ui_spinMaxHT", m_config_form.ui_spinMaxHT->value());
	m_settings.writeEntry("ui_spinVolumeTreshold", m_config_form.ui_spinVolumeTreshold->value());
	m_settings.writeEntry("ui_spinLatencyFactor", m_config_form.ui_spinLatencyFactor->value());
	m_settings.writeEntry("ui_spinMonoTestComplexity", m_config_form.ui_spinMonoTestComplexity->value());
	m_settings.writeEntry("ui_spinComponentsTreshold", m_config_form.ui_spinComponentsTreshold->value());
	m_settings.writeEntry("ui_spinFrequencyAverageSize", m_config_form.ui_spinFrequencyAverageSize->value());
	m_settings.writeEntry("ui_spinErrorVarianceTolerance", m_config_form.ui_spinErrorVarianceTolerance->value());

	View::saveAll();
}
void CustomInstrumentTunerForm::loadSettings()
{
	// views
	resize(m_settings.readNumEntry("width", 800), m_settings.readNumEntry("height", 600));
	ui_tbViews->setShown(m_settings.readBoolEntry("ui_tbViews", ui_tbViews->isShown()));
	ui_tbButtons->setShown(m_settings.readBoolEntry("ui_tbButtons", ui_tbButtons->isShown()));
	m_config_form.ui_chkFullScreen->setChecked(m_settings.readBoolEntry("ui_chkFullScreen", m_config_form.ui_chkFullScreen->isChecked()));
	m_config_form.ui_chkAutoSaveOnExit->setChecked(m_settings.readBoolEntry("ui_chkAutoSaveOnExit", m_config_form.ui_chkAutoSaveOnExit->isChecked()));
	m_config_form.ui_cbTonality->setCurrentItem(m_settings.readNumEntry("ui_cbTonality", m_config_form.ui_cbTonality->currentItem()));
	m_config_form.ui_cbNotesName->setCurrentItem(m_settings.readNumEntry("ui_cbNotesName", m_config_form.ui_cbNotesName->currentItem()));
	m_config_form.ui_spinErrorLatency->setValue(m_settings.readNumEntry(m_config_form.ui_spinErrorLatency->name(), m_config_form.ui_spinErrorLatency->value()));

	// A frequency
	ui_spinAFreq->setValue(m_settings.readNumEntry("ui_spinAFreq", ui_spinAFreq->value()));
	ui_spinA3Offset->setValue(m_settings.readNumEntry("ui_spinA3Offset", ui_spinA3Offset->value()));
	m_config_form.ui_chkShowA3Offset->setChecked(m_settings.readBoolEntry("ui_chkShowA3Offset", m_config_form.ui_chkShowA3Offset->isChecked()));

	// Sound Capture
	m_config_form.ui_chkAutoDetect->setChecked(m_settings.readNumEntry("ui_chkAutoDetect", (m_config_form.ui_chkAutoDetect->isChecked()?1:0)));
	m_config_form.ui_rdbtnALSA->setChecked(m_settings.readNumEntry("ui_rdbtnALSA", (m_config_form.ui_rdbtnALSA->isChecked()?1:0)));
	m_config_form.ui_rdbtnJACK->setChecked(m_settings.readNumEntry("ui_rdbtnJACK", (m_config_form.ui_rdbtnJACK->isChecked()?1:0)));
#ifdef CAPTURE_ALSA
	m_config_form.ui_txtALSAPCMName->setText(m_settings.readEntry("ui_txtALSAPCMName", (m_config_form.ui_txtALSAPCMName->text())));
	m_config_form.ui_chkALSASamplingRateMax->setChecked(m_settings.readNumEntry("ui_chkALSASamplingRateMax", (m_config_form.ui_chkALSASamplingRateMax->isChecked()?1:0)));
	m_config_form.ui_spinALSASamplingRate->setValue(m_settings.readNumEntry("ui_spinALSASamplingRate", m_config_form.ui_spinALSASamplingRate->value()));
	m_config_form.ui_chkALSAMixMultipleChannels->setChecked(m_settings.readNumEntry(m_config_form.ui_chkALSAMixMultipleChannels->name(), m_config_form.ui_chkALSAMixMultipleChannels->isChecked()?1:0));
#endif
#ifdef CAPTURE_JACK
	//	cout << m_config_form.ui_chkJACKAutoConnect->isChecked() << ":" << m_settings.readNumEntry("ui_chkJACKAutoConnect", 0) << endl;
	//	cout << m_config_form.ui_txtJACKSourcePort->text() << ":" << m_settings.readEntry("ui_txtJACKSourcePort", "prout") << endl;
	m_config_form.ui_chkJACKAutoConnect->setChecked(m_settings.readNumEntry("ui_chkJACKAutoConnect", (m_config_form.ui_chkJACKAutoConnect->isChecked()?1:0)));
	m_config_form.ui_txtJACKSourcePort->setText(m_settings.readEntry("ui_txtJACKSourcePort", m_config_form.ui_txtJACKSourcePort->text()));
#endif
	m_config_form.ui_spinRefreshTime->setValue(m_settings.readNumEntry("ui_spinRefreshTime", m_config_form.ui_spinRefreshTime->value()));

	// Sound Analysis
	m_config_form.ui_spinMinHT->setValue(m_settings.readNumEntry("ui_spinMinHT", m_config_form.ui_spinMinHT->value()));
	m_config_form.ui_spinMaxHT->setValue(m_settings.readNumEntry("ui_spinMaxHT", m_config_form.ui_spinMaxHT->value()));
	m_config_form.ui_spinVolumeTreshold->setValue(m_settings.readNumEntry("ui_spinVolumeTreshold", m_config_form.ui_spinVolumeTreshold->value()));
	m_config_form.ui_spinLatencyFactor->setValue(m_settings.readNumEntry("ui_spinLatencyFactor", m_config_form.ui_spinLatencyFactor->value()));
	m_config_form.ui_spinMonoTestComplexity->setValue(m_settings.readNumEntry("ui_spinMonoTestComplexity", m_config_form.ui_spinMonoTestComplexity->value()));
	m_config_form.ui_spinComponentsTreshold->setValue(m_settings.readNumEntry("ui_spinComponentsTreshold", m_config_form.ui_spinComponentsTreshold->value()));
	m_config_form.ui_spinFrequencyAverageSize->setValue(m_settings.readNumEntry("ui_spinFrequencyAverageSize", m_config_form.ui_spinFrequencyAverageSize->value()));
	m_config_form.ui_spinErrorVarianceTolerance->setValue(m_settings.readNumEntry("ui_spinErrorVarianceTolerance", m_config_form.ui_spinErrorVarianceTolerance->value()));

	View::loadAll();
}

void CustomInstrumentTunerForm::restoreFactorySettings()
{
	if(QMessageBox::question(this, tr("Restore Factory Settings"), tr("This operation is NOT reversible.\nAre you sure you want to lose all your current settings ?"), QMessageBox::Yes, QMessageBox::No)==QMessageBox::Yes)
	{
		m_settings.removeEntry("width");
		m_settings.removeEntry("height");
		m_settings.removeEntry("ui_chkFullScreen");
		m_settings.removeEntry("ui_tbViews");
		m_settings.removeEntry("ui_tbButtons");
		m_settings.removeEntry("ui_chkAutoSaveOnExit");

		m_settings.removeEntry("ui_cbTonality");
		m_settings.removeEntry("ui_cbNotesName");
		m_settings.removeEntry("ui_spinErrorLatency");

		m_settings.removeEntry("ui_spinAFreq");
		m_settings.removeEntry("ui_spinA3Offset");
		m_settings.removeEntry("ui_chkShowA3Offset");

		m_settings.removeEntry("ui_chkAutoDetect");
		m_settings.removeEntry("ui_rdbtnALSA");
		m_settings.removeEntry("ui_rdbtnJACK");
		m_settings.removeEntry("ui_txtALSAPCMName");
		m_settings.removeEntry("ui_chkALSASamplingRateMax");
		m_settings.removeEntry("ui_spinALSASamplingRate");

		m_settings.removeEntry("ui_chkJACKAutoConnect");
		m_settings.removeEntry("ui_txtJACKSourcePort");

		m_settings.removeEntry("ui_spinRefreshTime");

		m_settings.removeEntry("ui_spinMinHT");
		m_settings.removeEntry("ui_spinMaxHT");
		m_settings.removeEntry("ui_spinVolumeTreshold");
		m_settings.removeEntry("ui_spinLatencyFactor");
		m_settings.removeEntry("ui_spinMonoTestComplexity");
		m_settings.removeEntry("ui_spinComponentsTreshold");
		m_settings.removeEntry("ui_spinFrequencyAverageSize");
		m_settings.removeEntry("ui_spinErrorVarianceTolerance");

		View::clearAllSettings();

		QMessageBox::information(this, tr("Restore Factory Settings"), tr("You can now restart FMIT to get back factory settings"), QMessageBox::Ok, QMessageBox::NoButton);
	}
}

void CustomInstrumentTunerForm::helpAbout()
{
	QMessageBox::about(this, tr("About Free Music Instrument Tuner"), tr("<h3>Version ")+PACKAGE_VERSION+tr("</h3><p><h3>Website:</h3><p>homepage: <a href=\"http://home.gna.org/fmit\">http://home.gna.org/fmit</a><p>development site: <a href=\"http://gna.org/projects/fmit\">http://gna.org/projects/fmit</a><p><h3>Author:</h3><p>Gilles Degottex [gilles.degottex@net2000.ch]"));
}

CustomInstrumentTunerForm::~CustomInstrumentTunerForm()
{
	if(m_config_form.ui_chkAutoSaveOnExit->isChecked())
		saveSettings();
	else
		m_settings.writeEntry("ui_chkAutoSaveOnExit", m_config_form.ui_chkAutoSaveOnExit->isChecked());
}

