//  panel_io.cxx - I/O for 2D panel.
//
//  Written by David Megginson, started January 2000.
//
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License as
//  published by the Free Software Foundation; either version 2 of the
//  License, or (at your option) any later version.
// 
//  This program is distributed in the hope that it will be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  General Public License for more details.
// 
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
//  $Id: panel_io.cxx,v 1.30 2001/12/17 19:34:37 curt Exp $

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#ifdef HAVE_WINDOWS_H          
#  include <windows.h>
#endif

#include <simgear/compiler.h>
#include <simgear/misc/exception.hxx>

#include <simgear/misc/sg_path.hxx>
#include <simgear/debug/logstream.hxx>
#include <simgear/misc/props.hxx>

#include STL_IOSTREAM
#include STL_FSTREAM
#include STL_STRING

#include <Main/globals.hxx>
#include <Main/fg_props.hxx>

#include <GUI/gui.h>

#include "panel.hxx"
#include "steam.hxx"
#include "panel_io.hxx"

#if !defined (SG_HAVE_NATIVE_SGI_COMPILERS)
SG_USING_STD(istream);
SG_USING_STD(ifstream);
#endif
SG_USING_STD(string);



////////////////////////////////////////////////////////////////////////
// Built-in layer for the magnetic compass ribbon layer.
//
// TODO: move this out into a special directory for built-in
// layers of various sorts.
////////////////////////////////////////////////////////////////////////

class FGMagRibbon : public FGTexturedLayer
{
public:
  FGMagRibbon (int w, int h);
  virtual ~FGMagRibbon () {}

  virtual void draw ();
};

FGMagRibbon::FGMagRibbon (int w, int h)
  : FGTexturedLayer(w, h)
{
  FGCroppedTexture texture("Aircraft/Instruments/Textures/compass-ribbon.rgb");
  setTexture(texture);
}

void
FGMagRibbon::draw ()
{
  double heading = FGSteam::get_MH_deg();
  double xoffset, yoffset;

  while (heading >= 360.0) {
    heading -= 360.0;
  }
  while (heading < 0.0) {
    heading += 360.0;
  }

  if (heading >= 60.0 && heading <= 180.0) {
    xoffset = heading / 240.0;
    yoffset = 0.75;
  } else if (heading >= 150.0 && heading <= 270.0) {
    xoffset = (heading - 90.0) / 240.0;
    yoffset = 0.50;
  } else if (heading >= 240.0 && heading <= 360.0) {
    xoffset = (heading - 180.0) / 240.0;
    yoffset = 0.25;
  } else {
    if (heading < 270.0)
      heading += 360.0;
    xoffset = (heading - 270.0) / 240.0;
    yoffset = 0.0;
  }

  xoffset = 1.0 - xoffset;
				// Adjust to put the number in the centre
  xoffset -= 0.25;

  FGCroppedTexture &t = getTexture();
  t.setCrop(xoffset, yoffset, xoffset + 0.5, yoffset + 0.25);
  FGTexturedLayer::draw();
}



////////////////////////////////////////////////////////////////////////
// Read and construct a panel.
//
// The panel is specified as a regular property list, and each of the
// instruments is its own, separate property list (and thus, a separate
// XML document).  The functions in this section read in the files
// as property lists, then extract properties to set up the panel
// itself.
//
// A panel contains zero or more instruments.
//
// An instrument contains one or more layers and zero or more actions.
//
// A layer contains zero or more transformations.
//
// Some special types of layers also contain other objects, such as 
// chunks of text or other layers.
//
// There are currently four types of layers:
//
// 1. Textured Layer (type="texture"), the default
// 2. Text Layer (type="text")
// 3. Switch Layer (type="switch")
// 4. Built-in Layer (type="built-in", must also specify class)
//
// The only built-in layer so far is the ribbon for the magnetic compass
// (class="compass-ribbon").
//
// There are three types of actions:
//
// 1. Adjust (type="adjust"), the default
// 2. Swap (type="swap")
// 3. Toggle (type="toggle")
//
// There are three types of transformations:
//
// 1. X shift (type="x-shift"), the default
// 2. Y shift (type="y-shift")
// 3. Rotation (type="rotation")
//
// Each of these may be associated with a property, so that a needle
// will rotate with the airspeed, for example, or may have a fixed
// floating-point value.
////////////////////////////////////////////////////////////////////////


/**
 * Read a cropped texture from the instrument's property list.
 *
 * The x1 and y1 properties give the starting position of the texture
 * (between 0.0 and 1.0), and the the x2 and y2 properties give the
 * ending position.  For example, to use the bottom-left quarter of a
 * texture, x1=0.0, y1=0.0, x2=0.5, y2=0.5.
 */
static FGCroppedTexture
readTexture (const SGPropertyNode * node)
{
    FGCroppedTexture texture(node->getStringValue("path"),
			     node->getFloatValue("x1"),
			     node->getFloatValue("y1"),
			     node->getFloatValue("x2", 1.0),
			     node->getFloatValue("y2", 1.0));
    SG_LOG(SG_COCKPIT, SG_DEBUG, "Read texture " << node->getName());
    return texture;
}


/**
 * Test for a condition in the current node.
 */

////////////////////////////////////////////////////////////////////////
// Read a condition and use it if necessary.
////////////////////////////////////////////////////////////////////////

static void
readConditions (FGConditional * component, const SGPropertyNode * node)
{
  const SGPropertyNode * conditionNode = node->getChild("condition");
  if (conditionNode != 0)
				// The top level is implicitly AND
    component->setCondition(fgReadCondition(conditionNode));
}


/**
 * Read an action from the instrument's property list.
 *
 * The action will be performed when the user clicks a mouse button
 * within the specified region of the instrument.  Actions always work
 * by modifying the value of a property (see the SGPropertyNode
 * class).
 *
 * The following action types are defined:
 *
 * "adjust" - modify the value of a floating-point property by
 *    the increment specified.  This is the default.
 *
 * "swap" - swap the values of two-floating-point properties.
 *
 * "toggle" - toggle the value of a boolean property between true and
 *    false.
 *
 * For the adjust action, it is possible to specify an increment
 * (use a negative number for a decrement), a minimum allowed value,
 * a maximum allowed value, and a flag to indicate whether the value
 * should freeze or wrap-around when it reachs the minimum or maximum.
 *
 * The action will be scaled automatically if the instrument is not
 * being drawn at its regular size.
 */
static FGPanelAction *
readAction (const SGPropertyNode * node, float w_scale, float h_scale)
{
  string name = node->getStringValue("name");

  int button = node->getIntValue("button");
  int x = int(node->getIntValue("x") * w_scale);
  int y = int(node->getIntValue("y") * h_scale);
  int w = int(node->getIntValue("w") * w_scale);
  int h = int(node->getIntValue("h") * h_scale);

  FGPanelAction * action = new FGPanelAction(button, x, y, w, h);

  vector<const SGPropertyNode *>bindings = node->getChildren("binding");
  for (unsigned int i = 0; i < bindings.size(); i++) {
    SG_LOG(SG_INPUT, SG_INFO, "Reading binding "
	   << bindings[i]->getStringValue("command"));
    action->addBinding(new FGBinding(bindings[i])); // TODO: allow modifiers
  }

  readConditions(action, node);
  return action;
}


/**
 * Read a transformation from the instrument's property list.
 *
 * The panel module uses the transformations to slide or spin needles,
 * knobs, and other indicators, and to place layers in the correct
 * positions.  Every layer starts centered exactly on the x,y co-ordinate,
 * and many layers need to be moved or rotated simply to display the
 * instrument correctly.
 *
 * There are three types of transformations:
 *
 * "x-shift" - move the layer horizontally.
 *
 * "y-shift" - move the layer vertically.
 *
 * "rotation" - rotate the layer.
 *
 * Each transformation may have a fixed offset, and may also have
 * a floating-point property value to add to the offset.  The
 * floating-point property may be clamped to a minimum and/or
 * maximum range and scaled (after clamping).
 *
 * Note that because of the way OpenGL works, transformations will
 * appear to be applied backwards.
 */
static FGPanelTransformation *
readTransformation (const SGPropertyNode * node, float w_scale, float h_scale)
{
  FGPanelTransformation * t = new FGPanelTransformation;

  string name = node->getName();
  string type = node->getStringValue("type");
  string propName = node->getStringValue("property", "");
  SGPropertyNode * target = 0;

  if (type == "") {
    SG_LOG( SG_COCKPIT, SG_ALERT,
            "No type supplied for transformation " << name
            << " assuming \"rotation\"" );
    type = "rotation";
  }

  if (propName != (string)"") {
    target = fgGetNode(propName, true);
  }

  t->node = target;
  t->min = node->getFloatValue("min", -9999999);
  t->max = node->getFloatValue("max", 99999999);
  t->factor = node->getFloatValue("scale", 1.0);
  t->offset = node->getFloatValue("offset", 0.0);

				// Check for an interpolation table
  const SGPropertyNode * trans_table = node->getNode("interpolation");
  if (trans_table != 0) {
    SG_LOG( SG_COCKPIT, SG_INFO, "Found interpolation table with "
            << trans_table->nChildren() << "children" );
    t->table = new SGInterpTable();
    for(int i = 0; i < trans_table->nChildren(); i++) {
      const SGPropertyNode * node = trans_table->getChild(i);
      if (node->getName() == "entry") {
	double ind = node->getDoubleValue("ind", 0.0);
	double dep = node->getDoubleValue("dep", 0.0);
	SG_LOG( SG_COCKPIT, SG_INFO, "Adding interpolation entry "
                << ind << "==>" << dep );
	t->table->addEntry(ind, dep);
      } else {
	SG_LOG( SG_COCKPIT, SG_INFO, "Skipping " << node->getName()
                << " in interpolation" );
      }
    }
  } else {
    t->table = 0;
  }
  
				// Move the layer horizontally.
  if (type == "x-shift") {
    t->type = FGPanelTransformation::XSHIFT;
//     t->min *= w_scale; //removed by Martin Dressler
//     t->max *= w_scale; //removed by Martin Dressler
    t->offset *= w_scale;
    t->factor *= w_scale; //Added by Martin Dressler
  } 

				// Move the layer vertically.
  else if (type == "y-shift") {
    t->type = FGPanelTransformation::YSHIFT;
    //t->min *= h_scale; //removed
    //t->max *= h_scale; //removed
    t->offset *= h_scale;
    t->factor *= h_scale; //Added
  } 

				// Rotate the layer.  The rotation
				// is in degrees, and does not need
				// to scale with the instrument size.
  else if (type == "rotation") {
    t->type = FGPanelTransformation::ROTATION;
  } 

  else {
    SG_LOG( SG_COCKPIT, SG_ALERT, "Unrecognized transformation type " << type );
    delete t;
    return 0;
  }

  readConditions(t, node);
  SG_LOG( SG_COCKPIT, SG_DEBUG, "Read transformation " << name );
  return t;
}


/**
 * Read a chunk of text from the instrument's property list.
 *
 * A text layer consists of one or more chunks of text.  All chunks
 * share the same font size and color (and eventually, font), but
 * each can come from a different source.  There are three types of
 * text chunks:
 *
 * "literal" - a literal text string (the default)
 *
 * "text-value" - the current value of a string property
 *
 * "number-value" - the current value of a floating-point property.
 *
 * All three may also include a printf-style format string.
 */
FGTextLayer::Chunk *
readTextChunk (const SGPropertyNode * node)
{
  FGTextLayer::Chunk * chunk;
  string name = node->getStringValue("name");
  string type = node->getStringValue("type");
  string format = node->getStringValue("format");

				// Default to literal text.
  if (type == "") {
    SG_LOG( SG_COCKPIT, SG_INFO, "No type provided for text chunk " << name
            << " assuming \"literal\"");
    type = "literal";
  }

				// A literal text string.
  if (type == "literal") {
    string text = node->getStringValue("text");
    chunk = new FGTextLayer::Chunk(text, format);
  }

				// The value of a string property.
  else if (type == "text-value") {
    SGPropertyNode * target =
      fgGetNode(node->getStringValue("property"), true);
    chunk = new FGTextLayer::Chunk(FGTextLayer::TEXT_VALUE, target, format);
  }

				// The value of a float property.
  else if (type == "number-value") {
    string propName = node->getStringValue("property");
    float scale = node->getFloatValue("scale", 1.0);
    SGPropertyNode * target = fgGetNode(propName, true);
    chunk = new FGTextLayer::Chunk(FGTextLayer::DOUBLE_VALUE, target,
				   format, scale);
  }

				// Unknown type.
  else {
    SG_LOG( SG_COCKPIT, SG_ALERT, "Unrecognized type " << type
            << " for text chunk " << name );
    return 0;
  }

  readConditions(chunk, node);
  return chunk;
}


/**
 * Read a single layer from an instrument's property list.
 *
 * Each instrument consists of one or more layers stacked on top
 * of each other; the lower layers show through only where the upper
 * layers contain an alpha component.  Each layer can be moved
 * horizontally and vertically and rotated using transformations.
 *
 * This module currently recognizes four kinds of layers:
 *
 * "texture" - a layer containing a texture (the default)
 *
 * "text" - a layer containing text
 *
 * "switch" - a layer that switches between two other layers
 *   based on the current value of a boolean property.
 *
 * "built-in" - a hard-coded layer supported by C++ code in FlightGear.
 *
 * Currently, the only built-in layer class is "compass-ribbon".
 */
static FGInstrumentLayer *
readLayer (const SGPropertyNode * node, float w_scale, float h_scale)
{
  FGInstrumentLayer * layer = NULL;
  string name = node->getStringValue("name");
  string type = node->getStringValue("type");
  int w = node->getIntValue("w", -1);
  int h = node->getIntValue("h", -1);
  if (w != -1)
    w = int(w * w_scale);
  if (h != -1)
    h = int(h * h_scale);


  if (type == "") {
    SG_LOG( SG_COCKPIT, SG_ALERT,
            "No type supplied for layer " << name
            << " assuming \"texture\"" );
    type = "texture";
  }


				// A textured instrument layer.
  if (type == "texture") {
    FGCroppedTexture texture = readTexture(node->getNode("texture"));
    layer = new FGTexturedLayer(texture, w, h);
  }

				// A group of sublayers.
  else if (type == "group") {
    layer = new FGGroupLayer();
    for (int i = 0; i < node->nChildren(); i++) {
      const SGPropertyNode * child = node->getChild(i);
      cerr << "Trying child " << child->getName() << endl;
      if (child->getName() == "layer") {
	cerr << "succeeded!" << endl;
	((FGGroupLayer *)layer)->addLayer(readLayer(child, w_scale, h_scale));
      }
    }
  }


				// A textual instrument layer.
  else if (type == "text") {
    FGTextLayer * tlayer = new FGTextLayer(w, h); // FIXME

				// Set the text color.
    float red = node->getFloatValue("color/red", 0.0);
    float green = node->getFloatValue("color/green", 0.0);
    float blue = node->getFloatValue("color/blue", 0.0);
    tlayer->setColor(red, green, blue);

				// Set the point size.
    float pointSize = node->getFloatValue("point-size", 10.0) * w_scale;
    tlayer->setPointSize(pointSize);

				// Set the font.
    // TODO

    const SGPropertyNode * chunk_group = node->getNode("chunks");
    if (chunk_group != 0) {
      int nChunks = chunk_group->nChildren();
      for (int i = 0; i < nChunks; i++) {
	const SGPropertyNode * node = chunk_group->getChild(i);
	if (node->getName() == "chunk") {
	  FGTextLayer::Chunk * chunk = readTextChunk(node);
	  if (chunk != 0)
	    tlayer->addChunk(chunk);
	} else {
	  SG_LOG( SG_COCKPIT, SG_INFO, "Skipping " << node->getName()
                  << " in chunks" );
	}
      }
      layer = tlayer;
    }
  }

				// A switch instrument layer.
  else if (type == "switch") {
    SGPropertyNode * target =
      fgGetNode(node->getStringValue("property"), true);
    FGInstrumentLayer * layer1 =
      readLayer(node->getNode("layer[0]"), w_scale, h_scale);
    FGInstrumentLayer * layer2 =
      readLayer(node->getNode("layer[1]"), w_scale, h_scale);
    layer = new FGSwitchLayer(w, h, target, layer1, layer2);
  }

				// A built-in instrument layer.
  else if (type == "built-in") {
    string layerclass = node->getStringValue("class");

    if (layerclass == "mag-ribbon") {
      layer = new FGMagRibbon(w, h);
    }

    else if (layerclass == "") {
      SG_LOG( SG_COCKPIT, SG_ALERT, "No class provided for built-in layer "
              << name );
      return 0;
    }

    else {
      SG_LOG( SG_COCKPIT, SG_ALERT, "Unknown built-in layer class "
              << layerclass);
      return 0;
    }
  }

				// An unknown type.
  else {
    SG_LOG( SG_COCKPIT, SG_ALERT, "Unrecognized layer type " << type );
    delete layer;
    return 0;
  }
  
  //
  // Get the transformations for each layer.
  //
  const SGPropertyNode * trans_group = node->getNode("transformations");
  if (trans_group != 0) {
    int nTransformations = trans_group->nChildren();
    for (int i = 0; i < nTransformations; i++) {
      const SGPropertyNode * node = trans_group->getChild(i);
      if (node->getName() == "transformation") {
	FGPanelTransformation * t = readTransformation(node, w_scale, h_scale);
	if (t != 0)
	  layer->addTransformation(t);
      } else {
	SG_LOG( SG_COCKPIT, SG_INFO, "Skipping " << node->getName()
                << " in transformations" );
      }
    }
  }

  readConditions(layer, node);
  SG_LOG( SG_COCKPIT, SG_DEBUG, "Read layer " << name );
  return layer;
}


/**
 * Read an instrument from a property list.
 *
 * The instrument consists of a preferred width and height
 * (the panel may override these), together with a list of layers
 * and a list of actions to be performed when the user clicks 
 * the mouse over the instrument.  All co-ordinates are relative
 * to the instrument's position, so instruments are fully relocatable;
 * likewise, co-ordinates for actions and transformations will be
 * scaled automatically if the instrument is not at its preferred size.
 */
static FGPanelInstrument *
readInstrument (const SGPropertyNode * node)
{
  const string &name = node->getStringValue("name");
  int x = node->getIntValue("x", -1);
  int y = node->getIntValue("y", -1);
  int real_w = node->getIntValue("w", -1);
  int real_h = node->getIntValue("h", -1);
  int w = node->getIntValue("w-base", -1);
  int h = node->getIntValue("h-base", -1);

  if (x == -1 || y == -1) {
    SG_LOG( SG_COCKPIT, SG_ALERT,
            "x and y positions must be specified and > 0" );
    return 0;
  }

  float w_scale = 1.0;
  float h_scale = 1.0;
  if (real_w != -1) {
    w_scale = float(real_w) / float(w);
    w = real_w;
  }
  if (real_h != -1) {
    h_scale = float(real_h) / float(h);
    h = real_h;
  }

  SG_LOG( SG_COCKPIT, SG_DEBUG, "Reading instrument " << name );

  FGLayeredInstrument * instrument =
    new FGLayeredInstrument(x, y, w, h);

  //
  // Get the actions for the instrument.
  //
  const SGPropertyNode * action_group = node->getNode("actions");
  if (action_group != 0) {
    int nActions = action_group->nChildren();
    for (int i = 0; i < nActions; i++) {
      const SGPropertyNode * node = action_group->getChild(i);
      if (node->getName() == "action") {
	FGPanelAction * action = readAction(node, w_scale, h_scale);
	if (action != 0)
	  instrument->addAction(action);
      } else {
	SG_LOG( SG_COCKPIT, SG_INFO, "Skipping " << node->getName()
                << " in actions" );
      }
    }
  }

  //
  // Get the layers for the instrument.
  //
  const SGPropertyNode * layer_group = node->getNode("layers");
  if (layer_group != 0) {
    int nLayers = layer_group->nChildren();
    for (int i = 0; i < nLayers; i++) {
      const SGPropertyNode * node = layer_group->getChild(i);
      if (node->getName() == "layer") {
	FGInstrumentLayer * layer = readLayer(node, w_scale, h_scale);
	if (layer != 0)
	  instrument->addLayer(layer);
      } else {
	SG_LOG( SG_COCKPIT, SG_INFO, "Skipping " << node->getName()
                << " in layers" );
      }
    }
  }

  readConditions(instrument, node);
  SG_LOG( SG_COCKPIT, SG_DEBUG, "Done reading instrument " << name );
  return instrument;
}


/**
 * Construct the panel from a property tree.
 */
FGPanel *
readPanel (const SGPropertyNode * root)
{
  SG_LOG( SG_COCKPIT, SG_INFO, "Reading properties for panel " <<
          root->getStringValue("name", "[Unnamed Panel]") );

  FGPanel * panel = new FGPanel();
  panel->setWidth(root->getIntValue("w", 1024));
  panel->setHeight(root->getIntValue("h", 443));

  //
  // Grab the visible external viewing area, default to 
  //
  panel->setViewHeight(root->getIntValue("view-height",
					 768 - panel->getHeight() + 2));

  //
  // Grab the panel's initial offsets, default to 0, 0.
  //
  if (!fgHasNode("/sim/panel/x-offset"))
    fgSetInt("/sim/panel/x-offset", root->getIntValue("x-offset", 0));

  if (!fgHasNode("/sim/panel/y-offset"))
    fgSetInt("/sim/panel/y-offset", root->getIntValue("y-offset", 0));

  //
  // Assign the background texture, if any, or a bogus chequerboard.
  //
  string bgTexture = root->getStringValue("background");
  if (bgTexture == "")
    bgTexture = "FOO";
  panel->setBackground(FGTextureManager::createTexture(bgTexture.c_str()));
  SG_LOG( SG_COCKPIT, SG_INFO, "Set background texture to " << bgTexture );


  //
  // Create each instrument.
  //
  SG_LOG( SG_COCKPIT, SG_INFO, "Reading panel instruments" );
  const SGPropertyNode * instrument_group = root->getChild("instruments");
  if (instrument_group != 0) {
    int nInstruments = instrument_group->nChildren();
    for (int i = 0; i < nInstruments; i++) {
      const SGPropertyNode * node = instrument_group->getChild(i);
      if (node->getName() == "instrument") {
	FGPanelInstrument * instrument = readInstrument(node);
	if (instrument != 0)
	  panel->addInstrument(instrument);
      } else {
	SG_LOG( SG_COCKPIT, SG_INFO, "Skipping " << node->getName()
                << " in instruments section" );
      }
    }
  }
  SG_LOG( SG_COCKPIT, SG_INFO, "Done reading panel instruments" );


  //
  // Return the new panel.
  //
  return panel;
}


/**
 * Read a panel from a property list.
 *
 * Each panel instrument will appear in its own, separate
 * property list.  The top level simply names the panel and
 * places the instruments in their appropriate locations (and
 * optionally resizes them if necessary).
 *
 * Returns 0 if the read fails for any reason.
 */
FGPanel *
fgReadPanel (istream &input)
{
  SGPropertyNode root;

  try {
    readProperties(input, &root);
  } catch (const sg_exception &e) {
    guiErrorMessage("Error reading panel: ", e);
    return 0;
  }
  return readPanel(&root);
}


/**
 * Read a panel from a property list.
 *
 * This function opens a stream to a file, then invokes the
 * main fgReadPanel() function.
 */
FGPanel *
fgReadPanel (const string &relative_path)
{
  SGPath path(globals->get_fg_root());
  path.append(relative_path);
  SGPropertyNode root;

  try {
    readProperties(path.str(), &root);
  } catch (const sg_exception &e) {
    guiErrorMessage("Error reading panel: ", e);
    return 0;
  }
  return readPanel(&root);
}



// end of panel_io.cxx
