#include <gtk/gtk.h>
#include <stdlib.h>
#include <string.h>

#include "entity.h"
#include "gtk-common.h"

enum {
    TARGET_STRING,
    TARGET_URL,
    TARGET_XML_SRC
};

/* For now, I'm only going to do URI/string drags.. */

static GtkTargetEntry target_table[] = {
    {"STRING", 0, TARGET_STRING},
    {"text/plain", 0, TARGET_STRING},
    {"text/uri-list", 0, TARGET_URL},
    {"text/url-list", 0, TARGET_URL},
    {"text/url", 0, TARGET_URL},
    {"text/xml-source", 0, TARGET_XML_SRC}
};

static guint n_targets = sizeof (target_table) / sizeof (target_table[0]);

static GtkTargetEntry xml_node_target_table[] = {
    {"text/xml-source", 0, TARGET_XML_SRC}
};

static guint n_xml_node_targets =
    sizeof (xml_node_target_table) / sizeof (xml_node_target_table[0]);


/* Called when something is dragged onto something else. */
static void
builtins_drag_source_get_data (GtkWidget * widget,
			       GdkDragContext * context,
			       GtkSelectionData * selection_data,
			       guint info, guint time, gpointer data)
{
    gchar *value = NULL;
    gchar *function;
    ENode *node = data;

    /* If the node that's being dragged is a <object> element, we just send
     * the data to be the XML of the object */
    if (ebuf_equal_str (node->element, "object")) {
	EBuf *str;

	str = enode_get_xml (node);

	gtk_selection_data_set (selection_data,
				selection_data->target, 8, str->str, str->len);

	/* Unchecked.. might need to keep it around */
	ebuf_free (str);
	return;
    }

    function = enode_attrib_str (node, "ondrag", NULL);
    enode_call_ignore_return (node, function, "");

    if (info == TARGET_STRING) {
	value = enode_attrib_str (node, "dragdata-text", NULL);
	EDEBUG (("gtk-common", "drag dest wanted string"));
    }

    if (info == TARGET_URL) {
	EDEBUG (("gtk-common", "drag dest wanted url"));
	value = enode_attrib_str (node, "dragdata-url", NULL);
    }

    if (value)
	gtk_selection_data_set (selection_data,
				selection_data->target, 8, value,
				strlen (value));
}

/* Called when something receives a drop */
static void
builtins_drag_target_data_received (GtkWidget * widget,
				    GdkDragContext * context,
				    gint x,
				    gint y,
				    GtkSelectionData * data,
				    guint info, guint time)
{
    ENode *node;
    gchar *function;
    gchar *drag_data;

    node = gtk_object_get_data (GTK_OBJECT (widget), "xml-node");
    drag_data = (gchar *) data->data;

    if (!drag_data)
	return;

    if (info == TARGET_XML_SRC) {
	EDEBUG (("gtk-common", "XML Source dropped - '%s'", drag_data));

	if (context->suggested_action == GDK_ACTION_MOVE) {
	    EBuf *xml = ebuf_new_with_str (drag_data);

	    /* TODO: Should delete source tree if applicable */
	    xml_parse_string (node, xml);
	    ebuf_free (xml);
	} else {
	    EBuf *xml = ebuf_new_with_str (drag_data);
	    xml_parse_string (node, xml);
	    ebuf_free (xml);
	}

	return;
    }

    if ((data->length >= 0) && (data->format == 8)) {
	EDEBUG (("gtk-common", "Received \"%s\" for drag data", drag_data));
	gtk_drag_finish (context, TRUE, FALSE, time);

	if (node) {
	    function = enode_attrib_str (node, "ondrop", NULL);
	    EDEBUG (("gtk-common",
		     "Checking suggested drag type - %d",
		     context->suggested_action));

	    if (context->suggested_action == GDK_ACTION_MOVE) {
		enode_call_ignore_return (node, function, "ss",
					  drag_data, g_strdup ("move"));
	    } else {
		enode_call_ignore_return (node, function, "ss",
					  drag_data, g_strdup ("copy"));
	    }
	}
	return;
    } else {
	gtk_drag_finish (context, FALSE, FALSE, time);
    }
}


/* This one is for <object> tags */
void
rendgtk_dnd_dragtag_source_create (ENode * node, GtkWidget * widget)
{
    gtk_drag_source_set (widget, GDK_BUTTON1_MASK | GDK_BUTTON3_MASK,
			 xml_node_target_table, n_xml_node_targets,
			 GDK_ACTION_COPY | GDK_ACTION_MOVE);

    /* Insure previous handler is disconnected */
    /* gtk_signal_disconnect_by_data (GTK_OBJECT (widget), node); */

    if (0) {
	EBuf *path = enode_path (node);
	EDEBUG (("gtk-common", "Setting node %s as a drag target", path->str));
	ebuf_free (path);
    }

    gtk_signal_connect (GTK_OBJECT (widget), "drag_data_get",
			GTK_SIGNAL_FUNC (builtins_drag_source_get_data), node);
}


/* This one is to make it a viable destination for a dragged <drag> node */
void
rendgtk_dnd_dragtag_target_create (ENode * node, GtkWidget * widget)
{
    /* TODO: this should not be GTK_DEST_DEFAULT_ALL */
    gtk_drag_dest_set (widget,
		       GTK_DEST_DEFAULT_ALL,
		       target_table, n_targets,
		       GDK_ACTION_COPY | GDK_ACTION_MOVE);

    gtk_signal_connect (GTK_OBJECT (widget), "drag_data_received",
			GTK_SIGNAL_FUNC (builtins_drag_target_data_received),
			NULL);
    EDEBUG (
	    ("gtk-common", "Setting node %s as a drag target",
	     node->element->str));
    gtk_object_set_data (GTK_OBJECT (widget), "xml-node", node);
}


/* Set a widget as dragable */

void
rendgtk_dnd_dragable_set (ENode * node, GtkWidget * widget)
{

    gtk_drag_source_set (widget, GDK_BUTTON1_MASK | GDK_BUTTON3_MASK,
			 target_table, n_targets,
			 GDK_ACTION_COPY | GDK_ACTION_MOVE);

    /* 
     * gtk_drag_source_set_icon (widget, gtk_widget_get_colormap (window),
     * drag_icon, drag_mask);
     * 
     * gdk_pixmap_unref (drag_icon); gdk_pixmap_unref (drag_mask); */

    gtk_signal_connect (GTK_OBJECT (widget), "drag_data_get",
			GTK_SIGNAL_FUNC (builtins_drag_source_get_data), node);

    /* gtk_signal_connect (GTK_OBJECT (widget), "drag_data_delete",
     * GTK_SIGNAL_FUNC (source_drag_data_delete), NULL); */
}


/* Set as a viable target for dnd ops */
void
rendgtk_dnd_target_create (ENode * node, GtkWidget * widget)
{
    gtk_drag_dest_set (widget,
		       GTK_DEST_DEFAULT_ALL,
		       target_table, n_targets,
		       GDK_ACTION_COPY | GDK_ACTION_MOVE);

    gtk_signal_connect (GTK_OBJECT (widget), "drag_data_received",
			GTK_SIGNAL_FUNC (builtins_drag_target_data_received),
			NULL);

    gtk_object_set_data (GTK_OBJECT (widget), "xml-node", node);
}


static gint
rendgtk_widget_containerbox_label_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *box;
    GtkWidget *label;
    EBuf *styleval;

    box = enode_get_kv (node, "bottom-widget");
    label = enode_get_kv (node, "bottom-widget-label");

    if (!box)			/* DON'T TEST LABEL! It doesn't have to be
				 * created yet. */
	return FALSE;

    /* Unshow the label if its set to "". */
    if (ebuf_empty (value) && label) {
	gtk_widget_hide (GTK_WIDGET (label));
	return (TRUE);
    }

    if (!label) {
	label = gtk_label_new (value->str);

	enode_set_kv (node, "bottom-widget-label", label);
	gtk_box_pack_start (GTK_BOX (box), label, 0, 0, 0);
    } else {
	gtk_label_set_text (GTK_LABEL (label), value->str);
    }

    /* Using the node's style="" set the label's style. */
    styleval = enode_attrib (node, "style", NULL);
    if (ebuf_not_empty (styleval)) {
	GtkStyle *style;

	style = rendgtk_style_parser (styleval, rendgtk_rc_get_style (label));

	/* apply the new style. */
	gtk_widget_set_style (GTK_WIDGET (label), style);
    }

    gtk_widget_show (label);

    return (TRUE);
}

void
widget_containerbox_child_attr_set (ENode * parent_node,
				    ENode * child_node, EBuf * attr,
				    EBuf * value)
{
    EBuf *expandv;
    gint expand = BOX_PACK_EXPAND_DEFAULT;
    EBuf *fillv;
    gint fill = BOX_PACK_FILL_DEFAULT;
    EBuf *paddingv;
    guint padding = BOX_PACK_PADDING_DEFAULT;
    GtkWidget *box;
    GtkWidget *child_widget;

    box = enode_get_kv (parent_node, "bottom-widget");
    child_widget = enode_get_kv (child_node, "top-widget");

    if (!box || !child_widget)
	return;

    expandv = enode_attrib (child_node, "expand", NULL);
    if (ebuf_not_empty (expandv))
	expand = erend_value_is_true (expandv);

    fillv = enode_attrib (child_node, "fill", NULL);
    if (ebuf_not_empty (fillv))
	fill = erend_value_is_true (fillv);

    paddingv = enode_attrib (child_node, "padding", NULL);
    if (ebuf_not_empty (paddingv))
	padding = erend_get_integer (paddingv);

    gtk_box_set_child_packing (GTK_BOX (box), child_widget,
			       expand, fill, padding, GTK_PACK_START);
}

void
rendgtk_containerbox_attr_register (Element * element)
{
    ElementAttr *e_attr;

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "label";
    e_attr->description = "Add a label the the widgets's bottom level box.";
    e_attr->value_desc = "string";
    /* e_attr->possible_values = ""; */
    e_attr->set_attr_func = rendgtk_widget_containerbox_label_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "expand";
    e_attr->description =
	"Toggle whether the widgets area should 'expand' if given space to do so.";
    e_attr->value_desc = "boolean";
    e_attr->possible_values = "true,false";
    e_attr->set_child_attr_func = widget_containerbox_child_attr_set;
    element_register_child_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "fill";
    e_attr->description =
	"Toggle whether the widget itself should 'fill' any extra space given to it";
    e_attr->value_desc = "boolean";
    e_attr->possible_values = "false,true";
    e_attr->set_child_attr_func = widget_containerbox_child_attr_set;
    element_register_child_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "padding";
    e_attr->description = "Amount of padding in pixels to place around widget";
    e_attr->value_desc = "integer";
    e_attr->possible_values = "0,*";
    e_attr->set_child_attr_func = widget_containerbox_child_attr_set;
    element_register_child_attrib (element, e_attr);
}

static gint
rendgtk_widget_idle_visible_callback (gpointer user_data)
{
    GtkWidget *widget;
    ENode *node = user_data;

    widget = enode_get_kv (node, "top-widget");
    if (widget) 
      {
        EBuf *value;
        
        value = enode_attrib (node, "visible", NULL);

        if ((ebuf_empty (value)) || erend_value_is_true (value))
            gtk_widget_show (widget);
        else 
            gtk_widget_hide (widget);
      }

    /* We upped the refcount to make sure it doesn't go away before this
     * function comes around.  Decrement now. */
    enode_unref (node);

    return FALSE;
}

void
rendgtk_widget_idle_visible (ENode *node)
{
    /* Gets decremented when the window displays. */
    enode_ref (node);
    gtk_timeout_add (10, rendgtk_widget_idle_visible_callback, node);
}

/* only show if visible attribute is not "false" */
void
rendgtk_show_cond (ENode * node, GtkWidget * widget)
{
    EBuf *value;

    value = enode_attrib (node, "visible", NULL);

    if ((ebuf_empty (value)) || erend_value_is_true (value))
	gtk_widget_show (widget);
}

/* Default destroy for gtk widgets */
void
rendgtk_element_destroy (ENode * node)
{
    GtkWidget *top = enode_get_kv (node, "top-widget");
    GtkWidget *extra = enode_get_kv (node, "extra-destroy-widget");

    EDEBUG (("gtk-common", "destroying widgets associated with node %s.%s",
	     node->element->str, enode_attrib_str (node, "name", NULL)));

    if (top) {
	EDEBUG (("gtk-common", "destroying widget %p, of type %s",
		 top, gtk_type_name (GTK_WIDGET_TYPE (top))));
	gtk_widget_destroy (top);
	enode_set_kv (node, "top-widget", NULL);
	enode_set_kv (node, "bottom-widget", NULL);
    }

    if (extra) {
	gtk_widget_destroy (extra);
	enode_set_kv (node, "extra-destroy-widget", NULL);
    }
}

/* This box_pack is used mostly for inside other widgets */
void
rendgtk_box_pack (ENode * parent_node, ENode * child_node)
{
    gint fill = BOX_PACK_FILL_DEFAULT;
    EBuf *fillv;
    gint expand = BOX_PACK_EXPAND_DEFAULT;
    EBuf *expandv;
    gint padding = BOX_PACK_PADDING_DEFAULT;
    EBuf *paddingv;
    GtkWidget *child;
    GtkWidget *parent;

    child = enode_get_kv (child_node, "top-widget");
    parent = enode_get_kv (parent_node, "bottom-widget");

    if (!child || !parent)
	return;

    if (GTK_IS_WINDOW (child))
	return;

    expandv = enode_attrib (child_node, "expand", NULL);
    if (ebuf_not_empty (expandv))
	expand = erend_value_is_true (expandv);

    fillv = enode_attrib (child_node, "fill", NULL);
    if (ebuf_not_empty (fillv))
	fill = erend_value_is_true (fillv);

    paddingv = enode_attrib (child_node, "padding", NULL);
    if (ebuf_not_empty (paddingv))
	padding = erend_get_integer (paddingv);

    gtk_box_pack_start (GTK_BOX (parent), child, expand, fill, padding);
}

/* All init and mainloop code has been moved into gtk-mainloop.c MW. */
