/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: oxyd.cc,v 1.68.2.2 2003/10/06 19:30:49 dheck Exp $
 */

/*
 * This files contains the interface between Enigma and the original
 * Oxyd games. Stuff like converting level descriptions or sound
 * effects is done here.
 */

#include "px/tools.hh"

#include "oxyd.hh"
#include "enigma.hh"
#include "sound.hh"
#include "objects.hh"
#include "world.hh"
#include "display.hh"
#include "system.hh"

#include "oxydlib/DatFile.h"
#include "oxydlib/FileUtils.h"
#include "oxydlib/Level.h"
#include "oxydlib/enigma_mapping.h"

#include <string>
#include <cstdio>
#include <cstdlib>
#include <cassert>
#include <iostream>

using namespace std;
using namespace enigma;
using namespace enigma::oxyd;
using world::Stone;
using world::MakeStone;
using world::Item;
using world::MakeItem;
using namespace OxydLib;

static enigma::Direction 
direction_oxyd2enigma(OxydLib::Direction odir) 
{
    switch (odir) {
    case Direction_Up:    return NORTH; break;
    case Direction_Down:  return SOUTH; break;
    case Direction_Left:  return WEST; break;
    case Direction_Right: return EAST; break;
    default :
        fprintf(stderr, "Unknown OxydLib-direction %i!\n", int(odir));
        return NODIR;
    }
}

//----------------------------------------
// Oxyd level packs
//----------------------------------------
namespace
{
    class LevelPack_Oxyd : public LevelPack {
    public:
        LevelPack_Oxyd (OxydVersion ver, DatFile *dat, time_t created,
                        int         idx_start, int idx_end, bool twoplayers);

        virtual ~LevelPack_Oxyd() = 0; // class is interface!

        // LevelPack interface
        virtual string get_name() const;
        size_t size() const { return nlevels; }

        bool load_level (size_t index);

        const LevelInfo *get_info (size_t index);

        time_t get_modtime(size_t /*index*/) { // filedate of levelpack
            return m_created;
        }

        int get_default_SoundSet() const { return m_version+2; }

        bool needs_twoplayers() const {
            // fprintf(stderr, "m_twoplayers=%i\n", int(m_twoplayers));
            return m_twoplayers;
        }

        bool may_have_previews() const { return false; }

    protected:
        /* Conversion tables from Oxyd bytecodes to Enigma object names. */
        enigma_mapping::Mapping m_mapping;

        virtual Stone *make_stone (int type, int x, int y);
        virtual Item *make_item (int type);

        Level *get_level() const { return m_level; }

        Stone *makeLaser(int type);

    private:
        // Private methods
        void load_floor (const Level &level);
        void load_items (const Level &level);
        void load_stones (const Level &level);
        void load_actors (const Level &level);
        void scramble_puzzles (const Level &level);
        void connect_signals (const Level &level);

        GameType get_gametype() const;

        // Variables
        OxydVersion  m_version;
        DatFile     *m_datfile; // just a reference (owned by GameInfo)
        time_t       m_created; // creation date of datafile
        Level       *m_level;   // Level currently being loaded
        bool         m_twoplayers; // true -> twoplayer game

        int m_index_start; // first index of this level pack
        int level_index[200];
        int nlevels;
    };
}

LevelPack_Oxyd::LevelPack_Oxyd (OxydVersion ver, DatFile *dat, time_t created,
                                int idx_start, int idx_end, bool twoplayers)
: m_mapping(ver), m_version(ver), m_datfile(dat),
  m_created(created), m_twoplayers(twoplayers),
  m_index_start(idx_start)
{
    string msg;
    Level level;
    nlevels = 0;
    for (int i = idx_start; i <= idx_end; i++)
    {
        if (parseLevel (m_datfile->getLevel(i), &level, &msg))
        {
            if (!level.isEmpty())
            {
                level_index[nlevels] = i;
                nlevels++;
            }
        }
        else {
            Log << "Error parsing level at index " << i << ": " << msg << endl;
        }
    }
    Log << "Levelpack '" << get_name() << "' has " << nlevels << " levels." << endl;
}

LevelPack_Oxyd::~LevelPack_Oxyd()
{}

GameType
LevelPack_Oxyd::get_gametype() const 
{
    GameType typ = GAMET_UNKNOWN;
    switch (m_version) {
    case OxydVersion_Oxyd1: typ = GAMET_OXYD1; break;
    case OxydVersion_OxydMagnum:
    case OxydVersion_OxydMagnumGold: typ = GAMET_OXYDMAGNUM; break;
    case OxydVersion_OxydExtra: typ = GAMET_OXYDEXTRA; break;
    case OxydVersion_PerOxyd: typ = GAMET_PEROXYD; break;
    default :
        assert(0);
        break;
    }
    return typ;
}

string
LevelPack_Oxyd::get_name() const
{
    static const char *names1p[] = {
        "Oxyd 1", "Oxyd magnum", "Magnum Gold", "Per.Oxyd", "Oxyd extra"
    };
    static const char *names2p[] = {
        "Oxyd 1 (2p)", "", "", "Per.Oxyd (2p)", "Oxyd extra (2p)"
    };
    return level_index[0]>99 ? names2p[m_version] : names1p[m_version];
}

void
LevelPack_Oxyd::load_floor (const Level &level)
{
    using namespace world;

    const Grid &grid = level.getGrid (GridType_Surfaces);
    for (unsigned y=0; y<grid.getHeight(); ++y) {
        for (unsigned x=0; x<grid.getWidth(); ++x) {
            int         code = grid.get(x,y);
            const char *name = m_mapping.floorName(code);
            Floor      *fl;

            if( name == 0) {
                // fprintf(stderr, "Unknown floor %X\n",code);
                fl = MakeFloor("fl-dummy");
                fl->set_attrib("code", code);
            }
            else {
                fl = MakeFloor(name);
            }

            SetFloor (GridPos(x, y), fl);
        }
    }
}

Item *
LevelPack_Oxyd::make_item (int type)
{
    using namespace world;

    Item *it = 0;

    switch (type) {
    case 0x00: break;           // ignore
    case 0x02:                  // note 1
        it = MakeItem ("it-document");
        it->set_attrib ("text", m_level->getNoteText(0, Language_English).c_str());
        break;
    case 0x03:                  // note 2
        it = MakeItem ("it-document");
        it->set_attrib ("text", m_level->getNoteText(1, Language_English).c_str());
        break;

    default:
        {
            const char *name = m_mapping.itemName(type);
            if( name == 0) {
                // fprintf(stderr, "Unknown item %X\n",type);
                it = MakeItem ("it-dummy");
                it->set_attrib("code", type);
            }
            else
                it = MakeItem (name);
        }
    }
    return it;
}

void
LevelPack_Oxyd::scramble_puzzles (const Level &level) {
    int count = level.getNumScrambleItems();

    for (int i = 0; i<count; ++i) {
        const ScrambleItem& si = level.getScrambleItem(i);
        world::AddScramble(GridPos(si.getX(), si.getY()),
                           direction_oxyd2enigma(si.getDir()));
    }
}

void
LevelPack_Oxyd::load_items (const Level &level)
{
    const Grid &grid = level.getGrid (GridType_Objects);
    for (unsigned y=0; y<grid.getHeight(); ++y)
        for (unsigned x=0; x<grid.getWidth(); ++x)
            if (Item *it = make_item(grid.get(x,y)))
                SetItem (GridPos(x, y), it);
}

Stone *
LevelPack_Oxyd::makeLaser(int type) {
    assert(type >= 0 && type <= 2); // oxyd supports three different lasers per level
    const Laser& laser = get_level()->getLaser(type);

    enigma::Direction dir = direction_oxyd2enigma(laser.getDir());
    bool              on  = laser.getOn();

    Stone *st = 0;
    if (dir != NODIR) {
        string lasername("st-laser");
        lasername += to_suffix(dir);
        st         = MakeStone(lasername.c_str());
        st->set_attrib("on", Value(on)); // OnOffStone attribute
    }
    return st;
}

Stone *
LevelPack_Oxyd::make_stone (int type, int /*x*/, int /*y*/)
{
    using namespace world;

    Stone *st = 0;
    char buf[20] = "";

    switch (type) {
    case 0x00: return 0;  // ignore

    case 0x01: case 0x02: case 0x03: case 0x04:
    case 0x05: case 0x06: case 0x07: case 0x08:
    case 0x09: case 0x0a: case 0x0b: case 0x0c:
    case 0x0d: case 0x0e: case 0x0f: case 0x10:
        // Oxyd stones of different colors
        st = MakeStone("st-oxyd");
        snprintf (buf, sizeof(buf), "%d", int((type-1)/2));
        st->set_attrib("color", buf);
        switch (m_version) {
            case OxydVersion_PerOxyd:
                st->set_attrib("flavor", "c");
                break;
            default :
                st->set_attrib("flavor", "a");
                break;
        }
        break;

    default:
        // No special case -> get Stone from map
        const char *name = m_mapping.stoneName(type);
        if (name == 0) {
            // fprintf (stderr, "Unknown stone %X\n", type);
            st = MakeStone ("st-dummy");
            st->set_attrib("code", type);
        }
        else if (name[0] != '\0') { // ignore if name==""
            st = MakeStone (name);
        }
    }
    return st;
}

void
LevelPack_Oxyd::load_stones (const Level &level)
{
    using namespace world;

    const Grid &grid = level.getGrid (GridType_Pieces);
    for (unsigned y=0; y<grid.getHeight(); ++y) {
        for (unsigned x=0; x<grid.getWidth(); ++x) {
            if (Stone *st = make_stone(grid.get(x,y), x, y))
                SetStone (GridPos(x, y), st);
        }
    }
}

namespace {
    void dumpUnknownObjects(const Level& level) {
        set<int> stones, items, floors;

        const Grid &sgrid = level.getGrid (GridType_Pieces);
        for (unsigned y=0; y<sgrid.getHeight(); ++y)
            for (unsigned x=0; x<sgrid.getWidth(); ++x)
                if (Stone *st = world::GetStone(GridPos(x, y)))
                    if (int code = st->int_attrib("code"))
                        stones.insert(code);

        const Grid &igrid = level.getGrid (GridType_Objects);
        for (unsigned y=0; y<igrid.getHeight(); ++y)
            for (unsigned x=0; x<igrid.getWidth(); ++x)
                if (Item *it = world::GetItem(GridPos(x, y)))
                    if (int code = it->int_attrib("code"))
                        items.insert(code);

        const Grid &fgrid = level.getGrid (GridType_Objects);
        for (unsigned y=0; y<fgrid.getHeight(); ++y)
            for (unsigned x=0; x<fgrid.getWidth(); ++x)
                if (world::Floor *fl = world::GetFloor(GridPos(x, y)))
                    if (int code = fl->int_attrib("code"))
                        floors.insert(code);

        if (!stones.empty()) {
            fprintf(stderr, "Unknown stones:");
            for (set<int>::iterator i = stones.begin(); i != stones.end(); ++i)
                fprintf(stderr, " %i", *i);
            fprintf(stderr, "\n");
        }
        if (!items.empty()) {
            fprintf(stderr, "Unknown items:");
            for (set<int>::iterator i = items.begin(); i != items.end(); ++i)
                fprintf(stderr, " %i", *i);
            fprintf(stderr, "\n");
        }
        if (!floors.empty()) {
            fprintf(stderr, "Unknown floors:");
            for (set<int>::iterator i = floors.begin(); i != floors.end(); ++i)
                fprintf(stderr, " %i", *i);
            fprintf(stderr, "\n");
        }
    }
}

namespace {

#define MAX_MARBLE_INFO_FIELDS 11
#define DEFAULT_VALUE          (-1)
#define TRACE_MARBLE_INFO // check for uninterpreted fields

    enum MarbleInfoIndices {
        // valid for all actor types:
        MI_FORCE = 0,

        // marbles only:

        // Jack/Rotor only:
        MI_RANGE = 1,
    };

    class MarbleInfo {
        int  value[MAX_MARBLE_INFO_FIELDS];
#if defined(TRACE_MARBLE_INFO)
        bool interpreted[MAX_MARBLE_INFO_FIELDS];
#endif // TRACE_MARBLE_INFO

    public:
        MarbleInfo(const Marble& marble);
        ~MarbleInfo();

        bool is_default(int idx) {
            assert(idx >= 0 && idx<MAX_MARBLE_INFO_FIELDS);
            return value[idx] == DEFAULT_VALUE;
        }

        int get_value(int idx) {
            assert(!is_default(idx)); // you cannot ask for default value (not stored here)
#if defined(TRACE_MARBLE_INFO)
            interpreted[idx] = true;
#endif // TRACE_MARBLE_INFO
            return value[idx];
        }
    };

    MarbleInfo::MarbleInfo(const Marble& marble) {
        const string& data = marble.getData(options::Difficulty == DIFFICULTY_EASY ? GameMode_Easy : GameMode_Hard);
        size_t        from = 0;
        int           idx  = 0;

        while (from != string::npos) {
            size_t par_open = data.find('(', from);
            from            = string::npos;

            if (par_open != string::npos) {
                size_t par_close = data.find(')', par_open);
                if (par_close != string::npos) {
                    from = par_close;
                    if (par_close == par_open+1) {
                        value[idx++] = DEFAULT_VALUE;
                    }
                    else {
                        value[idx++] = atoi(data.substr(par_open+1, par_close-par_open-1).c_str());
                    }
                }
                else {
                    fprintf(stderr, "Error in MarbleInfo: missing closing parenthesis\n");
                }
            }
        }
        for (; idx<MAX_MARBLE_INFO_FIELDS; ++idx)
            value[idx] = DEFAULT_VALUE;

        for (idx = 0; idx<MAX_MARBLE_INFO_FIELDS; ++idx)
            interpreted[idx] = false;
    }

    MarbleInfo::~MarbleInfo() {
#if defined(TRACE_MARBLE_INFO)
        for (int idx = 0; idx<MAX_MARBLE_INFO_FIELDS; ++idx) {
            if (!interpreted[idx] && !is_default(idx)) {
                fprintf(stderr, "MarbleInfo[%i]=%i is not used yet.\n", idx, get_value(idx));
            }
        }
#endif // TRACE_MARBLE_INFO
    }

#undef DEFAULT_VALUE
#undef MAX_MARBLE_INFO_FIELDS
};

void
LevelPack_Oxyd::load_actors (const Level &level)
{
    using namespace world;

    int nmeditationmarbles = 0;

    int nmarbles = level.getNumMarbles();
    for (int i=0; i<nmarbles; ++i) {
        const Marble &marble = level.getMarble(i);
        double        x      = marble.getX()/32.0;
        double        y      = marble.getY()/32.0;
        Actor        *ac     = 0;

        MarbleInfo minfo(marble);

        switch (marble.getMarbleType()) {
        case MarbleType_Black:
            ac = MakeActor ("ac-blackball");
            ac->set_attrib ("player", Value(0.0));
            break;
        case MarbleType_White:
            ac = MakeActor ("ac-whiteball");
            ac->set_attrib ("player", Value(1.0));
            break;
        case MarbleType_Meditation:
            ac = MakeActor ("ac-whiteball-small");
	    nmeditationmarbles += 1;
	    if (needs_twoplayers() && (nmeditationmarbles % 2)==0)
		ac->set_attrib("player", Value(1.0));
	    else
		ac->set_attrib ("player", Value(0.0));

//             if (minfo.is_default(MI_FORCE)) {
//                 ac->set_attrib("mouseforce", Value(1.0));
//             }
//             else {
//                 ac->set_attrib("mouseforce", Value(minfo.get_value(MI_FORCE) / 32.0)); // just a guess
//             }
            break;
        case MarbleType_Jack:
            ac = MakeActor ("ac-top");
            if (!minfo.is_default(MI_FORCE)) {
                double force = minfo.get_value(MI_FORCE) / 10.0 * 2.0; // just a guess
                ac->set_attrib("force", Value(force) );
                fprintf(stderr, "Set jack force to %f\n", force);
            }
            if (!minfo.is_default(MI_RANGE)) {
                double range = minfo.get_value(MI_RANGE) / 32.0; // value seems to contain distance in pixels
                ac->set_attrib("range", Value(range) );
                fprintf(stderr, "Set jack range to %f\n", range);
            }
            break;
        case MarbleType_Rotor:
            ac = MakeActor ("ac-rotor");
            if (!minfo.is_default(MI_FORCE)) {
                double force = minfo.get_value(MI_FORCE) / 10.0 * 2.0; // just a guess
                ac->set_attrib("force", Value(force) );
                fprintf(stderr, "Set rotor force to %f\n", force);
            }
            if (!minfo.is_default(MI_RANGE)) {
                double range = minfo.get_value(MI_RANGE) / 32.0; // value seems to contain distance in pixels
                ac->set_attrib("range", Value(range) );
                fprintf(stderr, "Set rotor range to %f\n", range);
            }
            break;

        case MarbleType_Horse:
            ac = MakeActor("ac-horse");
            break;
        case MarbleType_Bug:
            ac = MakeActor ("ac-bug");
            break;
        default:
            fprintf(stderr, "Unhandled actor type %i\n", int(marble.getMarbleType()));
            break;
//         case MarbleType_LifeSpitter:
//         case MarbleType_DynamiteHolder:
//             break;
        }

        if (ac) {
            AddActor (x, y, ac);
        }
    }
}

GridLoc to_gridloc (const SignalLocation &a)
{
    assert (a.getGridType() >= GridType_First &&
            a.getGridType() <= GridType_Last);
    static GridLayer tab[3] = { GRID_FLOOR, GRID_STONES, GRID_ITEMS };
    return GridLoc (tab[a.getGridType()], GridPos(a.getX(), a.getY()));
}

void
LevelPack_Oxyd::connect_signals (const Level &level)
{
    using namespace world;

    set<SignalLocation> senders;
    level.getSenders(&senders);
    set<SignalLocation>::const_iterator senderIter = senders.begin();
    set<SignalLocation>::const_iterator senderEnd = senders.end();
    for (; senderIter != senderEnd; ++senderIter) {
        const SignalLocation &sender = *senderIter;

        int nrec = level.getNumRecipients(sender);
        for (int irec=0; irec<nrec; ++irec) {
            SignalLocation recipient = level.getRecipient(sender, irec);
            GridLoc src = to_gridloc(sender);
            GridLoc dst = to_gridloc(recipient);
            world::AddSignal (src, dst, "signal");
        }
    }
}

bool
LevelPack_Oxyd::load_level (size_t index)
{
    assert(index < size());

    string msg;
    Level  level;

    if (!parseLevel (m_datfile->getLevel(level_index[index]), &level, &msg)) {
        Log << "Could not load " << get_name() << " level #" << index+1
            << ":\n  " << msg;
        return false;
    }

    m_level = &level;

    world::Reset ();
    world::Create (level.getWidth(), level.getHeight());
    display::ResizeGameArea (20, 11);

    load_floor (level);
    load_items (level);
    load_stones (level);

    dumpUnknownObjects(level);

    scramble_puzzles(level);
    load_actors (level);
    connect_signals (level);


    bool ok = world::InitWorld();
    if (level.getScrolling())
        SetFollowMode (display::FOLLOW_SCROLLING);
    else
        SetFollowMode (display::FOLLOW_SCREEN);

    m_level = 0;

    return ok;
}

const LevelInfo *
LevelPack_Oxyd::get_info (size_t index)
{
    static LevelInfo info ("oxyd", "Oxyd", "Dongleware", get_gametype());

    index = level_index[index]-m_index_start;

    char name[200];
    sprintf (name, "%s #%d", get_name().c_str(), index+1);
    info.name = name;

    char filename[200];
    sprintf (filename, "Import %s %d", get_name().c_str(), index);
    info.filename = filename;

    return &info;
}


//----------------------------------------
// Oxyd 1 level pack
//----------------------------------------
namespace
{
    class LP_Oxyd1 : public LevelPack_Oxyd {
        Stone *make_stone (int type, int x, int y);
        Item *make_item (int type);
    public:
        LP_Oxyd1 (DatFile *dat, time_t created, bool twoplayers);
    };
}

LP_Oxyd1::LP_Oxyd1 (DatFile *dat, time_t created, bool twoplayers)
: LevelPack_Oxyd (OxydVersion_Oxyd1, dat, created,
                  twoplayers ? 100 : 0,
                  twoplayers ? 199 : 99,
                  twoplayers)
{
}

Item *
LP_Oxyd1::make_item (int type)
{
    Item *it = 0;
    switch (type) {
    default:
        it = LevelPack_Oxyd::make_item(type);
        break;
    }
    return it;
}

Stone *
LP_Oxyd1::make_stone (int type, int x, int y)
{
    Stone *st = 0;
    switch (type) {
    case 0x44: case 0x45: case 0x46:
        st = makeLaser(type-0x44);
        break;
    default :
        st = LevelPack_Oxyd::make_stone(type, x, y);
        break;
    }
    return st;
}



//----------------------------------------
// OxydExtra level pack
//----------------------------------------
namespace
{
    class LP_OxydExtra : public LevelPack_Oxyd {
        Stone *make_stone (int type, int x, int y);
        Item *make_item (int type);
    public:
        LP_OxydExtra (DatFile *dat, time_t created);
    };
}

LP_OxydExtra::LP_OxydExtra (DatFile *dat, time_t created)
    : LevelPack_Oxyd (OxydVersion_OxydExtra, dat, created, 0, 99, false)
{
}

Item *
LP_OxydExtra::make_item (int type)
{
    Item *it = 0;
    switch (type) {
    default:
        it = LevelPack_Oxyd::make_item(type);
        break;
    }
    return it;
}

Stone *
LP_OxydExtra::make_stone (int type, int x, int y)
{
    Stone *st = 0;
    switch (type) {
    case 0x3e: case 0x3f: case 0x40:
        st = makeLaser(type-0x3e);
        break;
    default :
        st = LevelPack_Oxyd::make_stone(type, x, y);
        break;
    }
    return st;
}


//----------------------------------------
// PerOxyd level pack
//----------------------------------------
namespace
{
    class LP_PerOxyd : public LevelPack_Oxyd {
        Stone *make_stone (int type, int x, int y);
    public:
        LP_PerOxyd (DatFile *dat, time_t created, bool twoplayers);

    };
}

LP_PerOxyd::LP_PerOxyd (DatFile *dat, time_t created, bool twoplayers)
    : LevelPack_Oxyd (OxydVersion_PerOxyd, dat, created,
                      twoplayers ? 100 : 0,
                      twoplayers ? 199 : 99,
                      twoplayers)
{
}

Stone *
LP_PerOxyd::make_stone (int type, int x, int y)
{
    Stone *st = 0;
    switch (type) {
    case 0x39: case 0x3a: case 0x3b:
        // Create magic stones only if they are absolutely necessary
        if (get_level()->getRequireMagicPiece()) {
            const char *names[] = { "st-magic", "st-magic", "st-magic" };
            st = MakeStone (names[type - 0x39]);
        }
        break;
    case 0x3e: case 0x3f:  case 0x40:
        st = makeLaser(type-0x3e);
        break;
    default:
        st = LevelPack_Oxyd::make_stone(type, x, y);
        break;
    }
    return st;
}


//----------------------------------------
// Oxyd Magnum level pack
//----------------------------------------
namespace
{
    class LP_OxydMagnum : public LevelPack_Oxyd {
        Stone *make_stone (int type, int x, int y);
    public:
        LP_OxydMagnum(OxydVersion, DatFile *dat, time_t created);
    };
}

LP_OxydMagnum::LP_OxydMagnum(OxydVersion version, DatFile *dat, time_t created)
: LevelPack_Oxyd (version, dat, created, 0,
                  (version==OxydVersion_OxydMagnumGold) ? 120 : 99, false)
{
}

Stone *
LP_OxydMagnum::make_stone (int type, int x, int y)
{
    Stone *st = 0;
    switch (type) {
    case 0x44: case 0x45: case 0x46:
        st = makeLaser(type-0x44);
        break;
    default:
        st = LevelPack_Oxyd::make_stone(type, x, y);
        break;
    }
    return st;
}



namespace
{
    class GameInfo {
    public:
        GameInfo();
        GameInfo (OxydVersion ver_, const string &game_, const string &datfile_name_);
        ~GameInfo() {
            // /*!!*/ fixme: sth goes wrong if this is deleted. why ?
	    if (datfile)
		delete datfile;
        }

        bool is_present() const { return m_present; }
        DatFile *getDatfile() { return datfile; }

    private:
        // Variables.
        OxydVersion  ver;
        string       game;
        DatFile     *datfile;
        string       datfile_path;
        bool         m_present;
        time_t       modified;  // !m_present -> 0

        void       openDatFile();
        LevelPack *makeLevelPack(bool twoplayer);
    };
}

GameInfo::GameInfo()
: ver(OxydVersion_Invalid), datfile(0), m_present(false), modified(0)
{}


GameInfo::GameInfo (OxydVersion ver_, const string &game_, const string &datfile_name_)
: ver(ver_), game(game_), datfile(0), /*datfile_name(datfile_name_), */m_present(false), modified(0)
{
    string fname;
    if (FindFile (datfile_name_, datfile_path)) {
        enigma::Log << "Found " << game << " data file\n";
        m_present = true;
        openDatFile();

        if (m_present) {
            if (LevelPack *lp = makeLevelPack(false))
                enigma::LevelPacks.push_back(lp);
            if (LevelPack *lp = makeLevelPack(true))
                enigma::LevelPacks.push_back(lp);
        }
    }
}

void GameInfo::openDatFile()
{
    assert(m_present);

    ByteVec data;
    readFile (datfile_path, &data);

    datfile  = new DatFile;
    modified = sysdep::FileModTime(datfile_path);

    string errmsg;
    if (!parseDatFile (data, ver, datfile, &errmsg)) {
        enigma::Log << "Error loading " << datfile_path << ": " << errmsg << endl;
        delete datfile;
        datfile    = 0;
        m_present = false;
    } else {
        enigma::Log << "Loaded "<< datfile_path << endl;
    }
}

LevelPack *GameInfo::makeLevelPack(bool twoplayers)
{
    if (datfile == 0 || ver == OxydVersion_Invalid)
        return 0;

    if (twoplayers && (ver == OxydVersion_OxydExtra ||
                       ver == OxydVersion_OxydMagnum ||
                       ver == OxydVersion_OxydMagnumGold))
        return 0;           // no twoplayer levels available

    switch (ver) {
    case OxydVersion_Oxyd1:
        return new LP_Oxyd1 (datfile, modified, twoplayers);
    case OxydVersion_OxydExtra:
        return new LP_OxydExtra(datfile, modified);
    case OxydVersion_PerOxyd:
        return new LP_PerOxyd (datfile, modified, twoplayers);
    case OxydVersion_OxydMagnum:
    case OxydVersion_OxydMagnumGold:
        return new LP_OxydMagnum (ver, datfile, modified);
    default:
        assert(0);
        break;
            //         {
            //             int firstlevel = twoplayers ? 100 : 0;
            //             int lastlevel = twoplayers ? 199 : 99;
            //             return new LevelPack_Oxyd(ver, datfile, modified, firstlevel, lastlevel, twoplayers);
            //         }
    }

}

//----------------------------------------
// Local variables
//----------------------------------------
namespace
{
    vector<GameInfo*> games;

    int active_soundset              = 1; // 1: enigma  2..: OxydVersion+2;
    DatFile *active_soundset_datfile = 0;

    map <string,string> soundfx_map;

    struct {
        const char *name, *datfile;
        int start1p, end1p;
        int start2p, end2p;
    } oxversions[] = {
        {"Oxyd 1",           "oxyd1ibm.dat", 0, 99, 100, 199},
        {"Oxyd magnum",      "oxydmibm.dat", 0, 99,  -1,  -1},
        {"Oxyd magnum gold", "oxydmgg.dat",  0, 119, -1,  -1},
        {"Oxyd extra",       "oxydex.dat",   0, 99,  -1,  -1},
        {"Per.Oxyd",         "peroxyd.dat",  0, 99, 100, 199}
    };
}

void
enigma::oxyd::Init()
{
    games.clear();
    games.resize(OxydVersion_Count);

    games[OxydVersion_Oxyd1]          = new GameInfo(OxydVersion_Oxyd1,          "Oxyd 1",           "oxyd1ibm.dat");
    games[OxydVersion_OxydMagnum]     = new GameInfo(OxydVersion_OxydMagnum,     "Oxyd magnum",      "oxydmibm.dat");
    games[OxydVersion_OxydMagnumGold] = new GameInfo(OxydVersion_OxydMagnumGold, "Oxyd magnum gold", "oxydmgg.dat");
    games[OxydVersion_OxydExtra]      = new GameInfo(OxydVersion_OxydExtra,      "Oxyd extra",       "oxydex.dat");
    games[OxydVersion_PerOxyd]        = new GameInfo(OxydVersion_PerOxyd,        "Per.Oxyd",         "peroxyd.dat");
}

void enigma::oxyd::Shutdown()
{
    px::delete_sequence(games.begin(), games.end());
}

bool
enigma::oxyd::FoundOxyd (OxydVersion ver)
{
    return games[ver]->is_present();
}


void
enigma::oxyd::ChangeSoundset (int sound_set, int default_sound_set)
{
    static int last_default_sound_set = 1;

    // if called without knowing default sound set
    // take last default or default to enigma
    // (e.g when called from option menu from inside game)
    if (default_sound_set == -1)
        default_sound_set = last_default_sound_set;
    else
        last_default_sound_set = default_sound_set;

    if (sound_set == 0) {       // use specific soundset for each levelpack
        sound_set = default_sound_set;
    }

    if (sound_set == active_soundset) {
        return;
    }

    // reset to enigma soundset
    soundfx_map.clear();
    active_soundset         = 1;
    active_soundset_datfile = 0;
    sound::ClearSoundCache();

    if (sound_set == 1) {       // enigma -> no mapping
        return;
    }

    OxydVersion ver = OxydVersion(sound_set-2);
    GameInfo&   gi  = *(games[ver]);

    if (!gi.is_present())
        return;                 // not installed -> use enigma soundset

    active_soundset         = sound_set;
    active_soundset_datfile = gi.getDatfile();

    soundfx_map["intro"]         = "OXINTRO.SDD";
    soundfx_map["exit"]          = "OXEXIT.SDD";
    soundfx_map["finished"]      = "OXFINITO.SDD";

    soundfx_map["boink"]         = "OXJUMP.SDD";
    soundfx_map["st-oxydopen"]   = "OXMEMOP.SDD";
    soundfx_map["st-oxydopened"] = "OXMEMOK.SDD";
    soundfx_map["st-oxydclose"]  = "OXMEMCL.SDD";
    soundfx_map["st-mirrorturn"] = "OXTURN.SDD";
    soundfx_map["st-coinslot"]   = "OXMONEY.SDD";
    soundfx_map["st-laseron"]    = "OXLASER.SDD";
    soundfx_map["dooropen"]      = "OXMOTOR.SDD";
    soundfx_map["doorclose"]     = "OXMOTOR.SDD";
    soundfx_map["fart"]          = "OXUNTITL.SDD";
    soundfx_map["pickup"]        = "OXINVENT.SDD";
    soundfx_map["invrotate"]     = "OXINVROT.SDD";
    soundfx_map["it-triggerdown"]= "OXSWON.SDD";
    soundfx_map["it-triggerup"]  = "OXSWOFF.SDD";
    soundfx_map["shatter"]       = "OXKLIRR.SDD";

    soundfx_map["explosion2"]    = "OXCRASH1.SDD";
    soundfx_map["explosion1"]    = "OXCRASH2.SDD";
    soundfx_map["impulse"]       = "OXWOUOU.SDD";
    soundfx_map["st-move"]       = "OXMOVE.SDD";
    soundfx_map["rotate-right"]  = "OXMAGIC4.SDD";
    soundfx_map["rotate-left"]   = "OXMAGIC3.SDD";
    soundfx_map["st-magic"]      = "OXMAGIC.SDD";
    soundfx_map["puller"]        = "OXPULLER.SDD";

    soundfx_map["st-stone"]      = "OXKLICK1.SDD";
    soundfx_map["st-metal"]      = "OXKLICK2.SDD";
    soundfx_map["st-thud"]       = "OXKLICK4.SDD";
    soundfx_map["st-thud"]       = "OXKLICK5.SDD";
    soundfx_map["electric"]      = "OXKLICK6.SDD";
    soundfx_map["drown"]         = "OXBLOOP.SDD";
    soundfx_map["thief"]         = "OXTHIEF.SDD";
    soundfx_map["warp"]          = "OXTRANS.SDD";
    soundfx_map["crack"]         = "OXCRACK.SDD";
    soundfx_map["squish"]        = "OXMATSCH.SDD";
    soundfx_map["snip"]           = "OXCUT.SDD";

    // ???
    soundfx_map["klick3"]        = "OXKLICK3.SDD";
    soundfx_map["drop"]          = "OXDROP.SDD";
    soundfx_map["boing"]         = "OXBOING.SDD";
    soundfx_map["bold"]          = "OXBOLD.SDD";
    soundfx_map["magic2"]        = "OXMAGIC2.SDD";
    soundfx_map["magic3"]        = "OXMAGIC3.SDD";
}


Mix_Chunk *
enigma::oxyd::LoadSound (const std::string &name)
{
    Mix_Chunk *chunk = 0;
    if (active_soundset_datfile) {
        string         chunkname = soundfx_map[name];
        const ByteVec *snddata   = active_soundset_datfile->getChunk(chunkname);

        if (snddata) {
            enigma::Log << "Loaded sound file " << name << " =^= " << chunkname<<endl;
            const int offset = 16;
            return sound::ChunkFromRaw (&(*snddata)[0], snddata->size()-offset,
                                        6000, AUDIO_S8, 1);
        }
    }
    return chunk;
}
