// This file is part of Eigen, a lightweight C++ template library
// for linear algebra. Eigen itself is part of the KDE project.
//
// Copyright (C) 2006-2007 Benoit Jacob <jacob@math.jussieu.fr>
//
// Eigen is free software; you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 or (at your option) any later version.
//
// Eigen is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Eigen; if not, write to the Free Software Foundation, Inc., 51
// Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
//
// As a special exception, if other files instantiate templates or use macros
// or inline functions from this file, or you compile this file and link it
// with other works to produce a work based on this file, this file does not
// by itself cause the resulting work to be covered by the GNU General Public
// License. This exception does not invalidate any other reasons why a work
// based on this file might be covered by the GNU General Public License.

#include "main.h"
#include "../src/linearsolver.h"

template< typename T, int Size, typename LinearSolverType,
          typename MatrixType, typename VectorType >
void helper_checkSolver( const char *strType )
{
    MatrixType lhs(Size);
    VectorType rhs(Size);
    do pickRandomMatrix(lhs); while( !lhs.isInvertible() );
    do pickRandomVector(rhs); while( rhs.norm() == static_cast<T>(0));
    LinearSolverType solver1( lhs, rhs );
    QVERIFY( solver1.thereExistSolutions() );
    QVERIFY( solver1.dimSolutions() == 0 );
    QVERIFY( compareVectors( lhs * solver1.someSolution(), rhs, strType ) );

    VectorType row(Size);
    row.loadZero();
    int dep = rand() % Size;
    lhs.setRow( dep, row );
    rhs(dep) = static_cast<T>(0);
    for( int j = 0; j < Size; j++ )
    {
        if( j != dep )
        {
            T t;
            pickRandom( t );
            VectorType deprow;
            deprow = lhs.row( dep );
            deprow += t * lhs.row( j );
            lhs.setRow( dep, deprow );
            rhs( dep ) += t * rhs( j );
        }
    }
    LinearSolverType solver2( lhs, rhs );
    if( ! solver2.thereExistSolutions() )
    {
        qDebug() << strType << endl;
    }
    QVERIFY( solver2.thereExistSolutions() );
    QVERIFY( solver2.dimSolutions() >= 1 );
    if( solver2.dimSolutions() > 1 ) return;
    QVERIFY( compareVectors( lhs * solver2.someSolution(), rhs, strType ) );
    T param;
    pickRandom( param );
    QVERIFY( compareVectors( lhs * solver2.genericSolution( &param ), rhs, strType ) );
}

void MainTest::checkSolver()
{
#define HELPER_CHECKSOLVER_FIX( T, Size ) \
    helper_checkSolver< T, Size, LinearSolver< T, Size >,\
        Matrix< T, Size >, Vector< T, Size > >\
            ("fixed<"__STRING(T)","__STRING(Size)">")

#define HELPER_CHECKSOLVER_DYN( T, Size ) \
    helper_checkSolver< T, Size, LinearSolverX<T>,\
        MatrixX<T>, VectorX<T> >\
            ("dynamic<"__STRING(T)">("__STRING(Size)")")

    for( int repeat = 0; repeat < REPEAT; repeat++ )
    {
        HELPER_CHECKSOLVER_FIX( float, 1 );
        HELPER_CHECKSOLVER_FIX( double, 2 );
        HELPER_CHECKSOLVER_FIX( float, 3 );
        HELPER_CHECKSOLVER_FIX( double, 4 );
        HELPER_CHECKSOLVER_FIX( double, 8 );
        HELPER_CHECKSOLVER_FIX( double, 12 );
        HELPER_CHECKSOLVER_FIX( complex<double>, 7 );
        HELPER_CHECKSOLVER_FIX( complex<float>, 4 );
        HELPER_CHECKSOLVER_FIX( complex<double>, 12 );
    
        HELPER_CHECKSOLVER_DYN( float, 1 );
        HELPER_CHECKSOLVER_DYN( double, 2 );
        HELPER_CHECKSOLVER_DYN( float, 3 );
        HELPER_CHECKSOLVER_DYN( double, 4 );
        HELPER_CHECKSOLVER_DYN( double, 8 );
        HELPER_CHECKSOLVER_DYN( double, 12 );
        HELPER_CHECKSOLVER_DYN( complex<double>, 7 );
        HELPER_CHECKSOLVER_DYN( complex<float>, 4 );
        HELPER_CHECKSOLVER_DYN( complex<double>, 12 );
    }
}
