/* Drip - a transcoder for Unix
 * Copyright (C) 2001-2003 Jarl van Katwijk
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "../config.h"
#include <gnome.h>
#include "process.h"
#include "callbacks.h"
#include "interface.h"
#include "support.h"
#include "drip.h"
#include <pthread.h>
#include <sys/stat.h>
#include <sys/statfs.h>
#include <sys/time.h>
#include <unistd.h>
#include <sys/types.h>
#include <fcntl.h>
#include <dvddb.h>
#include <sys/ioctl.h>
#include <inttypes.h>
#include <ctype.h>
#include <sys/mount.h>
#include "cdinfo_ext.h"
#include <errno.h>
#include <assert.h>
#include "../encoder/external.hh"
/* libdvdread includes */
#include <dvdread/dvd_reader.h>
#include <dvdread/ifo_types.h>
#include <dvdread/ifo_read.h>
#include <dvdread/nav_read.h>
#include <dvdread/nav_print.h>


#define EXT_START_CODE     0x1B5L
#define SEQ_START_CODE     0x1B3L

#define SEQ_ID       1
#define DISP_ID      2

//#define DEBUG

/* GLOBALS */
gboolean converting = FALSE;
gboolean paused = FALSE;
gboolean cacheing = FALSE;
gint selected_files = -1;
gint largest_vob_cached = 0;
gint titleset_largest;
u_int spu_clut[32]; // 0-15: BGR colour, 16-32: YCRCB colour
GString *cached_files;
gint cached_files_list = 0;
GString *cached_file_list[1000];
gint cached_file_list_title[1000];
gdouble CALCframerate = 25.00;
void _detect_largest_title(void);
gboolean quit_converting;

/* converting thread */
pthread_t ripthread;
gint ripthread_id;
#ifdef LM_SENSORS
/* heat monitor thread */
pthread_t heatthread;
gint heatthread_id;
#endif /* LM_SENSORS */



/* FUNTIONS */

/* Returns the 1st selected titel, used for audio\subpuctures info, or -1 */
gint first_selected_titel() {
    gint i = 0;

    while (i<RANGE) {
        if (Config.conversion_mode == DVDTITLE_DVDDISK || Config.conversion_mode == DVDTITLE_DVDIMAGE) {
            if (Config.selected_chapters[i]!=NULL) {
                return ((selected_chapters_t*)(Config.selected_chapters[i]->data))->PC;
            }
        } else {
            if (Config.CS[i] == TRUE) {
                return i;
            }
        }
        i++;
    }
    /* Nothing found.. */
    return -1;
}

/* Full language naming out 2char name */
gchar *fulllanguage(uint16_t descr) {
    gint i;
    for (i=0; lang_tbl[i].lang_long[0]; i++) {
        if (!memcmp (&descr, &lang_tbl[i].descr, 2)) {
            return (char *) lang_tbl[i].lang_long;
        }
    }
    return lang_tbl[i].lang_long;
}

/* Fetch CLUT;
	spu_clut 0 -> 15 contain RGB colours,
	spu_clut 16 -> 31 YCbCr colours */
gboolean parse_clut(ifo_handle_t *ifo,gint DVDtitle,gboolean lock) {
    gulong R,G,B,Y,CR,CB;
    gdouble Yf,CRf,CBf,Rf,Gf,Bf;
    const pgc_t *pgc;
    gint clutcounter = DVDtitle-1;
    gint num,fd;
    guint result;
    GIOChannel *clutfile_channel;
    pgc = ifo->vts_pgcit->pgci_srp[0].pgc;
    for (num=0;num<16;num++) {
         Yf = ((pgc->palette[num])>>16)&0xff;Y = (gulong)Yf;
         CBf = ((pgc->palette[num])>>8)&0xff;CB = (gulong)CBf;
         CRf = ((pgc->palette[num])>>0)&0xff;CR = (gulong)CRf;
         spu_clut[num+16] = (Y) + (CR<<8) + (CB<<16); // YCRCB
         // YUV = YCbCr
         Bf = 1.164*(Yf-16)+2.018*(CBf-128);
         Gf = 1.164*(Yf-16)-0.813*(CRf-128)-0.391*(CBf-128);
         Rf = 1.164*(Yf-16)+1.596*(CRf-128);

         B = (uint)(Bf) & 255;
         G = (uint)(Gf) & 255; 
         R = (uint)(Rf) & 255;
         spu_clut[num] = (R) + (G<<8) + (B<<16); // BGR 24bits
    }

    /* Write out clut data */
    g_string_sprintf(Config.clutfile,"%s/CLUT%i.tbl",Config.CACHElocation->str,clutcounter);
    if (lock) {
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Writing CLUT file %s",Config.clutfile->str);
    } else {
        g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Writing CLUT file %s",Config.clutfile->str);
    }

    fd = open(Config.clutfile->str,O_WRONLY|O_CREAT,S_IREAD|S_IWRITE);
    if (fd==-1) {
        g_log(DRIP_CB_LD,G_LOG_LEVEL_WARNING,"Writing CLUT file %s:  %s",Config.clutfile->str,strerror(errno));
    }

    clutfile_channel = g_io_channel_unix_new(fd);
    for (num=0;num<32;num++) {
        g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"parse_clut: I420 Colour %i : %x",num,spu_clut[num]);
        g_io_channel_write(clutfile_channel, (gchar*)&spu_clut[num],sizeof(u_int),&result);
    }

    /* Clean & Exit */
    g_io_channel_close(clutfile_channel);
    close(fd);
    return TRUE;
}


/* Parse IFO file, for now it handles:
     * - audio channels + language code
     * - subpicture + language code
     *
     * TODO: angles\others..
     */
gboolean parse_ifo(gint DVDtitleset,gboolean lock,gboolean save_clut) {
    ifo_handle_t *ifo;
    gint i;
    audio_attr_t *attr = NULL;
    subp_attr_t *attr_sub = NULL;
    GString *multiple = NULL;
    GString *description = g_string_new("");

    /* Open titleset ifo */
    ifo = ifoOpen (Config.DVDhandle, DVDtitleset);
    if (ifo == NULL) {
        if (lock) {
            g_log(DRIP_LD,G_LOG_LEVEL_ERROR,"Error initializing ifo for titleset %i",DVDtitleset);
        } else {
            g_log(DRIP_CB_LD,G_LOG_LEVEL_ERROR,"Error initializing ifo for titleset %i",DVDtitleset);
        }
    }
    if (lock) {
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Opened ifo for titleset %i",DVDtitleset);
    } else {
        g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Opened ifo for titleset %i",DVDtitleset);
    }
    if (ifo->vtsi_mat) {
        multiple = g_string_new("");
 
        /* Aspect */
        switch(ifo->vtsi_mat->vts_video_attr.display_aspect_ratio) {
        case 0:
            Config.VTS_Aspects[DVDtitleset-1] = (gdouble)4/(gdouble)3;
            break;
        case 3:
            Config.VTS_Aspects[DVDtitleset-1] = (gdouble)16/(gdouble)9;
            break;
        default:
            Config.VTS_Aspects[DVDtitleset-1] = 0;
            break;
        }

        /* Audio */
        Config.audio_channels_in_dvd[DVDtitleset] = ifo->vtsi_mat->nr_of_vts_audio_streams;
        if (Config.audio_channels_in_dvd[DVDtitleset] > 0) {
            /* Fillin audio channels names and types */
            for (i=Config.audio_channels_in_dvd[DVDtitleset]-1;i>-1;i--) {
                attr = &ifo->vtsi_mat->vts_audio_attr[i];
                /* Audio language */
                switch(attr->lang_extension) {
                case 0:
                    g_string_free(description,TRUE);description = g_string_new("");
                    break;
                case 1: // Normal audio
                    g_string_sprintf(description,"Normal Caption - ");
                    break;
                case 2: // visually imparied
                    g_string_sprintf(description,"Audio for visually impaired - ");
                    break;
                case 3: // Directors 1
                    g_string_sprintf(description,"Director's comments 1 - ");
                    break;
                case 4: // Directors 2
                    g_string_sprintf(description,"Director's comments 2 - ");
                    break;
                default:
                    break;
                }
                /* Audio type */
                Config.audio_channel_types[DVDtitleset][i] = attr->audio_format;
                g_string_sprintf(Config.audio_channel_names[DVDtitleset][i],"%s%s",description->str,fulllanguage((attr->lang_code>>8)+((attr->lang_code&0xff)<<8)));
            }
        }
        /* Display result and set Config */
        if (Config.audio_channels_in_dvd[DVDtitleset]>1 || Config.audio_channels_in_dvd[DVDtitleset]==0) {
            g_string_sprintf(multiple,"%s","s");
        } else {
            g_string_sprintf(multiple,"%s","");
        }
        if (lock) {
            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Stream contains %i audio channel%s",Config.audio_channels_in_dvd[DVDtitleset],multiple->str);
        } else {
            g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Stream contains %i audio channel%s",Config.audio_channels_in_dvd[DVDtitleset],multiple->str);
        }

        /* -- Number of subpicture streams -- */
        Config.subpictures_in_dvd[DVDtitleset] = ifo->vtsi_mat->nr_of_vts_subp_streams;
        /* check if subpictures are valid */
        for (i=Config.subpictures_in_dvd[DVDtitleset]-1;i>-1;i--) {
            attr_sub = &ifo->vtsi_mat->vts_subp_attr[i];
            if(isalpha((int)(attr_sub->lang_code >> 8)) && isalpha((int)(attr_sub->lang_code & 0xff))) {
                g_string_sprintf(Config.subpicture_names[DVDtitleset][i],"%s", fulllanguage((attr_sub->lang_code>>8)+((attr_sub->lang_code&0xff)<<8)));
            } else {
                g_string_sprintf(Config.subpicture_names[DVDtitleset][i],"%02x%02x ", 0xff & (unsigned)(attr_sub->lang_code >> 8),0xff & (unsigned)(attr_sub->lang_code & 0xff));
            }
        }
        /* Display result and set Config */
        if (Config.subpictures_in_dvd[DVDtitleset]>1 || Config.subpictures_in_dvd[DVDtitleset]==0) {
            g_string_sprintf(multiple,"%s","s");
        } else {
            g_string_sprintf(multiple,"%s","");
        }
        if (lock) {
            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Stream contains %i subpicture%s",Config.subpictures_in_dvd[DVDtitleset],multiple->str);
        } else {
            g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Stream contains %i subpicture%s",Config.subpictures_in_dvd[DVDtitleset],multiple->str);
        }

        /* Parse CLUT */
        if (save_clut == TRUE) {
            parse_clut(ifo,DVDtitleset,lock);
        }
    } else {
        if (lock) {
            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"No VTS IFO found at for titleset %i",DVDtitleset);
        } else {
            g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"No VTS IFO found at for titleset %i",DVDtitleset);
        }
    }

    /* Clean & Exit */
    ifoClose (ifo);
    g_string_free(description,TRUE);
    if (multiple) {
        g_string_free(multiple,TRUE);
    }
    return TRUE;
}


gboolean save_config(gboolean lock) {
    /* Save configuration */
    GString *configfile;
    GIOChannel *configfile_channel;
    guint bytesloaded;
    gint saveerror;
    gint fd,i;
    gboolean true_udf = FALSE;
    gboolean dummy_bool;

    /* Lock */
    if (lock==TRUE) drip_lock("save_config");
    configfile = g_string_new("");

    /* delete old config */
    g_string_sprintf(configfile,"rm %s/.drip/drip.cfg",getenv("HOME"));
    unlink(configfile->str);
    /* create new one */
    g_string_sprintf(configfile,"%s/.drip/drip.cfg",getenv("HOME"));

    fd = open(configfile->str,O_WRONLY|O_CREAT,S_IREAD|S_IWRITE);
    configfile_channel = g_io_channel_unix_new(fd);
    /* Save config version */
    i = strlen(CONFIG_VERSION);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&i,sizeof(glong),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)CONFIG_VERSION,strlen(CONFIG_VERSION),&bytesloaded);

    /* Save Drip config */
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.DVDdevice->len,sizeof(glong),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,Config.DVDdevice->str,Config.DVDdevice->len,&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.DIVXfileUser->len,sizeof(glong),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,Config.DIVXfileUser->str,Config.DIVXfileUser->len,&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.set_width,sizeof(glong),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.set_height,sizeof(glong),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.video_bitrate,sizeof(guint),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.video_stream,sizeof(guint),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.audio_bitrate,sizeof(gint),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.audio_1st_stream,sizeof(guint),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.encoding_video_codec,sizeof(gint),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.cliptop,sizeof(guint),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.clipbottom,sizeof(guint),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.blurradius,sizeof(guint),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&dummy_bool,sizeof(gboolean),&bytesloaded); //Was Config.cache, disabled since 0.8.0cvs
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.eject_dvdrom,sizeof(gboolean),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.use_dvddb,sizeof(gboolean),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.nice,sizeof(gint),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.conversion_mode,sizeof(cmode_t),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.maxsize,sizeof(glong),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&true_udf,sizeof(gboolean),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.encode,sizeof(gboolean),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.DVDsource->len,sizeof(glong),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,Config.DVDsource->str,Config.DVDsource->len,&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.useDVDdevice,sizeof(gboolean),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.cpus,sizeof(gint),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.done_warning,sizeof(gboolean),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.CACHElocation->len,sizeof(glong),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,Config.CACHElocation->str,Config.CACHElocation->len,&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.cache_delete_on_exit,sizeof(gboolean),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.cache_chunk,sizeof(gboolean),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.create_iso,sizeof(gboolean),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.hyper_zoom,sizeof(gboolean),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.autoclip,sizeof(gboolean),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.autoclip_frames,sizeof(guint),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.clipright,sizeof(guint),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.clipleft,sizeof(guint),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&dummy_bool,sizeof(gboolean),&bytesloaded); //Was Config.honor_aspect, removed since cvs 0.8.0
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.deinterlace,sizeof(gboolean),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.pulldown,sizeof(gboolean),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.encoding_audio_codec,sizeof(gint),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&mainwindow->allocation.width,sizeof(gint),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&mainwindow->allocation.height,sizeof(gint),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.debug_log,sizeof(gboolean),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.audio_norm,sizeof(gboolean),&bytesloaded);

    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.audio_preferedA->len,sizeof(glong),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,Config.audio_preferedA->str,Config.audio_preferedA->len,&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.audio_preferedB->len,sizeof(glong),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,Config.audio_preferedB->str,Config.audio_preferedB->len,&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.spu_preferedA->len,sizeof(glong),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,Config.spu_preferedA->str,Config.spu_preferedA->len,&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,(gchar*)&Config.spu_preferedB->len,sizeof(glong),&bytesloaded);
    saveerror = g_io_channel_write(configfile_channel,Config.spu_preferedB->str,Config.spu_preferedB->len,&bytesloaded);

    g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Saved configuration");

    /* Clean & exit */
    g_io_channel_close(configfile_channel);
    close(fd);
    g_string_free(configfile,TRUE);
    /* Unlock */
    if (lock==TRUE) drip_unlock("save_config");
    return TRUE;
}

void drip_error(gchar *text) {
    GtkWidget *warning_window;
    warning_window = create_infobox(text);
    gtk_widget_show(warning_window);
}

void drip_info(gchar *text) {
    GtkWidget *info_window = create_infobox(text);
    gtk_widget_show(info_window);
    return;
}

/* system() wrapper */
GString* drip_system(gchar *command,gint *exec_result,gboolean freebuffer,gboolean lock) {
    GString *format = g_string_new("");
    GString *result = g_string_new("");
    GString *wrapperfile = g_string_new("");
    GIOChannel *pipefile;
    glong pipefile_size;
    guint bytessaved;
    gint loaderror;
    gchar *buf;
    gint fd;
    struct stat file_stats;

    if (lock) {
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Executing |%s|",command);
        drip_lock("Drip exec");
    } else {
        g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Executing |%s|",command);
    }

    /* build wrapper filename */
    g_string_sprintf(wrapperfile,"/tmp/drip.system.wrapper.%d.tmp",getpid());

    /* execute system command */
    g_string_sprintf(format,"%s 1>%s 2>&1",command,wrapperfile->str);
    *exec_result = system(format->str);

    /* read system command output */
    fd = open(wrapperfile->str,O_RDONLY);
    pipefile = g_io_channel_unix_new(fd);

    stat(wrapperfile->str, &file_stats);
    pipefile_size = file_stats.st_size - 1; 
    if (pipefile_size>0 && pipefile_size<10000000) {
        buf = (gchar*)malloc(pipefile_size+1);
        loaderror = g_io_channel_read(pipefile,buf,pipefile_size,&bytessaved);
        g_string_sprintf(result,"%s",buf);
        result->str[bytessaved] = 0; // 0 terminate
        free(buf);
    } else {
        g_string_sprintf(result,"{empty output}");
    }

    g_io_channel_close(pipefile);
    close(fd);
#ifndef DEBUG
    unlink(wrapperfile->str);
#endif
    g_string_free(wrapperfile,TRUE);
    g_string_free(format,TRUE);

    if (lock) drip_unlock("Drip exec");

    if (freebuffer == TRUE) {
        g_string_free(result,TRUE);
        return NULL;
    } else {
        return result;
    }
}

/* Threaded wrapper for drip_system */
gpointer drip_system_threaded(gpointer data) {
    drip_system_t* ds_data = (drip_system_t*)data;
    drip_system(ds_data->command,ds_data->exec_result,ds_data->freebuffer,TRUE);
    pthread_exit(0);
    return NULL;
}


/* Format Drip-gui's Config structure to single command line */
gchar* mpeg2divx_parameters(gint avifile) {
    GString *param = g_string_new("");
    GString *temp = g_string_new("");
    
    g_string_sprintf(temp,"-b %i ",Config.video_bitrate);
    param = g_string_append(param,temp->str);

    if (Config.video_stream != 0) {
        g_string_sprintf(temp,"-vs %i ",Config.video_stream);
        param = g_string_append(param,temp->str);
    }

    /* Audio, set to 128kb bitrate if 2 channel encoding is requested */
    if (Config.audio_2nd_stream != -1) {
        Config.audio_bitrate = 128;
        g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Audio bitrate set to 128kb due to dual audio channel encoding");
    }

    if (Config.audio_bitrate != -1) {
        g_string_sprintf(temp,"-a %i ",Config.audio_bitrate);
        param = g_string_append(param,temp->str);
    }
    if (Config.audio_1st_stream != 0) {
        g_string_sprintf(temp,"-as %i ",Config.audio_1st_stream);
        param = g_string_append(param,temp->str);
    }
    if (Config.audio_2nd_stream != -1) { // 2nd audio channel
        g_string_sprintf(temp,"-a2 %i ",Config.audio_bitrate);
        param = g_string_append(param,temp->str);
        g_string_sprintf(temp,"-as2 %i ",Config.audio_2nd_stream);
        param = g_string_append(param,temp->str);
    }

    g_string_sprintf(temp,"-w %li ",Config.set_width);
    param = g_string_append(param,temp->str);

    g_string_sprintf(temp,"-h %li ",Config.set_height);
    param = g_string_append(param,temp->str);

    if (Config.cliptop != 0) {
        g_string_sprintf(temp,"-ct %i ",Config.cliptop);
        param = g_string_append(param,temp->str);
    }

    if (Config.clipleft != 0) {
        g_string_sprintf(temp,"-cl %i ",Config.clipleft);
        param = g_string_append(param,temp->str);
    }

    if (Config.clipright != 0) {
        g_string_sprintf(temp,"-cr %i ",Config.clipright);
        param = g_string_append(param,temp->str);
    }

    if (Config.clipbottom != 0) {
        g_string_sprintf(temp,"-cb %i ",Config.clipbottom);
        param = g_string_append(param,temp->str);
    }

    if (Config.subpicture_stream>-1) {
        g_string_sprintf(temp,"-s %i %s ",Config.subpicture_stream,Config.clutfile->str);
        param = g_string_append(param,temp->str);
    }

    if (Config.autoclip == TRUE) {
        g_string_sprintf(temp,"-ac ");
        param = g_string_append(param,temp->str);
    }
    if (Config.autoclip_frames > 0) {
        g_string_sprintf(temp,"-acf %i ",Config.autoclip_frames);
        param = g_string_append(param,temp->str);
    }


    if (Config.deinterlace == TRUE) {
        g_string_sprintf(temp,"-ip ");
        param = g_string_append(param,temp->str);
    }

    if (Config.pulldown == TRUE) {
        g_string_sprintf(temp,"-in ");
        param = g_string_append(param,temp->str);
    }

    /* Frames of source from DVD */
    if (Config.selected_chapters_frames[avifile] > 0) {
        g_string_sprintf(temp,"-e %i -E ",(gint)Config.selected_chapters_frames[avifile]);
        param = g_string_append(param,temp->str);
    } else {
        /* Frames of source from cache */
        if (Config.frames2encode[avifile-1] > 0) {
            g_string_sprintf(temp,"-e %i -E ",Config.frames2encode[avifile-1]);
            param = g_string_append(param,temp->str);
        } else {
            /* Other (MPEG source) frames unknown */
            g_string_sprintf(temp,"-e 999999 -E ");
            param = g_string_append(param,temp->str);
        }
    }


    if (Config.conversion_mode == DVDTITLE_DVDDISK || Config.conversion_mode == DVDTITLE_DVDIMAGE) {
printf("VTS = %i\n",((selected_chapters_t*)(Config.selected_chapters[avifile]->data))->VTS);
        if (Config.VTS_Aspects[((selected_chapters_t*)(Config.selected_chapters[avifile]->data))->VTS]!=0) {
            g_string_sprintf(temp,"-r %g ",Config.VTS_Aspects[((selected_chapters_t*)(Config.selected_chapters[avifile]->data))->VTS]);
            param = g_string_append(param,temp->str);
        }
    } else {
        if (Config.VTS_Aspects[Config.VT_belongs_to_VTS[Config.CS_belongs_to_VT[Config.CS_current]]]!=0) {
            g_string_sprintf(temp,"-r %g ",Config.VTS_Aspects[Config.VT_belongs_to_VTS[Config.CS_belongs_to_VT[Config.CS_current]]]);
            param = g_string_append(param,temp->str);
        }
    }


    if (Config.audio_norm == FALSE) {
        g_string_sprintf(temp,"-av 0 ");
    } else {
        g_string_sprintf(temp,"-av 1 ");
    }
    param = g_string_append(param,temp->str);

    /* Avifile encoding video codec */
    g_string_sprintf(temp,"-c %li ",Config.codecs[Config.encoding_video_codec].ccodec);
    param = g_string_append(param,temp->str);
    g_string_sprintf(temp,"-m %s ",Config.codecs[Config.encoding_video_codec].cmodule->str);
    param = g_string_append(param,temp->str);

    /* Avifile encoding audio codec */
    g_string_sprintf(temp,"-au %li ",Config.codecs[Config.encoding_audio_codec].ccodec);
    param = g_string_append(param,temp->str);

    /* Output chunk size */
    g_string_sprintf(temp,"-l %lu",Config.maxsize*1024*1024);
    param = g_string_append(param,temp->str);

    /* Free & Exit */
    g_string_free(temp,TRUE);
    return param->str;
}

gint dirfilter_all_vob(const struct dirent *entry) {
    /* checks if a vob file is a rip candidate */
    gchar *result1;
    gchar *result2;
    gchar *result3;
    gchar *result4;
    gchar *result5;
    gchar *result6;
    
    result1 = strstr(entry->d_name,".vob");
    result2 = strstr(entry->d_name,".VOB");
    result3 = strstr(entry->d_name,".mpg");
    result4 = strstr(entry->d_name,".MPG");
    result5 = strstr(entry->d_name,".mpeg");
    result6 = strstr(entry->d_name,".MPEG");

    if ((result1==NULL) && (result2==NULL) && (result3==NULL) && (result4==NULL) && (result5==NULL) && (result6==NULL))
        return FALSE;
    else
        return TRUE;
}

gint dirfilter_all(const struct dirent *entry) {
    /* true on any file on disk */
    if (strstr(entry->d_name,".")==entry->d_name)
        return 0;
    else
        return 1;
}

/* Amount of TitleSets \ Titles \ Chapters \ Angles on DVD */
gboolean detect_DVD_titles(gboolean _lock) {
    ifo_handle_t *ifohandle_vmg;
    ifo_handle_t *ifohandle_vts;
    pgc_t *pgc;
    gboolean NTSCwarningdisplayed = FALSE;
    guint i,title_nr;
    gint j = 0;
    gint previous_chapter_sequence_nr;
    gint title_largest = 0;
    gint seconds,frames;
    gint frames_largest = 0;
    gdouble framerate = 0,framerate_largest = 0;
    gint hour,minute,second;
    GString* time = g_string_new("");
    GString* hour_str = g_string_new("");
    GString* minute_str = g_string_new("");
    GString* second_str = g_string_new("");
    gint u_frames;
    gpointer LD;

    /* Logging type */
    if (_lock) {
        LD = DRIP_LD;
    } else {
        LD = DRIP_CB_LD;
    }

    /* Reset variables */
    Config.DVD_Total_ChapterSequences = 0;

    /* Fetch amount of title sets */
    ifohandle_vmg = ifoOpen(Config.DVDhandle, 0);
    if (ifohandle_vmg==NULL) {
        g_log(LD,G_LOG_LEVEL_WARNING,"Drive doesn't contain a valid DVD disk");
        return FALSE;
    }
    Config.DVD_TitleSets = ifohandle_vmg->vmgi_mat->vmg_nr_of_title_sets;
    g_log(LD,G_LOG_LEVEL_INFO,"DVD contains %i title sets", Config.DVD_TitleSets);
    Config.DVD_Total_Titles = ifohandle_vmg->tt_srpt->nr_of_srpts;

    /* Loop titles and parse the whereabouts of the stream chunks
       (titlesets, titles, chapters, angles) */
    for (i=0;i<ifohandle_vmg->tt_srpt->nr_of_srpts;i++) {
        previous_chapter_sequence_nr = -2;
        Config.VT_belongs_to_VTS[i] = ifohandle_vmg->tt_srpt->title[i].title_set_nr;
        Config.VT_is_Title_of_VTS[i] = ifohandle_vmg->tt_srpt->title[i].vts_ttn-1;
        /* Open VTS ifo */
        g_log(LD,G_LOG_LEVEL_DEBUG,"Opening DVD VTS %i",Config.VT_belongs_to_VTS[i]);
        ifohandle_vts = ifoOpen(Config.DVDhandle,Config.VT_belongs_to_VTS[i]);
        if(ifohandle_vts->vmgi_mat) {
            g_log(LD,G_LOG_LEVEL_WARNING,"Opened a VMG ifo, VTS was exspected");
        }
        if(ifohandle_vts->vtsi_mat) {
            Config.VTS_Titles[Config.VT_belongs_to_VTS[i]-1] = ifohandle_vts->vts_ptt_srpt->nr_of_srpts;
            Config.VT_Chapters[i] = ifohandle_vmg->tt_srpt->title[i].nr_of_ptts;
            Config.VT_Angles[i] = ifohandle_vmg->tt_srpt->title[i].nr_of_angles;
            for (j=0;j<ifohandle_vts->vts_ptt_srpt->title[Config.VT_is_Title_of_VTS[i]].nr_of_ptts;j++) {
                if (previous_chapter_sequence_nr != (ifohandle_vts->vts_ptt_srpt->title[Config.VT_is_Title_of_VTS[i]].ptt[j].pgcn-1)) {
                    /* New CS */
                    title_nr = i;
                    Config.DVD_Total_ChapterSequences++;
                    Config.CS_belongs_to_VT[Config.DVD_Total_ChapterSequences-1] = title_nr;
                    Config.CS_starts_at_Chapter[Config.DVD_Total_ChapterSequences-1] = j+1;
                    /* Fill chapters in previous PC */
                    if (Config.DVD_Total_ChapterSequences>1 && j>0) {
                        Config.CS_Chapters[Config.DVD_Total_ChapterSequences-2] = ifohandle_vts->vts_ptt_srpt->title[Config.VT_is_Title_of_VTS[i]].ptt[j-1].pgn;
                    }
                    pgc = ifohandle_vts->vts_pgcit->pgci_srp[ifohandle_vts->vts_ptt_srpt->title[Config.VT_is_Title_of_VTS[i]].ptt[j].pgcn-1].pgc;
                    if (pgc!=NULL) {
                        g_string_sprintf(hour_str,"%02x",pgc->playback_time.hour);
                        g_string_sprintf(minute_str,"%02x",pgc->playback_time.minute);
                        g_string_sprintf(second_str,"%02x",pgc->playback_time.second);
                        u_frames = pgc->playback_time.frame_u & 0x3f;

                        hour = atoi(hour_str->str);
                        minute = atoi(minute_str->str);
                        second = atoi(second_str->str);

                        switch((pgc->playback_time.frame_u & 0xc0) >> 6) {
                        case 1:
                            framerate = 25.00;
                            break;
                        case 3:
                            framerate = 29.97;
                            if (NTSCwarningdisplayed == FALSE) {
                                g_log(LD,G_LOG_LEVEL_WARNING,"This NTSC stream probably needs reverse 3:2 pulldown which is unsupported");
                                NTSCwarningdisplayed = TRUE;
                            }
                            break;
                        default:
                            framerate = 0;
                            break;
                        }
                    } else {
                        hour = 0;
                        minute = 0;
                        second = 0;
                        u_frames = 0;
                        framerate = 0;
                        g_log(LD,G_LOG_LEVEL_WARNING,"PGC %i is NULL",ifohandle_vts->vts_ptt_srpt->title[Config.VT_is_Title_of_VTS[i]].ptt[j].pgcn);
                    }
                    frames = (gint)((hour * 3600 + minute * 60 + second) * framerate
+ u_frames);
                    if (framerate>0) {
                        seconds = (gint)(frames / framerate);
                    } else {
                        seconds = 0;
                    }
                    seconds2time(seconds,time);

                    g_log(LD,G_LOG_LEVEL_INFO,"PC %i: %s  (%i frames at %gFPS)",Config.DVD_Total_ChapterSequences,time->str,frames,framerate);
                    Config.frames2encode[Config.DVD_Total_ChapterSequences-1] = frames;
                    if (frames>frames_largest) {
                        frames_largest = frames;
                        title_largest = Config.DVD_Total_ChapterSequences;
                        framerate_largest = framerate;
                        Config.framerate = framerate;
                    }
                }
                previous_chapter_sequence_nr = ifohandle_vts->vts_ptt_srpt->title[Config.VT_is_Title_of_VTS[i]].ptt[j].pgcn-1;
            }
            /* Fill chapters in last PC */
            Config.CS_Chapters[Config.DVD_Total_ChapterSequences-1] = ifohandle_vts->vts_ptt_srpt->title[Config.VT_is_Title_of_VTS[i]].ptt[j-1].pgn;
        } else {
            g_log(LD,G_LOG_LEVEL_ERROR,"Cant open VTS for titleset %i",i+1);
        }
        ifoClose(ifohandle_vts);
    }

    /* Largest title */
    largest_vob_cached = title_largest;
    Config.CS_current = title_largest-1;
    Config.CS[Config.CS_current] = TRUE;
    seconds2time(frames_largest/framerate,time);
    g_log(LD,G_LOG_LEVEL_INFO,"Largest title %i %s (%i frames at %gFPS)",Config.CS_current+1,time->str,Config.frames2encode[Config.CS_current],framerate_largest);

    /* Clean & Exit */
    g_string_free(time,TRUE);
    g_string_free(hour_str,TRUE);
    g_string_free(minute_str,TRUE);
    g_string_free(second_str,TRUE);
    ifoClose(ifohandle_vmg);
    return TRUE;
}

/* Check mount point for free space */
gboolean free_space(gchar* mountpoint,glong min_freespace,glong warning_freespace,gboolean _lock) {
    static struct statfs fsd;
    static struct stat fss;
    register gulong freeblocks;
    register gulong freespace;
    register gulong blocksize;
    register gchar *LD;

    /* Logging type */
    if (_lock) {
        LD = DRIP_LD;
    } else {
        LD = DRIP_CB_LD;
    }
    /* Stat mountpoint, for blocksize */
    if (stat(mountpoint,&fss) != 0) {
        g_log(LD,G_LOG_LEVEL_WARNING,"Filesystem at %s could not be checked (%s)",mountpoint,strerror(errno));
        return FALSE;
    }
    blocksize = fss.st_blksize; 
    blocksize = blocksize >> 10;
    /* Stat FS */
    if (statfs(mountpoint,&fsd) != 0) {
        g_log(LD,G_LOG_LEVEL_WARNING,"Filesystem at %s could not be checked for free space (%s)",mountpoint,strerror(errno));
        return FALSE;
    }
    freeblocks = fsd.f_bavail;
    freespace = blocksize * freeblocks;
    /* Enough free space? */
    if (freespace == 0) {
        g_log(LD,G_LOG_LEVEL_WARNING,"Filesystem at %s is full",mountpoint);
        return FALSE;
    } else if (freespace<warning_freespace) {
        g_log(LD,G_LOG_LEVEL_WARNING,"Filesystem at %s is low on free space (requested %luKb, available %luKb)",mountpoint,min_freespace,freespace);
    } else if (freespace<min_freespace) {
        g_log(LD,G_LOG_LEVEL_WARNING,"Filesystem at %s has not enough free space (requiered %luKb, available %luKb)",mountpoint,min_freespace,freespace);
        return FALSE;
    }
    #ifdef DEBUG
    g_log(LD,G_LOG_LEVEL_DEBUG,"Filesystem at %s contains enough free space (requiered %luKb, available %luKb, blocksize %luKb, freeblocks %lu)",mountpoint,min_freespace,freespace,blocksize,freeblocks);
    #endif
    return TRUE;
}

void _detect_largest_title(void) {
    gint title_largest;
    gint titlesets,title_nr;
    gint seconds,frames;
    gint frames_largest;
    gboolean result;
    gboolean NTSCwarningdisplayed = FALSE;
    ifo_handle_t *ifohandle;
    pgc_t *pgc;
    gdouble framerate = 0,framerate_largest = 0;
    gint hour,minute,second;
    GString* time = g_string_new("");
    GString* hour_str = g_string_new("");
    GString* minute_str = g_string_new("");
    GString* second_str = g_string_new("");
    gint u_frames;
    gint j,n;
    gint previous_chapter_sequence_nr;
    ifo_handle_t *ifohandle_vmg;
    guint i = 0;

    /* already done this? */
    if (largest_vob_cached>0) {
        return;
    }

    /* Fetch amount of title sets */
    ifohandle_vmg = ifoOpen(Config.DVDhandle, 0);

    /* Loop titlesets and titles in search for largest play time */
    result = detect_DVD_titles(TRUE);
    Config.DVD_Total_ChapterSequences = 0;
    title_largest = 0;
    titleset_largest = 0;
    frames_largest = 0;
    for (titlesets = 1;titlesets<Config.DVD_TitleSets+1;titlesets++) {
    
        ifohandle = ifoOpen(Config.DVDhandle, titlesets);

        /* Detect all Chapter Sequences (CS) in this title */
        previous_chapter_sequence_nr = -2;
        for (n=0;n<ifohandle->vts_ptt_srpt->nr_of_srpts;n++) {
            for (j=0;j<ifohandle->vts_ptt_srpt->title[n].nr_of_ptts;j++) {
                if (previous_chapter_sequence_nr != (ifohandle->vts_ptt_srpt->title[n].ptt[j].pgn-1)) {
                    /* New CS */
                    title_nr = i;

                    Config.DVD_Total_ChapterSequences++;
                    Config.CS_belongs_to_VT[Config.DVD_Total_ChapterSequences-1] = title_nr;
                    Config.CS_starts_at_Chapter[Config.DVD_Total_ChapterSequences-1] = j+1;
                    pgc = ifohandle->vts_pgcit->pgci_srp[ifohandle->vts_ptt_srpt->title[n].ptt[j].pgcn-1].pgc;
                    if (pgc!=NULL) {
                        g_string_sprintf(hour_str,"%02x",pgc->playback_time.hour);
                        g_string_sprintf(minute_str,"%02x",pgc->playback_time.minute);
                        g_string_sprintf(second_str,"%02x",pgc->playback_time.second);
                        u_frames = pgc->playback_time.frame_u & 0x3f;

                        hour = atoi(hour_str->str);
                        minute = atoi(minute_str->str);
                        second = atoi(second_str->str);

                        switch((pgc->playback_time.frame_u & 0xc0) >> 6) {
                        case 1:
                            framerate = 25.00;
                            break;
                        case 3:
                            framerate = 29.97;
                            if (NTSCwarningdisplayed == FALSE) {
                                g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"NTSC streams need reverse 3:2 pulldown which is not supported");
                                NTSCwarningdisplayed = TRUE;
                            }
                            if (Config.deinterlace == FALSE) {
                                Config.deinterlace = TRUE;
                                g_log(DRIP_LD,G_LOG_LEVEL_MESSAGE,"Enabled deinterlacing so NTSC video looks less bad");
                            }
                            break;
                        default:
                            framerate = 0;
                            break;
                        }
                    } else {
                        hour = 0;
                        minute = 0;
                        second = 0;
                        u_frames = 0;
                        framerate = 0;
                        g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"PGC %i is NULL",ifohandle->vts_ptt_srpt->title[n].ptt[j].pgcn);
                    }
                    frames = (gint)((hour * 3600 + minute * 60 + second) * framerate + u_frames);
                    if (framerate>0) {
                        seconds = (gint)(frames / framerate);
                    } else {
                        seconds = 0;
                    }
                    seconds2time(seconds,time);
                    g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Title %i: %s (%i frames %gFPS",Config.DVD_Total_ChapterSequences,time->str,frames,framerate);
                    Config.frames2encode[Config.DVD_Total_ChapterSequences-1] = frames;
                    if (frames>frames_largest) {
                        frames_largest = frames;
                        title_largest = Config.DVD_Total_ChapterSequences;//title_nr;
                        titleset_largest = titlesets;
                        framerate_largest = framerate;
                        Config.framerate = framerate;
                    }
                } 
                previous_chapter_sequence_nr = ifohandle->vts_ptt_srpt->title[n].ptt[j].pgn;
            }
        }
        ifoClose(ifohandle);
        ifohandle = NULL;
    }

    largest_vob_cached = title_largest;
    Config.CS_current = title_largest-1;
    Config.CS[Config.CS_current] = TRUE;
    seconds2time(frames_largest/framerate,time);
    g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Largest title %i %s (%i frames at %gFPS)",Config.CS_current,time->str,Config.frames2encode[Config.CS_current],framerate_largest);
    g_string_free(time,TRUE);
    g_string_free(hour_str,TRUE);
    g_string_free(minute_str,TRUE);
    g_string_free(second_str,TRUE);
    return;
}

void seconds2time(gint seconds,GString *time) {
    /* conversion int seconds to char time */
    if (time==NULL) {
        g_log(DRIP_CB_LD,G_LOG_LEVEL_WARNING,"Time == NULL..");
        time = g_string_new("");
    }
    if (seconds/3600 < 1) {
        g_string_sprintf(time,"%.2i:%.2i",(seconds%3600)/60,seconds%60);
    } else 
        g_string_sprintf(time,"%i:%.2i:%.2i",seconds/3600,(seconds%3600)/60,seconds%60);
    return;
}

gint dirfilter_divx(const struct dirent *entry) {
    /* filters on .avi */
    gchar *result;
    result = strstr(entry->d_name,".avi");

    if (result==NULL)
        return 0;
    else
        return 1;
}

gint manual_stream_select(gchar *video_location) {
    /* allowes the user to manually select streams that are to be encoded */
    gint i;
    struct dirent **vobsdir;

    selected_files = -1; /* convertion thread polls this variable,
                           -1  = not ready selecting,
                           >-1 = selected number of files,
                           files are stored in namelist */
    i = scandir(video_location,&vobsdir,dirfilter_all,alphasort);

    drip_lock("manual_stream_select");
    select_dialog = create_manual_selection_dialog(i,vobsdir,video_location);
    gtk_widget_show(select_dialog);
    drip_unlock("manual_stream_select");
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Manual stream selection mode");
    while (selected_files==-1) usleep(100000);
    return 0;
}


gchar* readlink_malloc (gchar *filename) {
    gint size = 100;

    while (1) {
        char *buffer = (char *)malloc(size);
        int nchars = readlink (filename, buffer, size);

        /* Is filename a symlink? */
        if (nchars == -1) {
            free(buffer);
            return NULL; /* No */
        }

        /* Do we have a good result so far? */
        if (nchars < size) {
            return buffer; /* Yes */
        }

        free (buffer);
        size *= 2;
    }

    /* dummy */
    return NULL;
}


gchar* detect_source_track_dir(gboolean lock) {
    GString *command = g_string_new("");
    GString *track_dir = g_string_new("");
    GString *LD = g_string_new("");

    /* Need locking? */
    if (lock) {
        g_string_sprintf(LD,DRIP_LD);
    } else {
        g_string_sprintf(LD,DRIP_CB_LD);
    }

    /* Are we about to use a DVDdevice or just source files */
    //if (Config.conversion_mode != MPEG_PLAIN && Config.useDVDdevice) {
        /* Fetch info of CD\DVD disk, fill various fields of Config */
        cdinfo(&Config);
        /* Display info */
        if (Config.DVDvolumeid->len > 0) {
            g_log(LD->str,G_LOG_LEVEL_INFO,"DVD volume id %s",Config.DVDvolumeid->str);
        }
        if (Config.DVDpublisher->len > 0) {
            g_log(LD->str,G_LOG_LEVEL_INFO,"DVD publisher %s",Config.DVDpublisher->str);
        }
        if (Config.DVDpreparer->len > 0) {
            g_log(LD->str,G_LOG_LEVEL_INFO,"DVD preparer %s",Config.DVDpreparer->str);
        }
        if (Config.DVDcreationdate->len > 0) {
            g_log(LD->str,G_LOG_LEVEL_INFO,"DVD creation date %s",Config.DVDcreationdate->str);
        }
        if (Config.DVDvolumesize_MB > 0) {
            g_log(LD->str,G_LOG_LEVEL_INFO,"DVD volume size %luMB",Config.DVDvolumesize_MB);
        }
        g_string_sprintf(track_dir,"%s",Config.DVDdevice->str);
        /* Check for free HD space if caching */
        //TODO
    //} else {
    //    /* Use user entered dir for source */
    //    g_string_sprintf(track_dir,"%s",Config.DVDsource->str);
    //}

    /* clean up & exit */
    g_log(LD->str,G_LOG_LEVEL_DEBUG,"detect_source_track_dir returned |%s|",track_dir->str);
    g_string_free(command,TRUE);
    g_string_free(LD,TRUE);
    return track_dir->str;
}

/* Read a title from source and output the stream */
gboolean read_dvd_title(gint avifile) {
    gint angle,pgc_id, len, start_cell, cur_cell;
    guint cur_pack;
    gint ttn, pgn, next_cell, end_cell;
    gint written_blocks = 0;
    gint written_blocks_prev = 0;
    guchar *data;
    dvd_file_t *title_handle;
    tt_srpt_t *tt_srpt;
    ifo_handle_t *vts_file = NULL;
    ifo_handle_t *vmg_file;
    vts_ptt_srpt_t *vts_ptt_srpt;
    pgc_t *cur_pgc;
    gint output_fd;
    GString *output_file = g_string_new("");
    GString *string = g_string_new("");
    GString *timestr = g_string_new("");
    guint sectors;
    guint output_file_counter = 0;
    gdouble perc;
    gint blocks_countdown = 2;
    gint bytes_written;
    struct timeval current_tv;
    struct timezone current_tz={0,0};
    struct timeval caching_tv;
    struct timezone caching_tz={0,0};
    glong seconds;
    gfloat speed;
    GList *list;
    gint i;
    gint skip_cell;
    gint cells_read_last = 0;
    drip_cell_t* cells_read[1024];
    

    /* malloc big buffer */
    data = (guchar*)malloc(1024*2048);

    /* Open VMG */
    drip_lock("read_dvd_title 1");
    vmg_file = ifoOpen(Config.DVDhandle,0);
    drip_unlock("read_dvd_title 1");
    if(!vmg_file) {
        g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Can't open VMG info");
        return -1;
    }
    tt_srpt = vmg_file->tt_srpt;

    /* Open output */
    g_string_sprintf(output_file,"%s/dripcache_%02d_%i.vob",Config.CACHElocation->str,avifile,output_file_counter);
    g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Reading DVD disk information, might take a long time");
    drip_lock("read_dvd_title 3");
    output_fd = open(output_file->str,O_WRONLY|O_CREAT,S_IREAD|S_IWRITE);
    drip_unlock("read_dvd_title 3");
    /* Add this output to input list for encoder */
    selected_files = 1;
    Config.CS_files[0][0] = g_string_new(output_file->str);
    /* Update cached file list, used for cache deletion */
    cached_files_list++;
    cached_file_list[cached_files_list] = g_string_new(output_file->str);
    cached_file_list_title[cached_files_list] = avifile;
    /* Start time */
    gettimeofday(&caching_tv,&caching_tz);

    /* Loop AVI chapter list */
    list = Config.selected_chapters[avifile];
    g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Reading chapters for Avifile %i from DVD disk  (%s)",avifile,output_file->str);
    while (list) {
        /* Open Title Set */
        drip_lock("read_dvd_title 2");
        vts_file = ifoOpen(Config.DVDhandle,((selected_chapters_t*)(list->data))->VTS+1);
        drip_unlock("read_dvd_title 2");
        if(!vts_file) {
            g_log(DRIP_LD,G_LOG_LEVEL_ERROR,"Can't open ifo of video title set %d",((selected_chapters_t*)(list->data))->VTS+1);
            return FALSE;
        } 
        #ifdef DEBUG
        else {
            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Opened ifo for video title set %d",((selected_chapters_t*)(list->data))->VTS+1);
        }
        #endif

        /* Open the title sets data */
        drip_lock("read_dvd_title 4");
        title_handle = DVDOpenFile(Config.DVDhandle,tt_srpt->title[((selected_chapters_t*)(list->data))->VT].title_set_nr,DVD_READ_TITLE_VOBS );
        drip_unlock("read_dvd_title 4");
        if(!title_handle) {
            g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Can't open content of titleset %i",((selected_chapters_t*)(list->data))->VT);
            return FALSE;
        } 
        #ifdef DEBUG
        else {
            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Opened content of titleset %i",((selected_chapters_t*)(list->data))->VT);
        }
        #endif

        /* Determine start cell of current chapter\angle */
        #ifdef DEBUG
        g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Config.CS_belongs_to_VT[Config.CS_current] = %i",((selected_chapters_t*)(list->data))->VT);
        #endif
        ttn = tt_srpt->title[((selected_chapters_t*)(list->data))->VT].vts_ttn;     // ... of Video Title
        vts_ptt_srpt = vts_file->vts_ptt_srpt;                                        // 
        #ifdef DEBUG
        g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"((selected_chapters_t*)(list->data))->CHAP_TOTAL = %i     ttn = %i",((selected_chapters_t*)(list->data))->CHAP_TOTAL,ttn);
        #endif
        pgc_id = vts_ptt_srpt->title[ttn-1].ptt[((selected_chapters_t*)(list->data))->CHAP_TOTAL].pgcn; 
        pgn = vts_ptt_srpt->title[ttn-1].ptt[((selected_chapters_t*)(list->data))->CHAP_TOTAL].pgn;
        #ifdef DEBUG
        g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"pgc_id = %i    pgn = %i",pgc_id,pgn);fflush(NULL);
        #endif
     
        cur_pgc = vts_file->vts_pgcit->pgci_srp[pgc_id-1].pgc;
        start_cell = cur_pgc->program_map[pgn-1]-1;
        end_cell = cur_pgc->nr_of_cells;
        angle = Config.angle-1;
        /* Playback by cell in this pgc, starting at the cell for our chapter */
        next_cell = start_cell;
        //g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"ttn = %i, pgc = %i, pgn = %i",ttn,pgc_id,pgn);

        /* Loop current chapter's cell instead of only reading it's 1st cell */
        for( cur_cell = start_cell; next_cell < end_cell; ) {
            cur_cell = next_cell;
            /* Check if this Cell already has been read */
            skip_cell = FALSE;
            for (i=0; i< cells_read_last; i++) {
                if (cur_cell == cells_read[i]->cell &&
                    angle == cells_read[i]->angle &&
                    ((selected_chapters_t*)(list->data))->PC == cells_read[i]->pc
                  ) {
                    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Already read DVD Cell %i / %i at angle %i chapter %i title %i, skipping",cur_cell,end_cell-1,angle,pgn,((selected_chapters_t*)(list->data))->PC);
                    skip_cell = TRUE;
                }
            } 

            if (skip_cell == FALSE) {
            g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Reading DVD Cell %i / %i at angle %i chapter %i title %i",cur_cell,end_cell-1,angle,pgn,((selected_chapters_t*)(list->data))->PC);
            /* Add this cell to cells_read list */
            cells_read[cells_read_last] = (drip_cell_t*)malloc(sizeof(drip_cell_t));
            cells_read[cells_read_last]->cell  = (gint)cur_cell;
            cells_read[cells_read_last]->angle = (gint)angle;
            cells_read[cells_read_last]->pc    = (gint)((selected_chapters_t*)(list->data))->PC;
            cells_read_last++;
            /* Check for 100Mb free space (1Gb warning level), TODO check for real amount needed */
            if (free_space(Config.CACHElocation->str,100*1024,1024*1024,TRUE)==FALSE) {
                g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Free space check not passed, need at least 100Mb free");
                drip_lock("read_dvd_title 61");
                ifoClose(vmg_file);
                ifoClose(vts_file);
                close(output_fd);
                DVDCloseFile(title_handle);
                drip_unlock("read_dvd_title 61");
                return FALSE;
            }
            /* Iterate sectors reading until we're out of this cell */
            sectors = cur_pgc->cell_playback[cur_cell].last_sector - cur_pgc->cell_playback[cur_cell].first_sector;
            for (cur_pack=cur_pgc->cell_playback[cur_cell].first_sector; cur_pack<cur_pgc->cell_playback[cur_cell].last_sector; ) {
                dsi_t dsi_pack;
                guint next_vobu, next_ilvu_start, cur_output_size;
                /* Read NAV packet */
                drip_lock("read_dvd_title 5");
                len = DVDReadBlocks(title_handle,(gint)cur_pack,1,data);
                drip_unlock("read_dvd_title 5");
                if(len!=1) {
                    g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"DVD needs cleaning?  (Read failed for block %d)",cur_pack);
                    break;
                }
                /* Parse the contained dsi packet */
                navRead_DSI( &dsi_pack, &(data[ DSI_START_BYTE ]) );
                /* Determine where we go next.  These values are the ones we mostly care about. */
                next_ilvu_start = cur_pack + dsi_pack.sml_agli.data[ angle ].address;
                cur_output_size = dsi_pack.dsi_gi.vobu_ea;

                /**
                 * If we're not at the end of this cell, we can determine the next
                 * VOBU to display using the VOBU_SRI information section of the
                 * DSI.  Using this value correctly follows the current angle,
                 * avoiding the doubled scenes in The Matrix, and makes our life
                 * really happy.
                 * Otherwise, we set our next address past the end of this cell to
                 * force the code above to go to the next cell in the program.
                 */
                if( dsi_pack.vobu_sri.next_vobu != SRI_END_OF_CELL ) {
                    next_vobu = cur_pack + ( dsi_pack.vobu_sri.next_vobu & 0x7fffffff );
                } else {
                    next_vobu = cur_pack + cur_output_size + 1;
                }
                cur_pack++;
    
                /* Read in and output cursize packs */
                drip_lock("read_dvd_title 8");
                len = DVDReadBlocks(title_handle,(gint)cur_pack,cur_output_size,data );
                drip_unlock("read_dvd_title 8");
                if( len!=(gint)cur_output_size) {
                    g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"DVD needs cleaning?  (Read failed for %d blocks at sector %d)",cur_output_size,cur_pack);
                    break;
                } //else 
                  //  g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Read sector %d",cur_pack);

                /* Reached maximum filesize? -> close output and open next (2Gb filesize limit) */
                if ((Config.cache_chunk == TRUE) && (written_blocks>990000)) {
                    drip_lock("read_dvd_title 10");
                    close(output_fd);
                    drip_unlock("read_dvd_title 10");
                    output_file_counter++;
                    written_blocks = 0;
                    written_blocks_prev = 0;
                    g_string_sprintf(output_file,"%s/dripcache_%02d_%i.vob",Config.CACHElocation->str,avifile,output_file_counter);
                    g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Caching to %s",output_file->str);
                    /* Update selected files list, used for encoding */
                    selected_files++;
                    Config.CS_files[0][output_file_counter] = g_string_new(output_file->str);
                    /* Update cached file list, used for cache deletion */
                    cached_files_list++;
                    cached_file_list[cached_files_list] = g_string_new(output_file->str);
                    cached_file_list_title[cached_files_list] = avifile;
                    output_fd = open(output_file->str,O_WRONLY|O_CREAT,S_IREAD|S_IWRITE);
                    /* Reset time */
                    gettimeofday(&caching_tv,&caching_tz);
                }

                /* Demux and filter out unwanted streams */
            
                /* Output small enough, write out all */
                bytes_written = write(output_fd,data,DVD_VIDEO_LB_LEN*cur_output_size);
                if (bytes_written != DVD_VIDEO_LB_LEN*cur_output_size) {
                    g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Caching could not be completed (%s)",strerror(errno));
                    return FALSE;
                }
                written_blocks+=cur_output_size;

                cur_pack = next_vobu;
                blocks_countdown--;
                if (blocks_countdown == 0) {
                    /* Display progress */
                    perc = (gdouble)(g_list_length(Config.selected_chapters[avifile])-g_list_length(list))/(gdouble)(g_list_length(Config.selected_chapters[avifile]));
                    perc = perc+((gdouble)(cur_pack-cur_pgc->cell_playback[cur_cell].first_sector)/(sectors+1))/(gdouble)(g_list_length(Config.selected_chapters[avifile]));

                    gettimeofday(&current_tv,&current_tz);
                    seconds = (gulong)((current_tv.tv_sec - caching_tv.tv_sec))+1;
                    speed = written_blocks / seconds * 2;
                    written_blocks_prev = written_blocks;
                    seconds2time(seconds/perc-seconds,timestr);

                    drip_lock("read_dvd_title 12");
                    g_string_sprintf(string,"   %.0f%%    Caching DVD data for avifile %i at %gKb/s    %s left",perc*100,avifile,speed,timestr->str);
                    gnome_appbar_set_status(GNOME_APPBAR(appbar1),string->str);
                    gnome_appbar_set_progress(GNOME_APPBAR(appbar1),perc);
                    drip_unlock("read_dvd_title 12");
                    blocks_countdown = 30;
                }

                /* Pause reading */
                while (paused==TRUE) {
                    sleep(1);
                }

                /* Should DVD reading be terminated? */
                if (stopped==TRUE) {
                    drip_lock("read_dvd_title 12b");
                    ifoClose(vts_file);
                    DVDCloseFile(title_handle);
                    drip_unlock("read_dvd_title 12b");
                    goto exit;
                }
            }
// ----
            } // skip_cell == FALSE

            /* Check if we're entering an angle block. */
            if (cur_pgc->cell_playback[cur_cell].block_type == BLOCK_TYPE_ANGLE_BLOCK) {
                gint i;
                cur_cell += angle;
                for (i=0; ;++i) {
                    if (cur_pgc->cell_playback[cur_cell+i].block_mode == BLOCK_MODE_LAST_CELL) {
                        next_cell = cur_cell + i + 1;
                        break;
                    }
                }
            } else {
                next_cell = cur_cell + 1;
            }

        }
        drip_lock("read_dvd_title 12c");
        ifoClose(vts_file);
        DVDCloseFile(title_handle);
        drip_unlock("read_dvd_title 12c");
        list = list->next;
    } 

    /* Clean & Exit */
    exit:
    /* Cleanup cells_read struct */
    for (i=0;i<cells_read_last;i++) {
        free(cells_read[i]);
    }
    /* Display progress */
    perc = (gdouble)(g_list_length(Config.selected_chapters[avifile])-g_list_length(list))/(gdouble)(g_list_length(Config.selected_chapters[avifile]));
    perc = perc+((gdouble)(cur_pack-cur_pgc->cell_playback[cur_cell].first_sector)/(sectors+1))/(gdouble)(g_list_length(Config.selected_chapters[avifile]));
    gettimeofday(&current_tv,&current_tz);
    seconds = (gulong)((current_tv.tv_sec - caching_tv.tv_sec))+1;
    speed = written_blocks / seconds * 2;
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Caching done, read %iKb at %gKb/s",written_blocks*2048,speed);
    drip_lock("read_dvd_title 13");
    ifoClose(vmg_file);
    close(output_fd);
    drip_unlock("read_dvd_title 13");
    g_string_free(output_file,TRUE);
    g_string_free(string,TRUE);
    g_string_free(timestr,TRUE);
    free(data);
    if (written_blocks>0)
        return TRUE;
    else
        return FALSE;
}

/* Save info of a dvd title */
void save_dvd_title_info(gboolean lock) {
    GString *infofile = g_string_new("");
    GString *temp = g_string_new("");
    GIOChannel *infofile_channel;
    gint fd,titleset,i,title,t,n;
    gint programchain;
    glong len;
    //GList *list;
    guint result;
    //gdouble frames;
    guint frames_ui;
    guchar* LD;

    if (lock) {LD=DRIP_LD;} else {LD=DRIP_CB_LD;}
    /* Create file */
    g_string_sprintf(infofile,"%s/DVDTITLE.tbl",Config.CACHElocation->str);
    g_log(LD,G_LOG_LEVEL_DEBUG,"Writing DVD title file %s",infofile->str);
    if (lock) drip_lock("save_dvd_title_info 1");
    fd = open(infofile->str,O_WRONLY|O_CREAT,S_IREAD|S_IWRITE);
    infofile_channel = g_io_channel_unix_new(fd);
    if (lock) drip_unlock("save_dvd_title_info 1");

    if (lock) drip_lock("save_dvd_title_info 1b");
    /* Writeout DVDTITLE version number */
    len = strlen(DVDTITLE_VERSION);
    g_io_channel_write(infofile_channel,(gchar*)&len,sizeof(len),&result);
    g_io_channel_write(infofile_channel,(gchar*)DVDTITLE_VERSION,len,&result);
    /* Writeout DVD disk title string */
    len = Config.DVDvolumeid->len;
    g_io_channel_write(infofile_channel,(gchar*)&len,sizeof(len),&result);
    g_io_channel_write(infofile_channel,(gchar*)Config.DVDvolumeid->str,len,&result);
    if (lock) drip_unlock("save_dvd_title_info 1b");

    /* Writeout amount of cached titles */
    title = 0;
    for (i=0;i<RANGE;i++) {
        if (g_list_length(Config.selected_chapters[i])>0) {
            title++;
        }
    }
    if (title==0) {
        g_log(LD,G_LOG_LEVEL_WARNING,"No DVD titles cached, could not writeout cache description");
        /* Clean & Exit */
        if (lock) drip_lock("save_dvd_title_info 2");
        g_io_channel_close(infofile_channel);
        close(fd);
        if (lock) drip_unlock("save_dvd_title_info 2");
        g_string_free(infofile,TRUE);
        g_string_free(temp,TRUE);
        return;
    } else {
        /* OK, save amount of cached titles */
        g_log(LD,G_LOG_LEVEL_DEBUG,"Writing info for %i DVD title(s) to cache description",title);
        if (lock) drip_lock("save_dvd_title_info 3");
        g_io_channel_write(infofile_channel,(gchar*)&title,sizeof(gint),&result);
        if (lock) drip_unlock("save_dvd_title_info 3");
    }

    /* Loop all cached titles */
    if (lock) drip_lock("save_dvd_title_info 4");
    for (t=0;t<RANGE;t++) {                                   
        #ifdef DEBUG
        g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Writing info for chain %i",t);
        #endif
        if (g_list_length(Config.selected_chapters[t])>0) {      
            title = ((selected_chapters_t*)(Config.selected_chapters[t]->data))->VT;
            titleset = ((selected_chapters_t*)(Config.selected_chapters[t]->data))->VTS;
            programchain = t;
            /* Writeout info */
            g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Writing info for title %i (titleset %i)",title,titleset);
            g_io_channel_write(infofile_channel,(gchar*)&title,sizeof(gint),&result);
            g_io_channel_write(infofile_channel,(gchar*)&titleset,sizeof(gint),&result);
            g_io_channel_write(infofile_channel,(gchar*)&Config.CS_starts_at_Chapter[title],sizeof(gint),&result);
            g_io_channel_write(infofile_channel,(gchar*)&Config.CS_Chapters[title],sizeof(gint),&result);
            g_io_channel_write(infofile_channel,(gchar*)&Config.VT_Chapters[title],sizeof(guint),&result);
            g_io_channel_write(infofile_channel,(gchar*)&Config.VT_Angles[title],sizeof(guint),&result);
            g_io_channel_write(infofile_channel,(gchar*)&Config.framerate,sizeof(gdouble),&result);
            /* Audio channels */
            g_io_channel_write(infofile_channel,(gchar*)&Config.audio_channels_in_dvd[titleset+1],sizeof(guint),&result);
            g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Writing info %i Audio channels",Config.audio_channels_in_dvd[titleset+1]);
            for (i=0;i<Config.audio_channels_in_dvd[titleset+1];i++) {
                g_io_channel_write(infofile_channel,(gchar*)&Config.audio_channel_names[titleset+1][i]->len,sizeof(glong),&result);
                g_io_channel_write(infofile_channel,(gchar*)Config.audio_channel_names[titleset+1][i]->str,Config.audio_channel_names[titleset+1][i]->len,&result);
                g_io_channel_write(infofile_channel,(gchar*)Config.audio_channel_types[titleset+1][i],sizeof(gint),&result);
                g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Writing info audio channel %i (%s)",i+1,Config.audio_channel_names[titleset+1][i]->str);
            }
            /* Subpictures */
            g_io_channel_write(infofile_channel,(gchar*)&Config.subpictures_in_dvd[titleset+1],sizeof(guint),&result);
            g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Writing info %i Subpictures",Config.subpictures_in_dvd[titleset+1]);
            for (i=0;i<Config.subpictures_in_dvd[titleset+1];i++) {
                g_io_channel_write(infofile_channel,(gchar*)&Config.subpicture_names[titleset+1][i]->len,sizeof(glong),&result);
                g_io_channel_write(infofile_channel,(gchar*)Config.subpicture_names[titleset+1][i]->str,Config.subpicture_names[titleset+1][i]->len,&result);
                g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Writing info subpicture %i (%s)",i+1,Config.subpicture_names[titleset+1][i]->str);
            }
            /* Frames */
            frames_ui = (guint)Config.selected_chapters_frames[t];
            g_io_channel_write(infofile_channel,(gchar*)&frames_ui,sizeof(guint),&result);
            g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Writing info %i frames",frames_ui);
            /* Titles */
            n = 0;
            for (i=0;i<cached_files_list;i++) {
                if (cached_file_list_title[i+1] == t) {
                    n++;
                }
            }
            g_io_channel_write(infofile_channel,(gchar*)&n,sizeof(gint),&result);
            g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Writing info %i input files",n);
            for (i=0;i<cached_files_list;i++) {
                if (cached_file_list_title[i+1] == t) {
                    g_string_sprintf(temp,"%s",cached_file_list[i+1]->str);
                    g_io_channel_write(infofile_channel,(gchar*)&temp->len,sizeof(glong),&result);
                    g_io_channel_write(infofile_channel,(gchar*)temp->str,temp->len,&result);
                    g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Writing info %s input file",temp->str);
                }
            }
            g_io_channel_write(infofile_channel,(gchar*)&Config.VTS_Aspects[titleset],sizeof(gdouble),&result);
            //g_log(DRIP_CB_LD,G_LOG_LEVEL_INFO,"Writing info VTS_Aspects[%i] = %i",titleset,Config.VTS_Aspects[titleset]);
            /* End writeout title info */
        }
        /* next title */
    }

    /* Clean & Exit */
    g_io_channel_close(infofile_channel);
    close(fd);
    if (lock) drip_unlock("save_dvd_title_info 4");
    g_string_free(infofile,TRUE);
    g_string_free(temp,TRUE);
    return;
}

/* Load info of a set of cached dvd titles */
gboolean load_dvd_title_info(gint requested_title,gboolean lock) {
    GString *infofile = g_string_new("");
    GString *temp = g_string_new("");
    GIOChannel *infofile_channel;
    gint fd,i,CS,title,titleset,len,titles;
    guint result;
    gchar *buf;
    gchar *LD;

    /* Locking */
    if (lock) {
        LD = DRIP_LD;
    } else {
        LD = DRIP_CB_LD;
    }
    
    /* Reset CS_files */
    for (i=0; i<RANGE; i++) {
        for (len=0; len<RANGE; len++) {
            if (Config.CS_files[i][len] != NULL) g_string_free(Config.CS_files[i][len],TRUE);
            Config.CS_files[i][len] = NULL;
        }
    }

    /* open file */
    g_string_sprintf(infofile,"%s/DVDTITLE.tbl",Config.DVDsource->str);
    g_log(LD,G_LOG_LEVEL_DEBUG,"Loading DVD title %i from file %s",requested_title,infofile->str);
    if (lock) drip_lock("load_dvd_title_info 1");
    fd = open(infofile->str,O_RDONLY);
    infofile_channel = g_io_channel_unix_new(fd);

    /* Init varis */
    titleset = 1;
    for (i=0;i<256;i++) { Config.CS[i] = FALSE; }
    buf = (gchar*)malloc(1024);

    /* Read DVDTITLE version number and check */
    g_io_channel_read(infofile_channel,(gchar*)&len,sizeof(glong),&result);
    memset(buf,0,1024);
    g_io_channel_read(infofile_channel,buf,len,&result);
    if (strcmp(buf,DVDTITLE_VERSION)!=0) {
        g_log(DRIP_CB_LD,G_LOG_LEVEL_WARNING,"Cache description file is of other version (%s) as was expected (%s)",buf,DVDTITLE_VERSION);
        return FALSE;
    }

    /* Read DVD disk title string */
    g_io_channel_read(infofile_channel,(gchar*)&len,sizeof(glong),&result);
    memset(buf,0,1024);
    g_io_channel_read(infofile_channel,buf,len,&result);
    g_string_sprintf(Config.DVDvolumeid,"%s",buf);
    g_string_sprintf(Config.DIVXfileUser,"%s",Config.DVDvolumeid->str);

    /* Read amount of titles in cache */
    g_io_channel_read(infofile_channel,(gchar*)&titles,sizeof(guint),&result);
    Config.DVD_Total_Titles = titles;
    Config.DVD_Total_ChapterSequences = titles;
    /* Read all or just one requested title? */
    if (requested_title!=-1) {
        titles = requested_title+1;
        g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Loading requested titel %i",requested_title);
    }

    g_log(DRIP_CB_LD,G_LOG_LEVEL_INFO,"Loading info for %i cached DVD title(s)",titles);
    /* Loop reading all titles */
    for (CS=0;CS<titles;CS++) {
        Config.CS_current = CS;
        Config.CS[Config.CS_current] = TRUE;
        /* Load info */
        g_io_channel_read(infofile_channel,(gchar*)&Config.CS_belongs_to_VT[CS],sizeof(gint),&result);
        title = Config.CS_belongs_to_VT[CS];
        g_io_channel_read(infofile_channel,(gchar*)&Config.VT_belongs_to_VTS[title],sizeof(gint),&result);
        titleset = Config.VT_belongs_to_VTS[title];
        g_io_channel_read(infofile_channel,(gchar*)&Config.CS_starts_at_Chapter[Config.CS_current],sizeof(gint),&result);
        g_io_channel_read(infofile_channel,(gchar*)&Config.CS_Chapters[Config.CS_current],sizeof(gint),&result);
        g_io_channel_read(infofile_channel,(gchar*)&Config.VT_Chapters[Config.CS_current],sizeof(guint),&result);
        g_io_channel_read(infofile_channel,(gchar*)&Config.VT_Angles[Config.CS_current],sizeof(guint),&result);
        g_io_channel_read(infofile_channel,(gchar*)&Config.framerate,sizeof(gdouble),&result);
        /* Audio */
        g_io_channel_read(infofile_channel,(gchar*)&Config.audio_channels_in_dvd[titleset],sizeof(guint),&result);
        g_log(DRIP_CB_LD,G_LOG_LEVEL_INFO,"Loading info title %i, %i audio channels",CS+1,Config.audio_channels_in_dvd[titleset]);
        for (i=0;i<Config.audio_channels_in_dvd[titleset];i++) {
            memset(buf,0,1024);
            g_io_channel_read(infofile_channel,(gchar*)&len,sizeof(glong),&result);
            g_io_channel_read(infofile_channel,buf,len,&result);
            g_string_sprintf(Config.audio_channel_names[titleset][i],"%s",buf);
            g_io_channel_read(infofile_channel,(gchar*)Config.audio_channel_types[titleset][i],sizeof(gint),&result);
            g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Loading info title %i, audio %i (%s)",CS+1,i,Config.audio_channel_names[titleset][i]->str);
        }
        /* SPU */
        g_io_channel_read(infofile_channel,(gchar*)&Config.subpictures_in_dvd[titleset],sizeof(guint),&result);
        g_log(DRIP_CB_LD,G_LOG_LEVEL_INFO,"Loading info title %i, %i subpictures",CS+1,Config.subpictures_in_dvd[titleset]);
        for (i=0;i<Config.subpictures_in_dvd[titleset];i++) {
            memset(buf,0,1024);
            g_io_channel_read(infofile_channel,(gchar*)&len,sizeof(glong),&result);
            g_io_channel_read(infofile_channel,buf,len,&result);
            g_string_sprintf(Config.subpicture_names[titleset][i],"%s",buf);
            g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Loading info title %i, spu %i (%s)",CS+1,i,Config.subpicture_names[titleset][i]->str);
        }
        /* Frames */
        g_io_channel_read(infofile_channel,(gchar*)&Config.frames2encode[Config.CS_current],sizeof(guint),&result);
        seconds2time(Config.frames2encode[Config.CS_current]/Config.framerate,temp);
        g_log(DRIP_CB_LD,G_LOG_LEVEL_INFO,"Loading info title %i, %i frames (%s)",CS+1,Config.frames2encode[Config.CS_current],temp->str);
        /* Amount of cache file per title */
        g_io_channel_read(infofile_channel,(gchar*)&selected_files,sizeof(gint),&result);
        g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Loading info title %i, selected_files = %i",CS+1,selected_files);
        for (i=0;i<selected_files;i++) {
            memset(buf,0,1024);
            g_io_channel_read(infofile_channel,(gchar*)&len,sizeof(glong),&result);
            g_io_channel_read(infofile_channel,buf,len,&result);
            if (Config.CS_files[Config.CS_current][i]!=NULL)
                g_string_free(Config.CS_files[Config.CS_current][i],TRUE);
            Config.CS_files[Config.CS_current][i] = g_string_new("");
            if (strncmp(buf,".",1)==0) {
                /* Relative path, add PWD */
                g_string_sprintf(Config.CS_files[CS][i],"%s/%s",Config.start_directory->str,buf);
            } else {
                /* Absolute path */
                g_string_sprintf(Config.CS_files[CS][i],"%s",buf);
            }
            g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Loading info title %i file %i filename %s",CS+1,i,Config.CS_files[CS][i]->str);
        }
        g_io_channel_read(infofile_channel,(gchar*)&Config.VTS_Aspects[titleset],sizeof(gdouble),&result);
        /* Next iteration title read */
    }

    /* Clean & Exit */
    g_io_channel_close(infofile_channel);
    close(fd);
    if (lock) drip_unlock("load_dvd_title_info 1");
    g_string_free(infofile,TRUE);
    g_string_free(temp,TRUE);
    free(buf);
    return TRUE;
}

/* Format filename to (name)-(videorate)-(audiorate).avi */
gboolean format_divxname(GString *inputDIVXfilename,gboolean lock) {
    GString *tempname = g_string_new("");
    GString *tempstore = g_string_new("");
    GString *DIVXfilename = g_string_new(inputDIVXfilename->str);
    gpointer LD;

    if (lock) {LD = DRIP_LD;} else {LD = DRIP_CB_LD;}
    /* ".avi" is end of name already? */
    if (strncasecmp(DIVXfilename->str + (DIVXfilename->len-4),".avi",4) == 0) {
        /* truncate .avi */
        DIVXfilename = g_string_truncate(DIVXfilename,DIVXfilename->len-4);
    }
    g_string_sprintf(tempname,"%s",DIVXfilename->str);
    /* Add VTS number to name */
    g_string_sprintf(tempstore,"%i",Config.CS_current);
    g_string_sprintf(tempstore,"-%i",Config.CS_current);
    tempname = g_string_append(tempname,tempstore->str);
    /* Add video rate to name */
    g_string_sprintf(tempstore,"%i",Config.video_bitrate);
    if (strstr(tempname->str,tempstore->str)!=NULL) {
        /* Name already contains video bitrate */
    } else {
        g_string_sprintf(tempstore,"-%i",Config.video_bitrate);
        tempname = g_string_append(tempname,tempstore->str);
    }
    /* Add ".avi" to name */
    tempname = g_string_append(tempname,".avi");
    /* Set DIVX output name to newly generated name */
    g_string_sprintf(Config.DIVXfileFormatted,"%s",tempname->str);
    /* Update GUI displayed filename */
    if (combo_entry_divx_filename!=NULL) {
        gtk_entry_set_text(GTK_ENTRY(combo_entry_divx_filename),Config.DIVXfileUser->str);
    }

    /* Clean & Exit */
    g_string_free(tempname,TRUE);
    g_string_free(tempstore,TRUE);
    g_string_free(DIVXfilename,TRUE);
    return TRUE;
}


gboolean config_ok() {
    /* checks if the Config structure is filled with same data before 
       the dvd->divx conversion is initiated */

    /* 1: MPEG-1, 0: MPEG-2 LSF, 1995-07-11 shn */
    gint bitrate_table[2][16] = {
              {-1,0,8,16,24,32,40,48,56,64,80,96,112,128,144,160},
              {-1,0,32,40,48,56,64,80,96,112,128,160,192,224,256,320}};
    gint index = 0;
    gboolean found = FALSE;

    /* Traceback if Config.DVDdevice is a symbolic link */
    DVDsymlink();
    /* check audio rate */
    while(!found && index<16) {
        if(bitrate_table[1][index] == Config.audio_bitrate) /* dvd = mpeg2 */
            found = TRUE;
        else
            ++index;
    }
    return found;
}

gboolean user_can_mount_dvd() {
    GString *command = g_string_new("");
    GString *mountoptions = g_string_new("");
    gint result = 0;
    gboolean allowed = FALSE;

    g_string_sprintf(command,"cat /etc/fstab | grep -w %s | awk '{print($4)}'",Config.DVDdevice->str);
    g_string_sprintf(mountoptions,"%s",drip_system(command->str,&result,FALSE,TRUE)->str);
    /* check if mountoptions contains 'user' and not 'nouser' */
    if (strstr(mountoptions->str,"nouser")) {
        allowed = FALSE;
    } else {
        if (strstr(mountoptions->str,"user")) {
            allowed = TRUE;
        }
    }
    /* clean & exit */
    g_string_free(command,TRUE);
    g_string_free(mountoptions,TRUE);
    return allowed;
}


/* Returns all *.avi file in entry directory */
gint dirfilter_avi(const struct dirent *entry) {
    gchar *result1;
    gchar *result2;
    gchar **name_parts;
    GString *basename = g_string_new("");
    gint i;
    /* Split avi filename */
    name_parts = g_strsplit(Config.DIVXfileFormatted->str,"/",0);
    /* basename, name without '.avi' and basedir */
    i = 0; while (name_parts[i+1]!=NULL) { i++; }
    g_string_sprintf(basename,"%s",name_parts[i]);
    basename = g_string_truncate(basename,basename->len-4);
    /* check if name matches *.avi */
    result1 = strstr(entry->d_name,".avi");
    result2 = strstr(entry->d_name,basename->str);
    if (result1==NULL || result2==NULL) {
        return FALSE;
    } else {
        return TRUE;
    }
}


/* Renames divx files to proper numbering (like : 'I_of_II') */
void rename_divx_files() {
    struct dirent **avi_files;
    gint avi_files_number,i;
    GString *basename = g_string_new("");
    GString *basedir = g_string_new("");
    GString *newname = g_string_new("");
    GString *command = g_string_new("");
    gchar **name_parts;

    /* Split avi filename */
    name_parts = g_strsplit(Config.DIVXfileFormatted->str,"/",0);
    /* Basename, name without '.avi', basedir */
    for (i=0;name_parts[i+1]!=NULL;i++) {
        basedir = g_string_append(basedir,name_parts[i]);
        basedir = g_string_append(basedir,"/");
    }
    if (basedir->len < 2) {
        g_string_sprintf(basedir,"%s/",Config.start_directory->str); 
    }
    g_string_sprintf(basename,"%s",name_parts[i]);
    basename = g_string_truncate(basename,basename->len-4);
    /* Find out how many chunks the result is spread over */
    avi_files_number = scandir(basedir->str,&avi_files,dirfilter_avi,alphasort);
    if (avi_files_number<1) {
        g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Could not rename generated AVI files (%s), %i found",basedir->str,avi_files_number);
    } else {
        /* Rename iteration */
        g_string_sprintf(newname,"%s%sbps-%s_%ibps-%s_%ix%i_%iof%i.avi",basedir->str,basename->str,Config.codecs[Config.encoding_video_codec].cmodule->str,Config.audio_bitrate,"mp3",Config.out_width,Config.out_height,1,avi_files_number);
        g_string_sprintf(command,"%s%s",basedir->str,avi_files[avi_files_number-1]->d_name);
g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Renaming %s to %s",command->str,newname->str);
        rename(command->str,newname->str);
        for (i=0;i<avi_files_number-1;i++) {
            g_string_sprintf(newname,"%s%sbps-%s_%ibps-%s_%ix%i_%iof%i.avi",basedir->str,basename->str,Config.codecs[Config.encoding_video_codec].cmodule->str,Config.audio_bitrate,"mp3",Config.out_width,Config.out_height,i+2,avi_files_number);
            g_string_sprintf(command,"%s%s",basedir->str,avi_files[i]->d_name);
g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Renaming %s to %s",command->str,newname->str);
            rename(command->str,newname->str);
        }
    }

    /* Clean & Exit */
    g_strfreev(name_parts);
    g_string_free(newname,TRUE);
    g_string_free(basename,TRUE);
    g_string_free(command,TRUE);
    g_string_free(basedir,TRUE);
    return;
}


/* Ejects DVDrom device and displays some info */
gboolean eject_dvdrom(gchar *track_dir,gchar *dvddevice,gboolean lock) {
    gint result = 0;
    GString *command = g_string_new("");
    gpointer LD;

    /* Logging domain: locking or non-locking */
    if (lock) {LD = DRIP_LD;} else {LD = DRIP_CB_LD;}

    /* Close opens on DVD */
    if (Config.DVDhandle!=NULL) {
        DVDClose(Config.DVDhandle);
        Config.DVDhandle = NULL;
    }
    if (track_dir==NULL) return FALSE;

    /* Log and build system command lines */
    if (lock) drip_lock("eject_dvdrom1");
    gnome_appbar_set_status(GNOME_APPBAR(appbar1),"   Ejecting DVDrom");
    gnome_appbar_set_progress(GNOME_APPBAR(appbar1),0);
    if (lock) drip_unlock("eject_dvdrom1");
    result = umount(track_dir);
    if (result==-1) {
        g_log(LD,G_LOG_LEVEL_DEBUG,"Umount failed for %s: %s",track_dir,strerror(errno));
    }
    g_string_sprintf(command,"eject %s",dvddevice);
    system(command->str);
    /* Clean & Exit */
    g_string_free(command,TRUE);
    return TRUE;
}


/* Builds iso images out of the generated divx's */
gpointer make_iso_images(gpointer data) {
    gint i;
    gint selected_files = 0;
    struct dirent **divxdir;
    GString *command = g_string_new("");
 
    selected_files = scandir(".",&divxdir,dirfilter_divx,alphasort);
    for (i=0;i<selected_files;i++) {
        g_string_sprintf(command,"Building ISO image %s.iso",divxdir[i]->d_name);
        g_log(DRIP_LD,G_LOG_LEVEL_INFO,command->str);
        g_string_sprintf(command,"mkisofs -J -l -T -R %s > %s.iso",divxdir[i]->d_name,divxdir[i]->d_name);
        system(command->str);
        unlink(divxdir[i]->d_name);
    }

    making_iso = FALSE;
    g_string_free(command,TRUE);
    pthread_exit(0);
    return NULL;
}

/* Remove cache */
gboolean remove_cache(gboolean lock) {
    gint cnt,result;
    GString *command = g_string_new("");
    GString *cachefile = g_string_new("");
    gpointer ld;
    /* Anything to remove? */
    if ((Config.cache_delete_on_exit) && (cached_files>0)) {
        /* Input file */
        for (cnt = 0; cnt < cached_files_list; ++cnt) {
            g_string_sprintf(command,"%s",cached_file_list[cnt+1]->str);
            g_string_free(cached_file_list[cnt+1],TRUE);
            /* loggin domain */
            if (lock) {
                ld = DRIP_LD;
            } else {
                ld = DRIP_CB_LD;
            }
            /* Build cache filename */
            if (strcmp(&command->str[0],".")==0) {
                g_string_sprintf(cachefile,"%s%s",Config.start_directory->str,command->str);
            } else {
                g_string_sprintf(cachefile,"%s",command->str);
            }
            /* Delete cache file */
            g_log(ld,G_LOG_LEVEL_MESSAGE,"Removing cache content %s",cachefile->str);
            result = unlink(cachefile->str);
            if (result == -1) {
                g_log(ld,G_LOG_LEVEL_WARNING,"Removal of %s failed: %s",cachefile->str,strerror(errno));
            }
        }
    }
    /* Clean & Exit */
    cached_files = 0;
    g_string_free(command,TRUE);
    g_string_free(cachefile,TRUE);
    return TRUE;
}


/* Checks dvdname for default dvdnames etc */
gboolean drip_dvdname_check(GString* dvdname) {
    /* Empty? */
    if (dvdname->len == 0) return FALSE;
    /* Default preparation names */
    if (strstr(dvdname->str,"DVD_VIDEO")!=NULL) return FALSE;
    if (strstr(dvdname->str,"DVDVOLUME")!=NULL) return FALSE;
    // TODO: more dummy DVD titles ?? 

    /* Checks passed, name OK */
    return TRUE;
}


/* Main conversion thread: handles user input & pre reading code, 
                           calls DVD reader
                           executes MPEG encoder
                           and all post encoding logic */
gpointer conversion_thread(gpointer data) {
    GString *command;
    GString *temp;
    GString *title;
    gint i,cnt;
    gchar* buf;
    gchar* source_track;
    gint result;
    GList *avi_list;
    /* Conversion started: mark flag */
    converting = TRUE;
    cached_files = g_string_new("");
    title = g_string_new("");
    command = g_string_new("");
    temp = g_string_new("");
    Config.CS_current = -1; //TODO: parsing available titles & user dialog
    Config.audio_1st_stream = 0; //Default to 1st channel, consistent with GUI
    Config.audio_2nd_stream = -1; //Default to None, consistent with GUI

    #ifdef DEBUG
    /* Debug */
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Started conversion_thread, pid = %i",getpid());
    #endif

    /* Set thread cancel status */
    result =          pthread_setcancelstate(PTHREAD_CANCEL_ENABLE,NULL);
    result = result & pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS,NULL);

    /* Check configuration */
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Checking configuration");
    if (!config_ok()) {
        g_string_sprintf(temp,"Bitrate %ikbs not legal for output sampling.\n",Config.audio_bitrate);
        drip_error(temp->str);
        converting = FALSE;
        g_string_free(command,TRUE);
        g_string_free(temp,TRUE);
        pthread_exit(0);
    }

    drip_domain();

    /* set status flags to states */
    converting = TRUE;             // set encoding flag
    encoding_time = FALSE;         // unset encoding time, cq encoding start time has tobe determined
    quit_converting = FALSE;       // unset request quit flag, used to shutdown threads by request
    stopped = FALSE;               // unset general idle mode flag
    paused = FALSE;                // lift pause block

    /* Overrule audio bitrate setting if needed */
    if (Config.two_audio_channels_support == TRUE) {
        Config.audio_bitrate = 128;
    }

    /* Show progres frame buttons */
    drip_lock("conversion_thread0");
    gtk_widget_show(hbox35);
    drip_unlock("conversion_thread0");

    /* Store current directory */
    buf = malloc(1024);
    if (getcwd(buf,1024)==NULL) {
        g_log(DRIP_LD,G_LOG_LEVEL_ERROR,"Error while trying to determine program working directory");
    } else {
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Current program working directory is %s",buf);
        Config.start_directory = g_string_new("");
        g_string_sprintf(Config.start_directory,"%s",buf);
    }
    free(buf);

    /* Expand cache location to full path and create cache location */
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Expand cache location to full path and create cache location");
    g_string_sprintf(temp,"%s",Config.CACHElocation->str);
    mkdir(temp->str,S_IRUSR|S_IWUSR|S_IXUSR);

    /* Output filename: DVDDB, DVD iso header or user given name */
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Output filename: DVDDB, DVD iso header or user given name");
    source_track = detect_source_track_dir(TRUE);
    if (Config.useDVDdevice && Config.encode && Config.use_dvddb) {
        int sock;
        char id[9];
        g_log(DRIP_LD,G_LOG_LEVEL_MESSAGE,"Querying DVDDB...");
        drip_lock("conversion_thread3");
        gnome_appbar_set_status(GNOME_APPBAR(appbar1),"   Querying DVDDB...");
        drip_unlock("conversion_thread3");
        g_string_sprintf(temp,"%s","");
        result = dvddbGetID(Config.DVDdevice->str,id);
        if ((sock=dvddbNetConnect()) < 0) {
            /* Use hardcoded name, no dvddb name available */
            g_log(DRIP_LD,G_LOG_LEVEL_MESSAGE,"No DVDdb hit: using hardcoded name");
            if (drip_dvdname_check(Config.DVDvolumeid)==TRUE) {
                g_string_sprintf(Config.DIVXfileUser,"%s",Config.DVDvolumeid->str);
            }
            format_divxname(Config.DIVXfileUser,TRUE);
        } else {
            g_string_sprintf(temp,"%s",dvddbNetGet(sock, id));
            if (strstr(temp->str,"(null)")!=NULL || temp->str == NULL || temp->len == 0) {
                /* Use hardcoded name, no dvddb name availeble */
                g_log(DRIP_LD,G_LOG_LEVEL_MESSAGE,"No DVDdb hit: using hardcoded name");
                if (drip_dvdname_check(Config.DVDvolumeid)==TRUE) {
                    g_string_sprintf(Config.DIVXfileUser,"%s",Config.DVDvolumeid->str);
                }
            } else {
                g_log(DRIP_LD,G_LOG_LEVEL_MESSAGE,"Found DVD: using DVDdb name");
                g_string_sprintf(Config.DIVXfileUser,"%s-%i-%i.avi",temp->str,Config.video_bitrate,Config.audio_bitrate);
            }
            /* Format filename, add bitrate and dvd title number */
            format_divxname(Config.DIVXfileUser,TRUE);
        }
        dvddbNetDisconnect (sock);
    } else {
        /* No DVDDB -> get name from iso header or user */
        if (drip_dvdname_check(Config.DVDvolumeid)) {
            g_string_sprintf(Config.DIVXfileUser,"%s",Config.DVDvolumeid->str);
        }
        format_divxname(Config.DIVXfileUser,TRUE);
    }
    /* Show backup druid and let user select input type */
    if (Config.conversion_mode != MPEG_PLAIN) {
        /* -- Show user dialog for audio channel and subpictures -- */
        drip_lock("druid");
        audio_subpicture_dialog = create_configdruid();
        gtk_widget_show(audio_subpicture_dialog);
        drip_unlock("druid");
        while (audio_subpicture_dialog!=NULL) { usleep(100000);}
    } else {
        /* Open the MPEG_PLAIN input */
        if (source_track == NULL) {
            /* Clean & exit */
            converting = FALSE;
            drip_lock("conversion_thread1");
            gnome_appbar_set_status(GNOME_APPBAR(appbar1),"   Ready");
            drip_unlock("conversion_thread1");
            g_string_free(command,TRUE);
            g_string_free(temp,TRUE);
            g_string_free(cached_files,TRUE);
            pthread_exit(0);
            return FALSE;
        }
    }

    /* User canceled? */
    if (converting==FALSE) {
        /* Clean & exit */
        converting = FALSE;
        drip_lock("conversion_thread2");
        gnome_appbar_set_status(GNOME_APPBAR(appbar1),"   Ready");
        drip_unlock("conversion_thread2");
        g_string_free(command,TRUE);
        g_string_free(temp,TRUE);
        pthread_exit(0);
        return FALSE;
    }
    /* Location of input content */
    if (Config.useDVDdevice==FALSE) {
        g_string_sprintf(temp,"%s",Config.DVDsource->str);
    } else {
        g_string_sprintf(temp,"%s",Config.DVDdevice->str);
    }
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Opening DVD content at %s (1)",temp->str);
    if (Config.conversion_mode == MPEG_PLAIN) {
        /* Fetch input file for MPEG_PLAIN input */
        manual_stream_select(Config.DVDsource->str);
    }

    /* Unset quitting flag */
    quit_converting = FALSE;
    #ifdef DEBUG
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Selected files = %i",selected_files);
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"conversion_mode = %i",Config.conversion_mode);
    #endif
    /* --- Anything to do? --- */
    if ((selected_files) > 0 && (stopped==FALSE)) {
        if (Config.conversion_mode != MPEG_PLAIN) {
            /* Get info for selected dvd title */
            if ((Config.conversion_mode == DVDTITLE_DVDDISK) || (Config.conversion_mode == DVDTITLE_DVDIMAGE) || (Config.conversion_mode == DVDTITLE_DVD2VCDDISK)) {
                parse_ifo(Config.VT_belongs_to_VTS[Config.CS_belongs_to_VT[Config.CS_current]],TRUE,FALSE);
            }
            /* Log languages */
            for (i=0;i<Config.audio_channels_in_dvd[Config.VT_belongs_to_VTS[Config.CS_belongs_to_VT[Config.CS_current]]];i++) {
                g_string_sprintf(temp,"Audio channel %i is %s",i,Config.audio_channel_names[Config.VT_belongs_to_VTS[Config.CS_belongs_to_VT[Config.CS_current]]][i]->str);
                g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,temp->str);
            }
            g_string_sprintf(temp,"%s audio (source channel %i) to 1st avi audio",Config.audio_channel_names[Config.VT_belongs_to_VTS[Config.CS_belongs_to_VT[Config.CS_current]]][Config.audio_1st_stream]->str,Config.audio_1st_stream);
            g_log(DRIP_LD,G_LOG_LEVEL_INFO,temp->str);
            if (Config.audio_2nd_stream!=-1) {
                g_string_sprintf(temp,"%s audio (source channel %i) to 2nd avi audio",Config.audio_channel_names[Config.VT_belongs_to_VTS[Config.CS_belongs_to_VT[Config.CS_current]]][Config.audio_2nd_stream]->str,Config.audio_2nd_stream);
                g_log(DRIP_LD,G_LOG_LEVEL_INFO,temp->str);
            }

            /* Log subpictures */
            for (i=0;i<Config.subpictures_in_dvd[Config.VT_belongs_to_VTS[Config.CS_belongs_to_VT[Config.CS_current]]];i++) {
                g_string_sprintf(temp,"Subpicture %i is %s",i,Config.subpicture_names[Config.VT_belongs_to_VTS[Config.CS_belongs_to_VT[Config.CS_current]]][i]->str);
                g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,temp->str);
            }
            if (Config.subpictures_in_dvd[Config.VT_belongs_to_VTS[Config.CS_belongs_to_VT[Config.CS_current]]]>0 && Config.subpicture_stream>-1) {
                g_string_sprintf(temp,"%s subpicture (source spu %i)",Config.subpicture_names[Config.VT_belongs_to_VTS[Config.CS_belongs_to_VT[Config.CS_current]]][Config.subpicture_stream]->str,Config.subpicture_stream);
                g_log(DRIP_LD,G_LOG_LEVEL_INFO,temp->str);
            }
        } else {
            /* conversion mode = MPEG_PLAIN */
            Config.CS_current = 0;
            Config.VT_belongs_to_VTS[Config.CS_belongs_to_VT[Config.CS_current]] = 1;
            Config.subpictures_in_dvd[Config.VT_belongs_to_VTS[Config.CS_belongs_to_VT[Config.CS_current]]] = 0;
        }

        /* Check for free space (100Mb min, 1Gb warning) */
        if (free_space(Config.CACHElocation->str,100*1024,1024*1024,TRUE) == FALSE) {
            g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Not enough free space available (min 100Mb)"); 
            /* Clean & exit */
            if (Config.DVDhandle!=NULL) {
                DVDClose(Config.DVDhandle);
                Config.DVDhandle = NULL;
            }
            pthread_exit(0);
        }


        /* ------------------------- Start converting --------------------- */


        if (Config.conversion_mode == MPEG_PLAIN) {
            /* -----------------------------------------------------------------
             * CONVERT DIRECTLY FROM SOURCE TO DIVX (source = dvdrom || cached files)
             * - if encoding is needed
             * - if direct conversion is requested
             * - if caching already has been done and we just need to encode
             */
            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Started Direct Encoding");
            g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Can't determine amount of frames in a plain MPEG file");
            drip_lock("conversion_thread5");
            gnome_appbar_set_status(GNOME_APPBAR(appbar1),"   Encoding DVD sources to DIVX. Full movies take a long time");
            drip_unlock("conversion_thread5");
            g_string_sprintf(command,"dripencoder %s ",mpeg2divx_parameters(-1));
            /* Input file */
            for (cnt = 0; cnt < selected_files; ++cnt) {
                command = g_string_append(command,Config.CS_files[0][cnt]->str);
                if (cnt!=(selected_files-1))
                    command = g_string_append(command,":");
                else
                    command = g_string_append(command," ");
            }

            /* Output filename */
            format_divxname(Config.DIVXfileUser,TRUE);
            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Generated AVI filename: %s",Config.DIVXfileFormatted->str);
            g_string_sprintf(title,"Drip %s  |  MPEG %s",VERSION,Config.DIVXfileFormatted->str);
            drip_lock("conversion_thread5.2");
            gtk_window_set_title((GtkWindow*)mainwindow,title->str);
            drip_unlock("conversion_thread5.2");
            command = g_string_append(command,Config.DIVXfileFormatted->str);
            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Activating backend: %s",command->str);
            /* Tell encoder to start encoding */
            encoder_start(command->str,Config.nice);
        } 

        if (Config.conversion_mode == DVDTITLE_DRIPCACHE) {
            /* -----------------------------------------------------------------
             * Use a drip cache set (dripcache_XX.vob, CLUT.tbl, DVDTITLE.tbl)
             * as input for the encoder backend
             */
            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Started Encoding of previously cached data");
            drip_lock("conversion_thread6");
            gnome_appbar_set_status(GNOME_APPBAR(appbar1),"   Encoding DVD sources to DIVX. Full movies take a long time");
            drip_unlock("conversion_thread6");
            /* Loop selected titles */
            for (Config.CS_current=0;Config.CS_current<Config.DVD_Total_ChapterSequences;Config.CS_current++) {
                if (Config.CS[Config.CS_current]==TRUE) {
                        if (quit_converting==TRUE) {
                            /* Clean & exit */
                            #ifdef DEBUG
                            g_log(DRIP_LD,G_LOG_LEVEL_INFO,"conversion_thread, exiting");
                            #endif
                            if (Config.DVDhandle!=NULL) {
                                DVDClose(Config.DVDhandle);
                                Config.DVDhandle = NULL;
                            }
                            pthread_exit(0);
                        }

                        /* Define CLUT file */
                        g_string_sprintf(Config.clutfile,"%s/CLUT%i.tbl",Config.DVDsource->str,Config.VT_belongs_to_VTS[Config.CS_belongs_to_VT[Config.CS_current]]);

                        /* Wait for encoder to be idle, this is used when backing up multiple titles only */
                        while (encoding) {
                            sleep(5);
                            #ifdef DEBUG
                            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"After save_dvd_title_info(TRUE)");
                            #endif
                        }

                        /* Show progres frame buttons */
                        drip_lock("conversion_thread0");
                        gtk_widget_show(hbox35);
                        drip_unlock("conversion_thread0");

                        /* Output filename */
                        format_divxname(Config.DIVXfileUser,TRUE);
                        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Generated AVI filename: %s",Config.DIVXfileFormatted->str);
                        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Encoding CS %i",Config.CS_current);
                        g_string_sprintf(title,"Drip %s  |  %s ",VERSION,Config.DIVXfileFormatted->str);
                        drip_lock("conversion_thread5.3");
                        gtk_window_set_title((GtkWindow*)mainwindow,title->str);
                        drip_unlock("conversion_thread5.3");
                        /* Backend command string */
                        g_string_sprintf(command,"dripencoder %s ",mpeg2divx_parameters(Config.CS_current+1));
                        /* Input file */
                        cnt = 0;
                        while (Config.CS_files[Config.CS_current][cnt] != NULL) {
                            command = g_string_append(command,Config.CS_files[Config.CS_current][cnt]->str);
                            if (Config.CS_files[Config.CS_current][cnt+1] != NULL)
                                command = g_string_append(command,":");
                            else
                                command = g_string_append(command," ");
                            cnt++;
                        }
                        command = g_string_append(command,Config.DIVXfileFormatted->str);
                        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Activating AVI backend: %s",command->str);

                        /* Tell encoder to start encoding */
                        converting = TRUE;
                        encoder_start(command->str,Config.nice);
                }
            }
        }

        if ( Config.conversion_mode == DVDTITLE_DVDDISK || Config.conversion_mode == DVDTITLE_DVDIMAGE ) {
            /* ----------------------------------------------------------------
             * CACHE DVD STREAMS TO HD 
             * and start AVI encoder once done 
             */
            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"CACHE DVD STREAMS TO HD");
            selected_files = 0;
            for (Config.CS_current=0;Config.CS_current<RANGE;Config.CS_current++) { //Config.DVD_Total_ChapterSequences;Config.CS_current++) {
                g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Trying to transcode part %i",Config.CS_current);
                avi_list = Config.selected_chapters[Config.CS_current+1];
                if (avi_list && avi_list->data) { //Config.CS[Config.CS_current]==1) {
                    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"avi_list && avi_list->data = TRUE");
                    /* Need to quit? */
                    if (quit_converting==TRUE) {
                        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Exiting because quit_converting==TRUE");
                        /* Clean & exit */
                        if (Config.DVDhandle!=NULL) {
                            DVDClose(Config.DVDhandle);
                            Config.DVDhandle = NULL;
                        }
                        pthread_exit(0);
                    }
                    /* Display progress */
                    cacheing = TRUE;
                    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Caching DVD stream(s)");
                    drip_lock("conversion_thread7");
                    gnome_appbar_set_status(GNOME_APPBAR(appbar1),"   Caching DVD stream...");
                    drip_unlock("conversion_thread7");

                    /* Cache source streams */
                    result = read_dvd_title(Config.CS_current+1);
                    if (result == FALSE) stopped = TRUE;
                    /* shutting down ? */
                    if (stopped==TRUE) goto exit;
                    /* set caching flag */
                    cacheing = FALSE;
                    /* Write out DVD titles info file, used for 'Encode from location' mode */
                    drip_lock("conversion_thread8");
                    gnome_appbar_set_status(GNOME_APPBAR(appbar1),"   Writing info about cached title");
                    drip_unlock("conversion_thread8");
                    save_dvd_title_info(TRUE);
                    /* Encode mpeg2 cache */
                    if (Config.encode) {
                        /* Wait for encoder to be idle, this is used when backing up multiple titles only */
                        while (encoding) {
                            sleep(5);
                            #ifdef DEBUG
                            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"After save_dvd_title_info(TRUE)");
                            #endif
                        }
                        /* Show progres frame buttons */
                        drip_lock("conversion_thread0");
                        gtk_widget_show(hbox35);
                        drip_unlock("conversion_thread0");

                        /* Output filename */
                        format_divxname(Config.DIVXfileUser,TRUE);
                        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Generated AVI filename: %s",Config.DIVXfileFormatted->str);
                        g_string_sprintf(title,"Drip %s  |  %s",VERSION,Config.DIVXfileFormatted->str);
                        drip_lock("conversion_thread7.2");
                        gtk_window_set_title((GtkWindow*)mainwindow,title->str);
                        drip_unlock("conversion_thread7.2");
                        encoding_time = FALSE;
                        converting = TRUE;
                        /* Define CLUT file */
                        g_string_sprintf(Config.clutfile,"%s/CLUT%i.tbl",Config.DVDsource->str,Config.VT_belongs_to_VTS[Config.CS_belongs_to_VT[Config.CS_current]]-1);
                        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"CLUT = %s",Config.clutfile->str);

                        /* Build filename\parameters */
                        g_string_sprintf(command,"dripencoder %s ",mpeg2divx_parameters(Config.CS_current+1));
                        for (cnt = 0; cnt<selected_files; ++cnt) {
                            command = g_string_append(command,Config.CS_files[0][cnt]->str);
                            if (cnt!=(selected_files-1))
                                command = g_string_append(command,":");
                            else
                                command = g_string_append(command," ");
                        }
                        /* Add output divx filename */
                        command = g_string_append(command,Config.DIVXfileFormatted->str);
                        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Activating backend: %s",command->str);
                        /* ---------- Tell encoder to start encoding */
                        drip_lock("conversion_thread9");
                        gnome_appbar_set_status(GNOME_APPBAR(appbar1),"   Initialising AVI encoding process...");
                        drip_unlock("conversion_thread9");
                        encoder_start(command->str,Config.nice);
                        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"returned from encoder_start");
                    } else {
                        /* Dont encode, only cache */
                        g_log(DRIP_LD,G_LOG_LEVEL_MESSAGE,"Ready.");
                        drip_lock("conversion_threadA");
                        gnome_appbar_set_status(GNOME_APPBAR(appbar1),"   Ready");
                        drip_unlock("conversion_threadA");
                        converting = FALSE;
                    }
                } else {
                    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"avi_list && avi_list->data = FALSE");
                }
            }

            exit:

            /* Eject DVDrom */
            if (Config.eject_dvdrom) {
                eject_dvdrom(Config.track_dir->str,Config.DVDdevice->str,TRUE);
            }
            if (stopped==TRUE) goto exit2;
        }

        if ( Config.conversion_mode == DVDTITLE_DVD2VCDDISK) {
            /* ----------------------------------------------------------------
             * CACHE DVD STREAMS TO HD 
             * and start encoder to (S)VCD once done 
             */
            g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"DVDTITLE_DVD2VCDDISK");
            selected_files = 0;
            for (Config.CS_current=0;Config.CS_current<Config.DVD_Total_ChapterSequences;Config.CS_current++) {
                if (Config.CS[Config.CS_current]==1) {
                    /* Need to quit? */
                    if (quit_converting==TRUE) {
                        /* Clean & exit */
                        if (Config.DVDhandle!=NULL) {
                            DVDClose(Config.DVDhandle);
                            Config.DVDhandle = NULL;
                        }
                        pthread_exit(0);
                    }
                    /* Display progress */
                    cacheing = TRUE;
                    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Caching DVD stream(s)");
                    drip_lock("conversion_thread7");
                    gnome_appbar_set_status(GNOME_APPBAR(appbar1),"   Caching DVD stream...");
                    drip_unlock("conversion_thread7");

                    /* Cache source streams */
                    result = read_dvd_title(Config.CS_current+1);
                    if (result == FALSE) stopped = TRUE;
                    /* shutting down ? */
                    if (stopped==TRUE) goto exit;
                    /* set caching flag */
                    cacheing = FALSE;
                    /* Write out DVD titles info file, used for 'Encode from location' mode */
                    drip_lock("conversion_thread8");
                    gnome_appbar_set_status(GNOME_APPBAR(appbar1),"   Writing info about cached title");
                    drip_unlock("conversion_thread8");
                    save_dvd_title_info(TRUE);
                    /* Encode mpeg2 cache */
                    if (Config.encode) {
                        /* Wait for encoder to be idle, this is used when backing up multiple titles only */
                        while (encoding) {
                            sleep(5);
                            #ifdef DEBUG
                            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"After save_dvd_title_info(TRUE)");
                            #endif
                        }
                        /* Show progres frame buttons */
                        drip_lock("conversion_thread0");
                        gtk_widget_show(hbox35);
                        drip_unlock("conversion_thread0");

                        /* Output filename */
                        format_divxname(Config.DIVXfileUser,TRUE);
                        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Generated AVI filename: %s",Config.DIVXfileFormatted->str);
                        g_string_sprintf(title,"Drip %s  |  %s",VERSION,Config.DIVXfileFormatted->str);
                        drip_lock("conversion_thread7.2");
                        gtk_window_set_title((GtkWindow*)mainwindow,title->str);
                        drip_unlock("conversion_thread7.2");
                        encoding_time = FALSE;
                        converting = TRUE;
                        /* Define CLUT file */
                        g_string_sprintf(Config.clutfile,"%s/CLUT%i.tbl",Config.DVDsource->str,Config.VT_belongs_to_VTS[Config.CS_belongs_to_VT[Config.CS_current]]);
                        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"CLUT = %s",Config.clutfile->str);

                        /* Build filename\parameters */
                        g_string_sprintf(command,"dripencoder %s ",mpeg2divx_parameters(-1));
                        for (cnt = 0; cnt<selected_files; ++cnt) {
                            command = g_string_append(command,Config.CS_files[0][cnt]->str);
                            if (cnt!=(selected_files-1))
                                command = g_string_append(command,":");
                            else
                                command = g_string_append(command," ");
                        }
                        /* Add output divx filename */
                        command = g_string_append(command,Config.DIVXfileFormatted->str);
                        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Activating backend: %s",command->str);
                        /* ---------- Tell encoder to start encoding */
                        drip_lock("conversion_thread9");
                        gnome_appbar_set_status(GNOME_APPBAR(appbar1),"   Initialising VCD encoding process...");
                        drip_unlock("conversion_thread9");
/* TODO */                       // encoder_start(command->str,Config.nice);
                    } else {
                        /* Dont encode, only cache */
                        g_log(DRIP_LD,G_LOG_LEVEL_MESSAGE,"Ready.");
                        drip_lock("conversion_threadA");
                        gnome_appbar_set_status(GNOME_APPBAR(appbar1),"   Ready");
                        drip_unlock("conversion_threadA");
                        converting = FALSE;
                    }
                }
            }
        }

        if ( Config.conversion_mode == DVDTITLE_DRIPCACHE2VCD) {
            /* ----------------------------------------------------------------
             * Use a drip cache set (dripcache_XX.vob, CLUT.tbl, DVDTITLE.tbl)
             * as input for the (S)VCD encoder backend 
             */
            g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"DVDTITLE_DRIPCACHE2VCD");
            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Started Encoding of previously cached data");
            drip_lock("conversion_thread6");
            gnome_appbar_set_status(GNOME_APPBAR(appbar1),"   Encoding DVD sources to DIVX. Full movies take a long time");
            drip_unlock("conversion_thread6");
            /* Loop selected titles */
            for (Config.CS_current=0;Config.CS_current<Config.DVD_Total_ChapterSequences;Config.CS_current++) {
                if (Config.CS[Config.CS_current]==TRUE) {
                        if (quit_converting==TRUE) {
                            /* Clean & exit */
                            #ifdef DEBUG
                            g_log(DRIP_LD,G_LOG_LEVEL_INFO,"conversion_thread, exiting");
                            #endif
                            if (Config.DVDhandle!=NULL) {
                                DVDClose(Config.DVDhandle);
                                Config.DVDhandle = NULL;
                            }
                            pthread_exit(0);
                        }

                        /* Define CLUT file */
                        g_string_sprintf(Config.clutfile,"%s/CLUT%i.tbl",Config.DVDsource->str,Config.VT_belongs_to_VTS[Config.CS_belongs_to_VT[Config.CS_current]]);

                        /* Wait for encoder to be idle, this is used when backing up multiple titles only */
                        while (encoding) {
                            sleep(5);
                            #ifdef DEBUG
                            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"After save_dvd_title_info(TRUE)");
                            #endif
                        }

                        /* Show progres frame buttons */
                        drip_lock("conversion_thread0");
                        gtk_widget_show(hbox35);
                        drip_unlock("conversion_thread0");

                        /* Output filename */
                        format_divxname(Config.DIVXfileUser,TRUE);
                        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Generated AVI filename: %s",Config.DIVXfileFormatted->str);
                        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Encoding CS %i",Config.CS_current);
                        g_string_sprintf(title,"Drip %s  |  %s ",VERSION,Config.DIVXfileFormatted->str);
                        drip_lock("conversion_thread5.3");
                        gtk_window_set_title((GtkWindow*)mainwindow,title->str);
                        drip_unlock("conversion_thread5.3");
                        /* Backend command string */
/* TODO */                        g_string_sprintf(command,"dripencoder %s ",mpeg2divx_parameters(-1));
                        /* Input file */
                        for (cnt = 0; cnt<selected_files; ++cnt) {
                            command = g_string_append(command,Config.CS_files[Config.CS_current][cnt]->str);
                            if (cnt!=(selected_files-1))
                                command = g_string_append(command,":");
                            else
                                command = g_string_append(command," ");
                        }
                        command = g_string_append(command,Config.DIVXfileFormatted->str);
                        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Activating VCD backend: %s",command->str);

                        /* Tell encoder to start encoding */
                        converting = TRUE;
/* TODO */                        //encoder_start(command->str,Config.nice);
                }
            }
        }

    } else {
        /* NO TRACKS?? This should be handled before IP ever reached this point */
        converting = FALSE;
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"NO TRACKS?? This should be handled before IP ever reached this point");
    }

    exit2: 

    /* Clean & exit */
    if (Config.DVDhandle!=NULL) {
        DVDClose(Config.DVDhandle);
        Config.DVDhandle = NULL;
    }
    g_string_free(command,TRUE);
    g_string_free(temp,TRUE);
    g_string_free(cached_files,TRUE);
    pthread_exit(0);
}

#ifdef LM_SENSORS
gpointer
heat_thread(gpointer data) {
    /* Thread that monitors lm_sensors if availeble and pauses encoding
       if temp1 > 50deg */
    gint result;
    glong temp_long;
    GString *temp;
    
    g_log(DRIP_LD,G_LOG_LEVEL_MESSAGE,"Hardware monitoring started");
    while (converting) {
        temp = drip_system("sensors | grep temp1 | awk '{printf($2)}'",&result,FALSE,FALSE);
        temp_long = strtol(temp->str,NULL,0);
        if (temp_long>49) {
            g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"CPU temp too high, sleeping...");
            encoder_pause(TRUE);
            paused = TRUE;
            while (temp_long>49) {
                sleep(30);
                temp = drip_system("sensors | grep temp1 | awk '{printf($2)}'",&result,FALSE,FALSE);
                temp_long = strtol(temp->str,NULL,0);
            }
            encoder_resume(TRUE);
            paused = FALSE;
        }
        sleep(60);
    }

    /* Exit */
    g_log(DRIP_LD,G_LOG_LEVEL_MESSAGE,"Hardware monitoring stopped");
    pthread_exit(0);
}
#endif /* LM_SENSORS */

