import java.util.Arrays;

/**
 * Class to act as a structure for holding file magic information.
 *
 * @author Ben Secrest
 * @version $Id: FileMagic.java,v 1.2 2002/08/26 02:32:24 blsecres Exp $
 */
public class FileMagic {
    /** The bytes that make up a file's magic signature */
    private byte[][] magic;

    /** Determine if signature may be offset in the file */
    private boolean isOffset;

    /** Determines whether or not text data is case sensitive */
    private boolean caseSensitive;


    /**
     * Construct a FileMagic object with the given magic and attributes
     */
    public FileMagic(byte[][] signature, boolean offset, boolean useCase) {
	magic = signature;
	isOffset = offset;
	caseSensitive = useCase;
    }


    /**
     * Check if a given set of bytes matches this objects signatures.  If the
     * buffer is shorter than the magic signatures, it is not considered a
     * match even if all characters in buffer equal the equivalent characters
     * in the signature.  If the buffer is longer than the magic signatures and
     * no offset is allowed, the buffer will be truncated before comparison.
     *
     * @param buffer The buffer to search for matches
     * @return <tt>true</tt> if a match occurs, <tt>false</tt> otherwise.
     */
    public boolean magicMatches(byte[] buffer) {
	if (! isOffset) {
	    // set up a temporary array the length of the magic
	    byte[] tmp = new byte[magic[0].length];

	    // copy the beginning of the buffer to the temp array
	    for (int i = 0; i < magic[0].length; i++)
		tmp[i] = buffer[i];

	    // process all magic signatures looking for a match
	    for (int j = 0; j < magic.length; j++)
		if ((caseSensitive ? Arrays.equals(magic[j], tmp)
			    : equalsNoCase(magic[j], tmp)))
		    return true;

	    return false;
	} else {
	    return contains(buffer);
	}
    }


    /**
     * Determine if the two buffers are equal, ignoring case on text
     * @param buffer1 The first buffer to compare
     * @param buffer2 The second buffer to compare
     * @return <tt>true</tt> if the buffers are equal, <tt>false</tt> otherwise
     */
    private static boolean equalsNoCase(byte[] buffer1, byte[] buffer2) {
	if (buffer1.length != buffer2.length)
	    return false;

	for (int i = 0; i < buffer1.length; i++)
	    if (Character.toLowerCase((char) buffer1[i])
		    != Character.toLowerCase((char) buffer2[i]))
		return false;

	return true;
    }


    /**
     * Determine if a buffer contains the magic signature anywhere inside it.
     * @param buffer The buffer to search for matches
     * @return <tt>true</tt> if a match occurs, <tt>false</tt> otherwise.
     */
    private boolean contains(byte[] buffer) {
	byte[] tmp;

	// cycle through all characters in the buffer
	for (int i = 0; i < buffer.length; i++) {
	    // cycle through all magic signatures
	    for (int j = 0; j < magic.length
		    && (i + magic[j].length) < buffer.length; j++) {
		if (caseSensitive) {
		    if (buffer[i] == magic[j][0]) {
			tmp = new byte[magic[j].length];

			for (int k = 0; k < magic[j].length; k++)
			    tmp[k] = buffer[i + k];

			if (Arrays.equals(tmp, magic[j]))
			    return true;
		    }
		} else {
		    // not case sensitive, convert both to lower for comparison
		    if (Character.toLowerCase((char) buffer[i])
			    == Character.toLowerCase((char) magic[j][0])) {
			tmp = new byte[magic[j].length];

			for (int k = 0; k < magic[j].length; k++)
			    tmp[k] = buffer[i + k];

			if (equalsNoCase(tmp, magic[j]))
			    return true;
		    }
		}
	    }
	}

	return false;
    }
}
