/***************************************************************************
 *   Copyright (C) 2006 by Peter Penz                                      *
 *   peter.penz@gmx.at                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

// Design and implementation notes:
// Currently the bookmarks are managed in a redundant way with the current
// design: the listview widget contains the content of each bookmark and also
// the bookmarks are stored as value list. This allows a straight forward
// implementation for storing the bookmarks in
// BookmarksSettingsPage::applySettings(), but is not very nice from a design
// point of view. Each time a bookmark is modified or moved, this operation
// must be done twice for two models. In Qt4 Interview might be an interesting
// approach to solve this issue.

#include "bookmarkssettingspage.h"

#include <qlayout.h>
#include <kdialog.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qvbox.h>
#include <klocale.h>
#include <kpushbutton.h>
#include <klistview.h>
#include <assert.h>
#include "dolphinsettings.h"
#include "editbookmarkdialog.h"

BookmarksSettingsPage::BookmarksSettingsPage(QWidget*parent) :
    SettingsPageBase(parent),
    m_addButton(0),
    m_removeButton(0),
    m_moveUpButton(0),
    m_moveDownButton(0)
{
    QVBoxLayout* topLayout = new QVBoxLayout(parent, 2, KDialog::spacingHint());

    const int spacing = KDialog::spacingHint();

    QHBox* hBox = new QHBox(parent);
    hBox->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Fixed);
    hBox->setSpacing(spacing);
    hBox->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Ignored);

    m_listView = new KListView(hBox);
    m_listView->addColumn(i18n("Icon"));
    m_listView->addColumn(i18n("Name"));
    m_listView->addColumn(i18n("Location"));
    m_listView->setResizeMode(QListView::LastColumn);
    m_listView->setColumnAlignment(0, Qt::AlignHCenter);
    m_listView->setAllColumnsShowFocus(true);
    m_listView->setSorting(-1);
    connect(m_listView, SIGNAL(selectionChanged()),
            this, SLOT(updateButtons()));
    connect(m_listView, SIGNAL(pressed(QListViewItem*)),
            this, SLOT(slotBookmarkPressed(QListViewItem*)));
    connect(m_listView, SIGNAL(doubleClicked(QListViewItem*, const QPoint&, int)),
            this, SLOT(slotBookmarkDoubleClicked(QListViewItem*, const QPoint&, int)));

    QVBox* buttonBox = new QVBox(hBox);
    buttonBox->setSpacing(spacing);

    const QSizePolicy buttonSizePolicy(QSizePolicy::Preferred, QSizePolicy::Maximum);

    m_addButton = new KPushButton(i18n("Add..."), buttonBox);
    connect(m_addButton, SIGNAL(clicked()),
            this, SLOT(slotAddButtonClicked()));
    m_addButton->setSizePolicy(buttonSizePolicy);

    m_editButton = new KPushButton(i18n("Edit..."), buttonBox);
    connect(m_editButton, SIGNAL(clicked()),
            this, SLOT(slotEditButtonClicked()));
    m_editButton->setSizePolicy(buttonSizePolicy);

    m_removeButton = new KPushButton(i18n("Remove..."), buttonBox);
    connect(m_removeButton, SIGNAL(clicked()),
            this, SLOT(slotRemoveButtonClicked()));
    m_removeButton->setSizePolicy(buttonSizePolicy);

    m_moveUpButton = new KPushButton(i18n("Move Up"), buttonBox);
    connect(m_moveUpButton, SIGNAL(clicked()),
            this, SLOT(slotMoveUpButtonClicked()));
    m_moveUpButton->setSizePolicy(buttonSizePolicy);

    m_moveDownButton = new KPushButton(i18n("Move Down"), buttonBox);
    connect(m_moveDownButton, SIGNAL(clicked()),
            this, SLOT(slotMoveDownButtonClicked()));
    m_moveDownButton->setSizePolicy(buttonSizePolicy);

    // Add a dummy widget with no restriction regarding a vertical resizing.
    // This assures that the spacing between the buttons is not increased.
    new QWidget(buttonBox);

    topLayout->addWidget(hBox);

    // insert all editable bookmarks.
    const QValueList<Bookmark>& bookmarks = DolphinSettings::instance().bookmarks();
    QListViewItem* prev = 0;
    const int count = bookmarks.count();
    for (int i = 0; i < count; ++i) {
        const Bookmark& bookmark = bookmarks[i];
        if (bookmark.isEditable()) {
            // insert bookmark into listview
            QListViewItem* item = new QListViewItem(m_listView);
            item->setPixmap(0, SmallIcon(bookmark.icon()));
            item->setText(1, bookmark.name());
            item->setText(2, bookmark.url().prettyURL());
            m_listView->insertItem(item);
            if (prev != 0) {
                item->moveItem(prev);
            }
            prev = item;

            // remember bookmark in custom list
            m_bookmarks.append(bookmark);
        }
    }
    m_listView->setSelected(m_listView->firstChild(), true);

    updateButtons();
}


BookmarksSettingsPage::~BookmarksSettingsPage()
{
}

void BookmarksSettingsPage::applySettings()
{
    DolphinSettings::instance().updateEditableBookmarks(m_bookmarks);
}

void BookmarksSettingsPage::updateButtons()
{
    const QListViewItem* selectedItem = m_listView->selectedItem();
    const bool hasSelection = (selectedItem != 0);

    m_editButton->setEnabled(hasSelection);
    m_removeButton->setEnabled(hasSelection);

    const bool enableMoveUp = hasSelection &&
                              (selectedItem != m_listView->firstChild());
    m_moveUpButton->setEnabled(enableMoveUp);

    const bool enableMoveDown = hasSelection &&
                                (selectedItem != m_listView->lastChild());
    m_moveDownButton->setEnabled(enableMoveDown);
}

void BookmarksSettingsPage::slotBookmarkDoubleClicked(QListViewItem*,
                                                      const QPoint&,
                                                      int)
{
    slotEditButtonClicked();
}

void BookmarksSettingsPage::slotBookmarkPressed(QListViewItem* item)
{
    if (item == 0) {
        m_listView->setSelected(m_listView->currentItem(), true);
    }
}

void BookmarksSettingsPage::slotAddButtonClicked()
{
    Bookmark bookmark(i18n("New bookmark"), KURL(), "bookmark");
    if (EditBookmarkDialog::getBookmark(bookmark, i18n("Add Bookmark"))) {
        // insert bookmark into listview
        QListViewItem* item = new QListViewItem(m_listView);
        item->setPixmap(0, SmallIcon(bookmark.icon()));
        item->setText(1, bookmark.name());
        item->setText(2, bookmark.url().prettyURL());
        m_listView->insertItem(item);

        QListViewItem* lastItem = m_listView->lastChild();
        if (lastItem != 0) {
            item->moveItem(lastItem);
        }

        m_listView->setSelected(item, true);

        // remember bookmark in custom list
        const int index = selectedBookmarkIndex();
        m_bookmarks.insert(m_bookmarks.at(index), bookmark);
        updateButtons();
    }
}

void BookmarksSettingsPage::slotEditButtonClicked()
{
    const int index = selectedBookmarkIndex();
    assert(static_cast<int>(m_bookmarks.count()) > index);

    Bookmark& bookmark = m_bookmarks[index];
    if (EditBookmarkDialog::getBookmark(bookmark, i18n("Edit Bookmark"))) {
        QListViewItem* item = m_listView->selectedItem();
        assert(item != 0);
        item->setPixmap(0, SmallIcon(bookmark.icon()));
        item->setText(1, bookmark.name());
        item->setText(2, bookmark.url().prettyURL());
    }
}

void BookmarksSettingsPage::slotRemoveButtonClicked()
{
    // remove bookmark from custom list
    m_bookmarks.remove(m_bookmarks.at(selectedBookmarkIndex()));

    // remove bookmark from listview
    QListViewItem* selectedItem = m_listView->selectedItem();
    assert(selectedItem != 0);
    QListViewItem* nextItem = selectedItem->itemBelow();
    if (nextItem == 0) {
        nextItem = selectedItem->itemAbove();
    }

    m_listView->takeItem(selectedItem);
    if (nextItem != 0) {
        m_listView->setSelected(nextItem, true);
    }
}

void BookmarksSettingsPage::slotMoveUpButtonClicked()
{
    moveBookmark(-1);
}

void BookmarksSettingsPage::slotMoveDownButtonClicked()
{
    moveBookmark(+1);
}

int BookmarksSettingsPage::selectedBookmarkIndex() const
{
    int index = -1;

    QListViewItem* selectedItem = m_listView->selectedItem();
    if (selectedItem != 0) {
        index = 0;
        QListViewItem* item = m_listView->firstChild();
        while (item != selectedItem) {
            item = item->nextSibling();
            ++index;
        }
    }

    return index;
}

void BookmarksSettingsPage::moveBookmark(int direction)
{
    // this implementation currently only allows moving of bookmarks
    // one step up or down
    assert((direction >= -1) && (direction <= +1));

    // swap bookmarks in custom list
    const int index = selectedBookmarkIndex();
    const Bookmark bookmark = m_bookmarks[index + direction];
    m_bookmarks[index + direction] = m_bookmarks[index];
    m_bookmarks[index] = bookmark;

    // swap bookmarks in listview
    QListViewItem* selectedItem = m_listView->selectedItem();
    assert(selectedItem != 0);
    QListViewItem* item = (direction < 0) ? selectedItem->itemAbove() :
                                            selectedItem->itemBelow();
    assert(item != 0);

    QPixmap pixmap;
    if (item->pixmap(0) != 0) {
        pixmap = *(item->pixmap(0));
    }
    QString name(item->text(1));
    QString url(item->text(2));

    if (selectedItem->pixmap(0) != 0) {
        item->setPixmap(0, *(selectedItem->pixmap(0)));
    }
    item->setText(1, selectedItem->text(1));
    item->setText(2, selectedItem->text(2));

    selectedItem->setPixmap(0, pixmap);
    selectedItem->setText(1, name);
    selectedItem->setText(2, url);

    m_listView->setSelected(item, true);
}

#include "bookmarkssettingspage.moc"
