
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nickycopeland@hotmail.com> 1996,2010
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifndef GRANULARDCO_H
#define GRANULARDCO_H

#define GRANULARDCO_WAVE_SZE 1024
#define GRANULARDCO_WAVE_SZE_M 1023
#define GRANULARDCO_SYNC 0x01

#define GDCO_WAVE_COUNT 8
#define GDCO_WAVE_MASK 0x07

#define GDCO_WAVE_1	0x0001
#define GDCO_WAVE_2	0x0002
#define GDCO_WAVE_3	0x0004
#define GDCO_WAVE_4	0x0008
#define GDCO_WAVE_5	0x0010
#define GDCO_WAVE_6	0x0020
#define GDCO_WAVE_7	0x0040
#define GDCO_WAVE_8	0x0080
#define GDCO_WAVE_RAND 0x0100

#define GRAIN_COUNT 4096

typedef struct Grain {
	unsigned int flags;
	float *wave;
	int delay;
	float frequency;
	float gain;
	float gainstep;
} grain;

typedef struct BristolGRANULARDCO {
	bristolOPSpec spec;
	float *wave[8];
	grain grains[GRAIN_COUNT];
	unsigned int flags;
} bristolGRANULARDCO;

/*
 *	gtp = pointer into gain table. float.
 *	gts = step rate through gain table. float.
 *	wtp = index into wave table. float.
 *	wtr = step rate through wave table. float.
 *	wts = size of wave table. float.
 */
typedef struct GRT {
	float gtp;
	float gts;
	float wtp;
	float wtr;
	float wts;
	float *cwt;
	int cg;
	float cdt;
	float cgtp;
	float gain;
	int cri;
	int wave;
} grt;

#define GDCO_LEFT 0
#define GDCO_RIGHT 1

#define MAX_GRAIN_THREAD 16

typedef struct BristolGRANULARDCOlocal {
	int flags;
	/*
	 * running parameters, should be reset at note on.
	 */
	float cgain;
	grt runtime[MAX_GRAIN_THREAD + 1]; /* Max simultaneous grains */
} bristolGRANULARDCOlocal;

/*
 * This is the gain table. The original papers referred to having a gaussian
 * gain, starting from zero and having even distribution. Later implementations
 * used bastardisations of this including csound's rather dumb 'AR' envelope.
 * We are going to use a normalised, inverted cosine. This is a reasonable
 * approximation of the gaussian 'bellcurve' that could be improved upon in
 * later releases.
 *
 * This table is resampled for the different grain sizes.
 */
float quantumgain[GRANULARDCO_WAVE_SZE + 1] = {
 0.000000, 0.000038, 0.000151, 0.000339,0.000602, 0.000941, 0.001355, 0.001844,
 0.002408, 0.003047, 0.003760, 0.004549,0.005412, 0.006349, 0.007361, 0.008447,
 0.009607, 0.010841, 0.012149, 0.013530,0.014984, 0.016512, 0.018112, 0.019785,
 0.021530, 0.023347, 0.025236, 0.027196,0.029228, 0.031330, 0.033504, 0.035747,
 0.038060, 0.040443, 0.042895, 0.045416,0.048005, 0.050663, 0.053388, 0.056180,
 0.059039, 0.061965, 0.064957, 0.068014,0.071136, 0.074322, 0.077573, 0.080888,
 0.084265, 0.087705, 0.091208, 0.094771,0.098396, 0.102082, 0.105827, 0.109631,
 0.113495, 0.117416, 0.121396, 0.125432,0.129524, 0.133673, 0.137876, 0.142135,
 0.146447, 0.150812, 0.155230, 0.159699,0.164221, 0.168792, 0.173414, 0.178084,
 0.182803, 0.187570, 0.192384, 0.197244,0.202150, 0.207101, 0.212096, 0.217134,
 0.222215, 0.227338, 0.232501, 0.237705,0.242949, 0.248231, 0.253551, 0.258908,
 0.264302, 0.269731, 0.275194, 0.280692,0.286222, 0.291785, 0.297379, 0.303004,
 0.308658, 0.314341, 0.320052, 0.325791,0.331555, 0.337345, 0.343159, 0.348997,
 0.354858, 0.360740, 0.366644, 0.372567,0.378510, 0.384471, 0.390449, 0.396444,
 0.402455, 0.408480, 0.414519, 0.420571,0.426635, 0.432710, 0.438795, 0.444889,
 0.450991, 0.457101, 0.463218, 0.469340,0.475466, 0.481596, 0.487729, 0.493864,
 0.500000, 0.506136, 0.512271, 0.518404,0.524534, 0.530660, 0.536782, 0.542899,
 0.549009, 0.555111, 0.561205, 0.567290,0.573365, 0.579429, 0.585481, 0.591520,
 0.597545, 0.603556, 0.609551, 0.615529,0.621490, 0.627433, 0.633356, 0.639260,
 0.645142, 0.651003, 0.656841, 0.662655,0.668445, 0.674209, 0.679947, 0.685659,
 0.691342, 0.696996, 0.702621, 0.708215,0.713778, 0.719308, 0.724806, 0.730269,
 0.735698, 0.741092, 0.746449, 0.751769,0.757051, 0.762295, 0.767499, 0.772662,
 0.777785, 0.782866, 0.787904, 0.792899,0.797850, 0.802756, 0.807616, 0.812430,
 0.817197, 0.821916, 0.826586, 0.831208,0.835779, 0.840300, 0.844770, 0.849188,
 0.853553, 0.857865, 0.862124, 0.866327,0.870476, 0.874568, 0.878604, 0.882584,
 0.886505, 0.890369, 0.894173, 0.897918,0.901604, 0.905229, 0.908792, 0.912295,
 0.915735, 0.919112, 0.922427, 0.925678,0.928864, 0.931986, 0.935044, 0.938035,
 0.940961, 0.943820, 0.946612, 0.949337,0.951995, 0.954584, 0.957105, 0.959557,
 0.961940, 0.964253, 0.966496, 0.968670,0.970772, 0.972804, 0.974764, 0.976653,
 0.978470, 0.980215, 0.981888, 0.983488,0.985016, 0.986470, 0.987851, 0.989159,
 0.990393, 0.991553, 0.992639, 0.993651,0.994588, 0.995451, 0.996240, 0.996953,
 0.997592, 0.998156, 0.998645, 0.999059,0.999398, 0.999661, 0.999849, 0.999962,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 1.000000, 1.000000, 1.000000,1.000000, 1.000000, 1.000000, 1.000000,
 1.000000, 0.999962, 0.999849, 0.999661,0.999398, 0.999059, 0.998645, 0.998156,
 0.997592, 0.996953, 0.996240, 0.995451,0.994588, 0.993651, 0.992639, 0.991553,
 0.990393, 0.989159, 0.987851, 0.986470,0.985016, 0.983488, 0.981888, 0.980215,
 0.978470, 0.976653, 0.974764, 0.972804,0.970772, 0.968670, 0.966496, 0.964253,
 0.961940, 0.959557, 0.957105, 0.954584,0.951995, 0.949337, 0.946612, 0.943820,
 0.940961, 0.938035, 0.935044, 0.931986,0.928864, 0.925678, 0.922427, 0.919112,
 0.915735, 0.912295, 0.908792, 0.905229,0.901604, 0.897918, 0.894173, 0.890369,
 0.886505, 0.882584, 0.878604, 0.874568,0.870476, 0.866327, 0.862124, 0.857865,
 0.853553, 0.849188, 0.844770, 0.840300,0.835779, 0.831208, 0.826586, 0.821916,
 0.817197, 0.812430, 0.807616, 0.802756,0.797850, 0.792899, 0.787904, 0.782866,
 0.777785, 0.772662, 0.767499, 0.762295,0.757051, 0.751769, 0.746449, 0.741092,
 0.735698, 0.730269, 0.724806, 0.719308,0.713778, 0.708215, 0.702621, 0.696996,
 0.691342, 0.685659, 0.679947, 0.674209,0.668445, 0.662655, 0.656841, 0.651003,
 0.645142, 0.639260, 0.633356, 0.627433,0.621490, 0.615529, 0.609551, 0.603556,
 0.597545, 0.591520, 0.585481, 0.579429,0.573365, 0.567290, 0.561205, 0.555111,
 0.549009, 0.542899, 0.536782, 0.530660,0.524534, 0.518404, 0.512271, 0.506136,
 0.500000, 0.493864, 0.487729, 0.481596,0.475466, 0.469340, 0.463218, 0.457101,
 0.450991, 0.444889, 0.438795, 0.432710,0.426635, 0.420571, 0.414519, 0.408480,
 0.402455, 0.396444, 0.390449, 0.384471,0.378510, 0.372567, 0.366644, 0.360740,
 0.354858, 0.348997, 0.343159, 0.337345,0.331555, 0.325791, 0.320052, 0.314341,
 0.308658, 0.303004, 0.297379, 0.291785,0.286222, 0.280692, 0.275194, 0.269731,
 0.264302, 0.258908, 0.253551, 0.248231,0.242949, 0.237705, 0.232501, 0.227338,
 0.222215, 0.217134, 0.212096, 0.207101,0.202150, 0.197244, 0.192384, 0.187570,
 0.182803, 0.178084, 0.173414, 0.168792,0.164221, 0.159699, 0.155230, 0.150812,
 0.146447, 0.142135, 0.137876, 0.133673,0.129524, 0.125432, 0.121396, 0.117416,
 0.113495, 0.109631, 0.105827, 0.102082,0.098396, 0.094771, 0.091208, 0.087705,
 0.084265, 0.080888, 0.077573, 0.074322,0.071136, 0.068014, 0.064957, 0.061965,
 0.059039, 0.056180, 0.053388, 0.050663,0.048005, 0.045416, 0.042895, 0.040443,
 0.038060, 0.035747, 0.033504, 0.031330,0.029228, 0.027196, 0.025236, 0.023347,
 0.021530, 0.019785, 0.018112, 0.016512,0.014984, 0.013530, 0.012149, 0.010841,
 0.009607, 0.008447, 0.007361, 0.006349,0.005412, 0.004549, 0.003760, 0.003047,
 0.002408, 0.001844, 0.001355, 0.000941,0.000602, 0.000339, 0.000151, 0.000038,
 0.000000
 };

/*
 * This is the 'rand' table. It was generated using rand() and it is a table
 * such that we can have reproducable random numbers so that generated sounds
 * can be put into memories and recalled verbatim. Ok, rand can actualy be used
 * to do this, however rand is already used in other algorithms (preacher) and
 * unless the quantum generator can garantee ordered randomness it will not be
 * able to have memories, hence this table. The reason that we cannot use rand()
 * to generate this table is that rand() may already be arbitrarily seeded by
 * the same process. We will consider the use of rand_r() as a development.
 * HOWEVER:
 * This table should really be dropped. Rand is only required once or twice per
 * grain and is reasonably fast, at least on Linux. The issue is that the table
 * cannot remain resident in on-chip L1 cache hence there will be a page fault
 * that is rather slow to field. A better implementation would be to implement
 * a rand() function as a part of the oscillator, hence it will typically be
 * resident when the oscillator is running and can be made predictable.
 *
 * With 4K grains and an average cycle of 10ms (lots of short grains and some
 * long grains then cycle time through the table will be 45 seconds, enough to
 * prevent excessive cyclical noise for the oscillator. The value of 10ms is
 * likely to be on the low side (grains of ~2ms sound like clicks anyway
 * which are not going to be audibly 'nice') so the eventual cycle time will
 * be a lot longer.
 */
int quantumrand[GRAIN_COUNT] = {
0x6B8B4567, 0x327B23C6, 0x643C9869, 0x66334873,
0x74B0DC51, 0x19495CFF, 0x2AE8944A, 0X625558EC,
0x238E1F29, 0x46E87CCD, 0x3D1B58BA, 0x507ED7AB,
0x2EB141F2, 0x41B71EFB, 0x79E2A9E3, 0X7545E146,
0x515F007C, 0x5BD062C2, 0x12200854, 0x4DB127F8,
0x0216231B, 0x1F16E9E8, 0x1190CDE7, 0X66EF438D,
0x140E0F76, 0x3352255A, 0x109CF92E, 0x0DED7263,
0x7FDCC233, 0x1BEFD79F, 0x41A7C4C9, 0X6B68079A,
0x4E6AFB66, 0x25E45D32, 0x519B500D, 0x431BD7B7,
0x3F2DBA31, 0x7C83E458, 0x257130A3, 0X62BBD95A,
0x436C6125, 0x628C895D, 0x333AB105, 0x721DA317,
0x2443A858, 0x2D1D5AE9, 0x6763845E, 0X75A2A8D4,
0x08EDBDAB, 0x79838CB2, 0x4353D0CD, 0x0B03E0C6,
0x189A769B, 0x54E49EB4, 0x71F32454, 0X2CA88611,
0x0836C40E, 0x02901D82, 0x3A95F874, 0x08138641,
0x1E7FF521, 0x7C3DBD3D, 0x737B8DDC, 0X6CEAF087,
0x22221A70, 0x4516DDE9, 0x3006C83E, 0x614FD4A1,
0x419AC241, 0x5577F8E1, 0x440BADFC, 0X05072367,
0x3804823E, 0x77465F01, 0x7724C67E, 0x5C482A97,
0x2463B9EA, 0x5E884ADC, 0x51EAD36B, 0X2D517796,
0x580BD78F, 0x153EA438, 0x3855585C, 0x70A64E2A,
0x6A2342EC, 0x2A487CB0, 0x1D4ED43B, 0X725A06FB,
0x2CD89A32, 0x57E4CCAF, 0x7A6D8D3C, 0x4B588F54,
0x542289EC, 0x6DE91B18, 0x38437FDB, 0X7644A45C,
0x32FFF902, 0x684A481A, 0x579478FE, 0x749ABB43,
0x3DC240FB, 0x1BA026FA, 0x79A1DEAA, 0X75C6C33A,
0x12E685FB, 0x70C6A529, 0x520EEDD1, 0x374A3FE6,
0x4F4EF005, 0x23F9C13C, 0x649BB77C, 0X275AC794,
0x39386575, 0x1CF10FD8, 0x180115BE, 0x235BA861,
0x47398C89, 0x354FE9F9, 0x15B5AF5C, 0X741226BB,
0x0D34B6A8, 0x10233C99, 0x3F6AB60F, 0x61574095,
0x7E0C57B1, 0x77AE35EB, 0x579BE4F1, 0X310C50B3,
0x5FF87E05, 0x2F305DEF, 0x25A70BF7, 0x1DBABF00,
0x4AD084E9, 0x1F48EAA1, 0x1381823A, 0X5DB70AE5,
0x100F8FCA, 0x6590700B, 0x15014ACB, 0x5F5E7FD0,
0x098A3148, 0x799D0247, 0x06B94764, 0X42C296BD,
0x168E121F, 0x1EBA5D23, 0x661E3F1E, 0x5DC79EA8,
0x540A471C, 0x7BD3EE7B, 0x51D9C564, 0X613EFDC5,
0x0BF72B14, 0x11447B73, 0x42963E5A, 0x0A0382C5,
0x08F2B15E, 0x1A32234B, 0x3B0FD379, 0X68EB2F63,
0x4962813B, 0x60B6DF70, 0x06A5EE64, 0x14330624,
0x7FFFCA11, 0x1A27709E, 0x71EA1109, 0X100F59DC,
0x7FB7E0AA, 0x06EB5BD4, 0x6F6DD9AC, 0x094211F2,
0x00885E1B, 0x76272110, 0x4C04A8AF, 0X1716703B,
0x14E17E33, 0x3222E7CD, 0x74DE0EE3, 0x68EBC550,
0x2DF6D648, 0x46B7D447, 0x4A2AC315, 0X39EE015C,
0x57FC4FBB, 0x0CC1016F, 0x43F18422, 0x60EF0119,
0x26F324BA, 0x7F01579B, 0x49DA307D, 0X7055A5F5,
0x5FB8370B, 0x50801EE1, 0x0488AC1A, 0x5FB8011C,
0x6AA78F7F, 0x7672BD23, 0x6FC75AF8, 0X6A5F7029,
0x7D5E18F8, 0x5F3534A4, 0x73A1821B, 0x7DE67713,
0x555C55B5, 0x3FA62ACA, 0x14FCE74E, 0X6A3DD3E8,
0x71C91298, 0x09DAF632, 0x53299938, 0x1FBFE8E0,
0x5092CA79, 0x1D545C4D, 0x59ADEA3D, 0X288F1A34,
0x2A155DBC, 0x1D9F6E5F, 0x097E1B4E, 0x51088277,
0x1CA0C5FA, 0x53584BCB, 0x415E286C, 0X7C58FD05,
0x23D86AAC, 0x45E6D486, 0x5C10FE21, 0x0E7FFA2B,
0x3C5991AA, 0x4BD8591A, 0x78DF6A55, 0X39B7AAA2,
0x2B0D8DBE, 0x6C80EC70, 0x379E21B5, 0x0069E373,
0x2C27173B, 0x4C9B0904, 0x6AA7B75C, 0X1DF029D3,
0x5675FF36, 0x3DD15094, 0x3DB012B3, 0x2708C9AF,
0x5B25ACE2, 0x175DFCF0, 0x4F97E3E4, 0X053B0A9E,
0x34FD6B4F, 0x5915FF32, 0x56438D15, 0x519E3149,
0x2C6E4AFD, 0x17A1B582, 0x4DF72E4E, 0X5046B5A9,
0x5D888A08, 0x2A082C70, 0x5EC6AFD4, 0x19E21BB2,
0x75E0858A, 0x57A61A29, 0x5399C654, 0X20EE1348,
0x4427069A, 0x0B37E80A, 0x2157F6BC, 0x704E1DD5,
0x57D2F10E, 0x0BFFAE18, 0x0E3E47A8, 0X2E48F044,
0x49D0FEAC, 0x4BEE5A5B, 0x5551B9F3, 0x24F6AB8E,
0x634C574C, 0x24E99DD7, 0x2A31B62D, 0X1849C29B,
0x7DFF9D09, 0x00754342, 0x69E7F3E5, 0x2A6DE806,
0x1816F8C4, 0x37DF2233, 0x7AB49DAF, 0X759F82CD,
0x61E74EA3, 0x597B4D84, 0x0F819E7F, 0x57C7D42D,
0x312167AD, 0x631B64D4, 0x78B5E776, 0X75486E47,
0x6E534CDE, 0x1A0DDE32, 0x65968C1C, 0x46263DEC,
0x260D8C4A, 0x73D4D3C4, 0x746F2E30, 0X6FDE8AF6,
0x3FC32E20, 0x49C0E823, 0x14D53685, 0x230F856C,
0x6EAA85FB, 0x3F06ECB2, 0x3B594807, 0X6CAA2304,
0x3F7C2FF4, 0x25413BEC, 0x17180B0B, 0x579328B9,
0x5D205E20, 0x11CCA8BA, 0x4D32AB86, 0X3F07ACC3,
0x6B47F63E, 0x5CB44A05, 0x16CF80F1, 0x1C695DEC,
0x3FCFAED9, 0x0F856867, 0x11B1CC33, 0X2E22FBB7,
0x29934699, 0x77485850, 0x744939A3, 0x4FA0D2E3,
0x6B1D2C14, 0x68B867D3, 0x3F7F5DD9, 0X2AE05A34,
0x32794FF7, 0x5454945E, 0x4DEFDFA0, 0x2123D5F2,
0x135B8110, 0x094927A8, 0x0DCDF8F6, 0X52D7B105,
0x2E8A6394, 0x24E60401, 0x2A6AD9BE, 0x0BAAC1B4,
0x36B2ACBC, 0x779D8544, 0x4AB26E78, 0X21FAA2FA,
0x5451CF49, 0x6181EF69, 0x3E6400E6, 0x14217E23,
0x710757D0, 0x5015CD1A, 0x424479DA, 0X1A9A9E69,
0x475E256A, 0x368DB37E, 0x6A3B714C, 0x327B517E,
0x1F461B51, 0x29BACF25, 0x5D5BABB3, 0X51BF6B48,
0x7E0F6384, 0x2B4B8B53, 0x72E3413A, 0x116AE494,
0x3494B2FB, 0x00B13A31, 0x64429599, 0X631F1690,
0x25973E32, 0x0EAD6F57, 0x6EC9D844, 0x5C49EAEE,
0x064AF49B, 0x397C46BC, 0x7E448DE9, 0X5A9CC3E5,
0x1AFE3625, 0x3CA88ECF, 0x6EBE4208, 0x0C058DF5,
0x0CBE5BE9, 0x3102BBE2, 0x26A02C5E, 0X541C8153,
0x67906F60, 0x10DB9DAA, 0x0697D2D2, 0x06D68AB2,
0x3A966CD0, 0x63F37E85, 0x5895F5FA, 0X38A5D054,
0x0F3F09D8, 0x4B793735, 0x4A10B4E8, 0x43D3BCD4,
0x4C2A7166, 0x2E534A82, 0x26F2D364, 0X71C1AF98,
0x3D00B9D9, 0x15BCABA8, 0x4E0B9A87, 0x434BAE75,
0x4F38F265, 0x4C502870, 0x1DE8725A, 0X6A37288A,
0x08F8B73F, 0x0CA6B462, 0x763CB680, 0x15B71329,
0x3DA97044, 0x1CDCE2DE, 0x69D3947C, 0X2539DFA5,
0x2DB88089, 0x706B674E, 0x2C106A57, 0x684EED59,
0x545EE5D3, 0x04A66051, 0x20F4BDAD, 0X639DEFAC,
0x501F9786, 0x6B057295, 0x2771AC80, 0x1C4A08EC,
0x1958BD17, 0x4E647FE4, 0x0E0BB885, 0X565976F1,
0x64212B8C, 0x5C17530C, 0x19A52566, 0x335A1DF1,
0x28677B7C, 0x378D97C0, 0x1D91467C, 0X316032BB,
0x44344C22, 0x13CDFCFC, 0x471745E4, 0x01DDBC66,
0x30AADFDA, 0x30EADA61, 0x27179C0B, 0X5E636063,
0x215641AF, 0x53280662, 0x46B24DBC, 0x75B52783,
0x57CE66B4, 0x67A70B69, 0x5953172F, 0X27EDFE3A,
0x52AC7DFF, 0x00C4C3AF, 0x44380727, 0x6C053B16,
0x4F294393, 0x5243BFAC, 0x425EB207, 0X334A6F1F,
0x2E5B12B8, 0x5C03D76D, 0x66A48D11, 0x56C28E34,
0x13916F2D, 0x0435D38D, 0x0822C0EF, 0X57C5BB4F,
0x1803D089, 0x4F3A06D4, 0x59A377B6, 0x48AEB063,
0x0024E135, 0x00BB13C1, 0x271210C7, 0X217B22E4,
0x53E31A24, 0x6DC45E83, 0x17304A67, 0x2BB180D8,
0x556B69ED, 0x70836196, 0x539F7F12, 0X2817E7EC,
0x71482545, 0x17D78639, 0x141D2302, 0x407168D8,
0x6A1B45E5, 0x567BD50A, 0x73BBD7F8, 0X1876589D,
0x327FAC77, 0x5A606509, 0x6F38E6D1, 0x46111BA5,
0x5E963896, 0x775BA7C1, 0x1DD6D6F4, 0X769A091F,
0x4695AE95, 0x777A4EAA, 0x3F48B982, 0x46BA8FCA,
0x7835626C, 0x665ACA49, 0x6835B2AE, 0X4C187C90,
0x541F28CD, 0x7F65FD16, 0x77C9FD68, 0x298A92BA,
0x6FE95EAC, 0x4B697C7A, 0x51A27AA6, 0X613183F2,
0x634102B4, 0x65BF9DA8, 0x21A2ECCA, 0x4D5C4899,
0x3C3B72B2, 0x155EC4C2, 0x65D2A137, 0X6EBB1F2A,
0x6FBF29CB, 0x550B8808, 0x34CC3ACF, 0x4E556261,
0x4C672FC9, 0x52A311C3, 0x44EF6B80, 0X12FCDE5E,
0x4A1D606E, 0x04382503, 0x59B76E28, 0x4252C2DA,
0x6A92EF4C, 0x41ED20D7, 0x0E6B3F6A, 0X3EB21819,
0x41531DED, 0x06353CD2, 0x683CAAD3, 0x313C7C99,
0x519EB94C, 0x39DF2579, 0x126E008B, 0X34DFBC00,
0x1F9EC322, 0x3410ED56, 0x023C049A, 0x5BDA35D4,
0x496FB218, 0x680EA5D1, 0x4A9554FE, 0X392EDBE4,
0x3D1A2DD9, 0x7F618FCD, 0x07843E45, 0x09815DA3,
0x5204A191, 0x4C73A9C6, 0x1C7E3C01, 0X1C2201FF,
0x50ABCEC9, 0x7635AA2A, 0x5E74C4D9, 0x3B3EBE15,
0x3822CB01, 0x6CE00443, 0x79F0D62F, 0X7975E8EE,
0x73154115, 0x622D8102, 0x2AB26587, 0x44B3FA61,
0x1C0CA67C, 0x3D206613, 0x7993B662, 0X3BAB699E,
0x71315369, 0x7BCFBAFC, 0x17859F72, 0x3AA10581,
0x63DE60CD, 0x621AF471, 0x73CFE165, 0X20F88EA6,
0x617C843E, 0x7B541FAB, 0x2A79EC49, 0x338125CF,
0x47C7C971, 0x46F8284B, 0x4FA327CE, 0X1873983A,
0x3D2DD275, 0x2E17ECA7, 0x53B2564F, 0x75509D76,
0x1AF7F0EA, 0x4DA32C7E, 0x6EC68664, 0X0E0D31FF,
0x2FD0AD81, 0x1978EBEB, 0x52C12C61, 0x4BDD53FD,
0x569951FE, 0x4C54E2C3, 0x0788BD9B, 0X47CAA567,
0x48249DBF, 0x1F0E5D0D, 0x026BAAE9, 0x2C02FE8C,
0x0129517E, 0x763B8C4E, 0x4CFB8D32, 0X62A5D5BD,
0x718FABF9, 0x7775797C, 0x1626FB8C, 0x3957756A,
0x3E6DA1C7, 0x65CA235B, 0x51CB0DA4, 0X7B9B743C,
0x13E21002, 0x257D63F4, 0x70EC11B2, 0x2EDA00ED,
0x73209072, 0x5FB29816, 0x3CE732EC, 0X22F13DF3,
0x792B8401, 0x0FA85F4D, 0x6ECE91F0, 0x4FC4D600,
0x5BFD4210, 0x76574F8B, 0x178F7B67, 0X2421DFCF,
0x1565AC99, 0x19FB2650, 0x5024DE5B, 0x168EFE17,
0x1036B29F, 0x1D206B8E, 0x7934D3D4, 0X01C65E98,
0x1495E50A, 0x0F5BCF61, 0x3B1DD403, 0x530386D1,
0x7525F2BC, 0x0CE8E1A7, 0x4E9EFB0D, 0X090802BE,
0x3266459B, 0x3F8B0CBF, 0x37E203AB, 0x2586D60E,
0x1F3DA4D5, 0x74C93698, 0x48781401, 0X186928D6,
0x047195E5, 0x3746A5F2, 0x682DFED6, 0x606ED7F6,
0x2D9DF57D, 0x7FBD7A3E, 0x0490B7C5, 0X4303A216,
0x19B8A08E, 0x54B59621, 0x5992A02E, 0x29EF532D,
0x71D601AF, 0x52C77402, 0x2BB5B1C6, 0X066BE6B9,
0x62234363, 0x66D385C9, 0x596F6D8A, 0x5749361F,
0x73BC6770, 0x280E6897, 0x605138DE, 0X2622AD0C,
0x67997556, 0x18333C89, 0x4BA9831A, 0x06D71A2B,
0x0CFC7321, 0x1421971B, 0x1F404301, 0X116E0907,
0x4B683D0D, 0x076E41D8, 0x71DCE0FD, 0x7906328B,
0x072BBC16, 0x766D98C2, 0x3C09D4A1, 0X20E45CA4,
0x4B232EE3, 0x159C74CF, 0x4AD3AFD2, 0x3CF93092,
0x6863E8D2, 0x76896198, 0x4365174B, 0X4A872C35,
0x5D5CE761, 0x1CD484D5, 0x21D06255, 0x51194ED1,
0x44E2ED6C, 0x02219B33, 0x773BFBDD, 0X2C7C62C2,
0x1A54D7BC, 0x42E57EF7, 0x33537CED, 0x27514ADE,
0x57071613, 0x5293BFEF, 0x38BF53E5, 0X226F5320,
0x5A0201C7, 0x2A9C34E2, 0x1B7585AB, 0x612DBDDD,
0x2109CDA4, 0x577F5A4D, 0x02121A81, 0X6C2CFC88,
0x6D1BCF1C, 0x4CE5CA53, 0x29262D1A, 0x557FB7EE,
0x436F2BEB, 0x6C8B4466, 0x2006E424, 0X20CC134C,
0x095FC93B, 0x41D74679, 0x71E5621E, 0x4E42B6A8,
0x43F8E1AC, 0x69215DFB, 0x7ABF196A, 0X5E4DB968,
0x2C06DCF3, 0x2E129658, 0x059F0446, 0x030DF306,
0x00A65647, 0x3E5E582B, 0x257D4626, 0X5AA8580E,
0x68FA8D0D, 0x40F2CBD2, 0x3BD615EB, 0x0A045AB2,
0x1872261F, 0x3DE8306C, 0x7631573A, 0X058DF53B,
0x0ACDFAC0, 0x1F578454, 0x5B0DAD2A, 0x4E3D26AB,
0x0BE2C8BA, 0x7B14914E, 0x6F0939F8, 0X154291F6,
0x3CEBD7C7, 0x60EE9C16, 0x6385489E, 0x00E4B973,
0x4A0FFA11, 0x5E446208, 0x5F3272DB, 0X7616D704,
0x0C56F860, 0x64D17722, 0x7924CA0A, 0x0CFD4EA7,
0x232FCF4D, 0x1EA21031, 0x67A5A6B5, 0X0C2A5C5B,
0x5F94DC03, 0x237BBCA0, 0x162EB70D, 0x78070222,
0x6163ED0D, 0x0C600E47, 0x7D94F75D, 0X6C31E7CD,
0x2BB7929B, 0x58A2A487, 0x3A6F0E78, 0x379A5B56,
0x53B735D5, 0x29784870, 0x4CDCED4C, 0X10A30D9C,
0x0A66E486, 0x306235EA, 0x1187C70F, 0x5476DE98,
0x0EA697F2, 0x70BA39EB, 0x4A8DB59C, 0X1AFD9053,
0x558BB10D, 0x43B27FA7, 0x27FADEFA, 0x78BB805A,
0x62548FD8, 0x0FA085B0, 0x04E5DCB5, 0X41E96BDB,
0x331C4250, 0x1B1493C2, 0x39F06DFD, 0x14802F5D,
0x2774A209, 0x3785655A, 0x00B2172A, 0X532C34A5,
0x102809E2, 0x3B2125A3, 0x0AC68FFB, 0x63DF3FB7,
0x64996E13, 0x57A37D47, 0x74824D54, 0X6F00529A,
0x0805B331, 0x060A1463, 0x43773132, 0x16AC4B23,
0x76C44E4E, 0x0E04E6CE, 0x31A9DB76, 0X4C4FFF5B,
0x51B76675, 0x59A4BA71, 0x450B7FB6, 0x340BF64D,
0x69454021, 0x49F15C6B, 0x75F56228, 0X1C618271,
0x6505F02E, 0x2FE5D025, 0x30E1B1CF, 0x0C7A9237,
0x676B3580, 0x3193C8F9, 0x5FA6C6DC, 0X77933F62,
0x6CB4EE9C, 0x6A6D56D7, 0x5B727F19, 0x514E5CB0,
0x4210D41E, 0x4FF4CC6D, 0x404EAF4A, 0X4A16874F,
0x55FEE0D1, 0x03C5E07C, 0x60C2D273, 0x4CC32F1F,
0x11CAC74A, 0x126CADE9, 0x19132E7B, 0X63822DC0,
0x6C11685A, 0x5E1EAE31, 0x178E240D, 0x5556A87B,
0x28100A9C, 0x0D838636, 0x71B82AED, 0X0D15FACA,
0x3D69565B, 0x2299DCBC, 0x19908D02, 0x24D48BDB,
0x542DA5B5, 0x793753DE, 0x1C67CB3D, 0X40E29452,
0x63A4AAB6, 0x77DA4A57, 0x1230F102, 0x25B57ED4,
0x47CF16C4, 0x527FA04C, 0x6FCC0624, 0X1DCDF795,
0x564580C8, 0x508ED897, 0x6A9126B5, 0x68104812,
0x62FB8680, 0x03A45530, 0x4B9275D2, 0X4F0CEEDB,
0x61C30361, 0x632099E0, 0x24639756, 0x09D30DFD,
0x70A42016, 0x161BC243, 0x16E908C8, 0X2E0D7671,
0x38B59EFF, 0x307995CA, 0x52E2024D, 0x0CE344B5,
0x29B0E9A8, 0x6F49CD8A, 0x4DC5D907, 0X0D55945E,
0x672417E1, 0x5FF6CA09, 0x330B1333, 0x2EF32EA6,
0x32766A55, 0x22D71957, 0x4CC1263B, 0X08BBEB1D,
0x7365F1EE, 0x37524CF0, 0x70CC332F, 0x5661786E,
0x3AF6A220, 0x3C5EA902, 0x256E6749, 0X1CB9A581,
0x1F7F42E2, 0x49D1FEA0, 0x268CB37F, 0x102362F8,
0x5FEDC0E3, 0x3D75BC47, 0x3E30D969, 0X18A35FE3,
0x6DEF5211, 0x1112DBB6, 0x2586A498, 0x17A03BB9,
0x005CA941, 0x734C7D9F, 0x24F5D018, 0X6780C122,
0x534347A8, 0x5800E34B, 0x1673EFC8, 0x05B9B1FD,
0x7AD7FCA2, 0x63351604, 0x0E759D1A, 0X6E3DEE90,
0x1A8762F4, 0x7F41D049, 0x449F66FE, 0x557E0515,
0x3BA0794B, 0x6A0DCE48, 0x7237AA96, 0X5B1FBC2D,
0x33DFCCE8, 0x18C45E15, 0x6B431F25, 0x13CD8DCB,
0x563A1A5C, 0x2973F88F, 0x2C70EDAE, 0X44296C6D,
0x3A86D445, 0x51F79246, 0x5BC9A827, 0x3AE37D86,
0x45440FE5, 0x00BF783F, 0x22643EA9, 0X1887578D,
0x58C05B8A, 0x38D82E71, 0x1E41098A, 0x5398582C,
0x1C0D4475, 0x2CB6A6A4, 0x41D646BC, 0X3694A76A,
0x2BF876EE, 0x0675ADBA, 0x0C12AC7F, 0x6798F039,
0x70837C02, 0x7E4A5715, 0x42B8AC67, 0X246348EA,
0x170EB52B, 0x2DFBCB8C, 0x3830D6B6, 0x6D48CF87,
0x576FC41B, 0x64A1C464, 0x31723BF5, 0X11F69861,
0x369956AB, 0x0D3BE41C, 0x4CDA15E7, 0x7BDD6690,
0x0DFB5C5B, 0x6F3E5490, 0x1464BE1E, 0X66BBB7E5,
0x28168302, 0x32A5C7A8, 0x3A541011, 0x4423C777,
0x5F5C6E4D, 0x7C2A56CD, 0x7AB86EE1, 0X0B54E53B,
0x02A00487, 0x06CB1B60, 0x72EDD574, 0x7323808A,
0x05157276, 0x35A681DB, 0x1786C974, 0X1C2427A1,
0x63A24D68, 0x4FB7A02A, 0x096CF728, 0x3B121183,
0x3459648F, 0x3ADF331D, 0x4D08A9E4, 0X6AF2BB3A,
0x481B1739, 0x19E2BFCC, 0x66D021CA, 0x56167394,
0x0921145C, 0x7B34DFE8, 0x3CD22B79, 0X3137975E,
0x2DDAA791, 0x77263B8A, 0x755B5ED6, 0x0D3715DE,
0x73509257, 0x7013CDB7, 0x188BFB19, 0X75F096DF,
0x76DEE918, 0x0B79D08D, 0x69141769, 0x7BF45B8E,
0x41205269, 0x009AE0DD, 0x1818832F, 0X24C29FD1,
0x50528108, 0x21857A57, 0x5FD4B154, 0x04ABE597,
0x5C64AD75, 0x2CDD5B39, 0x6F9EA0D1, 0X247FC4AE,
0x46C01B05, 0x566EC29B, 0x7A963843, 0x4FE12F61,
0x51A3A284, 0x376863BC, 0x0118C6C0, 0X7F7E4A15,
0x2E8E9F47, 0x76742596, 0x0CB55FF3, 0x21DF319E,
0x6687F34D, 0x25415B0C, 0x17CFC87D, 0X5D66DC65,
0x30BB2B99, 0x00E3DFE6, 0x595B37F3, 0x71DB7E02,
0x017EC0C4, 0x7173BB22, 0x169E1DD3, 0X51D141CC,
0x12F9357A, 0x7672CF28, 0x567D2763, 0x6F5DE2EF,
0x23502A61, 0x461BC834, 0x13DDA79D, 0X6A104566,
0x1C8A8ACF, 0x0E73DFE0, 0x39F174C7, 0x6E2E2D53,
0x45DC439D, 0x3B0A3B87, 0x6DAC7768, 0X746AE2E4,
0x317E611D, 0x7A61D75B, 0x164A1482, 0x1806546B,
0x1FA33267, 0x2E19DD00, 0x756D30D0, 0X505E5E01,
0x2EFDBCE6, 0x4EC868C4, 0x4239DC03, 0x307C7DAA,
0x403C23E6, 0x58D7F9D7, 0x024DBF76, 0X53355960,
0x4F4AC8FF, 0x58CAE6D9, 0x42933C4F, 0x729AF360,
0x1EE6AF0D, 0x5670E3ED, 0x5CAB38C6, 0X3B7139DD,
0x64E4C3CD, 0x169CAD8D, 0x299F6730, 0x2AC1076A,
0x51A6E915, 0x174BDE99, 0x1F2BEA4E, 0X03254A32,
0x11ADB5F4, 0x3575FED1, 0x1B2B9E9D, 0x3150E85C,
0x638FDBD1, 0x1098CF6E, 0x01AF465D, 0X128D98B7,
0x5F613832, 0x43E92260, 0x430A1662, 0x1F9D5C18,
0x1CC11C37, 0x4557D5D8, 0x72D2B579, 0X6C0BE536,
0x1E22BCB2, 0x3565F1C8, 0x5EA6D896, 0x3D096BBF,
0x0BD6D5B5, 0x3B52115C, 0x787AA59C, 0X70BB9983,
0x51EEBEEA, 0x221A0CCD, 0x1B7CA0ED, 0x2395A7FF,
0x3965EB66, 0x3AA88B3C, 0x26BAF231, 0X4B13A15A,
0x701E8A0D, 0x41E690CF, 0x7C6489B6, 0x53AE65DE,
0x527F603D, 0x7E13D013, 0x663BFE95, 0X31E0986F,
0x41FCF274, 0x294614F7, 0x517DF487, 0x5EBE0EAB,
0x6E9DEAD0, 0x4450AA00, 0x4AC9F3E2, 0X0CC0A782,
0x79B69BC9, 0x2970CC78, 0x49CA1341, 0x058D717E,
0x64C2DDD5, 0x4244B8DE, 0x76490B01, 0X36B19CBF,
0x645EC5AB, 0x11C5ABEF, 0x5A4744BE, 0x1DC4B111,
0x4C6E372B, 0x010236EF, 0x68D8526B, 0X3C8CC138,
0x42E8C7BE, 0x653CDC22, 0x103B2716, 0x156827FB,
0x6350AC35, 0x767725AB, 0x4748C06A, 0X254D9EA9,
0x1FBD3AA3, 0x18C6B4F2, 0x040BAD55, 0x0E5B2573,
0x5D175EF2, 0x4ED5A137, 0x1B1BCCF5, 0X56CDFABB,
0x78466DAF, 0x64E5E036, 0x5C5B6C3A, 0x5D094B84,
0x272A9914, 0x52A4773B, 0x13BAE843, 0X0B895EBF,
0x646A232A, 0x6E022D01, 0x294E0FD0, 0x30D85A55,
0x6F0463F1, 0x1226623C, 0x6D651B8D, 0X31ED2BAF,
0x77633E5E, 0x7DA042A3, 0x475553AB, 0x5AB3EA93,
0x7417684F, 0x0E9E1415, 0x0001893D, 0X13D4A2F2,
0x2764C907, 0x040D3692, 0x222FC865, 0x047C27FA,
0x52E2D7C9, 0x3D4B955A, 0x5B4A22B5, 0X4B294578,
0x22317590, 0x37A58EEF, 0x283290FD, 0x495C0EA5,
0x0A4A062B, 0x3BED7940, 0x54E56D64, 0X6EB42955,
0x29EFA642, 0x7E337D35, 0x1F8C83AB, 0x18F40A33,
0x1059DF71, 0x0CF19F38, 0x4AE135E2, 0X07BD1DCF,
0x0A91E1DC, 0x1236898D, 0x62710862, 0x7EA94A2B,
0x20D49DA3, 0x6272919F, 0x127DED1D, 0X483966AA,
0x667FC831, 0x34ADB582, 0x4CB58EA4, 0x39629FFA,
0x71F94ADC, 0x27FFB15A, 0x048BE573, 0X142AC06C,
0x5FA54049, 0x2CBE7670, 0x5D86CF11, 0x69EF4674,
0x68ABEFB0, 0x326C3C76, 0x58A36FCA, 0X129B95F2,
0x309FB9AB, 0x782FF375, 0x2B8FA025, 0x40F9991C,
0x052192AD, 0x7670D608, 0x48B6B6EB, 0X0FB37489,
0x08A75F95, 0x2B27BF4D, 0x0E5CBEB4, 0x297BFD38,
0x0D9A50ED, 0x20DAABD1, 0x71B563E3, 0X741A191E,
0x55886153, 0x3E6AF287, 0x2D7CB919, 0x4781AC2F,
0x666AA3E1, 0x32089E8C, 0x5BAC6C9C, 0X460FE42B,
0x5EC714FC, 0x39333BAD, 0x2FFF2A9F, 0x477304AC,
0x6B9F7823, 0x08A29A69, 0x5A0E9A9F, 0X1C3F31CE,
0x00D28DDE, 0x059E3AC4, 0x5D38CAEA, 0x05F4208C,
0x7C0F10CC, 0x25EF81D5, 0x15A79515, 0X04B67062,
0x51174123, 0x240453CA, 0x2E326D9A, 0x5EB19210,
0x44DEFF9B, 0x1FE7D17D, 0x52CBAB2E, 0X1A6760EF,
0x5E52C405, 0x00486447, 0x61E90D1E, 0x44BD67E6,
0x325102D3, 0x3D9579BA, 0x0ACD4C11, 0X111817CF,
0x76C8B568, 0x3ACC76B1, 0x588B1C7C, 0x62682D8B,
0x436F111A, 0x3299B71B, 0x7EA75F5A, 0X44419EF9,
0x3837F1DF, 0x5BE02A44, 0x4A35BF85, 0x344702AC,
0x01CFAC1A, 0x5FDD549A, 0x38FD730E, 0X52E6ED3D,
0x03E1A864, 0x672FE0A8, 0x31987F4D, 0x48C0A800,
0x0717B226, 0x04642A7B, 0x632808EF, 0X656A762B,
0x04AC8EC3, 0x4511160D, 0x2A27DE11, 0x36FD9196,
0x02A68FC8, 0x34F52A23, 0x4815A966, 0X796F4530,
0x6FC1A0D4, 0x20A0C5E2, 0x5BD772BB, 0x3330B1EE,
0x533A7CFD, 0x5A7ED215, 0x777250E7, 0X0B726EDC,
0x365EFC5A, 0x41A8106C, 0x3FB97188, 0x382EA874,
0x21856507, 0x78B6E496, 0x0B1595B1, 0X25670D6B,
0x5FE6C53F, 0x3CAE14FE, 0x6E27B56B, 0x66FE7765,
0x41123F79, 0x514FBE5A, 0x4C68ED90, 0X45BECE3C,
0x1660D468, 0x7690CBA1, 0x7CBC5FD3, 0x19076430,
0x2B85F5C4, 0x44D20939, 0x1276A960, 0X1B479698,
0x6572CF1B, 0x6E4E1C1B, 0x4E784887, 0x38AD4C18,
0x48CCEE31, 0x45EA996E, 0x441FBAF4, 0X7F2BEA8B,
0x0792A9DB, 0x03D92C7D, 0x375A92FF, 0x29180EE2,
0x7C901113, 0x427028B0, 0x4E7F1C4D, 0X5C76D652,
0x7F1E3DAE, 0x3CA6D1B9, 0x43754DB7, 0x40307D27,
0x0DF69013, 0x0FDE3B47, 0x05EF4B64, 0X2457647B,
0x066F06E9, 0x02ABAB37, 0x3D5EC8AB, 0x31F4FCAD,
0x477DB470, 0x4FD5720B, 0x4D3C9346, 0X2CF0838B,
0x3E238E27, 0x1BB4DBCD, 0x659DCFA3, 0x06F07C58,
0x619F753B, 0x29BD8A97, 0x061C66E3, 0X69321F16,
0x2D96B714, 0x3D76F9E2, 0x124A2DF8, 0x2A26C828,
0x7FE72292, 0x60C94A46, 0x069D9E7A, 0X7F056040,
0x1D701BFF, 0x4A12EC32, 0x3F35DD67, 0x2B66AC12,
0x59F12779, 0x452528CB, 0x4FBE108E, 0X60602E62,
0x47D0D402, 0x0D1CD939, 0x12552B10, 0x0F4E8872,
0x5CF24B45, 0x5F91BE56, 0x3C3F0BFD, 0X1B15D96C,
0x7B469A23, 0x21DCDBA0, 0x220655C4, 0x5CE60F5E,
0x4B9A6638, 0x2822BCA7, 0x46182E75, 0X79311D4C,
0x6599B689, 0x58625C6D, 0x2357E574, 0x6580D91B,
0x392BA6B3, 0x29F583EF, 0x6486395B, 0X569BC2B2,
0x74087021, 0x23BC16C2, 0x02026EC5, 0x4DF9979A,
0x68E13F8E, 0x51C07F53, 0x2E59C5FD, 0X30B21390,
0x5EDD588C, 0x40AEF10D, 0x40009C03, 0x3BCFA3D1,
0x2040AF63, 0x7C3FA800, 0x56E57D3D, 0X1B874986,
0x1E1C83A1, 0x78EBD301, 0x786D58E4, 0x69B6E9D9,
0x210E8FA8, 0x3E858759, 0x62E80725, 0X06A84631,
0x16E7E3C7, 0x063FEC9A, 0x6C291F4C, 0x50138A7A,
0x30357089, 0x50AF58A7, 0x26AF4D2D, 0X243DE0AA,
0x746B6F6A, 0x28B1BBF2, 0x72377844, 0x5D4CAEF8,
0x7A723B45, 0x20913E41, 0x0DFEC288, 0X594F93D1,
0x61402F4E, 0x4DFF5E8B, 0x151F37A3, 0x0180DEB1,
0x4A3F068C, 0x6C04B4E0, 0x1D082837, 0X685B8A2D,
0x64F087E2, 0x1575811C, 0x52127406, 0x05FF178A,
0x53FB0875, 0x34FA7B2B, 0x0CA75DBC, 0X6AE2EC3C,
0x3B3A67C5, 0x78D07D08, 0x3AF676B7, 0x6B6FD84E,
0x497FD5B0, 0x61A5C3E4, 0x0FADB8F8, 0X3DEB451A,
0x0A577FD6, 0x01E5313D, 0x1B37F412, 0x04C9BB1B,
0x22766F7E, 0x2936B69A, 0x5E194EEC, 0X03B69ECD,
0x77361526, 0x7338868F, 0x05377D7E, 0x41751BB2,
0x5F3D3B70, 0x223FA5B6, 0x29D0A5DF, 0X442DC352,
0x37B526D2, 0x7BE319E5, 0x4A2CDADC, 0x0BB02F47,
0x30DD9510, 0x56D43898, 0x76931B84, 0X6C17FCD6,
0x4FA4B5A1, 0x3189923B, 0x5787D524, 0x19248B51,
0x132F561F, 0x67358E1D, 0x570FD06B, 0X1D86D5F5,
0x691ABF5A, 0x7247C47D, 0x22509110, 0x0B912ED8,
0x1B7E7B17, 0x0069DFFC, 0x0F47CDA5, 0X12B4903D,
0x73A2668C, 0x147F4B24, 0x5429ABEF, 0x52DFA1FC,
0x36BEF0DA, 0x7DFA51CE, 0x170D654E, 0X6E7417AC,
0x79DD6BB3, 0x613A402A, 0x7A2446F3, 0x2ABB00C4,
0x380E78C3, 0x70B76277, 0x16D2FD9A, 0X07B32E64,
0x2240F4B2, 0x6E5AD2BE, 0x20D7B9B5, 0x35704AD1,
0x559060DB, 0x77E78A20, 0x52F720C6, 0X3EAB2035,
0x6A2F4E9D, 0x7547B1D6, 0x4A3C4F0E, 0x05ADC9B4,
0x75B191D3, 0x59841CB3, 0x186259F2, 0X6953F85F,
0x6E0367D7, 0x6C8C05E1, 0x3C339A5B, 0x24C258B1,
0x6A8657B0, 0x5340FFA9, 0x1336705D, 0X6463C363,
0x347B3FD3, 0x0D5AB751, 0x0F1EC427, 0x6C89B896,
0x7E1219C8, 0x25F1C1C1, 0x743CE6FA, 0X20530E7B,
0x144C9480, 0x1514A0AF, 0x55C3594C, 0x69DCF55B,
0x0CFC2ACF, 0x28BA7A13, 0x28881591, 0X772B796C,
0x1E022BE9, 0x72C4649F, 0x7CD94321, 0x13B3BDBC,
0x4C488152, 0x153B9D13, 0x7D07B61B, 0X3A4BE92A,
0x01C7A2F4, 0x393B5076, 0x5F0E41DB, 0x6C4DFAA4,
0x0C7C501F, 0x7244B239, 0x50B1BE08, 0X40F78FF3,
0x7F9F698A, 0x5FD0822F, 0x2D814889, 0x7DB18352,
0x05C243F1, 0x21BE2F84, 0x1E0491CD, 0X1A0ED871,
0x36D2D033, 0x73C7EB1A, 0x03EBCDCC, 0x43CEFB03,
0x1C82652D, 0x2C73E35D, 0x3AFA746F, 0X3A849116,
0x1F3847FC, 0x37D3B790, 0x4E384ED3, 0x6B80C94F,
0x4D0F54A3, 0x4B4004EE, 0x25CCB279, 0X4ED6F798,
0x047B5565, 0x04DAF454, 0x3B24F23C, 0x10F7A584,
0x771FA68D, 0x0BD6B044, 0x51EF3577, 0X76BF1017,
0x6BA73274, 0x7F707E01, 0x7470936A, 0x71697665,
0x212EAD85, 0x12752537, 0x0B784ED6, 0X58017DB8,
0x063D1051, 0x0F641CA2, 0x1BD078BB, 0x22BF757E,
0x3BD80000, 0x56CAED2B, 0x5D440695, 0X5B1047FC,
0x0E9EA4BB, 0x2B7C5568, 0x4691114B, 0x5BADF95F,
0x76BC5A56, 0x6C5DC3C4, 0x2A84F0F7, 0X7B37AFBB,
0x7138B819, 0x65A9E333, 0x0C2F5540, 0x68585EA6,
0x71809378, 0x5E1E8AB7, 0x5F176EBE, 0X5D27C5EC,
0x5D8F08B8, 0x53880228, 0x4E913C51, 0x7EBDB63D,
0x65FD275F, 0x5A098B27, 0x56BF33F6, 0X6C3A37B1,
0x696DA7C9, 0x728FACB1, 0x0EF9AD2F, 0x2545A7C9,
0x495A99DC, 0x6C3DB3C4, 0x0055EFC6, 0X57F93E98,
0x17BA092C, 0x46E70111, 0x33A737F7, 0x0E766383,
0x3344C4D6, 0x5E2C28EE, 0x09AE133E, 0X247D7CEF,
0x43D60C21, 0x15DD687E, 0x0CD5DB95, 0x35569F99,
0x73FBF336, 0x6BED4A53, 0x127E6585, 0X518AFBEE,
0x3F754C7B, 0x610FA1D6, 0x5048B22C, 0x257273DB,
0x3B192CFD, 0x2707E622, 0x11ACAB8C, 0X2486D4C7,
0x199792D3, 0x20A658BB, 0x49CC7C90, 0x62F22CB0,
0x0CE40C80, 0x4A226C56, 0x3AEB6B48, 0X249E15AC,
0x11096D68, 0x6E92A33F, 0x3314792F, 0x444E323E,
0x4CBECC2D, 0x3CC28C6E, 0x68CBAF2D, 0X1094D84E,
0x529FF4EC, 0x75A18AC2, 0x45EB77E8, 0x469BE822,
0x618ED516, 0x5869DD6D, 0x1826E411, 0X21042191,
0x39797F44, 0x686F963D, 0x4676956C, 0x7492AC41,
0x0F777C5F, 0x582340F8, 0x19198108, 0X290F0F32,
0x78C999B4, 0x62E5FD99, 0x0C013BE2, 0x05ADA634,
0x2D0869EF, 0x46ECA72A, 0x2A4BBBE0, 0X3E11D757,
0x357F4A69, 0x5D603510, 0x02600995, 0x023E1696,
0x1A22C17E, 0x6B2BB8C2, 0x12D2EEE5, 0X6CC2B66A,
0x60CD4385, 0x58BE66CD, 0x335E9E8D, 0x425C189B,
0x3128443A, 0x4B85829E, 0x63603A2C, 0X6AA1C37E,
0x33F518DB, 0x29D6CF99, 0x5F346FC0, 0x436C953A,
0x01FA1091, 0x784DF0C8, 0x6C7BA46C, 0X7AC3AA45,
0x5B33EE61, 0x787CE04F, 0x00715079, 0x083C5851,
0x3F698779, 0x2ABD0C5A, 0x464E2FA8, 0X74E8D1E3,
0x081D416A, 0x48AE393E, 0x7726E879, 0x224002E8,
0x33D9F200, 0x09F9D75E, 0x0F02B952, 0X14A73585,
0x62B83E2B, 0x426157DF, 0x57034E20, 0x13E08266,
0x0DE6DA7D, 0x3A63884D, 0x7E8245E4, 0X41DBF358,
0x643A57E6, 0x5DB6B5A4, 0x05488892, 0x66346877,
0x5604A66D, 0x71C42CFF, 0x60F812BD, 0X313894CE,
0x6A410D4E, 0x61696336, 0x3974ED1F, 0x29AA94C7,
0x0C266F90, 0x7FC31CC8, 0x1E9366AA, 0X1443B0FA,
0x48715606, 0x15BA4F24, 0x3683B3E2, 0x7C4B4806,
0x1FB42682, 0x45866D35, 0x10F27D8C, 0X026C64AE,
0x07E7C514, 0x67F5CBAC, 0x164CE714, 0x15CE9F92,
0x225953F9, 0x14CF2CF8, 0x57AA92EA, 0X0693ABDF,
0x7285E29D, 0x5CF31B7D, 0x6CC81457, 0x488A890A,
0x4EB7487C, 0x4DC02714, 0x79C31DD8, 0X38F855CA,
0x2F298A4A, 0x33380AF8, 0x62A2EA91, 0x3B4FF9DB,
0x32FB27C0, 0x0136513C, 0x4F93AAD5, 0X7B6C7DC6,
0x16F0A060, 0x06175EB8, 0x77B7C5CC, 0x36A4C6E2,
0x4B9DCBED, 0x08AA4358, 0x39112B90, 0X53859101,
0x70A00F05, 0x4F5E12A4, 0x69543093, 0x12F962FE,
0x642D3F9D, 0x40FEC37E, 0x198D0EDE, 0X56B3223A,
0x1DF1DEFB, 0x06552335, 0x1F3DAB44, 0x6CA92777,
0x54154A49, 0x1900C91C, 0x25A17D41, 0X033ED493,
0x4C38D414, 0x084467D2, 0x3E8ECE6E, 0x7F33FBD4,
0x097AB90E, 0x0E227944, 0x7AA0799A, 0X206B596E,
0x1439D7FC, 0x72583F67, 0x57102051, 0x5FD7A3E9,
0x7B0282BF, 0x10214BE1, 0x335D34EA, 0X6BA291C4,
0x5F7F5E86, 0x1CB1657E, 0x7E9BF4C3, 0x43AC9E23,
0x5DB028FC, 0x182903A1, 0x1A5FC05D, 0X7BA207F7,
0x1E7E26D6, 0x399D6BA1, 0x684B2F6E, 0x7293711F,
0x529E34BD, 0x0DECACAF, 0x75D245B2, 0X1ED708D2,
0x16311481, 0x34611421, 0x1E0B04A6, 0x1FABCD90,
0x42838D65, 0x18AB7E41, 0x401726FE, 0X56BD6561,
0x0B03BDA8, 0x1727474F, 0x3695094A, 0x06064067,
0x27489331, 0x69F23E34, 0x71A8D22C, 0X06C7F1B7,
0x06A3A3B2, 0x7044C6EF, 0x4A748FDA, 0x6453CCAE,
0x086DCA90, 0x64D45037, 0x5FF5D4A5, 0X26EBF166,
0x1E71BBD8, 0x48410413, 0x197F6285, 0x710FF095,
0x562DB0C2, 0x0F51A837, 0x0FE6F967, 0X6C5EC544,
0x43B2BC58, 0x2DF1FE0E, 0x0C0A92D4, 0x063649BD,
0x469D7C4F, 0x4C21B9D2, 0x5CF3AF1E, 0X51A139F7,
0x63490122, 0x1388B868, 0x57A77A5E, 0x0A919453,
0x7D7AF69D, 0x49504C8A, 0x1159860A, 0X041E9A4F,
0x39951379, 0x5BCE15E4, 0x687266FE, 0x4202DE09,
0x40A2661B, 0x48683BA3, 0x68EECF6F, 0X5F1421F3,
0x10A93FB7, 0x026E31F4, 0x50241288, 0x66D6F079,
0x11BFDA2C, 0x600B0BF0, 0x5335B5BD, 0X55729684,
0x0DFD09FE, 0x5F404891, 0x5BA8E042, 0x549A864D,
0x2B620264, 0x389C8F60, 0x263BC044, 0X0EAB0386,
0x4C2547C9, 0x7DE33AA2, 0x193C97D9, 0x49A03E66,
0x4733872D, 0x2A961DE3, 0x4DBED8B5, 0X00C89AA6,
0x066433C7, 0x36313FB3, 0x42CB78B0, 0x470699E2,
0x7E997B57, 0x2BBA481F, 0x261ABBD5, 0X0F42BB0E,
0x2E287A14, 0x763ECE5D, 0x7619AB87, 0x3FE85440,
0x5649DA4D, 0x494F6145, 0x155AEAC4, 0X6446E44B,
0x288FA9D6, 0x7103CB06, 0x38E16A98, 0x53F1AC3A,
0x29A05A67, 0x5F1D2ADC, 0x629CAFC0, 0X75C5A230,
0x5D00657F, 0x7BD94799, 0x3F65E096, 0x2433ECAC,
0x266F657C, 0x0D24B94B, 0x24FC8752, 0X2CD39943,
0x4355F8FF, 0x67C80002, 0x73DA3325, 0x41EF7456,
0x13824822, 0x19F4EEFA, 0x51322F64, 0X41AAC236,
0x1033BD58, 0x474BDAEB, 0x01931676, 0x667D97A5,
0x109B3C30, 0x16EE013A, 0x4AC47BF1, 0X392AE607,
0x07F1CC41, 0x03A5E689, 0x0D1C9241, 0x319226A8,
0x62C31166, 0x6FB94202, 0x2757C8D8, 0X3FC376E5,
0x6B92899B, 0x66BDA96E, 0x63F76391, 0x1201EF18,
0x73E262B9, 0x08F3EAE3, 0x3ED5885B, 0X37385BB8,
0x70BBEAE6, 0x32AFBB81, 0x7927D00E, 0x043E3308,
0x4CA4AA7B, 0x4A59FF72, 0x45E8F53E, 0X5CD867D3,
0x11A5DA5E, 0x477C0BB4, 0x4355FF79, 0x2241168E,
0x5E6A0CEE, 0x0E1A7B6A, 0x5B6BFC95, 0X665BD92F,
0x11C061F3, 0x68888ED7, 0x17EDFFD7, 0x74837359,
0x5841D0D9, 0x3F45C8AF, 0x3446EA3E, 0X43D45A74,
0x2603721D, 0x183E4DCF, 0x55D6498C, 0x19E5D4D7,
0x213238B3, 0x14ABD1E8, 0x511E308F, 0X11EE2399,
0x475B8D69, 0x4A46009E, 0x162C56A1, 0x140037E4,
0x14A00010, 0x5C154BDF, 0x70D89FB8, 0X2645DA6E,
0x23915793, 0x342E9F31, 0x4886F0FD, 0x01FB6481,
0x42491A9B, 0x23F2ED92, 0x68573DB1, 0X54097C8E,
0x0C7B7C69, 0x00453D88, 0x488CEFE8, 0x64BD4D42,
0x3F8B0638, 0x7CD3DA26, 0x2891A7B7, 0X658E7855,
0x151227F6, 0x7E67F143, 0x7F744D2C, 0x364460A9,
0x1313C32B, 0x50927DBC, 0x48328442, 0X5A6F5094,
0x1AD87E5A, 0x5E5EDAE3, 0x6E6F8879, 0x2F787E6A,
0x3A7426C2, 0x5F482831, 0x55BE58D9, 0X5E057E55,
0x1376C762, 0x1E4549D6, 0x6000E2D6, 0x55BFE1FD,
0x42383768, 0x48582087, 0x29C95E8B, 0X4EB3B3D2,
0x489D5E10, 0x72564E73, 0x33710114, 0x08286448,
0x6F2A289A, 0x5C02A8CB, 0x6DB6DC9D, 0X043C5090,
0x5A6A9A0F, 0x6D2B29CA, 0x3A80B139, 0x6D7E5D3A,
0x3DBDA786, 0x02B3357B, 0x47EDADCF, 0X589625E0,
0x6112105E, 0x365D3648, 0x080EA44A, 0x1B863720,
0x15A55E79, 0x5DCCFD23, 0x798BB575, 0X291C25DB,
0x7C1246F9, 0x598C984B, 0x7EDC07D8, 0x3E4A7E62,
0x21E4B8D3, 0x28A56663, 0x0CFE3234, 0X6A8216E3,
0x1AFBB4D7, 0x406F3348, 0x72AA7B2B, 0x0A25DD71,
0x1C71DC14, 0x606157C8, 0x0E622E01, 0X76DC7623,
0x4D8C8192, 0x48E2DF3A, 0x645AD35D, 0x0B4A2918,
0x4B9614B5, 0x2C48812C, 0x63E04EF8, 0X2CA82513,
0x62A5B774, 0x6BEEF343, 0x482E5C33, 0x784B15ED,
0x49BBF066, 0x41BA11A8, 0x21673BC8, 0X45CE3760,
0x1B46A9F3, 0x204343A0, 0x0418B5C2, 0x3D2B62C6,
0x48E8AA04, 0x1116E7F6, 0x27AD79A9, 0X63E45EDB,
0x51861B3E, 0x1A57F4D4, 0x6E0A3C4C, 0x6DF7F752,
0x7AB94C9D, 0x7C6C6A4D, 0x64D46D75, 0X4845CE2F,
0x454F4987, 0x492F40D3, 0x538FF748, 0x10E55E3C,
0x7577C1FF, 0x37704640, 0x3D8D834F, 0X581D7974,
0x235F3983, 0x05BBDF82, 0x50688F61, 0x6D1B29EA,
0x4775F12A, 0x71CFCB2A, 0x32E9614A, 0X62BC9B1D,
0x12130ECA, 0x3702170C, 0x1FE7FDE4, 0x5AFBB8CE,
0x4818FF02, 0x4795778D, 0x3EE017A9, 0X199F1A40,
0x61ED6C62, 0x2CEA53F5, 0x07971193, 0x5CA6B8FF,
0x2956BE42, 0x6C6B7F08, 0x24EC872E, 0X6EA607C9,
0x359ABFDB, 0x787C7E76, 0x7F8B6605, 0x2B1281DB,
0x2FECC4B7, 0x3D18E954, 0x032FFB4F, 0X534BFE3A,
0x42D4C8D6, 0x53988AB0, 0x40672824, 0x0A4ABA00,
0x456855DA, 0x7350896E, 0x6D07551E, 0X577B64A5,
0x2A52A07A, 0x0CEF5302, 0x32771D73, 0x726B9F7C,
0x5484CA8F, 0x7157351D, 0x0C0AB9BD, 0X367236F1,
0x1E418912, 0x13A1CB50, 0x1318EFF0, 0x47984755,
0x000D4A58, 0x3805771F, 0x363E4F1E, 0X35A80A34,
0x3081F595, 0x35C9B524, 0x60BA8C0F, 0x606EBA4C,
0x72E29E78, 0x63EA875E, 0x33BAB887, 0X35B7674F,
0x3783120E, 0x7421E0AB, 0x4002214F, 0x7CEB67E9,
0x67726A1A, 0x2D09766D, 0x5466CC8E, 0X11C50A94,
0x39F8C96F, 0x06DDEA01, 0x0430AA11, 0x0E7D93FF,
0x78351F1E, 0x103B63CE, 0x44EFCAF0, 0X1676A831,
0x23DD2F1E, 0x5808BAE1, 0x5E0EEF86, 0x23EA7976,
0x100E3200, 0x144D3EA4, 0x599283AA, 0X40902795,
0x4A16F3C8, 0x3A4D0FB9, 0x20FEE1E2, 0x3CF99241,
0x1E379717, 0x54B99A69, 0x72B0F990, 0X55BAA926,
0x48DB7B14, 0x32B31ADF, 0x52A6110F, 0x304DE52E,
0x5FBC914D, 0x270CDD9D, 0x4212EFC3, 0X19B55ABC,
0x2DEAC79E, 0x464399D4, 0x2832EEBB, 0x261FE6BD,
0x567EFDA2, 0x6D22B9AC, 0x3C968EEE, 0X7A5C2CC0,
0x452B748D, 0x1AA57E74, 0x1E46A636, 0x5539A68D,
0x2EF2BD18, 0x77D929E1, 0x15C9CE22, 0X7909B0E1,
0x3226399A, 0x36C8B004, 0x36034322, 0x505DD0B2,
0x0B824A6D, 0x28B43CB2, 0x261879D8, 0X545DC582,
0x5B675791, 0x78BE8AE7, 0x04ABAAB0, 0x3B23E8DE,
0x1FCB6884, 0x46BE9A73, 0x54D9439B, 0X4DB63022,
0x0D023447, 0x7D0C3256, 0x73D616DF, 0x638131E9,
0x6A2EEC02, 0x306CA5CD, 0x5DDD5EA9, 0X2F5A608F,
0x4B122441, 0x7C2404E0, 0x0494071C, 0x7A04E15A,
0x73FD2EC1, 0x1A5DD53F, 0x730E923B, 0X2623685B,
0x51268543, 0x2911D55D, 0x7681390D, 0x5CA8CFB1,
0x51C6120F, 0x1C99B2E5, 0x31069533, 0X2D2D69A0,
0x15583DCC, 0x35B23FE3, 0x6851527F, 0x3523A650,
0x7C70DA57, 0x3D2A961A, 0x02D9D673, 0X09730E9E,
0x3A36C870, 0x76AFED52, 0x6CF44088, 0x2465B473,
0x271C9320, 0x4AD19F31, 0x53C01502, 0X722EB761,
0x46F5A411, 0x58541C1F, 0x6C3398BB, 0x3AF2D2D2,
0x72B1F15E, 0x5F422AF6, 0x61163B2E, 0X43D876A1,
0x08540053, 0x5797743B, 0x20814652, 0x5A1A1262,
0x74312721, 0x5187DB85, 0x07477C03, 0X098964ED,
0x073A1B69, 0x6F98CE82, 0x3EAD0B3E, 0x03AAF5C0,
0x2CC3649C, 0x4186E1B1, 0x0D1E045E, 0X66FA2D0C,
0x3836CF03, 0x7A1244E6, 0x0B5FE17F, 0x5F536223,
0x44E3E418, 0x5F1FF682, 0x51821985, 0X0BD98829,
0x377412A1, 0x3DB5B240, 0x46CC5AFC, 0x2A2603FF,
0x1CF7DD37, 0x27E2962A, 0x6DFE7AA0, 0X254BDD8A,
0x7F7A0A65, 0x0E7FC0F3, 0x7F65EFED, 0x73AB3186,
0x60079C78, 0x06AD6BF0, 0x7D349674, 0X6741B7E1,
0x76463A72, 0x3BE1A1B2, 0x6AECADA1, 0x23099F0E,
0x7D688363, 0x780AB200, 0x0A03CC1A, 0X359F5266,
0x721CF6E6, 0x1563AD9A, 0x14F2B48A, 0x3700DAFE,
0x7483A41C, 0x6674CE0F, 0x42DA6328, 0X2BF7B6BD,
0x242A804F, 0x09A6BE24, 0x561DBABC, 0x41225D86,
0x3189544E, 0x441C355C, 0x666E3B11, 0X31035EB3,
0x529BF64F, 0x65D42AFE, 0x24AE903A, 0x32A392C8,
0x6C8196EE, 0x21E326AE, 0x19E54AA9, 0X62C7D160,
0x5DC4C860, 0x04D1F84B, 0x05D1706E, 0x5B2D4BC3,
0x7CDCAA4B, 0x0FD53C88, 0x10CC9E29, 0X6EF9A131,
0x2538EA22, 0x25BF52B3, 0x25FA7C30, 0x19BC8E3E,
0x0C3420C2, 0x68D4DF58, 0x45B444FB, 0X305EA112,
0x727B9D7C, 0x1BD1FFB7, 0x7180FE98, 0x2404F1CA,
0x5FEE3514, 0x57EF39A9, 0x5508507D, 0X328A2B63,
0x3DC364A7, 0x79B6E0B7, 0x652DBE2B, 0x2A44FB95,
0x1B9A0765, 0x7F1308D5, 0x0D0CCCF5, 0X795ECFC5,
0x03E50120, 0x12DE3D63, 0x548C1B88, 0x00C1AB6B,
0x22B379EC, 0x6558B9B2, 0x6FBB4C9C, 0X47EC640E,
0x0B180C65, 0x15B5C8CC, 0x61A8F24D, 0x174C2D28,
0x7E8AA824, 0x275D3748, 0x47AACE3A, 0X710645A0,
0x432F3700, 0x392BCCD2, 0x150B376A, 0x231D6C14,
0x111B067C, 0x6A1387E8, 0x55A79777, 0X4EDE6B23,
0x63CA689F, 0x3AD555A3, 0x792366B9, 0x7F647005,
0x39E85E78, 0x063033AE, 0x78C33FCA, 0X3DCD5F98,
0x190E7112, 0x4D4F5B53, 0x3E8F0B03, 0x3BC1EAFE,
0x32A81505, 0x2E4A579F, 0x03AE4F0C, 0X3DC0216A,
0x4400206C, 0x65574159, 0x550C4E92, 0x428AC890,
0x0CB478A2, 0x1CB71CCC, 0x33910E31, 0X4FE3AFA2,
0x55E2E99F, 0x489C459B, 0x73011BB6, 0x66FDF01B,
0x32AFCD83, 0x48A8B32D, 0x35DC5B3E, 0X167A3623,
0x037E08D0, 0x2EFFC1F7, 0x15DEA628, 0x3D666748,
0x352FF5A6, 0x0EA1E5F2, 0x7B33C6E0, 0X4E3E66B8,
0x5BF14145, 0x39C2D1E3, 0x0A0051B6, 0x0E99564A,
0x680D2983, 0x0DAEA0C2, 0x4C5977B5, 0X2C0D49EF,
0x7305E21C, 0x2165C647, 0x6E98127F, 0x7FBA5ABE,
0x3E1CE314, 0x222920B0, 0x4F9E0A60, 0X13FFCCB3,
0x6AC5664C, 0x429F2616, 0x7AFDBCCE, 0x1D7533CF,
0x0B47D943, 0x30DA180C, 0x33EF69F2, 0X0EC5E214,
0x5FD9DA04, 0x49CE101A, 0x4C2C495C, 0x1509CFAA,
0x586FF60D, 0x4760103D, 0x63483662, 0X34613752,
0x0122E220, 0x6D488818, 0x42FA8D9D, 0x69300BA3,
0x7AF728DA, 0x0F540552, 0x153D5592, 0X6DFD0AF6,
0x30B9CB99, 0x03D56812, 0x6DB765B4, 0x6ED6AEAD,
0x25FE88C2, 0x3D557014, 0x02D67B60, 0X10C3EF0E,
0x7FF4962A, 0x7DD4382E, 0x2E3922DE, 0x0B3C6F6E,
0x2EAE503B, 0x62288CD0, 0x1A025182, 0X0E882A3F,
0x2BF69CEB, 0x662E9ADE, 0x2391F9E9, 0x046692F8,
0x2D8EAB1B, 0x06DA304B, 0x38C7CA4A, 0X2EB18D3C,
0x7422B863, 0x7BC257E7, 0x17E198DF, 0x6F19E13D,
0x0B165D39, 0x2D1EEE72, 0x5D16EC34, 0X3BD028D3,
0x30F45684, 0x4ACE51E8, 0x2AA6D780, 0x56F2DF46,
0x0823C1FD, 0x2D7D52E1, 0x67B6CE55, 0X08185827,
0x2B518B0F, 0x15EFF133, 0x1354C795, 0x59FFDB4A,
0x78187E03, 0x2D571917, 0x68880589, 0X240F1AEE,
0x1385B3F6, 0x0C19FF72, 0x2875ADE6, 0x41145F11,
0x12F42FBD, 0x613D7831, 0x6FC5EC4D, 0X0716E820,
0x5CFFD018, 0x07A7852D, 0x7630C95E, 0x68162D52,
0x34C6739F, 0x5347B592, 0x23E65625, 0X65BACA23,
0x1E16077A, 0x4E8D2DA5, 0x3CADA969, 0x2639C977,
0x7C0A8086, 0x246477BE, 0x2E52219F, 0X275C0B96,
0x3A5468F1, 0x41A6E934, 0x015BE6E0, 0x326CE6F5,
0x6EFE024C, 0x69E3EC6A, 0x567C01E3, 0X0283B642,
0x75FDEBDC, 0x7EF1AFCA, 0x43981553, 0x08F21B9A,
0x602F27FB, 0x335E01A1, 0x100903BA, 0X3D2EF813,
0x3B0586CE, 0x0639CD18, 0x25452565, 0x6FCBFA6D,
0x598182AA, 0x492B7B8A, 0x5586C490, 0X77978A25,
0x17B8A930, 0x12346DF9, 0x1DD1539C, 0x13C329B6,
0x3698E5B8, 0x4C23753B, 0x3B1F354C, 0X70ED4EA9,
0x0DCA5E70, 0x3C7B1C2D, 0x235A359E, 0x7CC860BC,
0x265F0897, 0x79D63782, 0x7F4C16FE, 0X1C5CF473,
0x78C7E74C, 0x42E42C51, 0x254F100D, 0x58F70F47,
0x76422DF2, 0x355813C8, 0x1626075A, 0X3147B4C0,
0x3B91E0E0, 0x3B6B2CC0, 0x2113AF2D, 0x1513638B,
0x0496A84A, 0x769A73BD, 0x0CAAEDB0, 0X1C4F517A,
0x08CEE1B7, 0x2A7C414C, 0x30127B31, 0x3F67C76F,
0x769FB688, 0x6B31B07D, 0x30551618, 0X046A14F8,
0x27ACCCAA, 0x53AF4BB7, 0x013275B4, 0x4E0BD541,
0x4D858339, 0x007E8CB2, 0x6A68C9B5, 0X464D6A85,
0x4362B903, 0x0FB7D9C2, 0x1F4479CC, 0x39A4E6F6,
0x450FED8A, 0x356A8126, 0x6AEC9BB6, 0X00A1CE6B,
0x70D5ADE6, 0x0C004AE4, 0x15B531F6, 0x756C5631,
0x029ABEA1, 0x22601FA6, 0x11BBA7AB, 0X0B69A058,
0x4CDC60F2, 0x41CE22DC, 0x4AD167C7, 0x437C177A,
0x2CFFD35A, 0x7B267DE0, 0x47E62C72, 0X54ACA004,
0x4ED5C997, 0x4918A226, 0x22B87546, 0x1C5B4CD0,
0x49972ED8, 0x0D213EFB, 0x62A8B755, 0X0CF9E7DC,
0x1CD918BD, 0x01ED3121, 0x469ECED2, 0x61E90648,
0x3757B247, 0x318B6A88, 0x628AD4B3, 0X282D602E,
0x3D8BB56C, 0x784006A9, 0x1D99B65F, 0x4026740E,
0x1AA0264F, 0x2F555E0A, 0x4B901466, 0X677C8741,
0x712380E7, 0x16617C2E, 0x2AF89EBC, 0x1E235441,
0x1187FA0E, 0x72DECB2E, 0x72CFF445, 0X605DC3A5,
0x3BF76D55, 0x1588698B, 0x7CB91075, 0x058E9C2D,
0x22A9A886, 0x5F61C7CA, 0x12888409, 0X3F82C144,
0x614EF8EB, 0x592752DB, 0x216BC78C, 0x18A6AB32,
0x0AB2BD64, 0x03F69C3F, 0x40D40B60, 0X483E72D0,
0x7C36A2E8, 0x5E6DC1BF, 0x0864E6DE, 0x16D6C937,
0x0DC31FCA, 0x53F4FB45, 0x7E535078, 0X7EE6A0B1,
0x6A567773, 0x294BEF34, 0x1D09F4F2, 0x7BDE7181,
0x1C2ABA63, 0x0FD9E937, 0x5C3C3526, 0X582227B8,
0x256252C3, 0x58F5459B, 0x5DB0C3E5, 0x480BFB49,
0x38570D65, 0x703947EF, 0x078EBC8D, 0X19A60650,
0x49609ACA, 0x28FA8419, 0x324CB182, 0x5413582E,
0x2CF12058, 0x7320BCE3, 0x1C51CAFF, 0X2927C340,
0x518E7EA2, 0x24B6B1DD, 0x3FFE8C77, 0x5F519E6C,
0x78ABAD22, 0x3E51DCF0, 0x5E383F1D, 0X63022495,
0x679DCC24, 0x7B42340F, 0x5EE09616, 0x03C88687,
0x0B1C1D47, 0x3B1CCB3C, 0x5BEAAE3F, 0X307E700A,
0x141210D7, 0x399B7225, 0x788A6B53, 0x4C691E3C,
0x29D4BA14, 0x001927E1, 0x660F248C, 0X733554DE,
0x2913ABFA, 0x185BD60F, 0x4748AD0D, 0x5604CC53,
0x0B7C92F2, 0x639A780C, 0x7F2C8F93, 0X5D0B1194,
0x085129E9, 0x3F2B1C0B, 0x3C5CB001, 0x00FCD70C,
0x7D7CF8FB, 0x1A94EF1E, 0x63FEFBA1, 0X651AC51F,
0x15D7232E, 0x42DF91B8, 0x68E34BA7, 0x20F34075,
0x7DFC5CF4, 0x44CDF9E6, 0x5171B07F, 0X120E6DCC,
0x7E696C0B, 0x49FC1BD2, 0x5E778C08, 0x283E261F,
0x4A1543B3, 0x4486B095, 0x1B737AFE, 0X7328EFAE,
0x5CE286A4, 0x62BC280B, 0x492DBC01, 0x685F1996,
0x4656A017, 0x485A4B94, 0x456A2B2A, 0X4EA7CA00,
0x0785679F, 0x01C6DB2B, 0x4FA4A10C, 0x0502609A,
0x1C5BCA4A, 0x33A39CAE, 0x6A1D25BA, 0X3232ED78,
0x76832E66, 0x53007161, 0x53262DED, 0x747F8B5A,
0x17CE6B47, 0x2497DE6C, 0x068DF926, 0X1637D753,
0x6E93FA3E, 0x6505852F, 0x3E75FD72, 0x38A93DF2,
0x298C35C4, 0x59E97870, 0x2BD22DA0, 0X066EBC68,
0x3CA5A07B, 0x74FFE9A1, 0x6ECDD5FE, 0x02FC4092,
0x3D5A3535, 0x34380128, 0x51A40A93, 0X44DF9CD5,
0x35FEDC54, 0x2148AB9F, 0x49E1FD6F, 0x525AA69E,
0x54EC484D, 0x33FF2329, 0x048D9416, 0X4B6F76B3,
0x06FF948A, 0x57B3C203, 0x3FEF020E, 0x1ECDFFD2,
0x7C4BA06F, 0x467CFB34, 0x3505D725, 0X6ADF9AAD,
0x2B828063, 0x737BD497, 0x2388D89F, 0x550EB627,
0x4D654D08, 0x4F5B063F, 0x5B7D728F, 0X0A0AED83,
0x445AEFE0, 0x4A4B488D, 0x0D072E16, 0x01B52516,
0x7E8349B6, 0x5EAB38A9, 0x4694C1EB, 0X3482260A,
0x7FF3E448, 0x1076BF5A, 0x06DCCCA8, 0x54E02C96,
0x4475E284, 0x0B6A60BE, 0x204FA349, 0X4B75770E,
0x631E22C1, 0x603EA557, 0x6A4376E0, 0x5F69C330,
0x26BBA08C, 0x1F494E05, 0x4A495DDD, 0X523E20EF,
0x12C5229D, 0x6DD2367D, 0x274CD717, 0x602A6FA5,
0x3D2D3CBC, 0x02CA49A6, 0x6A355D28, 0X01882C9D,
0x4D159234, 0x773C8B3E, 0x033D51B3, 0x4B98DBEA,
0x55E7C3E7, 0x49D2139E, 0x001B01F4, 0X55DBA830,
0x5A48D2F8, 0x06F7CE9C, 0x2ABBD4C6, 0x1EBEB57C,
0x12622F5A, 0x4B0B780F, 0x6A342C8B, 0X7580521B,
0x2B4A1D67, 0x5477A36B, 0x54EA154B, 0x5205BDF3,
0x73C0F171, 0x1F337328, 0x2443DEE2, 0X0686140E,
0x0D05A9A5, 0x4B90B5F9, 0x66B083B3, 0x4A32E662,
0x4E5AFFA0, 0x50E5E0DB, 0x4BBB12FF, 0X1B7091D4,
0x48226C1A, 0x4EF864B2, 0x67096DBE, 0x1E0A3001,
0x18CA7850, 0x67246FB2, 0x73E5D831, 0X73134B48,
0x6E1C3E4E, 0x1EA1ACF7, 0x11D200C5, 0x007E6DA8,
0x69AD2507, 0x7C062D50, 0x75FEBFC3, 0X14F7426E,
0x507DD0BB, 0x4AE8D50E, 0x66FD0061, 0x443EC22C,
0x6A1C4836, 0x0B40DF43, 0x4AC4D63A, 0X7721F1DC,
0x56D1953D, 0x317559ED, 0x4154D83E, 0x252C94DD,
0x025B3AC9, 0x0D0FEB3D, 0x409D26B1, 0X4A7DA6E3,
0x5C084FEF, 0x27A6946F, 0x6887D6E4, 0x74D2C83F,
0x0ECB0421, 0x5C6DAF16, 0x67E61387, 0X7CE7426F,
0x7B0F5C0D, 0x79B8144C, 0x7D65B017, 0x64BC8114,
0x75BE419C, 0x73646FDA, 0x79B3C382, 0X463C1258,
0x3E4D44E8, 0x60B0C3E3, 0x0A7AD484, 0x28698D1E,
0x6BF1A327, 0x553FAABF, 0x1F8B7EFA, 0X42C33864,
0x06B504AC, 0x60E05738, 0x67EFCD41, 0x09103F75,
0x6DF04275, 0x288CF3F2, 0x538DE658, 0X49F89264,
0x50338861, 0x3C15BD3D, 0x3ECB5AA3, 0x5EFE8C82,
0x18836C53, 0x26B16E2B, 0x5BE5CEF1, 0X1392C860,
0x20698277, 0x594B7F08, 0x784F4975, 0x1627C414,
0x4CAFEEE2, 0x72030CF7, 0x5C63D66C, 0X0AFD33CA,
0x52B3D0DB, 0x66DEAAF0, 0x3366C0E8, 0x3EA57402,
0x3C1E55AF, 0x52F23FE3, 0x0168AC66, 0X42D35A5C,
0x33D2971B, 0x695879A7, 0x4BE399D1, 0x21C2D991,
0x11E56D99, 0x1F71802A, 0x6BBB6BF5, 0X6218F5FA,
0x5B873D67, 0x2A86C699, 0x4117827C, 0x740AA9BA,
0x513834C4, 0x1CFD516D, 0x079D721A, 0X71A1B73B,
0x7648D075, 0x7FECBB8F, 0x07C97B4F, 0x42F8BF57,
0x71EFC887, 0x642D51BB, 0x4DF5F321, 0X44A39962,
0x4B0BFCAC, 0x015CB409, 0x03490D64, 0x072A525B,
0x544EF3EC, 0x04B1B9CA, 0x49FDACB7, 0X08218B08,
0x6E0A3371, 0x15E14689, 0x29E46499, 0x7FEFA10A,
0x3552C6B3, 0x159FD08E, 0x62089704, 0X10DA041A,
0x40269727, 0x23201980, 0x04E4ADD4, 0x115ECBEB,
0x401D6AED, 0x0C821FEE, 0x03008327, 0X36663B62,
0x0C6EDB7E, 0x0AC9FE76, 0x795EFAB9, 0x7E5EA405,
0x6EF75032, 0x4754EDDA, 0x43023D67, 0X3A034CDE,
0x48B1A1E3, 0x464B4ACB, 0x412D9F39, 0x1D0095D0,
0x4AFD0495, 0x0B2B4BF1, 0x252220D8, 0X39073806,
0x210C927A, 0x4F068571, 0x38F6D910, 0x565F592D,
0x64A655FF, 0x1AFF7014, 0x67395D47, 0X24CCED27,
0x3E1F8994, 0x6C1E0B1B, 0x362BB912, 0x7E3CF481,
0x78A02B09, 0x392C3C39, 0x34A32FE3, 0X050F0687,
0x43F63AB0, 0x2E022A9C, 0x036DAA8C, 0x32ED8AE2,
0x75571876, 0x466FE7F3, 0x6CF0D7C0, 0X3E08BA59,
0x0CBB32BE, 0x2E1E76F9, 0x5B095029, 0x57B83753,
0x3949C2EA, 0x002B7101, 0x10BF6F59, 0X5A565564,
0x4F31F672, 0x49B64869, 0x30B5AE91, 0x33D84C72,
0x64B5B87D, 0x17EF0BD8, 0x58A53999, 0X22D54211,
0x040D16F3, 0x0ED0F2AB, 0x21123692, 0x7CAD41FD,
0x47FD2EE5, 0x55B56675, 0x01BC4884, 0X0BF36995,
0x03B79111, 0x0529F311, 0x3EE0F477, 0x790EA987,
0x4B99DB04, 0x2BD1CC37, 0x371763E1, 0X58550DC3,
0x59F04330, 0x1220B40A, 0x300D4516, 0x133A061B,
0x124C250C, 0x40CCB470, 0x6D905B7F, 0X617E1B7E,
0x0A82FCD9, 0x1E460A11, 0x155667F0, 0x6F38B557,
0x363515E9, 0x6DFBA189, 0x120DF768, 0X3A422CDD,
0x7CCC9435, 0x33202DFB, 0x36EF6EDA, 0x44C9C31A,
0x08D59470, 0x38ABB75E, 0x50BD2CAF, 0X0C8D2582,
0x3DD5AA6F, 0x0F9E2126, 0x059BCF09, 0x096F8574,
0x3B6FED5D, 0x3CB332EA, 0x61C49337, 0X1560308D,
0x4ED3E6F5, 0x11D1D84D, 0x289A36A8, 0x61200C01,
0x529E8CBD, 0x162A9228, 0x429E277F, 0X5D218997,
0x34709C39, 0x57F48F70, 0x4C5A3EEE, 0x6AA5B222,
0x45F030F9, 0x5E683656, 0x24E7DEFF, 0X42BCC52E,
0x11886451, 0x5BD74DD9, 0x07868848, 0x1A5DF8C2,
0x14830538, 0x5843B4F7, 0x26EB1E44, 0X5258AFA7,
0x67E1D61D, 0x2C86ED4D, 0x5BC8351B, 0x2351C37A,
0x693A2038, 0x3D8CC852, 0x38B1F408, 0X380E072D,
0x4F5EA0A0, 0x614C2AB0, 0x192E132E, 0x21FD2D5D,
0x7776BCD8, 0x5BCC3AAD, 0x7F1EB6F4, 0X2BE75911,
0x33C0CA1D, 0x4B78F5E2, 0x168D0B34, 0x79B0FB17,
0x29E12C39, 0x3B74EA33, 0x3C6DC045, 0X3B69908A,
0x174C380D, 0x43F4488E, 0x55C7894C, 0x2BCF3D45,
0x1C37FD85, 0x7CB2A790, 0x7E27ECEC, 0X0419D3A3,
0x293994DE, 0x59F02208, 0x276B971D, 0x1273B516,
0x177CEA5A, 0x601D8B25, 0x4A81BC43, 0X66DB8AFA,
0x4169B5D6, 0x63AFCF71, 0x08D8B858, 0x38E072AE,
0x3F7C0A1E, 0x07F76F4C, 0x64C7CBC0, 0X733CD43C,
0x5370652F, 0x7B54D6F4, 0x6CEDCF53, 0x7D519168,
0x36C9C127, 0x295B8F98, 0x38BB21F2, 0X4E15F934,
0x6D4FD826, 0x0E82AB3F, 0x79E53679, 0x0987D5AC,
0x0B3552CF, 0x780D2366, 0x0DA1A94F, 0X346EE7AD,
0x51FD456E, 0x350D406C, 0x46E29CC3, 0x697A2FC8,
0x152ACB92, 0x11645906, 0x5055BAC3, 0X56948168,
0x75142877, 0x592E731B, 0x0F74F416, 0x34903296,
0x6125E267, 0x743CBFD6, 0x27CD06D2, 0X34964796,
0x6F9196CA, 0x14BAD625, 0x31E7D8FE, 0x265B57F2,
0x3E1665BD, 0x6AA2FAF1, 0x74715126, 0X2B663DE4,
0x7925A630, 0x6E5687A0, 0x34EE1390, 0x045AF8FF,
0x6663AB06, 0x428FBCDF, 0x38C9E0AD, 0X3860F074,
0x779CFD4B, 0x7FAC7D70, 0x21DB203C, 0x0CC7C8DD,
0x1110D677, 0x7230DAFF, 0x635C4A45, 0X0624FEEE,
0x4B5F4E1A, 0x72D13E5C, 0x3AB53184, 0x2C853082,
0x670DFE32, 0x62823856, 0x611B7818, 0X569F94FD,
0x773D0E7B, 0x13035117, 0x7CFAECEF, 0x35537439,
0x7DA64C08, 0x716C3E15, 0x60B9B21D, 0X76CBF238,
0x5FC2C5B5, 0x15A7C5AD, 0x7B26EB37, 0x462670BB,
0x5837828C, 0x33F0CBE4, 0x7E87612F, 0X4FD47FD7,
0x339D4955, 0x2062816C, 0x5C9C48B5, 0x44AE1FCC,
0x12935C6B, 0x3FF892FA, 0x4AD31EBA, 0X5DF2AA86,
0x32C9D156, 0x0588503F, 0x0A77DB08, 0x19D7CF89,
0x680A8895, 0x6B935320, 0x70776486, 0X5F479711,
0x7E96A437, 0x6D725175, 0x149B0B4A, 0x7C3CF03F,
0x5EDE8F8A, 0x7554BD67, 0x7308E277, 0X3EA15540,
0x0AFC8314, 0x6E2FCDAF, 0x04C7C5FB, 0x633405A0,
0x22209993, 0x034F272B, 0x33088577, 0X55BDE2E8,
0x23B1A897, 0x0FA4CE2C, 0x1A6C02B4, 0x36450502,
0x4F9D6127, 0x653F216F, 0x1437AF88, 0X0267327D,
0x6AC771AE, 0x1EAF8A90, 0x1C3F0206, 0x52D1FA43,
0x0A42DDB1, 0x0CB6668C, 0x32199154, 0X08D981E8,
0x7A28B801, 0x46B49C9E, 0x05167228, 0x5907478C,
0x3C095A05, 0x781F549F, 0x17A89CCC, 0X4705DD19,
0x664F224E, 0x1C7062C7, 0x2A39E2B9, 0x086FBBE2,
0x1FBF89F2, 0x5D426831, 0x5E2D9ECA, 0X43713289,
0x6CE7365D, 0x7899A17F, 0x79B6378C, 0x3C849784,
0x5DD8C2EE, 0x0DEDE714, 0x3EEBCA02, 0X48A0349C,
0x2C9D71A5, 0x5B2ACC08, 0x1B722EDF, 0x36E04F56,
0x67E13295, 0x4D8BC034, 0x3FB9D13E, 0X6209EA96,
0x14405CD2, 0x44D04366, 0x3B113222, 0x5049B6D8,
0x3CEF9806, 0x52B9CEEE, 0x174F93F1, 0X233EBA54,
0x6F2A31B6, 0x418976AB, 0x2BAE7636, 0x0EE9BBA8,
0x1ECBDEDC, 0x09DC1501, 0x525AEE32, 0X0BB31539,
0x0275B680, 0x4C1125BE, 0x4837ACBE, 0x604E796E,
0x59FF0CD2, 0x072376C0, 0x28EEAE0A, 0X069C7E77,
0x624E42C8, 0x4460DCE9, 0x3D7CCDCD, 0x4A2F755D,
0x11EC9D1D, 0x7D369F0C, 0x2C395FF4, 0X262CF9F0,
0x4206E272, 0x674A9216, 0x7676B0C8, 0x7EF67A78,
0x3A046105, 0x0DC644B9, 0x223534CD, 0X292E92BB,
0x4F4FBB64, 0x4DE3AB03, 0x38184E63, 0x6E1B9A40,
0x57BFC004, 0x0A733C95, 0x79CEAF7A, 0X5A357684,
0x56846253, 0x42065C38, 0x3A83EFF2, 0x30836F26,
0x4929D2F8, 0x63729DFC, 0x371FED9D, 0X2B7815C0,
0x27D37AE6, 0x749CBB6B, 0x75A78B1E, 0x39C01803,
0x71D35A77, 0x21E0EB12, 0x5FED11F3, 0X33DA3CE9,
0x092B7D28, 0x5663C2BB, 0x32D0B762, 0x432FDE2D,
0x642A0775, 0x5505EC2F, 0x6C5E70E8, 0X3379C2D9,
0x22E99732, 0x2476BF4C, 0x21955D1A, 0x7AA95737,
0x2EE9FBE1, 0x1B640C94, 0x54DECDBB, 0X056E5E35,
0x5D6A68CC, 0x0F62BDAE, 0x35F1CD5B, 0x26943BC4,
0x72D55BAA, 0x6D11BAF8, 0x520C5184, 0X1AA8D690,
0x61AE7663, 0x47B3DCA2, 0x5468EE94, 0x5381D0DA,
0x6994C7B4, 0x34560087, 0x075C0DC4, 0X72C044DD,
0x0AB9C343, 0x3A2CC526, 0x35F0230A, 0x6EE3CAB8,
0x0F32B155, 0x224E93F3, 0x225D8D91, 0X321C4887,
0x46C5533F, 0x43F2EAAB, 0x2CC59FBE, 0x75AF4F20,
0x5F56F73F, 0x01A46D7A, 0x7B1DAD55, 0X3CC1600B,
0x11072B28, 0x310F7AB0, 0x63559BCF, 0x03DC86D2,
0x1E2135A9, 0x3561ED54, 0x1E855D63, 0X7FCFAC0C,
0x7D15C9F6, 0x72EE4BF7, 0x53517CE7, 0x66AA91AB,
0x27444C7E, 0x5AAD8AAB, 0x596AD688, 0X31FE0FC1,
0x14DA4FD1, 0x0F5AF992, 0x20E1DA79, 0x240D0126,
0x31A98D85, 0x433F680B, 0x562949AD, 0X786EE0C4,
0x073252B6, 0x02EEE96C, 0x6E1E2FE5, 0x668949F6,
0x049356E6, 0x693BDD3A, 0x234AAA01, 0X159A820E,
0x1A4B57EB, 0x06A045D1, 0x197708E0, 0x386C8D94,
0x3C023325, 0x37FC6643, 0x383C39A0, 0X3917FD1B,
0x2AEAB23A, 0x0B8DB687, 0x1FC28EC6, 0x522EFEB9,
0x663B4132, 0x792D654E, 0x042D0E7A, 0X7B159103,
0x08885EE1, 0x250EE8F4, 0x1F229229, 0x3A31EC66,
0x684E50FF, 0x754BDBD7, 0x32A0CD2B, 0X6F80A3B5,
0x783AC543, 0x20BEFD10, 0x5609EDAB, 0x7CCE1C29,
0x09FADA4A, 0x795497AD, 0x12689E37, 0X24463235,
0x7FF4DD7E, 0x2BDFA717, 0x5CB2BFC9, 0x3BF710A3,
0x63DC0D5B, 0x14EEF96A, 0x750F0DBE, 0X0EC6BF95,
0x207CAFF1, 0x14D19C85, 0x60F5BE4E, 0x06B7F124,
0x0DFF01D3, 0x6522CCC9, 0x01CD8227, 0X168760B4,
0x0A31B5BD, 0x20F01451, 0x50B94D1B, 0x728006BC,
0x163BF028, 0x035A1A46, 0x6200AA71, 0X0E76B56B,
0x24191756, 0x380A981D, 0x0B44D194, 0x2E13F1A0,
0x315F2FCA, 0x1DAD6FCB, 0x525A23D6, 0X31540D48,
0x498D16E2, 0x2F0CE39F, 0x6D4B1DEB, 0x2D69243D,
0x43FBDD09, 0x625A2BA9, 0x3C2FE3D3, 0X64788CFB,
0x772BC82E, 0x1D25A221, 0x6B307E1F, 0x052ACA02,
0x02486EEA, 0x6CFE0046, 0x1BB22AB6, 0X0C7A24A7,
0x0DEE1497, 0x6C6B77D1, 0x7EFA2B63, 0x242A04BF,
0x6FC59217, 0x60FAD5D5, 0x32A0BA2A, 0X13DEA96D,
0x19056DF2, 0x3DE58BBE, 0x41F29B0E, 0x4A649DBC,
0x5B92FB89, 0x144CBEE4, 0x7BB8AB04, 0X2520126C,
0x4359A283, 0x6903C8EF, 0x528936A9, 0x07557F8D,
0x4B5DF498, 0x0EB91A7C, 0x6BCE0C88, 0X4289BCC7,
0x2BDEBC9E, 0x56FE8AA7, 0x47B486C9, 0x2E272B88,
0x43FC8AED, 0x6366B17F, 0x3AA15030, 0X51EA9F85,
0x4FD22951, 0x399B7B93, 0x7614A444, 0x3F97BB68,
0x1A965168, 0x28B55E6F, 0x537664D6, 0X339BBF5A,
0x669AEA2D, 0x1568FFE4, 0x7E005D16, 0x422DE5B7,
0x29B5BEC8, 0x79B9081A, 0x674DF823, 0X6D0F614B,
0x62BCD109, 0x39D72ECC, 0x7464E0D8, 0x2E1AC5A2,
0x48904949, 0x6032ED60, 0x70A48269, 0X746F05E7,
0x37317807, 0x38590932, 0x2296316F, 0x7B2E02F5,
0x1BBFBAB1, 0x5D37819F, 0x4D18A27A, 0X6B91E402,
0x16D2FD33, 0x432D46BE, 0x2B299F6B, 0x31694E9B,
0x6BE2A52D, 0x7EA00441, 0x65050DF6, 0X527D8F5B,
0x14090425, 0x63056B0C, 0x14AB7512, 0x3DBEC2ED,
0x5CBE7327, 0x7BF96D35, 0x2ACE2438, 0X3F7B4430,
0x35D09C01, 0x1F330511, 0x6D9609D2, 0x7E60E54A,
0x7F65F271, 0x5E3A8C3B, 0x72CFEB31, 0X36976A79,
0x1693956D, 0x15661CA1, 0x31C56D6E, 0x3253501F,
0x729D9E40, 0x7EDE0FE8, 0x1DE53421, 0X09709B73,
0x420B56A6, 0x490ED38C, 0x3AD9EA0F, 0x2DEDFBD4,
0x47AED7CD, 0x1FDEF805, 0x006B8B2F, 0X5BB7DBF2,
0x02E46311, 0x15170041, 0x19769EDF, 0x5FA2D638,
0x11106D76, 0x4444C318, 0x1F1E1A69, 0X46E10977,
0x6377C829, 0x0CB4243B, 0x4541EEC2, 0x62DDBA9A,
0x6AEEB077, 0x3811D9F3, 0x19752513, 0X018245E4,
0x4D77F694, 0x4B3A9281, 0x33D59603, 0x401594D5,
0x4A18A269, 0x51BACA25, 0x49863048, 0X0C23F910,
0x1AC99DB1, 0x04601A57, 0x3A11F4E4, 0x6278757F,
0x243F125C, 0x3A7D8013, 0x3E305171, 0X2723756E,
0x4F948054, 0x57A6F051, 0x06C64BA6, 0x60A4EDCA,
0x1BEBB369, 0x25E4660F, 0x2785F741, 0X7F637B92,
0x32988A4B, 0x6CC7E603, 0x6241362C, 0x1D873AC2,
0x24D9BFF7, 0x7BB65B40, 0x1F0980A6, 0X7251B68B,
0x46F0EDC1, 0x52DF16AA, 0x32674B60, 0x1109902B,
0x2499E0CF, 0x7BED7BA9, 0x1D2D893B, 0X3F637E80,
0x004D9600, 0x573F7E1F, 0x21DBF3FF, 0x248CA85D,
0x11BCFE32, 0x600C4571, 0x4BB01DCB, 0X61517E86,
0x37B335C2, 0x52766971, 0x41F66C50, 0x539EE92B,
0x785ACF81, 0x697C6391, 0x530264BD, 0X2AF359CC,
0x56444995, 0x35439AE9, 0x487A948E, 0x7B1E098C,
0x30F9F629, 0x67841534, 0x6D6FC017, 0X77EAE3EB,
0x3A632BDE, 0x1FD70B78, 0x08F47416, 0x5EFD0CAD,
0x1BC48721, 0x2621FD51, 0x1E608B2E, 0X1C121D21,
0x7D617B70, 0x403C7F2D, 0x409EC57E, 0x0F1E79A2,
0x2048C49E, 0x0C4EE349, 0x706FF828, 0X57FBFA60,
0x5EC54CBB, 0x32666478, 0x2B9AE38B, 0x57201C3C,
0x1BE2C809, 0x7E9D4848, 0x02137608, 0X7227119E,
0x33E0E332, 0x4A8E0A96, 0x6D451B2A, 0x64DAD95B,
0x32121FCA, 0x5AB4DB42, 0x5CC5BD46, 0X6C754BA9,
0x7A8BE6BA, 0x65BA315C, 0x4B725856, 0x16506DDB,
0x0BDC2EAD, 0x69D2E384, 0x32628AFC, 0X093DAA1D,
0x2A0F62B2, 0x7301507B, 0x185C23BF, 0x4A582750,
0x7F5033C4, 0x08CC1BE7, 0x225421B1, 0X5E15807F,
0x3B32805F, 0x4DEF053C, 0x35359CBB, 0x57154869,
0x4C8C4D85, 0x374912C3, 0x493C5A07, 0X006D30B7,
0x01D71D59, 0x36817532, 0x65480A12, 0x33E93D24,
0x11365074, 0x420DC759, 0x205E88CD, 0X0BC2372E,
0x27C7F8B5, 0x6BD0E123, 0x2212A509, 0x33A42763,
0x55A3C4A8, 0x54753005, 0x3CE1D180, 0X7FB3275A,
0x47768080, 0x553DF540, 0x4A0B4EAA, 0x46C6B445,
0x5E0A1127, 0x6C5F705B, 0x24DC34C4, 0X193C9187,
0x3A4E7598, 0x5A11D180, 0x7051D9F0, 0x06DAC31D,
0x115AE443, 0x398E33F7, 0x0747F3D4, 0X1332019D,
0x700FA929, 0x6C8FFDE6, 0x471B3EC1, 0x0145F99D,
0x2E9DC53F, 0x6779C78E, 0x0D0830CB, 0X5665BDF5,
};

#endif /* GRANULARDCO_H */

