------------------------------------------------------------------------------
--                                                                          --
--                      GNAT METRICS TOOLS COMPONENTS                       --
--                                                                          --
--           M E T R I C S . M E T R I C _ D E F I N I T I O N S            --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                    Copyright (C) 2002-2005, AdaCore                      --
--                                                                          --
-- GNAT Metrics Toolset  is free software;  you can  redistribute it and/or --
-- modify it under terms of the  GNU General Public License as published by --
-- the Free Software Foundation;  either version 2, or (at your option) any --
-- later version.  GNAT Metrics Toolset is  distributed in the hope that it --
-- will be useful, but  WITHOUT ANY WARRANTY; without even the implied war- --
-- ranty of  MERCHANTABILITY or  FITNESS FOR A PARTICULAR PURPOSE.  See the --
-- GNU General Public License for more details.  You should have received a --
-- copy of the  GNU General Public License distributed with  GNAT; see file --
-- COPYING.  If not,  write to the  Free  Software  Foundation,   59 Temple --
-- Place - Suite 330, Boston,                                               --
--                                                                          --
-- GNAT Metrics Toolset is maintained by ACT Europe                         --
-- (http://www.act-europe.fr).                                              --
--                                                                          --
------------------------------------------------------------------------------

--  This package contains the definition of different metrics and the
--  corresponding data structures

package METRICS.Metric_Definitions is

   Metric_Count_Disabled : constant := -2;
   --  Used to indicate the situation that the given metric should not be
   --  computed and reported

   Metric_Count_Undefined : constant := -1;
   --  Used to indicate the situation that the given metric should be
   --  computed and reported but has not been computed

   Max_Metric_Count : constant := Integer'Last;

   type Metric_Count is range Metric_Count_Disabled .. Max_Metric_Count;

   ------------------
   -- Line metrics --
   ------------------

   --  We define the set of line metrics as a record type to make it possible
   --  to store these metrics for each source being processed

   type Line_Metrics_Record is record
      All_Lines : Metric_Count;
      --  The total number of lines in the given source

      Code_Lines : Metric_Count;
      --  Lines containing at least one character belonging to the non-comment
      --  Ada code

      Comment_Lines : Metric_Count;
      --  Comment lines

      EOL_Comments : Metric_Count;
      --  Lines containing both Ada code and comments, or, in other words,
      --  End-Of-Line comments

      Blank_Lines : Metric_Count;
      --  Lines containing only space characters and/or format effectors

   end record;

   Zero_Line_Metrics : constant Line_Metrics_Record :=
     (All_Lines     => 0,
      Code_Lines    => 0,
      Comment_Lines => 0,
      EOL_Comments  => 0,
      Blank_Lines   => 0);

   ---------------------
   -- Element Metrics --
   ---------------------

   --  We define the set of line metrics as a record type to make it possible
   --  to store these metrics for each source being processed

   type Element_Metrics_Record is record
      All_Statements : Metric_Count;
      --  The total number of statements;

      All_Declarations : Metric_Count;
      --  The total number of declarations;

      Max_Program_Unit_Nesting : Metric_Count := 0;
      --  Maximal number of nesting level of program units nested into the
      --  given element.

      Max_Construct_Nesting : Metric_Count := 0;
      --  Maximal number of construct nesting.

--      Max_Instantiation_Depth : Metric_Count := 0;
      --  Maximal number of the instantiation depth computed over all the
      --  instantiation contained into the given element

   end record;

   Zero_Element_Metrics : constant Element_Metrics_Record :=
     (All_Statements           => 0,
      All_Declarations         => 0,
      Max_Program_Unit_Nesting => 0,
      Max_Construct_Nesting        => 0);
--      Max_Instantiation_Depth  => 0);

   type Public_Types_Details is record
      Abstract_Types  : Metric_Count;
      Tagged_Types    : Metric_Count;
      Private_Types   : Metric_Count;
      Task_Types      : Metric_Count;
      Protected_Types : Metric_Count;
   end record;

   Zero_Public_Types_Details : constant Public_Types_Details :=
     (Abstract_Types  => 0,
      Tagged_Types    => 0,
      Private_Types   => 0,
      Task_Types      => 0,
      Protected_Types => 0);

   type All_Metrics is record
      Computed_Line_Metrics : Natural;
      --  The number of units for which line metrics are safely computed

      Computed_Element_Metrics : Natural;
      --  The number of units for which element metrics are safely computed

      Computed_Public_Subprograms : Natural;
      --  The number of units for which public subprograms are safely
      --  computed

      Computed_All_Subprograms : Natural;
      --  The number of units for which all the subprograms are safely
      --  computed

      Computed_Public_Types : Natural;
      Computed_All_Types    : Natural;

      Line_Metrics    : Line_Metrics_Record;
      Element_Metrics : Element_Metrics_Record;

      Public_Subprograms : Metric_Count;
      All_Subprograms    : Metric_Count;

      Public_Types          : Metric_Count;
      Public_Types_Detailed : Public_Types_Details;
      All_Types             : Metric_Count;

      --  We count public and all subprograms and types separately from
      --  Element metrics because these metrics are computed for compilation
      --  units only, not for nested program units.
   end record;

   Global_Statistics : All_Metrics;
   --  Here we collect the integrated metrics values for all the units being
   --  processed

   function Computed (Value : Metric_Count) return Boolean;
   --  Checks if Value is computed (that is, neither disabled nor undefined)

   procedure Init_Global_Statistics;
   --  This procedure initializes the fields of Global_Statistics global
   --  variable according to the metrics computation options

   procedure Add_Public_Types_Details (Value : Public_Types_Details);
   --  Adds detailed counters for public types to the corresponding fields
   --  of Global_Statistics.

   function Details_Present (Value : Public_Types_Details) return Boolean;
   --  Checks if at least one field of the argument is non-zero.

   procedure Set_Global_Metrics_Flags;
   --  Sets values of the flags defined in METRICS.Common which indicate if
   --  this or that "number of..." metric should be computed for a currently
   --  processed unit.

end METRICS.Metric_Definitions;
