/*

*************************************************************************

ArmageTron -- Just another Tron Lightcycle Game in 3D.
Copyright (C) 2000  Manuel Moos (manuel@moosnet.de)

**************************************************************************

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  
***************************************************************************

*/

#include "gCycle.h"
#include "nConfig.h"
#include "rModel.h"
//#include "eTess.h"
#include "eGrid.h"
#include "rTexture.h"
#include "eTimer.h"
#include "tInitExit.h"
#include "rScreen.h"
#include "rFont.h"
#include "gSensor.h"
#include "ePlayer.h"
#include "eSound.h"
#include "gSparks.h"
#include "gExplosion.h"
#include "gWall.h"

#include <math.h>
#include <stdlib.h>
#include <assert.h>

#ifndef DEDICATED
#include "rRender.h"
#include <SDL.h>
#endif

#ifdef WIN32
#include <float.h>
#define finite _finite
#endif

void clamp(REAL &c, REAL min, REAL max){
  assert(min < max);

  if (!finite(c))
    c = 0;

  if (c<min)
    c = min;

  if (c>max)
    c = max;
}


// base speed of cycle im m/s
static REAL sg_speedCycle=10;
static nSettingItem<REAL> c_s("CYCLE_SPEED","Basic speed of your cycle if you drive straight and not close to eWalls",sg_speedCycle);

// start speed of cycle im m/s
static REAL sg_speedCycleStart=20;
static tSettingItem<REAL> c_st("CYCLE_START_SPEED",
			       "Initial cycle speed",sg_speedCycleStart);

// min time between turns
REAL sg_delayCycle=.1;
static nSettingItem<REAL> c_d("CYCLE_DELAY",
			      "Minimum time between turns",sg_delayCycle);

// acceleration multiplicator
static REAL sg_accelerationCycle=1;
static nSettingItem<REAL> c_a("CYCLE_ACCELL","Wall acceleration factor"
			      ,sg_accelerationCycle);

// acceleration multiplicator
static REAL sg_accelerationCycleOffs=1;
static nSettingItem<REAL> c_ao("CYCLE_ACCELL_OFFSET","Minimum numeric eWall distance",sg_accelerationCycleOffs);

// when is a eWall near?
static REAL sg_nearCycle=6;
static nSettingItem<REAL> c_n("CYCLE_WALL_NEAR","Maximum accelerating eWall distance",sg_nearCycle);

// sound speed divisor
static REAL sg_speedCycleSound=10;
static nSettingItem<REAL> c_ss("CYCLE_SOUND_SPEED","Sound speed divisor",sg_speedCycleSound);

// strength of brake
static REAL sg_brakeCycle=30;
static nSettingItem<REAL> c_ab("CYCLE_BRAKE","Brake intensity",sg_brakeCycle);

#ifndef DEDICATED
#define MAXRUBBER 1
#else
#define MAXRUBBER 3
#endif

// niceness when crashing a eWall
static REAL sg_rubberCycle=MAXRUBBER;
static tSettingItem<REAL> c_r("CYCLE_RUBBER","Niceness factor to allow you drive really close to a eWall",sg_rubberCycle);


// niceness when crashing a eWall, influence of your ping
static REAL sg_rubberCyclePing=3;
static tSettingItem<REAL> c_rp("CYCLE_PING_RUBBER","Addidional niceness for "
"high ping players",sg_rubberCyclePing);


// moviepack hack
//static bool moviepack_hack=false;       // do we use it?
//static tSettingItem<bool> ump("MOVIEPACK_HACK",moviepack_hack);



static int score_die=-2;
static tSettingItem<int> s_d("SCORE_DIE","What you get for dying",score_die);

static int score_kill=3;
static tSettingItem<int> s_k("SCORE_KILL","What you get for killing someone",score_kill);

static int score_suicide=-4;
static tSettingItem<int> s_s("SCORE_SUICIDE","What you get for stupidly dying",score_suicide);

// input control

uActionPlayer gCycle::s_brake("CYCLE_BRAKE","brake",
			   "decreases your speed in critical situations");

uActionPlayer eGameObject::se_turnRight("CYCLE_TURN_RIGHT","turn right",
	       "Makes a 90 degrees turn to the right");

uActionPlayer eGameObject::se_turnLeft("CYCLE_TURN_LEFT","turn left",
			       "Makes a 90 degrees turn to the left");

static eWavData cycle_run("moviesounds/engine.wav","sound/cyclrun.wav");
static eWavData turn_wav("moviesounds/cycturn.wav","sound/expl.wav");

// a class of textures where the transparent part of the
// image is replaced by the player color
class gTextureCycle: public rTexture{
  REAL r,g,b; // player color
  bool wheel; // wheel or body
 public:
  gTextureCycle(const char *fileName,REAL R,REAL G,REAL B,bool repx=0,bool repy=0,bool wheel=false);

  virtual void ProcessImage(SDL_Surface *im);

  void Select();
};


gTextureCycle::gTextureCycle(const char *fileName,REAL R,REAL G,REAL B,bool repx,bool repy,bool w)
  :rTexture(rTEX_OBJ,fileName,repx,repy),
   r(R),g(G),b(B),wheel(w)
{
  Select();
}

void gTextureCycle::ProcessImage(SDL_Surface *im){
#ifndef DEDICATED
  GLubyte R=int(r*255);
  GLubyte G=int(g*255);
  GLubyte B=int(b*255);
  
  GLubyte *pixels =reinterpret_cast<GLubyte *>(im->pixels);

  for(int i=im->w*im->h-1;i>=0;i--){
    GLubyte alpha=pixels[4*i+3];
    pixels[4*i  ] = (alpha * pixels[4*i  ] + (255-alpha)*R) >> 8;
    pixels[4*i+1] = (alpha * pixels[4*i+1] + (255-alpha)*G) >> 8;
    pixels[4*i+2] = (alpha * pixels[4*i+2] + (255-alpha)*B) >> 8;
    pixels[4*i+3] = 255;
  }
#endif
}

void gTextureCycle::Select(){
#ifndef DEDICATED
  rTexture::Select();

  if(TextureMode[rTEX_OBJ]<0){
    REAL R=r,G=g,B=b;
    if(wheel){
      R*=.7;
      G*=.7;
      B*=.7;
    }
    glColor3f(R,G,B);
    GLfloat color[4]={R,G,B,1};
    
    glMaterialfv(GL_FRONT_AND_BACK,GL_SPECULAR,color);
    glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,color);
  }
#endif
}


//  *****************************************************************



//  *****************************************************************



//  *****************************************************************


// take pos,dir and time from a cycle
gDestination::gDestination(const gCycle &c)
  :position(c.Position()),
   direction(c.Direction()),
   gameTime(c.lastTime),
   distance(c.distance),
   speed(c.speed),
   braking(c.braking),
   hasBeenUsed(false),
   next(NULL),
   list(NULL){
#ifdef DEBUG
  if (!finite(gameTime) || !finite(speed) || !finite(distance))
    st_Breakpoint();
#endif
}

// or from a message
gDestination::gDestination(nMessage &m)
:gameTime(0),distance(0),speed(0),
 hasBeenUsed(false),
 next(NULL),list(NULL){
  m >> position;
  m >> direction;
  m >> distance;
  m >> braking;
}
  
// write all the data into a nMessage
void gDestination::WriteCreate(nMessage &m){
  m << position;
  m << direction;
  m << distance;
  m << braking;
}


gDestination *gDestination::RightBefore(gDestination *list, REAL dist){
  if (!list || list->distance > dist)
    return NULL;

  gDestination *ret=list;
  while (ret && ret->next && ret->next->distance < dist)
    ret=ret->next;

  return ret;
}


// insert yourself into a list ordered by gameTime
void gDestination::InsertIntoList(gDestination **l){
  if (!l)
    return;
  
  RemoveFromList();
  
  list=l;

  while (l && *l && (*l)->distance < distance)
    l = &((*l)->next);

  assert(l);
  
  next=*l;
  *l=this;
}



void gDestination::RemoveFromList(){
  if (!list)
    return;
    
  while (list && *list && *list != this)
    list = &((*list)->next);
  
  assert(list);
  assert(*list);
  
  (*list) = next;
  next=NULL;
  list=NULL;
}

//  *****************************************************************

static nNOInitialisator<gCycle> cycle_init(22,"cycle");

static void new_destination_handler(nMessage &m){
  // read the destination 
  gDestination *dest=new gDestination(m);

  // and the ID of the cycle the destination is added to
  unsigned short cycle_id;
m.Read(cycle_id);
 nNetObject *o=nNetObject::ObjectDangerous(cycle_id);
 if (o && &o->CreatorDescriptor() == &cycle_init){
   if ((sn_GetNetState() == nSERVER) && (m.SenderID() != o->Owner()))
     Cheater(m.SenderID());
   if(o->Owner() != ::sn_myNetID)
     dynamic_cast<gCycle *>(o)->AddDestination(dest);
 }
}

static nDescriptor destination_descriptor(27,&new_destination_handler,"destinaton");

static void BroadCastNewDestination(gCycle *c, gDestination *dest){
  nMessage *m=new nMessage(destination_descriptor);
  dest->WriteCreate(*m);
  m->Write(c->ID());
  m->BroadCast();
}

//  *****************************************************************

#define MAXAI_COLOR 13

static int current_ai;
static REAL rgb_ai[MAXAI_COLOR][3]={
  {1,.2,.2},
  {.2,1,.2},
  {.2,.2,1},
  {1,1,.2},
  {1,.2,1},
  {.2,1,1},
  {1,.6,.2},      
  {1,.2,.6},      
  {.6,.2,1},      
  {.2,.6,1},      
  {1,1,1},      
  {.2,.2,.2},       
  {.5,.5,.5}       
};

const eTempEdge* gCycle::Edge(){
  if (currentWall)
    return currentWall->Edge();
  else
    return NULL;
}

const gPlayerWall* gCycle::CurrentWall(){
  if (currentWall)
    return currentWall->Wall();
  else
    return NULL;
}

const gPlayerWall* gCycle::LastWall(){
  if (lastWall)
    return lastWall->Wall();
  else
    return NULL;
}


void gCycle::MyInitAfterCreation(){
#ifdef DEBUG
  //con << "creating cycle.\n";
#endif
  engine  = tNEW(eSoundPlayer)(cycle_run,true);
  turning = tNEW(eSoundPlayer)(turn_wav);

  correctDistSmooth=correctTimeSmooth=correctSpeedSmooth=0;

  currentDestination = NULL;
  destinationList = NULL;

  braking = false;
  mp=sg_MoviePack();

  lastTimeAnim = 0;

 if (mp)
   customModel=new rModel("moviepack/cycle.ASE","moviepack/cycle.ase");
 else{
   body=new rModel("models/cycle_body.mod");
   front=new rModel("models/cycle_front.mod");
   rear=new rModel("models/cycle_rear.mod");
 }

 correctPosSmooth=eCoord(0,0);

 rotationFrontWheel=rotationRearWheel=eCoord(1,0);

 acceleration=skew=skewDot=0;

 dir=dirDrive;
 alive=1;


 if (sn_GetNetState()!=nCLIENT){
   if(!ePlayer){ // distribute AI colors
     
     current_ai=(current_ai+1) % MAXAI_COLOR;
     int take_ai=current_ai;
     int try_ai=current_ai;
     
     REAL maxmindist=-10000;
     
     
     for(int i=MAXAI_COLOR-1;i>=0;i--){
       REAL mindist=4;
       REAL score=0;
       for (int j=grid->GameObjects().Len()-1;j>=0;j--){
	 eGameObject *go=grid->GameObjects()(j);
	 gCycle *c=dynamic_cast<gCycle*>(go);
	 if (c && c!=this){
	   REAL dist=
	     fabs(c->r - rgb_ai[try_ai][0])+
	     fabs(c->g - rgb_ai[try_ai][1])+
	     fabs(c->b - rgb_ai[try_ai][2]);
	   
	   score+=exp(-dist*dist*4);
	   
	   if (dist<mindist){
	     mindist=dist;
	     /*	   con << c->r << ":" << rgb_ai[try_ai][0] << '\t'
		   << c->g << ":" << rgb_ai[try_ai][1] << '\t'
		   << c->b << ":" << rgb_ai[try_ai][2] << '\t' << dist << '\n'; */
	   }
	 }
       }
       //con << "md=" << mindist << "\n\n";
       if (mindist>2)
	 mindist=2;
       
       mindist=-score;
       
       if (mindist>maxmindist){
	 maxmindist=mindist;
	 take_ai=try_ai;
       }
       
       try_ai = ((try_ai+1) % MAXAI_COLOR);
     }
     
     
     // con << current_ai << ':' << take_ai << ':' << maxmindist <<  "\n\n\n";
     
     r=rgb_ai[take_ai][0];
     g=rgb_ai[take_ai][1];
     b=rgb_ai[take_ai][2];
   }
   else{
     r=ePlayer->r/15.0;
     g=ePlayer->g/15.0;
     b=ePlayer->b/15.0;
     
     if (r+g+b<1){
       REAL add=(1-r+g+b)/3;
       r+=add;
       g+=add;
       b+=add;
     }
   }
 }
 
 if (mp)
   customTexture=new gTextureCycle("moviepack/bike.png",r,g,b,0,0);
 else{
   wheelTex=new gTextureCycle("textures/cycle_wheel.png",r,g,b,0,0,true);
   bodyTex=new gTextureCycle("textures/cycle_body.png",r,g,b,0,0);
 }

 distance=0;
 //last_turn=new ePoint(pos);
 //last_turnTime=lastTime;
 //currentPos=new ePoint(pos);
 //last_turn->InsertIntoGrid();
 // current_eEdge=new eEdge(last_turn,currentPos,
 //		       new gPlayerWall(this,lastTime,lastTime,distance,distance));
 z=2.5;
 pendingTurns=0;
 nextTurn=lastTime-10;
#ifdef DEBUG
 //con << "Created cycle.\n";
#endif
 nextSync=lastTime+1;
 
 if (!currentWall){
   currentWall=new gNetPlayerWall(this,pos,dirDrive,se_GameTime(),0);
   if (sn_GetNetState()!=nCLIENT)
     currentWallID=currentWall->ID();
 }
 lastWall=currentWall;

 if (sn_GetNetState()!=nCLIENT)
   RequestSync();

  grid->AddGameObjectInteresting(this);

  lastTimeAnim=lastTime;

  engine->Reset(10000);
  turning->End();

  nextChatAI=lastTime;

  turns=0;

  if (!finite(speed)){
    st_Breakpoint();
    speed = 1;
  }

  if (speed < .1)
    speed=.1;
}

void gCycle::InitAfterCreation(){
#ifdef DEBUG
  if (!finite(speed))
    st_Breakpoint();
#endif
  eNetGameObject::InitAfterCreation();
#ifdef DEBUG
  if (!finite(speed))
    st_Breakpoint();
#endif
  MyInitAfterCreation();
}

gCycle::gCycle(eGrid *grid, const eCoord &pos,const eCoord &d,ePlayerNetID *p,bool autodelete)
:eNetGameObject(grid, pos,d,p,autodelete),
 engine(NULL),
 turning(NULL),
 destinationList(NULL),currentDestination(NULL),
 dirDrive(d),skew(0),skewDot(0),speed(sg_speedCycleStart*2),acceleration(0),
 rotationFrontWheel(1,0),rotationRearWheel(1,0),heightFrontWheel(0),heightRearWheel(0),
 alive(1),
 currentWall(NULL),
 lastWall(NULL),
 currentWallID(0)
{  
  rubber=0;
  r=g=b=1;
  pendingTurns=0;
  turns=0;
  MyInitAfterCreation();
}

gCycle::~gCycle(){
#ifdef DEBUG
  //con << "deleting cycle.\n";
#endif
  // clear the destination list
  currentDestination = NULL;

  tDESTROY(engine);
  tDESTROY(turning);

  while(destinationList)
    delete destinationList;

  if (currentWall)
    currentWall->CopyIntoGrid();
  currentWall=NULL;
  lastWall=NULL;
  if (mp){
    delete customModel;
    delete customTexture;
  }
  else{
    delete body;
    delete front;
    delete rear;
    delete wheelTex;
    delete bodyTex;
  }
#ifdef DEBUG
  //con << "Deleted cycle.\n";
#endif
  /*
  delete currentPos;
  delete last_turn;
  */
  speed=distance=0;
}


inline void rotate(eCoord &r,REAL angle){
  REAL x=r.x;
  r.x+=r.y*angle;
  r.y-=x*angle;
  r=r*(1/sqrt(r.Norm_squared()));
}


bool crash_sparks=true;
static bool forceTime=false;

// from nNetwork.C
extern REAL planned_rate_control[MAXCLIENTS+2];

bool gCycle::Timestep(REAL currentTime){
  // remove old destinations
  while(destinationList && 
	destinationList != currentDestination && 
	destinationList->distance < distance - 100)
    delete destinationList;

  // nothing special if simulating backwards
  if (currentTime < lastTime)
    return TimestepCore(currentTime);

  // no targets are given
  
  if (!currentDestination || pendingTurns){
    if (currentTime>nextChatAI && bool(ePlayer) 
	&& ePlayer->chatting && ePlayer->Owner()==sn_myNetID){
      nextChatAI=currentTime+1;

      gSensor front(this,pos,dir);
      gSensor left(this,pos,dir.Turn(eCoord(0,1)));
      gSensor right(this,pos,dir.Turn(eCoord(0,-1)));
 
      REAL range=speed*4;
      
      front.detect(range);
      left.detect(range);
      right.detect(range);
      
      if (front.hit<left.hit*.9 || front.hit<right.hit*.9){
	REAL lr=front.lr;
	if (front.type==gSENSOR_SELF) // NEVER close yourself in.
	  lr*=100; 
	lr-=left.hit-right.hit;
	if (lr>0)
	  Act(&se_turnRight,1);
	else
	  Act(&se_turnLeft,1);
      }
    }


    bool simulate=(alive!=0);
    
  // if we predict the object, stop simulation if there is a eWall ahead
    // as long as possible.
#ifndef DEDICATED
    if (sr_predictObjects)
#endif
      if (!forceTime) {
	REAL ts=currentTime-lastTime;
	if (ts>0){
	  gSensor fr(this,pos,dirDrive);
	  fr.detect(speed*1.5*ts);
	  REAL wishTime=currentTime;
	  if (fr.hit<speed*1.3*ts){ // problem: we are hitting a eWall
	    wishTime=lastTime;
	    if (wishTime<currentTime-Lag()*3)
	      wishTime=currentTime-Lag()*3;
	    else
	      simulate=false; // no not simulate right now
	  }
	  currentTime=wishTime;
	}
      }

    if (pendingTurns)
      simulate=true;
    
    // do no simulation if currentTime is smaller than the last
    // client action
    
    if (lastClientsideAction>currentTime)
      simulate=false;
    

    if (simulate)
      return TimestepCore(currentTime);
    else
      return !alive;
  }
  
  int timeout=10;
  while (!pendingTurns && currentDestination && timeout > 0){
    timeout --;

    // the distance to the next destination
    REAL dist_to_dest = eCoord::F(currentDestination->position - pos, dirDrive);

    REAL ts=currentTime - lastTime;

    // our speed
    REAL avgspeed=speed;
    if (acceleration > 0)
      avgspeed += acceleration * ts * .5;
    
    if (dist_to_dest > ts * avgspeed)
      break; // no need to worry; we won't reach the next destination

    if (dist_to_dest < 0.01){
      // the destination is very close. Now is the time to turn towards
      // it or turn to the direction it gives us

      REAL t = currentDestination->direction * dirDrive;
      
      currentDestination->hasBeenUsed = true;

      bool missed=(fabs(t)<.5);
      if (int(braking) != int(currentDestination->braking))
	missed=!missed;
      
      if (!missed){ // then we did not miss a destination
	if (fabs(t)>.5)
	  Turn(-t);
	else{
	  braking = currentDestination->braking;
	  RequestSync();
	}

	/*
	con << "turning alon " << currentDestination->position << "," 
	    << currentDestination->direction << "," 
	    << currentDestination->distance << "\n";
	*/
	do
	  currentDestination = currentDestination->next;
	while (currentDestination && currentDestination->hasBeenUsed);
      }
      else{ // OK, we missed a turn. Don't panic. Just turn
	// towards the destination:
	REAL side = (currentDestination->position - pos) * dirDrive;
	if (fabs(side)>1)
	    Turn(-side);
	/*
	con << "turning to   " << currentDestination->position << "," 
	    << currentDestination->direction << "," 
	    << currentDestination->distance << "\n";
	*/
      }
    }
    // ok, the dest is right ahead, but not close enough.
    // how long does it take at least
    // (therefore the strange average speed) to get there?
    TimestepCore(lastTime + dist_to_dest/avgspeed);
  }
  
  // do the rest of the timestep
  return TimestepCore(currentTime);
}




bool gCycle::TimestepCore(REAL currentTime){
  if (!finite(skew))
    skew=0;
  if (!finite(skewDot))
    skewDot=0;
  
  eCoord oldpos=pos;
  
  
  REAL ts=(currentTime-lastTime);

  clamp(ts, -10, 10);
  clamp(correctTimeSmooth, -100, 100);
  clamp(correctDistSmooth, -100, 100);
  clamp(correctSpeedSmooth, -100, 100);
  clamp(speed, -1000, 1000);
  
  if (ts > 0){
    REAL scts = correctTimeSmooth * ts;
    if (scts > ts)
      scts = ts;
    correctTimeSmooth -= scts;
    ts -= scts;

    REAL scs = correctSpeedSmooth * ts;
    speed += scs;
    correctSpeedSmooth -= scs;
    
    REAL scd = correctDistSmooth * ts;
    distance += scd;
    correctDistSmooth -= scd;
  }


  REAL step=speed*ts+.5*acceleration*ts*ts;
  
  
  // be a little nice and don't drive into the eWall
  if (ePlayer){
    gSensor fr(this,pos+dirDrive,dirDrive);
    fr.detect(step*1.5+rubber);
    if (fr.hit<step*1.2+rubber){
      REAL rubberneeded=1.2*step-fr.hit;
      if (rubberneeded < 0)
	rubberneeded=0;
      
      REAL rubber_granted=sg_rubberCycle+Lag()*sg_rubberCyclePing;
      
      if (rubberneeded+rubber>rubber_granted)
	rubberneeded=rubber_granted-rubber;
      
      rubber+=rubberneeded;
      step-=rubberneeded;
      
      if (step<0){
	rubber+=step;
	step=0;
      }
    }

    if (ts > 0)
      rubber/=(1+ts*.1);
  }
  eCoord nextpos;
  if (speed >0)
    nextpos=pos+dirDrive*step;
  else
    nextpos=pos;
  
  if (alive>=1){
    distance+=step;
    Move(nextpos,lastTime,currentTime);
  }
  
  if (alive==-1){
    Move(oldpos,currentTime,currentTime);
    
    if (currentWall){
      currentWall->Update(lastTime,pos);
      currentWall->CopyIntoGrid();
      currentWall=NULL;	
    }
    
    alive=0;
    //    eEdge::SeethroughHasChanged();
  }
  
  
  REAL animts=currentTime-lastTimeAnim;
  if (animts<0 || !finite(animts))
    animts=0;
  else
    lastTimeAnim=currentTime;
  
  if (animts>.2)
    animts=.2;
  
  rotate(rotationFrontWheel,.5*speed*animts/.43);
  rotate(rotationRearWheel,.5*speed*animts/.73);
  
  //const REAL extension=.5;
  const REAL extension=1;
  
  // sr_laggometer
  if (eNetGameObject::Timestep(currentTime))
    return true;
  
  
  eCoord correct=correctPosSmooth*animts*3;
  correctPosSmooth=correctPosSmooth-correct;
  pos=pos+correct;
  
  dir=dir+dirDrive*.2*animts*speed*3;
  dir=dir*(1/sqrt(dir.Norm_squared()));
  
  
  if (alive){
    if (currentWall)
      currentWall->Update(currentTime,pos);
    
    gSensor fl(this,pos,dirDrive.Turn(1,1));
    gSensor fr(this,pos,dirDrive.Turn(1,-1));
    
    fl.detect(extension);
    fr.detect(extension);
    REAL lr=(fl.hit-fr.hit)/extension;
    
    // ODE for skew
    const REAL skewrelax=8;
    skewDot-=128*(skew+lr/2)*animts;
    skewDot/=1+skewrelax*animts;
    if (skew*skewDot>4) skewDot=4/skew;
    skew+=skewDot*animts;
    
    eCoord sparkpos,sparkdir;
    if (fl.hit<extension){
      sparkpos=pos+dirDrive.Turn(1,1)*fl.hit;
      sparkdir=dirDrive.Turn(0,-1);
    }
    if (fr.hit<extension){
      sparkpos=pos+dirDrive.Turn(1,-1)*fr.hit;
      sparkdir=dirDrive.Turn(0,1);
    }
    
    /*
      if (crash_sparks && animts>0)
      new gSpark(pos,dirDrive,currentTime);
    */
    
    if (fabs(skew)<fabs(lr*.8) ){
      skewDot-=lr*1000*animts;
      if (crash_sparks && animts>0)
	new gSpark(grid, sparkpos-dirDrive*.1,sparkdir,currentTime);
    }
    
    if (fabs(skew)<fabs(lr*.9) ){
      skewDot-=lr*100*animts;
      if (crash_sparks && animts>0)
	new gSpark(grid, sparkpos,sparkdir,currentTime);
    }
    /*
      if (fl.hit+fr.hit<extension*.4)
      Kill();
    */
  }

  if (skew>.5){
    skew=.5;
    skewDot=0;
  }

  if (skew<-.5){
    skew=-.5;
    skewDot=0;
  }

  // sense near eWalls and calculate acceleration
  
  acceleration=0;
  
  if(braking)
    acceleration-=sg_brakeCycle;
  
  if (speed<sg_speedCycle*2) 
    acceleration+=(sg_speedCycle*2-speed)*5;
  else
    acceleration+=(sg_speedCycle*2-speed)/10;
  
  for(int d=1;d>=-1;d-=2){
    gSensor rear(this,pos,dirDrive.Turn(-1,d));
    rear.detect(sg_nearCycle);
    if (rear.type!=gSENSOR_RIM)
      acceleration+=sg_accelerationCycle*20*((1/(rear.hit+sg_accelerationCycleOffs))
				     -(1/(sg_nearCycle+sg_accelerationCycleOffs)));
  }    
  
  if (speed >0){
    speed+=acceleration*ts;
    if (speed<sg_speedCycle*.5){
      speed=sg_speedCycle*.5;
      acceleration=0;
    }
  }

  lastTime=currentTime;
  static bool recursion=0;
  if (pendingTurns && lastTime>nextTurn){
    if (!recursion){
      recursion=1;
      Timestep(nextTurn);
      recursion=0;
    }
    //con << "Executing delayed turn at time " << lastTime << "\n";
    if (pendingTurns>0){
      Turn(1);
      pendingTurns--;
    }
    if (pendingTurns<0){
      Turn(-1);
      pendingTurns++;
    }
  }
  // if (last_turnTime+4 < currentTime) Turn(0);
  
  if (nextSync< currentTime && sn_GetNetState()==nSERVER){
    RequestSync(false);
    nextSync=currentTime+1;
  }  
  
  /*
    if (sn_GetNetState()==nSERVER && owner!=0 && 
    planned_rate_control[owner]>200)
    RequestSync(owner,false);
  */
  
  return (!alive);
}


void gCycle::InteractWith(eGameObject *target,REAL,int){
  /*
   if (alive && target->type()==ArmageTron_CYCLE){
      gCycle *c=(gCycle *)target;
      if (c->alive){
	 const eEdge *current_eEdge=Edge();
	 const eEdge *other_eEdge=c->Edge();
	 if(current_eEdge && other_eEdge){
	    ePoint *meet=current_eEdge->IntersectWith(other_eEdge);
	    if (meet){ // somebody is going to die!
		 REAL ratio=current_eEdge->Ratio(*meet);
	       REAL time=CurrentWall()->Time(ratio);
	       PassEdge(other_eEdge,time,other_eEdge->Ratio(*meet),0);
	    }
	 }
      }
   }
  */
}
   

void gCycle::Hunt(gCycle *hunter, gCycle *prey){
  if (prey->alive<1 || sn_GetNetState()==nCLIENT)
    return;

  if (hunter==prey){
    if (hunter->ePlayer)
      hunter->ePlayer->AddScore(score_suicide,
				"because of stupidity.");
  }
  else{
    if (hunter->ePlayer)
      hunter->ePlayer->AddScore
	(score_kill,"for core dumping another program.");
    if (prey->ePlayer)
      prey->ePlayer->AddScore
	(score_die,"since it caused a general protection fault.");
  }
  prey->Kill();
}

bool gCycle::EdgeIsDangerous(const eWall* ww, REAL time, REAL a){
  if (!ww)
    return false;

  const gPlayerWall *w = dynamic_cast<const gPlayerWall*>(ww);
  if (w)
    {
      if(currentWall && 
	 (w == currentWall->Wall() || (lastWall && lastWall->Wall() == w)))
	return false;  	     // this is our current eWall. Ignore it.

      gCycle *otherPlayer=w->Cylce();
      if (otherPlayer==this && time < w->Time(a)+2.5*sg_delayCycle) 
	return false;        // impossible to make such a small circle

      if (otherPlayer->alive < 1 && otherPlayer->deathTime+.2<=time)
	return false;        // the other guy is already dead
    }
  
  return true; // it is a real eWall.
}

void gCycle::PassEdge(const eWall *ww,REAL time,REAL a,int){
  if (!EdgeIsDangerous(ww,time,a))
    return;

  const gPlayerWall *w = dynamic_cast<const gPlayerWall*>(ww);

  if (w)
    {
      gCycle *otherPlayer=w->Cylce();
      
      if(time < w->Time(a)-EPS) // we were first!
	Hunt(this,otherPlayer);
      else if (time > w->Time(a)+EPS) // sad but true
	if (otherPlayer->alive>=1 || otherPlayer->deathTime+.2>time)
	  Hunt(otherPlayer,this);
    }
  else
    {
      if (bool(ePlayer) && sn_GetNetState()!=nCLIENT && alive>=1)
	ePlayer->AddScore(score_suicide,
			  "for trying to escape from the game grid.");
      Kill();
    }
}

bool gCycle::Act(uActionPlayer *Act,REAL x){
  if (!alive && sn_GetNetState()==nSERVER)
    RequestSync(false);

  if(se_turnLeft==*Act && x>.5){
    //SendControl(lastTime,&se_turnLeft,1);
    Turn(-1);
    return true;
  }
  else if(se_turnRight==*Act && x>.5){
    //SendControl(lastTime,&se_turnRight,1);
    Turn(1);
    return true;
  }
  else if(s_brake==*Act){
    //SendControl(lastTime,&brake,x);
    braking=(x>0);
    AddDestination();
    return true;
  }
  return false;
}

 void gCycle::Turn(REAL d){
   if (d>0)
     Turn(1);
   else if (d<0)
     Turn(-1);
 }

void gCycle::Turn(int d){
  if (alive){
    turning->Reset();

    clientside_action();

    //if (sn_GetNetState()==nCLIENT)
    //turns++;
    /*
    cerr << "Turning " << d << " at time " << lastTime 
	 << ", pos " << pos << ", speed " << speed << ".\n";
    */


    if (lastTime>=nextTurn-.001){
      //if (sn_GetNetState()!=nCLIENT)
      /*
      if (sn_GetNetState()==nCLIENT && owner==::sn_myNetID){
	if(d<0)
	  SendControl(lastTime,&se_turnLeft,1);
	else if(d>0)
	  SendControl(lastTime,&se_turnRight,1);
      }
      turns++;
      */
      /*
      else{
	if (d<0)
	  SendControl(lastTime,&se_turnLeft,1);
	if (d>0)
	  SendControl(lastTime,&se_turnRight,1);
      }
      */

      speed*=.95;
      skewDot+=4*d;
      if (d){
	REAL swap=dirDrive.x;
	dirDrive.x=d*dirDrive.y;
	dirDrive.y=-d*swap;
      }

      if (sn_GetNetState() == nCLIENT && owner == ::sn_myNetID)
	AddDestination();

      /*
      eSensor kil(this,pos,dirDrive);
      kil.detect(speed/10.0);
      if (kil.hit<speed/15.0)
	Kill();
      */

      //con << "self-assigning new eWall..\n";

      if (sn_GetNetState()!=nCLIENT){
	RequestSync();
      }
      
      if(currentWall){
	lastWall=currentWall;
	currentWall->Update(lastTime,pos);
	currentWall->CopyIntoGrid();
	currentWall=NULL;
      }
      currentWall=new gNetPlayerWall
	(this,pos,dirDrive,lastTime,distance);
      if (sn_GetNetState()!=nCLIENT)
	currentWallID=currentWall->ID();
#ifndef DEDICATED
      nextTurn=lastTime+sg_delayCycle;
#else
      nextTurn=lastTime+sg_delayCycle*.7;
#endif
    }
    else
      pendingTurns+=d;
  }
}

void gCycle::Kill(){
   if (sn_GetNetState()!=nCLIENT){
     RequestSync(true);
      if (alive>=1){
	 deathTime=lastTime;
	 alive=-1;
	 new gExplosion(grid, pos,lastTime,r,g,b);
	 //	 eEdge::SeethroughHasChanged();
      }
   }
   else if (owner == ::sn_myNetID)
     AddDestination();
   /*
   else if (owner!=::sn_myNetID)
     speed=-.01;
   */
}
   
   
void gCycle::Turbo(bool turbo){
  if (turbo && speed<30){
    speed=40;
    Turn(0);
  }

  if (!turbo && speed>=30){
    speed=20;
    Turn(0);
  }
}

static rTexture cycle_shad(rTEX_FLOOR,"textures/shadow.png",0,0,true);

#ifndef DEDICATED
void gCycle::Render(const eCamera *cam){
  if (!finite(z) || !finite(pos.x) ||!finite(pos.y)||!finite(dir.x)||!finite(dir.y)
      || !finite(skew))
    st_Breakpoint();
  if (Alive()){
     //con << "Drawing cycle at " << pos << '\n';

#ifdef DEBUG
     /*     {
       gDestination *l = destinationList;
       glDisable(GL_LIGHTING);
       glColor3f(1,1,1);
       while(l){
	 if (l == currentDestination)
	   glColor3f(0,1,0);

	 glBegin(GL_LINES);
	 glVertex3f(l->position.x, l->position.y, 0);
	 glVertex3f(l->position.x, l->position.y, 100);
	 glEnd();
	 
	 if (l == currentDestination)
	   glColor3f(0,1,1);
	 
	 l=l->next;
       }
       } */
#endif

     GLfloat color[4]={1,1,1,1};
     static GLfloat lposa[4] = { 320, 240, 200,0};
     static GLfloat lposb[4] = { -240, -100, 200,0};
     static GLfloat lighta[4] = { 1, .7, .7, 1 };   
     static GLfloat lightb[4] = { .7, .7, 1, 1 };   
     
     glMaterialfv(GL_FRONT_AND_BACK,GL_SPECULAR,color);
     glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,color);
     
     glLightfv(GL_LIGHT0, GL_DIFFUSE, lighta);
     glLightfv(GL_LIGHT0, GL_SPECULAR, lighta);
     glLightfv(GL_LIGHT0, GL_POSITION, lposa);
     glLightfv(GL_LIGHT1, GL_DIFFUSE, lightb);
     glLightfv(GL_LIGHT1, GL_SPECULAR, lightb);
     glLightfv(GL_LIGHT1, GL_POSITION, lposb);
     
     
     ModelMatrix();
     glPushMatrix();
     eCoord p = PredictPosition();
     glTranslatef(p.x,p.y,0);
     glScalef(2,2,2);


     eCoord ske(1,skew);
     ske=ske*(1/sqrt(ske.Norm_squared()));

     GLfloat m[4][4]={{dir.x,dir.y,0,0},
		      {-dir.y,dir.x,0,0},
		      {0,0,1,0},
		      {0,0,0,1}};
     glMultMatrixf(&m[0][0]);

     glPushMatrix();
     //glTranslatef(-1.84,0,0);     
     if (!mp)
       glTranslatef(-1.5,0,0);

     glPushMatrix();
     
     GLfloat sk[4][4]={{1,0,0,0},
		       {0,ske.x,ske.y,0},
		       {0,-ske.y,ske.x,0},
		       {0,0,0,1}};
     
     glMultMatrixf(&sk[0][0]);

   
     glEnable(GL_LIGHT0);
     glEnable(GL_LIGHT1);
     glEnable(GL_LIGHTING);



     TexMatrix();
     IdentityMatrix();

     if (mp){

       ModelMatrix();
       glPushMatrix();
       /*
       GLfloat sk[4][4]={{0,.1,0,0},
			 {-.1,0,0,0},
			 {0,0,.1,0},
			 {1,.2,-1.05,1}};

       if (moviepack_hack)
	 glMultMatrixf(&sk[0][0]);
       */

       customTexture->Select();
       //glDisable(GL_TEXTURE_2D);
       //glDisable(GL_TEXTURE);
       glColor3f(1,1,1);

       //glPolygonMode(GL_FRONT, GL_FILL);
       //glDepthFunc(GL_LESS);
       //glCullFace(GL_BACK);
       customModel->Render();
       //glLineWidth(2);
       //glPolygonMode(GL_BACK,GL_LINE);
       //glDepthFunc(GL_LEQUAL);
       //glCullFace(GL_FRONT);
       //customModel->Render();
       //sr_ResetRenderState();

       glPopMatrix();
       glPopMatrix();
       glTranslatef(-1.5,0,0);
     }
     else{
       /*
       glTexGeni(GL_S,GL_TEXTURE_GEN_MODE,GL_OBJECT_LINEAR);
       glEnable(GL_TEXTURE_GEN_S);
       
       glTexGeni(GL_T,GL_TEXTURE_GEN_MODE,GL_OBJECT_LINEAR);
       glEnable(GL_TEXTURE_GEN_T);
       
       glTexGeni(GL_R,GL_TEXTURE_GEN_MODE,GL_OBJECT_LINEAR);
       glEnable(GL_TEXTURE_GEN_R);
       
       glTexGeni(GL_Q,GL_TEXTURE_GEN_MODE,GL_OBJECT_LINEAR);
       glEnable(GL_TEXTURE_GEN_Q);
       */
       
       glEnable(GL_TEXTURE_2D);

       /*       
       static    GLfloat tswap[4][4]={{1,0,0,0},
				      {0,0,1,0},
				      {0,-1,0,0},
				      {.5,.5,0,1}};
       
       static    GLfloat tswapb[4][4]={{1,0,0,0},
				       {0,0,1,0},
				       {0,-1,0,0},
				       {.2,1.2,0,1}};
       */
       
       //       TexMatrix();
       //       glLoadMatrixf(&tswapb[0][0]);
       //       glScalef(.4,.4,.8);
       ModelMatrix();
       
       
       bodyTex->Select();
       body->Render();
       
       wheelTex->Select();
       
       glPushMatrix();
       glTranslatef(0,0,.73);
       
       GLfloat mr[4][4]={{rotationRearWheel.x,0,rotationRearWheel.y,0},
			 {0,1,0,0},
			 {-rotationRearWheel.y,0,rotationRearWheel.x,0},
			 {0,0,0,1}};
       
       
       glMultMatrixf(&mr[0][0]);
       
       
       //       TexMatrix();
       //       glLoadMatrixf(&tswap[0][0]);
       //       glScalef(.65,.65,.65);
       //       ModelMatrix();
       
       rear->Render();
       glPopMatrix();
       
       glPushMatrix();
       glTranslatef(1.84,0,.43);
       
       GLfloat mf[4][4]={{rotationFrontWheel.x,0,rotationFrontWheel.y,0},
			 {0,1,0,0},
			 {-rotationFrontWheel.y,0,rotationFrontWheel.x,0},
			 {0,0,0,1}};
       
       glMultMatrixf(&mf[0][0]);
       
       
       //       TexMatrix();
       //       glLoadMatrixf(&tswap[0][0]);
       //       glScalef(1.2,1.2,1.2);
       //       ModelMatrix();
       
       front->Render();
       glPopMatrix();
       glPopMatrix();
     

     }
       

     //     TexMatrix();
     //     IdentityMatrix();
     ModelMatrix();

     /*
     glDisable(GL_TEXTURE_GEN_S);
     glDisable(GL_TEXTURE_GEN_T);
     glDisable(GL_TEXTURE_GEN_Q);
     glDisable(GL_TEXTURE_GEN_R);
     */

     glDisable(GL_LIGHT0);
     glDisable(GL_LIGHT1);
     glDisable(GL_LIGHTING);
       
     //glDisable(GL_TEXTURE);
     glDisable(GL_TEXTURE_2D);
     glColor3f(1,1,1);

     if (bool(ePlayer) && ePlayer->chatting){

       GLfloat s=sin(lastTime);
       GLfloat c=cos(lastTime);

       GLfloat m[4][4]={{c,s,0,0},
			{-s,c,0,0},
			{0,0,1,0},
			{0,0,1,1}};

       glPushMatrix();

       glMultMatrixf(&m[0][0]);
       glScalef(.5,.5,.5);


       BeginTriangles();

       glColor3f(1,1,0);
       glVertex3f(0,0,3);
       glVertex3f(0,1,4.5);
       glVertex3f(0,-1,4.5);

       glColor3f(.7,.7,0);
       glVertex3f(0,0,3);
       glVertex3f(1,0,4.5);
       glVertex3f(-1,0,4.5);
       
       RenderEnd();

       glPopMatrix();
     }


     // shadow:

     sr_DepthOffset(true);
       

     REAL h=0;//se_cameraZ*.005+.03;

     glEnable(GL_CULL_FACE);

     if(sr_floorDetail>rFLOOR_GRID && TextureMode[rTEX_FLOOR]>0 && sr_alphaBlend){
       glColor3f(0,0,0);
       cycle_shad.Select();
       BeginQuads();
       glTexCoord2f(0,1);
       glVertex3f(-.6,.4,h);
       
       glTexCoord2f(1,1);
       glVertex3f(-.6,-.4,h);

       glTexCoord2f(1,0);
       glVertex3f(2.1,-.4,h);
       
       glTexCoord2f(0,0);
       glVertex3f(2.1,.4,h);
       
       RenderEnd();
     }

     glDisable(GL_CULL_FACE);

     // sr_laggometer;


     REAL f=speed;

     REAL l=Lag();

     glPopMatrix();

     h=cam->CameraZ()*.005+.03;

     if (sn_GetNetState() != nSTANDALONE && sr_laggometer && f*l>.5){
       //&& owner!=::sn_myNetID){
       glPushMatrix();
       
       glColor3f(1,1,1);
       //glDisable(GL_TEXTURE);
       glDisable(GL_TEXTURE_2D);
       
       glTranslatef(0,0,h);
       //glScalef(.5*f,.5*f,.5*f);
       glScalef(f,f,f);

       // move the sr_laggometer ahead a bit
       if (!sr_predictObjects || sn_GetNetState()==nSERVER)
	 glTranslatef(l,0,0);

       
       BeginLineLoop();
       
       
       glVertex2f(-l,-l);
       glVertex2f(0,0);
       glVertex2f(-l,l);
       if(l>.2){
	 glVertex2f(-2*l+.1,.1);
	 glVertex2f(-2*l+.2,0);
	 glVertex2f(-2*l+.1,-.1);
       }
       else if (l>.1){
	 glVertex2f(-2*l+.1,.1);
	 glVertex2f(-l,.2-l);
	 glVertex2f(-l,-(.2-l));
	 glVertex2f(-2*l+.1,-.1);
       }
       
       RenderEnd();
       glPopMatrix();
     }
     sr_DepthOffset(false);

     glPopMatrix();

   }
}


bool gCycle::RenderCockpitFixedBefore(bool){
     /*
  if (alive)
    return true;
  else{
    REAL rd=se_GameTime()-deathTime;
    if (rd<1)
      return true;
    else{
      REAL d=1.25-rd;
      d*=8;
      if (d<0) d=0;
      glColor3f(d,d/2,d/4);
      glDisable(GL_TEXTURE_2D);
      glDisable(GL_TEXTURE);
      glDisable(GL_DEPTH_TEST);
      glRectf(-1,-1,1,1);
      glColor4f(1,1,1,rd*(2-rd/2));
      DisplayText(0,0,.05,.15,"You have been deleted.");
      return false;
    }
  }
  */
  return true;
}

void gCycle::SoundMix(Uint8 *dest,unsigned int len,
		     int viewer,REAL rvol,REAL lvol){
  if (Alive()){
    /*
    if (!cycle_run.alt){
      rvol*=4;
      lvol*=4;
    }
    */
    
	if (engine)
      engine->Mix(dest,len,viewer,rvol,lvol,speed/(sg_speedCycleSound*2));
    
    if (turning)
      if (turn_wav.alt)
        turning->Mix(dest,len,viewer,rvol,lvol,5);
      else
        turning->Mix(dest,len,viewer,rvol,lvol,1);
  }
}
#endif

eCoord gCycle::PredictPosition(){
  eCoord p = pos + dir * (speed * se_PredictTime());
  gSensor s(this,pos, p-pos);
  s.detect(1);
  return s.before_hit;
} 

eCoord gCycle::CamPos(){
  return PredictPosition() + dir.Turn(0,-skew*z);
}

eCoord  gCycle::CamTop(){
  return dir.Turn(0,-skew);
}


#ifdef POWERPAK_DEB
void gCycle::PPDisplay(){
  int R=int(r*255);
  int G=int(g*255);
  int B=int(b*255);
  PD_PutPixel(DoubleBuffer,
	      se_X_ToScreen(pos.x),
	      se_Y_ToScreen(pos.y),
	      PD_CreateColor(DoubleBuffer,R,G,B));
  /*
  PD_PutPixel(DoubleBuffer,
	      se_X_ToScreen(pos.x+1),
	      se_Y_ToScreen(pos.y),
	      PD_CreateColor(DoubleBuffer,R,G,B));
  PD_PutPixel(DoubleBuffer,
	      se_X_ToScreen(pos.x-1),
	      se_Y_ToScreen(pos.y),
	      PD_CreateColor(DoubleBuffer,R,G,B));
  PD_PutPixel(DoubleBuffer,
	      se_X_ToScreen(pos.x),
	      se_Y_ToScreen(pos.y+1),
	      PD_CreateColor(DoubleBuffer,R,G,B));
  PD_PutPixel(DoubleBuffer,
	      se_X_ToScreen(pos.x),
	      se_Y_ToScreen(pos.y-1),
	      PD_CreateColor(DoubleBuffer,R,G,B));
  */
}
#endif





// cycle network routines:
gCycle::gCycle(nMessage &m)
  :eNetGameObject(m),
   engine(NULL),
   turning(NULL),
   destinationList(NULL),currentDestination(NULL),
   dirDrive(1,0),
   skew(0),skewDot(0),speed(5),acceleration(0),
   rotationFrontWheel(1,0),rotationRearWheel(1,0),heightFrontWheel(0),heightRearWheel(0),
   alive(1),
   r(1),g(.2),b(.2),
   currentWall(NULL),
   lastWall(NULL),
   currentWallID(0)
{
  rubber=0;
  pendingTurns=0;
  turns=0;
  speed=0;
  correctTimeSmooth =0;
  correctDistSmooth =0;
  correctSpeedSmooth =0;
  distance = 0;
  deathTime = 0;

  m >> r;
  m >> g;
  m >> b;
}

void gCycle::WriteCreate(nMessage &m){
  eNetGameObject::WriteCreate(m);
  m << r;
  m << g;
  m << b;
}

void gCycle::WriteSync(nMessage &m){
  eNetGameObject::WriteSync(m);
  m << speed;
  m << alive;
  m << distance;
  if (!currentWall || currentWall->preliminary)
    m.Write(0);
  else
    m.Write(currentWall->ID());

  m.Write(turns);
  m.Write(braking);
}

bool gCycle::SyncIsNew(nMessage &m){
  bool ret=eNetGameObject::SyncIsNew(m);

  REAL dummy2;
  short al;
  unsigned short Turns;
  unsigned short b;

  m >> dummy2;
  m >> al;
  m >> dummy2;
  m.Read(Turns);

  if (!m.End())
    m.Read(Turns);
  else
    Turns=turns;

  if (!m.End())
    m.Read(b);

  // con << turns << ":" << Turns << ":" << al << "\n";

  //Update(turns,Turns);

  if (turns>Turns){

    if (al==1 && 
	lastAttemptedSyncTime<lastClientsideAction+1.2*sn_Connections[0].ping+.1){
      #ifdef DEBUG
      //con << "rejecting sync message.\n";
      #endif
      return false;
    }
#ifdef DEBUG
    else   if (ret){
      con << "accepting sync message in spite of obvious latency:"
	  << lastAttemptedSyncTime << ":" << lastClientsideAction << "\n";
    }
#endif
  }  
  
  if (ret || al!=1){
    // con << "accepting..\n";
    return true;
  }
  else
    return false;

  return ret || al!=1;
}

void gCycle::ReadSync(nMessage &m){
  REAL syncTime,sync_dist,sync_speed;
  short sync_alive;
  unsigned short sync_turns=turns,sync_braking=braking,sync_eWall;
  eCoord sync_dir,sync_pos;

  // warning: depends on the implementation of eNetGameObject::WriteSync.
  m >> syncTime;
  m >> sync_dir;
  m >> sync_pos;

  m >> sync_speed;
  m >> sync_alive;
  m >> sync_dist;
  m.Read(sync_eWall);
  if (!m.End())
    m.Read(sync_turns);
  if (!m.End())
    m.Read(sync_braking);

  gDestination emergency_aft(*this);

#ifdef DEBUG
    if (!finite(sync_speed) || !finite(sync_dist))
      st_Breakpoint();
#endif
  
  // all the data was read. check where it fits in our destination list:
  gDestination *bef=gDestination::RightBefore(destinationList, sync_dist);
  gDestination *aft=NULL;
  if (bef){
    aft=bef->next;
    if (!aft)
      aft=&emergency_aft;
  }

  REAL better_distance = sync_dist;

  bool distanceBased = aft && owner==::sn_myNetID;
  
  if (aft){
    better_distance = aft->distance - sqrt((sync_pos-pos).Norm_squared());

    /*
    if (fabs(better_distance - sync_dist) > 100){
      // con << "Wanring: bad sync.\n";
      distanceBased = false;
    }
    */
  }
  
  if (distanceBased){
    // new way: correct our time and speed

#ifdef DEBUG
    // destination *save = bef;
#endif

    REAL ratio = (better_distance - bef->distance)/
      (aft->distance - bef->distance);

    if (!finite(ratio))
      ratio = 0;

    // TODO: better interpolation
    REAL interpolatedTime = bef->gameTime * (1-ratio) + 
      aft->gameTime * ratio; 
    REAL interpolated_speed = bef->speed * (1-ratio) + aft->speed * ratio; 


    REAL correctTime  = (syncTime  - interpolatedTime);
    REAL correct_speed = (sync_speed - interpolated_speed);
    REAL correct_dist  = (sync_dist - better_distance);

    clamp(correctTime , -10, 10);
    clamp(correct_speed, -10, 10);
    clamp(correct_dist , -10, 10);

    if (correctTime > 0){
      correctTime*=.5;
      correct_speed*=.5;
      correct_dist*=.5;
    }

    correctTimeSmooth  += correctTime;
    correctSpeedSmooth += correct_speed;
    correctDistSmooth  += correct_dist;

#ifdef DEBUG
    if (!finite(correctTime) || !finite(correct_speed))
      st_Breakpoint();
#endif
    
    while (bef){
      bef->gameTime += correctTime;
      bef->speed    += correct_speed;
      //bef->distance += correct_dist;
      bef = bef->next;
    }

    if (fabs(correctTimeSmooth > 5))
      st_Breakpoint();


#ifdef DEBUG
    if (!finite(speed) || !finite(correctTimeSmooth) || fabs(correctTimeSmooth)> 100)
      st_Breakpoint();
#endif

    clamp(correctTimeSmooth,-100,100);
  }
  
  else
    {
      // the sync is newer than our last turn. correct our position.
      correctTimeSmooth = 0;
      
      REAL t=(dirDrive*sync_dir);
      if (fabs(t)>.2){
#ifdef DEBUG
	//	if (owner==sn_myNetID)
	//con << "Warning! Turned cycle!\n";
#endif
	if (turning)
	  turning->Reset();

	eCoord crossPos = sync_pos;

	if(currentWall){
	// make sure the position lies on the last path
	  if (owner != ::sn_myNetID){
	    crossPos = crossPos - dirDrive * eCoord::F(dirDrive,crossPos-sync_pos);
	    crossPos = crossPos - sync_dir * eCoord::F(sync_dir,crossPos-pos);
	  }


	  lastWall=currentWall;
	  currentWall->Update(syncTime,crossPos);
	  if (owner == ::sn_myNetID)
	    currentWall->CopyIntoGrid();
	  else
	    delete currentWall;
	  currentWall=NULL;
	}
	distance = sync_dist;
	currentWall=new gNetPlayerWall
	  (this,crossPos,sync_dir,syncTime,sync_dist);
	if (sn_GetNetState()!=nCLIENT)
	  currentWallID=currentWall->ID();
	 
      }
      
      // side bending
      skewDot+=4*t;
      
      REAL ts = syncTime - lastTime;

      eCoord intPos = pos + dirDrive * (ts * speed + .5 * ts*ts*acceleration);
      REAL  int_speed = speed + acceleration*ts;

      dirDrive = sync_dir;
      
      if (owner!=::sn_myNetID){
	pos = sync_pos;
	speed = sync_speed;
	REAL oldTime = lastTime;
	lastTime = syncTime;
	clientside_action();
	forceTime=true;
	TimestepThis(oldTime, this);
	forceTime=false;
      }
      else if (fabs(t)>.1 || (intPos-sync_pos).Norm_squared() > 10){
	pos = sync_pos + pos - intPos;
#ifdef DEBUG
	if (!finite(int_speed))
	  st_Breakpoint();
#endif
	speed = sync_speed + speed - int_speed;
      }
      else{
	correctPosSmooth = sync_pos - intPos;
	correctSpeedSmooth = sync_speed - int_speed;
      }

      braking = sync_braking;
      distance = sync_dist - speed * ts - acceleration * ts*ts*.5;

      correctTimeSmooth = 0;
    }
  
  if (alive && sync_alive!=1){
    new gExplosion(grid, sync_pos,lastTime,r,g,b);
    deathTime=lastTime;
    //    eEdge::SeethroughHasChanged();
    pos = sync_pos;
  }
  alive=sync_alive;
  
  sn_Update(turns,sync_turns);

  if (fabs(correctTimeSmooth > 5))
    st_Breakpoint();

}

/*
void gCycle::old_ReadSync(nMessage &m){
  REAL oldTime=lastTime;
  eCoord oldpos=pos;
  //+correctPosSmooth;
  //correctPosSmooth=0;
  eCoord olddir=dir;
  eNetGameObject::ReadSync(m);

  REAL t=(dirDrive*dir);
  if (fabs(t)>.2){
#ifdef DEBUG
    if (owner==sn_myNetID)
      con << "Warning! Turned cycle!\n";
#endif
    turning.Reset();
  }

  // side bending
  skewDot+=4*t;


  dirDrive=dir;
  dir=olddir;

  m >> speed;
  short new_alive;
  m >> new_alive;
  if (alive && new_alive!=1){
    new gExplosion(pos,oldTime,r,g,b);
    deathTime=oldTime;
    eEdge::SeethroughHasChanged();
  }
  alive=new_alive;

  m >> distance;

  // go to old time frame

  eCoord realpos=pos;
  REAL realtime=lastTime;
  REAL realdist=distance;

  if (currentWall)
    lastWall=currentWall;

  m.Read(currentWallID);

  unsigned short Turns;
  if (!m.End())
    m.Read(Turns);
  else
    Turns=turns;

  if (!m.End())
    m.Read(braking);
  else
    braking=false;

  TimestepThis(oldTime,this);

  if (!currentWall || fabs(t)>.3 || currentWall->inGrid){
    //REAL d=eCoord::F(pos-realpos,olddir);
    //eCoord crosspos=realpos+olddir*d;
    //d=eCoord::F(oldpos-crosspos,dir);
    //crosspos=crosspos+dir*d;

    eCoord crosspos=realpos;

    if (currentWall){
      currentWall->Update(realtime,crosspos);
      //currentWall->Update(realtime,realpos);
      currentWall->CopyIntoGrid();
    }
    //con << "NEW\n";
    currentWall=new gNetPlayerWall
      (this,crosspos,dirDrive,realtime,realdist);
  }


  // smooth correction
  if ((oldpos-pos).Norm_squared()<4 && fabs(t)<.5){
    correctPosSmooth=pos-oldpos;
    pos=oldpos;
  }
  


#ifdef DEBUG
  //int old_t=turns;
  //if(sn_Update(turns,Turns))
  //con << "Updated turns form " << old_t << " to " << turns << "\n";
#endif
  sn_Update(turns,Turns);
}
*/

void gCycle::ReceiveControl(REAL time,uActionPlayer *act,REAL x){
  forceTime=true;
  TimestepThis(time,this);
  Act(act,x);
  forceTime=false;
}

nDescriptor &gCycle::CreatorDescriptor() const{
  return cycle_init;
}

void gCycle::AddDestination(gDestination *dest){
  //  con << "got new dest " << dest->position << "," << dest->direction 
  // << "," << dest->distance << "\n";

  dest->InsertIntoList(&destinationList);

  // if the new destination was inserted at the end of the list
  //if (!currentDestination && !dest->next && 
  //if ((dest->distance >= distance || (!currentDestination && !dest->next)) && 

  if (dest->next && dest->next->hasBeenUsed){
    delete dest;
    return;
  }
    

  if ((!currentDestination || currentDestination->distance > dest->distance) && 
      sn_GetNetState()!=nSTANDALONE && owner!=::sn_myNetID){
    currentDestination=dest;
    // con << "setting new cd\n";
  }

  //  if (sn_GetNetState()==nSERVER || ::sn_myNetID == owner)
  if (sn_GetNetState()==nCLIENT && ::sn_myNetID == owner)
    BroadCastNewDestination(this,dest);
}



// ************************************************************




// ************************************************************

