/* Copyright (C) 2002, 2003, 2004 Jan Wedekind.
   This file is part of the recipe database application AnyMeal.

   AnyMeal is free software; you can redistribute it and/or modify it under
   the terms of the GNU GENERAL PUBLIC LICENSE as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   AnyMeal is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTIBILITY or FITNESS
   FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
   details.

   You should have received a copy of the GNU General Public License along
   with AnyMeal; if not, contact one of the authors of this software. */
/****************************************************************************
 ** ui.h extension file, included from the uic-generated form implementation.
 **
 ** If you wish to add, delete or rename functions or slots use
 ** Qt Designer which will update this file, preserving your code. Create an
 ** init() function in place of a constructor, and a destroy() function in
 ** place of a destructor.
 *****************************************************************************/

#ifndef NDEBUG
#include <iostream>
#endif

void RecipeEditor::init()
{
#ifndef NDEBUG
  std::cerr << "Initialising recipe-editor." << std::endl;
#endif
  QDoubleValidator *amountValidator = new QDoubleValidator( amountEdit );
  amountValidator->setBottom( 0.0 );
  amountEdit->setValidator( amountValidator );
  unitComboBox->insertStringList( createUnitStringList() );
  ingredientsView->setSorting( -1 );
}

void RecipeEditor::setRecipe( const std::string &recipe )
{
  // Title.
  XMLDocument xmlDocument( "" ); xmlDocument.fromString( recipe );
  titleEdit->setText( xmlDocument.getDocumentElement().
                      selectNode( "recipe/title" ).getNodeText() );

  // Categories.
  XMLNodeReferenceList categories
    ( xmlDocument.getDocumentElement().
      selectNodes( "recipe/categories/category" ) );
  for ( int i=0; i<categories.getLength(); i++ )
    categoriesListBox->insertItem
      ( dynamic_pointer_cast< XalanElement >( categories.item( i ) ).
        getNodeText() );

  // Servings: amount, unit
  servingsSpinBox->setValue
    ( atoi( xmlDocument.getDocumentElement().
            selectNode( "recipe/servings/amount" ).getNodeText().c_str() ) );
  servingsEdit->setText
    ( xmlDocument.getDocumentElement().selectNode( "recipe/servings/unit" ).
      getNodeText() );
  
  // Ingredients.
  XMLNodeReferenceList ingredientSections
    ( xmlDocument.getDocumentElement().
      selectNodes( "recipe/ingredients/section" ) );

  for ( int i=ingredientSections.getLength()-1;i>=0; i-- ) {
    XMLReference< XalanElement > section
      ( dynamic_pointer_cast< XalanElement >( ingredientSections.item( i ) ) );
    std::string title;
    if ( section.existNode( "title" ) )
      title = section.selectNode( "title" ).getNodeText();
#ifndef NDEBUG
    std::cerr << "editor: " << i << "th ingredient-section is \"" << title
              << "\"." << std::endl;
#endif
    QListViewItem *sectionItem =
      new QListViewItem( ingredientsView, title.c_str() );
    XMLNodeReferenceList ingredients = section.selectNodes( "ingredient" );

    for ( int j=ingredients.getLength()-1; j>=0; j-- ) {

      XMLReference< XalanElement > ingredient
        ( dynamic_pointer_cast< XalanElement >( ingredients.item( j ) ) );

      std::string amount1, amount2, amount3;
      if ( ingredient.existNode( "amount/float" ) )
        amount1 = ingredient.selectNode( "amount/float" ).getNodeText();
      if ( ingredient.existNode( "amount/fraction/nominator" ) )
        amount2 = ingredient.selectNode( "amount/fraction/nominator" ).
          getNodeText();
      if ( ingredient.existNode( "amount/fraction/denominator" ) )
        amount3 = ingredient.selectNode( "amount/fraction/denominator" ).
          getNodeText();

      std::string unit;
      if ( ingredient.existNode( "unit" ) )
        unit = ingredient.selectNode( "unit" ).getNodeText();

      std::string name = ingredient.selectNode( "name" ).getNodeText();

      std::string prep;
      if ( ingredient.existNode( "prep" ) )
        prep = ingredient.selectNode( "prep" ).getNodeText();

      new IngredientViewItem( sectionItem, amount1, amount2, amount3,
                              unit, name, prep );

    };

  };
  ingredientsView->setCurrentItem( NULL );
  
  // Instructions
  XMLNodeReferenceList instructionsSections
    ( xmlDocument.getDocumentElement().selectNodes
      ( "recipe/instructions/section" ) );
  for ( int i=0; i<instructionsSections.getLength(); i++ ) {
    XMLReference< XalanElement > section
      ( dynamic_pointer_cast< XalanElement >
        ( instructionsSections.item( i ) ) );
    std::string title;
    if ( section.existNode( "title" ) )
      title = section.selectNode( "title" ).getNodeText();
    instructionsGroupListBox->insertItem( title );
    QTextEdit *textEdit = new QTextEdit;
    textEdit->setTextFormat( Qt::PlainText );

    XMLNodeReferenceList paragraphs( section.selectNodes( "par" ) );
    for ( int j=0; j<paragraphs.getLength(); j++ ) {
#ifndef NDEBUG
      std::cerr << j << ": " << paragraphs.item( j ).getNodeText()
                << std::endl;
#endif
      textEdit->append( paragraphs.item( j ).getNodeText() );
    };

    instructionsWidgetStack->addWidget( textEdit );
    editors.push_back( textEdit );
  };

  if ( !editors.empty() )
      instructionsWidgetStack->raiseWidget( editors[ 0 ] );

  updateOkEnabled();
  categorySelected( categoriesListBox->currentItem() );
  ingredientSelected( ingredientsView->currentItem() );
  instructionSelected( instructionsGroupListBox->currentItem() );
}

void RecipeEditor::deleteCategory(void)
{
  int currentItem = categoriesListBox->currentItem();
  assert( currentItem >= 0 );
  assert( currentItem < categoriesListBox->numRows() );
  categoriesListBox->removeItem( currentItem );
  if ( currentItem >= categoriesListBox->numRows() )
    currentItem--;
  categoriesListBox->setCurrentItem( currentItem );
  categorySelected( currentItem );
  updateOkEnabled();
}


void RecipeEditor::categorySelected( int item )
{
  categoryRemoveButton->setEnabled( item >= 0 );
  if ( item >= 0 ) {
    categoryRemoveButton->setEnabled( true );
    categoryUpButton->setEnabled( item > 0 );
    categoryDownButton->setEnabled( item < categoriesListBox->numRows() - 1 );
    categoryEdit->setEnabled( true );
    if ( categoriesListBox->text( item ) != categoryEdit->text() )
      categoryEdit->setText( categoriesListBox->text( item ) );
  } else {
    categoryRemoveButton->setEnabled( false );
    categoryUpButton->setEnabled( false );
    categoryDownButton->setEnabled( false );
    categoryEdit->setEnabled( false );
    // categoryEdit->clear();
  };
}


void RecipeEditor::categoryText( const QString &text )
{
  if ( categoriesListBox->currentItem() >= 0 ) {
    // Prevent recursion.
    if ( !text.isEmpty() &&
         categoriesListBox->text( categoriesListBox->currentItem() ) !=
         text ) {
      int cursorPos = categoryEdit->cursorPosition();
      categoriesListBox->changeItem( text, categoriesListBox->currentItem() );
      categoryEdit->setFocus();
      categoryEdit->setCursorPosition( cursorPos );
      updateOkEnabled();
    };
  };
}


void RecipeEditor::addCategory(void)
{
  categoriesListBox->insertItem( i18n( "Category" ) );
  categoriesListBox->setCurrentItem( categoriesListBox->count() - 1 );
  updateOkEnabled();
}


void RecipeEditor::categoryUp()
{
  int currentItem = categoriesListBox->currentItem();
  assert( currentItem >= 1 );
  assert( currentItem < categoriesListBox->numRows() );
  QString text = categoriesListBox->text( currentItem - 1 );
  categoriesListBox->changeItem( categoriesListBox->text( currentItem ),
                                 currentItem - 1 );
  categoriesListBox->changeItem( text, currentItem );
  categoriesListBox->setCurrentItem( currentItem - 1 );
}


void RecipeEditor::categoryDown()
{
  int currentItem = categoriesListBox->currentItem();
  assert( currentItem >= 0 );
  assert( currentItem < categoriesListBox->numRows() - 1 );
  QString text = categoriesListBox->text( currentItem + 1 );
  categoriesListBox->changeItem( categoriesListBox->text( currentItem ),
                                 currentItem + 1 );
  categoriesListBox->changeItem( text, currentItem );
  categoriesListBox->setCurrentItem( currentItem + 1 );
}


void RecipeEditor::ingredientSelected( QListViewItem *item )
{
  if ( item != NULL ) {
    bool isGroup = item->parent() == NULL;
    if ( isGroup ) {

      // Activate group editor.
      ingredientsGroupEdit->setText( item->text( 0 ) );
      ingredientsGroupEdit->setEnabled( true );
      ingredientsUpButton->setEnabled( item != ingredientsView->firstChild() );
      ingredientsDownButton->setEnabled( item->nextSibling() != NULL );

      // Deactive ingredient widgets.
      amountTypeCombo->setEnabled( false );
      amountEdit->setEnabled( false );
      naturalNumSpinBox->setEnabled( false );
      nominatorSpinBox->setEnabled( false );
      denominatorSpinBox->setEnabled( false );
      unitComboBox->setEnabled( false );
      nameEdit->setEnabled( false );
      prepEdit->setEnabled( false );

    } else {

      IngredientViewItem *ingredientItem = (IngredientViewItem *)item;

      // Deactivate group editor.
      ingredientsGroupEdit->setEnabled( false );
      ingredientsUpButton->setEnabled( false );
      ingredientsDownButton->setEnabled( false );

      // Activate ingredient widgets.
      amountTypeCombo->setEnabled( true );
      amountEdit->setEnabled( true );
      naturalNumSpinBox->setEnabled( true );
      nominatorSpinBox->setEnabled( true );
      denominatorSpinBox->setEnabled( true );
      unitComboBox->setEnabled( true );
      nameEdit->setEnabled( true );
      prepEdit->setEnabled( true );
      
      if ( !ingredientItem->isFraction() ) {
        amountTypeCombo->setCurrentItem( 0 );
        amountWidgetStack->raiseWidget( 0 );
        amountEdit->setText( QString( "%1" ).
                             arg( ingredientItem->getAmountDouble() ) );
      } else {
        amountTypeCombo->setCurrentItem( 1 );
        amountWidgetStack->raiseWidget( 1 );
        // Read out joined values, before setting the controls.
        int
          number = ingredientItem->getAmountNumber(),
          nominator = ingredientItem->getAmountNominator(),
          denominator = ingredientItem->getAmountDenominator();
        naturalNumSpinBox->setValue( number );
        nominatorSpinBox->setValue( nominator );
        denominatorSpinBox->setValue( denominator );
      };

      unitComboBox->setCurrentText( item->text( 1 ) );
      nameEdit->setText( item->text( 2 ) );
      prepEdit->setText( item->text( 3 ) );
      ingredientsUpButton->setEnabled
        ( item != item->parent()->firstChild() ||
          item->parent() != ingredientsView->firstChild() );
      ingredientsDownButton->setEnabled
        ( item->nextSibling() != NULL ||
          item->parent()->nextSibling() != NULL );

    };

    addIngredientButton->setEnabled( true );
    removeIngredientButton->setEnabled( true );

  } else {

    // Deactivate group editor.
    ingredientsGroupEdit->setEnabled( false );
    ingredientsUpButton->setEnabled( false );
    ingredientsDownButton->setEnabled( false );

    // Deactivate ingredient widgets.
    amountTypeCombo->setEnabled( false );
    amountEdit->setEnabled( false );
    naturalNumSpinBox->setEnabled( false );
    nominatorSpinBox->setEnabled( false );
    denominatorSpinBox->setEnabled( false );
    unitComboBox->setEnabled( false );
    nameEdit->setEnabled( false );
    prepEdit->setEnabled( false );

    addIngredientButton->setEnabled( false );
    removeIngredientButton->setEnabled( false );

  };
}


void RecipeEditor::addIngredientsGroup(void)
{
  QListViewItem *currentGroup = ingredientsView->currentItem();
  if ( currentGroup != NULL ) {
    if ( currentGroup->parent() != NULL )
      currentGroup = currentGroup->parent();
  };

  QListViewItem *newItem =
    new QListViewItem( ingredientsView,
                       currentGroup != NULL ? currentGroup :
                       ingredientsView->lastItem(),
                       "" );
  ingredientsView->setCurrentItem( newItem );

  // Make sure, item is visible.
  ingredientsView->ensureItemVisible( newItem );

  ingredientSelected( newItem );
}


void RecipeEditor::ingredientsGroupText( const QString &text )
{
  QListViewItem *currentItem = ingredientsView->currentItem();
  if ( currentItem != NULL ) currentItem->setText( 0, text );
}


void RecipeEditor::deleteIngredient()
{
  QListViewItem *currentItem = ingredientsView->currentItem();
  assert( currentItem != NULL );
  delete currentItem;
  ingredientsView->setCurrentItem( NULL );
}


void RecipeEditor::amountTextChanged( const QString &text )
{
  IngredientViewItem *item =
    (IngredientViewItem *)ingredientsView->currentItem();

  if ( item != NULL ) {
    if ( item->parent() != NULL ) item->setAmount( atof( text ) );
  };
}


void RecipeEditor::unitTextChanged( const QString &text )
{
  QListViewItem *item = ingredientsView->currentItem();

  if ( item != NULL ) {
    if ( item->parent() != NULL ) item->setText( 1, text );
  };
}


void RecipeEditor::nameTextChanged( const QString &text )
{
  QListViewItem *item = ingredientsView->currentItem();

  if ( item != NULL && !text.isEmpty() ) {
    if ( item->parent() != NULL ) item->setText( 2, text );
  };
}


void RecipeEditor::preparationTextChanged( const QString &text )
{
  QListViewItem *item = ingredientsView->currentItem();

  if ( item != NULL ) {
    if ( item->parent() != NULL ) item->setText( 3, text );
  };
}


void RecipeEditor::addIngredient()
{
  assert( ingredientsView->currentItem() != NULL );

  QListViewItem
    *current = ingredientsView->currentItem(),
    *parent,
    *after;

  if ( current->parent() != NULL ) {
    parent = current->parent();
    after = current;
  } else {
    parent = current;
    after = NULL;
  };

  ingredientsView->setOpen( parent, true );

  IngredientViewItem *newItem =
    new IngredientViewItem( parent, after, "", "", "", "", "Ingredient", "" );

  // Make sure, item is visible.
  ingredientsView->ensureItemVisible( newItem );

  ingredientsView->setCurrentItem( newItem );
}


void RecipeEditor::instructionSelected( int index )
{
  if ( index >= 0 && index < (signed)editors.size() ) {

    instructionsWidgetStack->raiseWidget( editors[ index ] );
    instructionsGroupUpButton->setEnabled( index > 0 );
    instructionsGroupDownButton->setEnabled
      ( index < instructionsGroupListBox->numRows() - 1 );

    if ( instructionsGroupEdit->text() !=
         instructionsGroupListBox->text( index ) )
      instructionsGroupEdit->setText
        ( instructionsGroupListBox->text( index ) );
    instructionsGroupEdit->setEnabled( true );
    instructionsGroupRemoveButton->setEnabled( true );

  } else {

    instructionsWidgetStack->raiseWidget( 0 );
    instructionsGroupUpButton->setEnabled( false );
    instructionsGroupDownButton->setEnabled( false );

    instructionsGroupEdit->setEnabled( false );
    instructionsGroupRemoveButton->setEnabled( false );

  };
}


void RecipeEditor::addInstructions()
{
  int currentItem = instructionsGroupListBox->currentItem();
  instructionsGroupListBox->insertItem( "",
                                        currentItem >= 0 ?
                                        currentItem + 1 : -1 );
  QTextEdit *textEdit = new QTextEdit;
  textEdit->setTextFormat( Qt::PlainText );
  instructionsWidgetStack->addWidget( textEdit );
  editors.insert( currentItem >= 0 ?
                  editors.begin() + currentItem + 1 : editors.end(),
                  textEdit );
  instructionsGroupListBox->setCurrentItem
    ( currentItem >= 0 ?
      currentItem + 1 : instructionsGroupListBox->numRows() - 1 );
  instructionSelected( instructionsGroupListBox->currentItem() );
}


void RecipeEditor::deleteInstructions()
{
  int currentItem = instructionsGroupListBox->currentItem();
  assert( currentItem >= 0 );
  assert( currentItem < (signed)editors.size() );
  QWidget *widget = editors[ currentItem ];
  editors.erase( editors.begin() + currentItem );
  instructionsWidgetStack->removeWidget( widget );
  delete widget;
  instructionsGroupListBox->removeItem( currentItem );
  instructionSelected( instructionsGroupListBox->currentItem() );
}


void RecipeEditor::instructionsUp()
{
  int currentItem = instructionsGroupListBox->currentItem();
  assert( currentItem >= 1 );
  assert( currentItem < instructionsGroupListBox->numRows() );
  QString text = instructionsGroupListBox->text( currentItem - 1 );
  QTextEdit *widget = editors[ currentItem - 1 ];
  instructionsGroupListBox->changeItem
    ( instructionsGroupListBox->text( currentItem ), currentItem - 1 );
  editors[ currentItem - 1 ] = editors[ currentItem ];
  instructionsGroupListBox->changeItem( text, currentItem );
  editors[ currentItem ] = widget;
  instructionsGroupListBox->setCurrentItem( currentItem - 1 );
}


void RecipeEditor::instructionsDown()
{
  int currentItem = instructionsGroupListBox->currentItem();
  assert( currentItem >= 0 );
  assert( currentItem < instructionsGroupListBox->numRows() - 1 );
  QString text = instructionsGroupListBox->text( currentItem + 1 );
  QTextEdit *widget = editors[ currentItem + 1 ];
  instructionsGroupListBox->changeItem
    ( instructionsGroupListBox->text( currentItem ), currentItem + 1 );
  editors[ currentItem + 1 ] = editors[ currentItem ];
  instructionsGroupListBox->changeItem( text, currentItem );
  editors[ currentItem ] = widget;
  instructionsGroupListBox->setCurrentItem( currentItem + 1 );
}


void RecipeEditor::instructionsGroupText( const QString &text )
{
  int currentItem = instructionsGroupListBox->currentItem();
  if ( currentItem >= 0 ) {
    // Prevent recursion.
    if ( instructionsGroupListBox->text( currentItem ) !=
         text ) {
      int cursorPos = instructionsGroupEdit->cursorPosition();
      categoriesListBox->changeItem( text, categoriesListBox->currentItem() );
      instructionsGroupListBox->changeItem( text, currentItem );
      instructionsGroupEdit->setFocus();
      instructionsGroupEdit->setCursorPosition( cursorPos );
    };
  };
}


std::string RecipeEditor::getRecipe( void )
{
  std::ostringstream stream;
  // Title
  assert( !titleEdit->text().isEmpty() );
  stream << "  <recipe xml:lang='"
         << anyMealLanguage() << "'>" << std::endl
         << "    <title>" << xmlText( titleEdit->text() ) << "</title>"
         << std::endl << "    <categories>" << std::endl;

  // Categories
  assert( categoriesListBox->numRows() > 0 );
  for ( int i=0; i<categoriesListBox->numRows(); i++ )
    stream << "      <category>" << xmlText( categoriesListBox->text( i ) )
           << "</category>" << std::endl;

  // Servings
  stream << "    </categories>" << std::endl << "    <servings>" << std::endl
         << "      <amount>" << servingsSpinBox->value() << "</amount>"
         << std::endl << "      <unit>" << xmlText( servingsEdit->text() )
         << "</unit>" << std::endl << "    </servings>" << std::endl
         << "    <ingredients>" << std::endl;

  // Ingredients
  QListViewItem *sectionItem = ingredientsView->firstChild();

  while ( sectionItem != NULL ) {

    stream << "      <section>" << std::endl;

    if ( !sectionItem->text( 0 ).isEmpty() )
      stream << "        <title>" << xmlText( sectionItem->text( 0 ) )
             << "</title>" << std::endl;

    IngredientViewItem *ingredientItem =
      (IngredientViewItem *)sectionItem->firstChild();

    while ( ingredientItem != NULL ) {

      stream << "        <ingredient>" << std::endl;

      if ( !ingredientItem->isNull() ) {
        stream << "        <amount>" << std::endl;

        if ( ingredientItem->isFraction() ) {

          int
            denominator = ingredientItem->getAmountDenominator(),
            nominator = ingredientItem->getAmountNumber() * denominator +
                        ingredientItem->getAmountNominator();
          
          stream << "          <fraction>" << std::endl
                 << "            <nominator>"
                 << nominator << "</nominator>" << std::endl
                 << "            <denominator>"
                 << denominator << "</denominator>" << std::endl
                 << "          </fraction>" << std::endl;

        } else

          stream << "          <float>"
                 << ingredientItem->getAmountDouble() << "</float>"
                 << std::endl;

        stream << "        </amount>" << std::endl;
      };

      if ( !ingredientItem->text( 1 ).isEmpty() )
        stream << "        <unit>" << ingredientItem->text( 1 )
               << "</unit>" << std::endl;

      stream << "        <name>" << ingredientItem->text( 2 )
             << "</name>" << std::endl;

      if ( !ingredientItem->text( 3 ).isEmpty() )
        stream << "        <prep>" << ingredientItem->text( 3 )
               << "</prep>" << std::endl;

      stream << "        </ingredient>" << std::endl;

      ingredientItem = (IngredientViewItem *)ingredientItem->nextSibling();

    };

    stream << "      </section>" << std::endl;
    sectionItem = sectionItem->nextSibling();

  };

  // Instructions
  stream << "    </ingredients>" << std::endl
         << "    <instructions>" << std::endl;
  for ( int i=0; i<instructionsGroupListBox->numRows(); i++ ) {
    stream << "      <section>" << std::endl;
    if ( !instructionsGroupListBox->text( i ).isEmpty() )
      stream << "        <title>"
             << xmlText( instructionsGroupListBox->text( i ) )
             << "</title>" << std::endl;
    stream << generateParagraphs( editors[ i ]->text() ) << std::endl
           << "      </section>" << std::endl;
  };
  stream << "    </instructions>" << std::endl;
  
  stream << "  </recipe>" << std::endl;
  return stream.str();
}

std::string RecipeEditor::generateParagraphs( const std::string &text )
{
  return
    std::string( "        <par>" ) +
    replaceAll( xmlText( text ), "\n", "]]></par>\n        <par><![CDATA[" ) +
    "</par>";
};


void RecipeEditor::updateOkEnabled()
{
  bool enableOk;
  if ( !titleEdit->text().isEmpty() && categoriesListBox->numRows() > 0 ) {

    // Check for duplicate categories.
    QStringList categories;
    for ( int i=0; i<categoriesListBox->numRows(); i++ )
      categories.append( categoriesListBox->text( i ) );
    categories.sort();
    enableOk = true;
    for ( int i=0; i<categoriesListBox->numRows() - 1; i++ )
      if ( categories[ i ] == categories[ i + 1 ] ) {
        enableOk = false;
        break;
      };

  } else

    enableOk = false;

  okButton->setEnabled( enableOk );
}


void RecipeEditor::ingredientUp()
{
  QListViewItem *currentItem = ingredientsView->currentItem();
  assert( currentItem != NULL );

  bool isGroup = currentItem->parent() == NULL;

  if ( isGroup ) {

    assert( currentItem != ingredientsView->firstChild() );
    // Get previous group.
    QListViewItem *previousGroup = currentItem->itemAbove();
    assert( previousGroup != NULL );
    if ( previousGroup->parent() != NULL )
      previousGroup = previousGroup->parent();

    // Move previous group below current group.
    previousGroup->moveItem( currentItem );

  } else {

    QListViewItem *previousItem = currentItem->itemAbove();

    if ( previousItem->parent() != NULL ) {

      assert( previousItem->parent() == currentItem->parent() );
      // Move previous item below current item.
      previousItem->moveItem( currentItem );

    } else {

      // Top of group has been reached.
      // Item needs to be shifted to previous group.
      currentItem->parent()->takeItem( currentItem );
      QListViewItem *previousGroup = previousItem->itemAbove();
      assert( previousGroup != NULL );
      if ( previousGroup->parent() != NULL )
        previousGroup = previousGroup->parent();

      // Open previous group.
      ingredientsView->setOpen( previousGroup, true );

      // Get last item of previous group (or previous group itself if empty).
      QListViewItem *previousItem2 = previousItem->itemAbove();

      // Insert item into previous group.
      previousGroup->insertItem( currentItem );

      // Move it to the last position if necessary.
      if ( previousItem2->parent() != NULL )
        currentItem->moveItem( previousItem2 );

      // Select item.
      ingredientsView->setCurrentItem( currentItem );

    };

  };

  // Make sure, item is visible.
  ingredientsView->ensureItemVisible( currentItem );

  ingredientSelected( ingredientsView->currentItem() );
}


void RecipeEditor::ingredientDown()
{
  QListViewItem *currentItem = ingredientsView->currentItem();
  assert( currentItem != NULL );

  bool isGroup = currentItem->parent() == NULL;

  if ( isGroup ) {

    assert( currentItem->nextSibling() != NULL  );

    // Move group below next group.
    currentItem->moveItem( currentItem->nextSibling() );

  } else {

    QListViewItem *nextItem = currentItem->itemBelow();

    if ( nextItem->parent() != NULL ) {

      assert( nextItem->parent() == currentItem->parent() );
      // Move item below next item.
      currentItem->moveItem( nextItem );

    } else {

      // Bottom of group has been reached.
      // Item needs to be shifted to next group.
      QListViewItem *nextGroup = currentItem->parent()->nextSibling();
      assert( nextGroup != NULL );
      currentItem->parent()->takeItem( currentItem );

      // Insert item into next group.
      nextGroup->insertItem( currentItem );

      // Select item.
      ingredientsView->setCurrentItem( currentItem );

    };

  };

  // Make sure, item is visible.
  ingredientsView->ensureItemVisible( currentItem );

  ingredientSelected( ingredientsView->currentItem() );
}


void RecipeEditor::setAmountType( int _type )
{
  QListViewItem *currentItem = ingredientsView->currentItem();
  assert( currentItem != NULL );
  assert( currentItem->parent() != NULL );

  IngredientViewItem *ingredientItem = (IngredientViewItem *)currentItem;
  ingredientItem->setFraction( _type == 1 );
  ingredientSelected( ingredientItem );
}


void RecipeEditor::amountNumberChanged( int )
{
  amountFractionChanged();
}


void RecipeEditor::amountNominatorChanged( int )
{
  amountFractionChanged();
}


void RecipeEditor::amountDenominatorChanged( int )
{
  amountFractionChanged();
}


void RecipeEditor::amountFractionChanged()
{
  QListViewItem *currentItem = ingredientsView->currentItem();
  assert( currentItem != NULL );
  assert( currentItem->parent() != NULL );

  IngredientViewItem *ingredientItem = (IngredientViewItem *)currentItem;
  ingredientItem->setAmount
    ( naturalNumSpinBox->value() * denominatorSpinBox->value()+
      nominatorSpinBox->value(),
      denominatorSpinBox->value() );
}
