// ---------------------------------------------------------------------------
// - Key.hpp                                                                 -
// - afnix cryptographic library - key class definition                      -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_KEY_HPP
#define  AFNIX_KEY_HPP

#ifndef  AFNIX_STRING_HPP
#include "String.hpp"
#endif

namespace afnix {

  /// The Key class is an original class used to store a particular key
  /// or to generate one. A key can be set at the class construction or
  /// be generated from a pass-phrase with the help of hasher function.
  /// @author amaury darsch

  class Key : public Object {
  public:
    enum t_type{
      K128, // 128 bits key - 16 bytes
      K192, // 192 bits key - 24 bytes
      K256  // 256 bits key - 32 bytes
    };

  private:
    /// the key type
    t_type  d_type;
    /// the key buffer
    t_byte* p_kbuf;

  public:
    /// create a 128 bits random key
    Key (void);

    /// create a 128 bits message key
    /// @param msg the key message
    Key (const String& msg);

    /// create a new random key by type
    /// @param type the key type
    Key (const t_type type);

    /// create a new key by type and data
    /// @param type the cipher key type
    /// @param data the key data
    Key (const t_type type, const t_byte* data);

    /// create a new key by type and string
    /// @param type the cipher key type
    /// @param msg  the key message
    Key (const t_type type, const String& msg);

    /// copy construct this key
    /// @param that the key key to copy
    Key (const Key& that);

    /// destroy this key
    ~Key (void);

    /// assign a key to this one
    /// @param that the key to assign
    Key& operator = (const Key& that);

    /// @return the class name
    String repr (void) const;

    /// @return the key type
    t_type gettype (void) const;

    /// @return  the key size in bytes
    long getsize (void) const;

    /// @return a key byte by index
    t_byte get (const long index) const;

    /// @return the formatted key result
    String format (void) const;

  public:
    /// evaluate an object data member
    /// @param robj  the current runnable
    /// @param nset  the current nameset
    /// @param quark the quark to evaluate
    static Object* meval (Runnable* robj, Nameset* nset, const long quark);

    /// create a new object in a generic way
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);
    
    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;
 
    /// apply this object with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
                   Vector* argv);
  };
}

#endif
